/****************************************************************************
** clparam_list.h
**
** Michael S. Borella <mike@borella.net>
**
** Class definition for a command line parameter list
**
** $Id: clparam_list.h,v 1.4 2000/10/25 22:52:05 mborella Exp $
**
*****************************************************************************/

#ifndef CLPARAM_LIST_H
#define CLPARAM_LIST_H

#include <list>
#include <algorithm>
#include <string>
#include "logfile.h"
#include "clparam.h"
#include "eh.h"

#define MAX_CLPARAMS 128   // This should be way more than enough

//---------------------------------------------------------------------------
//
// class Clparam_list
//
//---------------------------------------------------------------------------

class Clparam_list
{
private:
  list<Clparam> _params;
  list<string>  _global_callback_list;
  list<string>  _include_list;
  list<string>  _invocation_list;
  list<string>  _mandatory_list;
  string        _parsefunc;
  string        _cppext;
  bool          _quiet;

  // These are fairly general
  string    str2upper(string);
  bool      have_callbacks();

  // These are for both C and C++
  void      output_comments(ofstream&, string, string);
  void      output_comments_function(ofstream&, string, string);
  void      output_includes(ofstream&);
  string    type2str(param_t,string);
  string    str2comment(string);

  // C-only: These are called by the output_c_parser()
  void      output_c_usage(ofstream&);
  void      output_c_freeargs(ofstream&);

  // C-only: These each create one C file
  void      output_c_header(string) throw (EH);
  void      output_c_parser(string) throw (EH);
  void      output_c_callbacks(string) throw (EH);

  // C++-only
  void      output_cpp_comments_class(ofstream&, string, string);
  void      output_cpp_header(string) throw (EH);
  void      output_cpp_class(string) throw (EH);
  void      output_cpp_callbacks(string) throw (EH);

public:
  // Constructor, destructor and copy constructor
  Clparam_list() { _cppext = "cc"; _quiet = false; }
  ~Clparam_list() {}
  Clparam_list& operator=(const Clparam_list &);

  // Add a parameter to the list
  void      add(Clparam p) throw(EH);

  // Add an include file
  void      add_include(string s) { _include_list.push_back(s); }

  // Add a mandatory
  void      add_mandatory(string s) { _mandatory_list.push_back(s); }

  // Add an invocation
  void      add_invocation(string s) { _invocation_list.push_back(s); }

  // Add a global callbacks
  void      add_global_callback(string s) { _global_callback_list.push_back(s); }

  // output C code
  void      output_c(string) throw (EH);

  // output C++ code
  void      output_cpp(string);

  // dump to a logfile
  void      logdump(Logfile&);

  // these tell the caller if a parameter is in the list or not, based on the
  // short representation or internal name
  bool      contains(char);
  bool      contains(string);

  // set the parsing function / class name
  void      set_parsefunc(string s) { _parsefunc = s; }

  // set the c++ extension
  void      set_cppext(string s) { _cppext = s; }

  // set quiet mode
  void      quiet() { _quiet = true; }
};

#endif
