/*  $Id$
    couple.h - Class representing a couple of integers.

    gengameng - Generic 2D Game Engine library
    Copyright (C) 1999 Pierre Sarrazin <sarrazip@iname.com>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/


#ifndef __cplusplus
#error Must use C++ with Couple.h
#endif


#ifndef H_Couple
#define H_Couple

#include "goodies.h"

#include <math.h>


class Couple {
//: Class representing a couple of integers.
public:
    int x, y;
    //: The components.

    Couple ();
    //: Builds a couple with undefined values for the components.

    Couple (int a, int b);
    //: Initiliazes a couple with x equal to a and y equal to b.

    Couple (const Couple &c);
    //: Copies the components of the couple 'c' into the current couple.

    Couple &operator = (const Couple &c);
    //: Replaces the components of the current couple by those of 'c'.

    ~Couple ();
    //: Does nothing.

    Couple &zero ();
    //: Assign zero to both components.
    /*  Returns a reference to this object.
	Was void until version 0.1.
    */

    bool is_zero () const;
    //: Returns 1 if both components are zero; returns 0 otherwise.
    /*  Returned int until version 0.1.
    */

    double length () const;
    //: Returns the length of the vector represented by the couple.

    Couple &operator += (const Couple &c);
    Couple &operator -= (const Couple &c);
    Couple &operator *= (int n);
    Couple &operator /= (int n);
    friend Couple operator + (const Couple &c1, const Couple &c2);
    friend Couple operator - (const Couple &c1, const Couple &c2);
    friend Couple operator * (const Couple &c1, int n);
    friend Couple operator * (int n, const Couple &c1);
    friend Couple operator / (const Couple &c1, int n);
    friend bool operator == (const Couple &c1, const Couple &c2);
    friend bool operator != (const Couple &c1, const Couple &c2);
    //: Miscellaneous operators.
    /*  Returns the current object as modified.
        The friend functions don't need to be friends since the two
        components are public data members, but this documents the
        functions in a useful place...
        operator << prints the two components between parentheses and
        the string ", " between them.

	bool used since version 0.1.
    */

};


//
//  IMPLEMENTATION (inline functions)
//


inline Couple::Couple ()
{
}


inline Couple::Couple (int a, int b)
  : x(a), y(b)
{
}


inline Couple::Couple (const Couple &c)
  : x(c.x), y(c.y)
{
}


inline Couple &Couple::operator = (const Couple &c)
{
    x = c.x;
    y = c.y;
    return *this;
}


inline Couple::~Couple ()
{
}


#if 0  /* Obsolete @sarrazip 20010203 */
inline Couple Couple::eoap () const
{
    return Couple(x << 3, y << 3);
}


inline Couple Couple::uneoap () const
{
    return Couple((x + 4) >> 3, (y + 4) >> 3);
}
#endif  /* 0 */


inline Couple &Couple::zero ()
{
    x = y = 0;
    return *this;
}


inline bool Couple::is_zero () const
{
    return (x == 0 && y == 0);
}


inline Couple &Couple::operator += (const Couple &c)
{
    x += c.x;
    y += c.y;
    return *this;
}


inline Couple &Couple::operator -= (const Couple &c)
{
    x -= c.x;
    y -= c.y;
    return *this;
}


inline Couple &Couple::operator *= (int n)
{
    x *= n;
    y *= n;
    return *this;
}


inline Couple &Couple::operator /= (int n)
{
    x /= n;
    y /= n;
    return *this;
}


inline Couple operator + (const Couple &c1, const Couple &c2)
{
    Couple c(c1);
    return c += c2;
}


inline Couple operator - (const Couple &c1, const Couple &c2)
{
    Couple c(c1);
    return c -= c2;
}


inline Couple operator * (const Couple &c1, int n)
{
    Couple c(c1);
    return c *= n;
}


inline Couple operator * (int n, const Couple &c1)
{
    Couple c(c1);
    return c *= n;
}


inline Couple operator / (const Couple &c1, int n)
{
    Couple c(c1);
    return c /= n;
}


inline bool operator == (const Couple &c1, const Couple &c2)
{
    return (c1.x == c2.x && c1.y == c2.y);
}


inline bool operator != (const Couple &c1, const Couple &c2)
{
    return !(c1 == c2);
}


inline double Couple::length () const
{
    return sqrt(double(x) * x + double(y) * y);
}


#endif  /* H_Couple */
