/*
   objectstore.h - Public Interface to the objectstore library and
                   therefore the database backend(s)

   Copyright (C) 2001 Free Software Foundation

   This file is part of the GNU Enterprise Application Server (GEAS)

   GEAS is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2, or (at your option) any later
   version.

   GEAS is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with GEAS; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   $Id: objectstore.h,v 1.22 2001/06/13 00:47:47 treshna Exp $
*/

#ifndef _OBJECTSTORE_H
#define _OBJECTSTORE_H

#include <glib.h>
#include "oql/oql.h"
#include "config/configuration.h"

typedef void         *DatabaseResultRow;

/* THESE ARE ONLY VALID AFTER A SUCCESSFUL QUERY */
/* FAILED QUERIES CAN INVALIDATE THESE WITHOUT WARNING OR */
/* ANY INDICATION */
/* get the contents of an arbitrary row in the table */

/* do not rely on the current implementation of these macros */

#define count_fields_in_row(result) \
   ((unsigned long int)(result)->field_count)

#define count_rows_in_result(result) \
   ((unsigned long int)(result)->rows_affected)

#define get_result_row(result,_row) \
   ((DatabaseResultRow)g_list_nth_data(((result)->data),(_row)))

#define get_result_field(result,_row,field) \
     ((const char *)g_list_nth_data( get_result_row((result),(_row)) , field ))

#define get_result_field_name(result,field) \
     ((const char *)oql_query_get_field_name(((result)->query),(field)))

#define get_result_class_name(result) \
     ((const char *)oql_query_get_classname(((result)->query)))

#define get_field_in_row(_row,field) \
     g_list_nth_data( ((GList *)_row) , (field) )

/* probably should be private - do not use this
 * structure, as it may be removed or changed without notice.
 * use the macros defined above
 */
struct query_result
{
  QueryData            *query;          /* what were we trying to do? */
  gboolean              success;        /* did it actually succeed? */
  int                   rows_affected;  /* number of rows affected, usually -
                                           not guaranteed; will be set to the
                                           number of rows returned, if any */
  int                   field_count;    /* and the number of fields per row */

  GList                *data;           /* data found : SELECT query only,
                                           otherwise NULL; NULL also means no
                                           data - check success field */
};

/** \brief Connect and initialize a database connection.
 */
gboolean              initialise_objectstore (configuration config);

/** \brief Close connection to database.
 */
gboolean              close_object_store (void);

/** \brief Free query result.
 */
void                  free_query_result (struct query_result *result);

/** \brief Perform a query.
 * NOTE: pass in NULL for errcode and/or errmsg if you don't wish
 * to receive them. if you do pass values in, you must free the string
 * allocated and stored in *errmsg.
 */
struct query_result  *query_objectstore (QueryData * query, int *err,
                                         char **errmsg);

/** \brief Delete a matching object from the database.
 */
struct query_result  *delete_from_objectstore (const char *classname,
                                               const char *key, int *err,
                                               char **errmsg);
/** \brief Delete all matching objects from a database.
 */
struct query_result  *delete_all_from_objectstore (const char *classname,
                                                   const char *field,
                                                   const char *key, int *err,
                                                   char **errmsg);

/** \brief Write data from object cache to the datastore
 *
 * Once this function returns, the object may be removed instantly,
 *  so should not be accessed again, such as from another thread.
 */
struct query_result  *write_to_objectstore (const gchar *classname,
                                            const gchar *key,
                                            GHashTable *values,
                                            gboolean update,
                                            int *err, char **errmsg);

/** \brief Go through all active databases, and make sure they contain
 * the appropriate tables.
 */
void                  update_objectstore_databases (gboolean remove_items);

/* start another hack... noone wants to waste time thinking about    */
/* the future effects of code - it's much more fun to solev problems */
/* when they happen, rather than avoid causing problems              */
/* btw, they're postgresql only features - will die if you're using mysql */
void start_objectstore_transaction( void );

/* and the rest of it */
void commit_objectstore_transaction( void );

int using_postgresql( void );

#endif /* not _OBJECTSTORE_H */
