/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <ctype.h>

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "file.h"
#include "parse.h"
#include "matrix.h"
#include "numeric.h"
#include "surface.h"
#include "interface.h"
#include "opengl.h"

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/* TODO - XML & add diffraction data to elements file */
/**************************************/
/* parse a file for gdis element data */
/**************************************/
/* type - should the element data be treated as default or patched */
/* ie the standard library, or modified values from a gdisrc file */
gint read_elem_data(FILE *fp, gint type)
{
gint i, n, key, flag, num_tokens; 
gint *set;
gchar **buff, *str, *symbol, line[LINELEN];
gdouble *sfc;
struct elem_pak elem, elem_default;
GSList *list;

/* checks */
g_return_val_if_fail(fp != NULL, 1);

n = num_keys();
set = g_malloc(n * sizeof(gint));

while(!fgetline(fp,line))
  {
  list = get_keywords(line);
  if (list) 
    {
    switch(GPOINTER_TO_INT(list->data))
      {
      case GDIS_ELEM_START:
/* init for element */
         elem.shell_charge = 0.0;
/* init the set flags */
        for (i=n ; i-- ; )
          set[i] = 0;
/* keep looping until end of element data, or EOF */
        flag=0;
        while(!flag)
          {
/* get next line & process for keywords */
          if (fgetline(fp, line))
            break;
          list = get_keywords(line);
          buff = tokenize(line, &num_tokens);

/* process 1st keyword, and subsequent data (if any) */
          if (list)
            {
            key = GPOINTER_TO_INT(list->data);
            switch(key)
              {
/* these shouldn't be modified if not default */
              case SYMBOL:
                g_assert(num_tokens > 1);
                elem.symbol = g_strdup(*(buff+1));
                break; 
              case NAME:
                g_assert(num_tokens > 1);
                elem.name = g_strdup(*(buff+1));
                break; 
              case NUMBER:
                g_assert(num_tokens > 1);
                elem.number = (gint) str_to_float(*(buff+1));
                set[key]++;
                break; 
              case WEIGHT:
                g_assert(num_tokens > 1);
                elem.weight = str_to_float(*(buff+1));
                set[key]++;
                break; 
              case COVALENT:
                g_assert(num_tokens > 1);
                elem.cova = str_to_float(*(buff+1));
                set[key]++;
                break; 
              case VDW:
                g_assert(num_tokens > 1);
                elem.vdw = str_to_float(*(buff+1));
                set[key]++;
                break; 
              case CHARGE:
                g_assert(num_tokens > 1);
                elem.charge = str_to_float(*(buff+1));
                set[key]++;
                break; 
              case COLOUR:
                g_assert(num_tokens > 3);
                elem.colour[0] = (gint) str_to_float(*(buff+1));
                elem.colour[1] = (gint) str_to_float(*(buff+2));
                elem.colour[2] = (gint) str_to_float(*(buff+3));
                set[key]++;
                break; 
              case GDIS_END:
                flag++;
                break; 
              }
            }
          g_strfreev(buff); 
          }
/* store the element data according to type (ie global/model only etc.) */
        switch(type)
          {
          case DEFAULT:
            if (elem.number < 0 || elem.number > MAX_ELEMENTS-1)
              printf("Error: Element number %d outside allocated bounds.\n", elem.number);
            else
              memcpy(&elements[elem.number], &elem, sizeof(struct elem_pak));
            break;
          case MODIFIED:
/* TODO - scan for symbol as well (elem_seek with data = NULL) */
            if (!set[NUMBER])
              {
              printf("Warning: missing element number.\n");
              elem.number = 0;
              }
/* get default values */
            get_elem_data(elem.number, &elem_default, NULL);
/* fill out specified values */
            if (set[WEIGHT])
              elem_default.weight = elem.weight;
            if (set[COVALENT])
              elem_default.cova = elem.cova;
            if (set[VDW])
              elem_default.vdw = elem.vdw;
            if (set[CHARGE])
              elem_default.charge = elem.charge;
            if (set[COLOUR])
              {
              elem_default.colour[0] = elem.colour[0];
              elem_default.colour[1] = elem.colour[1];
              elem_default.colour[2] = elem.colour[2];
              }
            put_elem_data(&elem_default, NULL);
            break;
/* ignore other types */
          default:
            break;
          }
/* ignore all other keywords */
      default:
        break;
      }
    }

/* scattering factor coeffs */
  if (g_ascii_strncasecmp("\%gdis_sfc", line, 9) == 0)
    {
    while ((str = file_read_line(fp)))
      {
      buff = tokenize(str, &num_tokens);

if (num_tokens != 12)
  {
  /*
  printf("suspicious line: %s\n", str);
  */
  g_strfreev(buff);
  g_free(str);
  break;
  }
      symbol = g_strdup(*(buff+0));
      if (elem_symbol_test(symbol))
        {
        list = NULL;
        for (i=1 ; i<num_tokens ; i++)
          {
          sfc = g_malloc(sizeof(gdouble));
          *sfc = str_to_float(*(buff+i));
          list = g_slist_prepend(list, sfc);
          }
        list = g_slist_reverse(list);

        g_hash_table_insert(sysenv.sfc_table, symbol, list);

	g_strfreev(buff);
        }
      g_free(str);
      }
    }
  }
/* done */
g_free(set);
return(0);
}

/*****************************************************/
/* write the global exceptions to a specified stream */
/*****************************************************/
/* NB: should be suitable for appending (hence *fp) to gdisrc file */
gint write_elem_data(FILE *fp)
{
GSList *list;
struct elem_pak *elem;

/* checks */
g_return_val_if_fail(fp != NULL, 1);

list = sysenv.elements;
while(list != NULL)
  {
/* CURRENT - only write those values that are different to default library */
  elem = (struct elem_pak *) list->data;
  if (elem)
    {
    fprintf(fp, "%%gdis_elem\n");
    fprintf(fp, "number: %d\n", elem->number);
    fprintf(fp, "weight: %f\n", elem->weight);
    fprintf(fp, "  cova: %f\n", elem->cova);
    fprintf(fp, "   vdw: %f\n", elem->vdw);
    fprintf(fp, "charge: %f\n", elem->charge);
    fprintf(fp, "colour: %f %f %f\n", elem->colour[0], 
                                      elem->colour[1],
                                      elem->colour[2]);
    fprintf(fp, "%%gdis_end\n");
    }
  list = g_slist_next(list);
  }
return(0);
}

void write_sfc(gpointer key, gpointer val, gpointer data)
{
GSList *list;
FILE *fp = data;

fprintf(fp, "%s", (gchar *) key);
for (list=val ; list ; list=g_slist_next(list))
  fprintf(fp, " %f", *((gdouble *) list->data)); 
fprintf(fp, " \n");
}

void write_sfc_data(FILE *fp)
{
fprintf(fp, "%%gdis_sfc\n");
g_hash_table_foreach(sysenv.sfc_table, &write_sfc, fp);
fprintf(fp, "%%gdis_end\n");
}

/***************************************/
/* find all unique elements in a model */
/***************************************/
#define DEBUG_FIND_UNIQUE 0
GSList *find_unique(gint mode, struct model_pak *model)
{
gint found;
GSList *list, *types, *unique;
struct core_pak *core;

unique = NULL;

/* go through all atoms */
for (list=model->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

/* determine if current atom type is in the list */
  found=0;
  for (types=unique ; types ; types = g_slist_next(types))
    {
    switch(mode)
      {
      case ELEMENT:
        if (GPOINTER_TO_INT(types->data) == core->atom_code)
          found++;
        break;

      case LABEL:
      case LABEL_NORMAL:
      case LABEL_GHOST:
        if (g_ascii_strcasecmp(types->data, core->label) == 0)
          found++;
        break;
      }
    if (found)
      break;
    }

/* not in list, so add it */
  if (!found)
    {
    switch(mode)
      {
      case ELEMENT:
        unique = g_slist_prepend(unique, GINT_TO_POINTER(core->atom_code));
        break;
      case LABEL:
        unique = g_slist_prepend(unique, g_strdup(core->label));
        break;
      case LABEL_NORMAL:
        if (!core->ghost)
          unique = g_slist_prepend(unique, g_strdup(core->label));
        break;
      case LABEL_GHOST:
        if (core->ghost)
          unique = g_slist_prepend(unique, g_strdup(core->label));
        break;
      }
    }
  }

#if DEBUG_FIND_UNIQUE
printf("Found: ");
for (types=unique ; types ; types = g_slist_next(types))
  {
  switch(mode)
    {
    case ELEMENT:
      printf("[%3d] ", GPOINTER_TO_INT(types->data));
      break;
    case LABEL:
    case LABEL_NORMAL:
    case LABEL_GHOST:
      printf("[%s] ", (gchar *) types->data);
      break;
    }
  }
printf("\n");
#endif

return(unique);
}

/***********************/
/* set the atom colour */
/***********************/
void init_atom_colour(struct core_pak *core, struct model_pak *model)
{
struct elem_pak elem_data;

get_elem_data(core->atom_code, &elem_data, model);
ARR3SET(core->colour, elem_data.colour);
if (core->ghost)
  core->colour[3] = 0.5;
else
  core->colour[3] = 1.0;
}

/***********************/
/* set the atom charge */
/***********************/
void init_atom_charge(struct core_pak *core, struct model_pak *model)
{
struct elem_pak elem_data;

if (core->lookup_charge)
  {
  get_elem_data(core->atom_code, &elem_data, model);
  core->charge = elem_data.charge;
  }
}

/************************/
/* set the atom charges */
/************************/
void init_model_charges(struct model_pak *model)
{
GSList *list;

for (list=model->cores ; list ; list=g_slist_next(list))
  init_atom_charge(list->data, model);
}

/*********************************************************/
/* compute net charge and dipole (if model is a surface) */
/*********************************************************/
#define DEBUG_CALC_EMP 0
void calc_emp(struct model_pak *data)
{
gdouble zdip, qsum, x[3];
GSList *list;
struct core_pak *core;
struct shel_pak *shell;

/* checks */
g_assert(data != NULL);

/* TODO - speed up by getting to a local var the elem data for unique atoms */
/* cores & shell dipole calc */
zdip=qsum=0.0;
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = list->data;
/*
  if (core->region != REGION1A)
    continue;
*/
  if (core->status & DELETED)
    continue;

/* core contribution */
  ARR3SET(x, core->x);
  vecmat(data->latmat, x);

  zdip += core->charge * x[2];
  qsum += core->charge;

/* NB: add shell constribution at the core's z location */
  if (core->shell)
    {
    shell = core->shell;

    ARR3SET(x, shell->x);
    vecmat(data->latmat, x);
    zdip += shell->charge * x[2];
    qsum += shell->charge;
    }
  }

#if DEBUG_CALC_EMP
printf("surface: %f %f %f  (%f)\n", 
data->surface.miller[0], data->surface.miller[1], data->surface.miller[2], 
data->surface.shift);
printf("net dipole: %e\n", zdip);
printf("sum charge: %e\n", qsum);
if (qsum*qsum > FRACTION_TOLERANCE)
  printf("Warning: your model has a net charge = %f\n", qsum);
#endif

/* only surfaces get the z dipole */
if (data->periodic == 2)
  data->gulp.sdipole = zdip;

data->gulp.qsum = qsum;
}

/***************************************************/
/* does a given string represent an atomic number? */
/***************************************************/
gint elem_number_test(const gchar *input)
{
gint n=0;

if (str_is_float(input))
  {
  n = str_to_float(input);
  if (n<0 || n>MAX_ELEMENTS-1)
    n=0; 
  }
return(n);
}

/***************************************************/
/* does a given string represent an element symbol */
/***************************************************/
#define DEBUG_ELEM_TYPE 0
gint elem_symbol_test(const gchar *input)
{
gint i, j, m, n;
gchar *elem1, **buff;

/* checks */
if (input == NULL)
  return(0);
/* zero length string matches something, so force it to return 0 */
if (!strlen(input))
  return(0);

/* duplicate for manipulation */
elem1 = g_strdup(input);

/* remove anything but alphabetic chars */
for (i=0 ; i<strlen(elem1) ; i++)
  if (!g_ascii_isalpha(*(elem1+i)))
    *(elem1+i) = ' ';

/* remove trailing/leading spaces & use only the first token */
buff = tokenize(elem1, &n);
if (n)
  {
  g_free(elem1);
  elem1 = g_strdup(*buff);
  }
g_strfreev(buff);

m = strlen(elem1);

/* catch Deuterium */
if (g_ascii_strncasecmp(elem1, "D", m) == 0)
  *elem1 = 'H';

#if DEBUG_ELEM_TYPE
printf("Looking for [%s]...", elem1);
#endif

/* attempt to match atom type with database */
j=0;
/* FIXME - elim dependence on const */
for(i=0 ; i<sysenv.num_elements ; i++)
  {
/* only compare if lengths match */
  n = strlen(elements[i].symbol);
  if (n == m)
    {
    if (g_ascii_strncasecmp(elem1, elements[i].symbol, m) == 0)
      {
      j = i;
      break;
      }
    }
  }

#if DEBUG_ELEM_TYPE
if (j)
  printf("found.\n");
else
  printf("not found.\n");
#endif

/* done */
g_free(elem1);
return(j);
}

/*************************************************************/
/* does a given string represent an element symbol or number */
/*************************************************************/
gint elem_test(const gchar *input)
{
gint n;

n = elem_symbol_test(input);
if (!n)
  n = elem_number_test(input);

return(n);
}

/*********************************/
/* PDB hack for element matching */
/*********************************/
#define DEBUG_PDB_ELEM_TYPE 0
gint pdb_elem_type(const gchar *input)
{
gint i, j, len;
gchar *tmp;

#if DEBUG_PDB_ELEM_TYPE
printf("pdb1: [%s]\n", input);
#endif

/* checks */
if (input == NULL)
  return(0);
/* zero length string matches something, so force it to return 0 */
len = strlen(input);
if (!len)
  return(0);

/* if we encounter C, O, or H then ignore the rest of the string */
for (i=0 ; i<len ; i++)
  {
  switch(*(input+i))
    {
    case 'c':
    case 'C':
    case 'o':
    case 'O':
    case 'h':
    case 'H':
      tmp = g_strndup(input, i+1);
      j = elem_symbol_test(tmp);
      g_free(tmp);
      return(j);
    }
  }
/* default element matching */
return(elem_symbol_test(input));
}

/*************************/
/* retrieve element data */
/*************************/
/* will return non-default data (if it exists) otherwise the default */
#define DEBUG_GET_ELEM 0
gint get_elem_data(gint code, struct elem_pak *elem, struct model_pak *model)
{
gint i;
struct elem_pak *elem_data;
GSList *list;

/* TODO - seek in sysenv (cope with data == NULL) */

/* search the model exceptions database */
if (model)
  {
#if DEBUG_GET_ELEM
printf("Searching %d model items items for code = %d.\n",model->num_elem,code);
#endif
  for (i=model->num_elem ; i-- ; )
    {
    if (code == (model->elements+i)->number)
      {
      memcpy(elem, model->elements+i, sizeof(struct elem_pak));
      return(0);
      }
    }
#if DEBUG_GET_ELEM
printf("Not found.\n");
#endif
  }

/* search the global exceptions database */
#if DEBUG_GET_ELEM
printf("Searching linked list for code = %d.\n",code);
#endif
list = sysenv.elements;
while(list != NULL)
  {
  elem_data = (struct elem_pak *) list->data;
  if (code == elem_data->number)
    {
    memcpy(elem, elem_data, sizeof(struct elem_pak));
    return(0);
    }
  list = g_slist_next(list);
  }
#if DEBUG_GET_ELEM
printf("Not found.\n");
#endif

/* return from the standard database */
if (code >= 0 && code < sysenv.num_elements)
  {
#if DEBUG_GET_ELEM
printf("Retrieving default.\n");
#endif
  memcpy(elem, &elements[code], sizeof(struct elem_pak));
  return(0);
  }

/* done - no match */
#if DEBUG_GET_ELEM
printf("ERROR: bad element code.\n");
#endif
return(1);
}

/**********************************/
/* store non-default element data */
/**********************************/
#define DEBUG_PUT_ELEM_DATA 0
void put_elem_data(struct elem_pak *elem, struct model_pak *model)
{
gint i, ix=0, flag=0;
GSList *list;
struct elem_pak *elem_data;
struct core_pak *core;

/* checks */
g_return_if_fail(elem != NULL);

#if DEBUG_PUT_ELEM_DATA
printf(" *** Saving element ***\n");
printf("symbol: %s\n", elem->symbol);
printf("  name: %s\n", elem->name);
printf("number: %d\n", elem->number);
printf("weight: %f\n", elem->weight);
printf("  cova: %f\n", elem->cova);
printf("   vdw: %f\n", elem->vdw);
printf("charge: %f\n", elem->charge);
printf("colour: %f %f %f\n", elem->colour[0], elem->colour[1], elem->colour[2]);
#endif

/* FIXME - crude hack for element data being passed in range 0.0 - 1.0 */
/* (this happens in gperiodic - and prob should be default) */
if (VEC3MAGSQ(elem->colour) < 3.1)
  {
  VEC3MUL(elem->colour, 65535.0);
  }

/* if data = NULL (ie no model) -> global -> put in sysenv list */
if (model == NULL)
  {
/* search for existing item */
  list = sysenv.elements;
  while(list != NULL)
    {
    elem_data = (struct elem_pak *) list->data;
    if (elem_data->number == elem->number)
      {
#if DEBUG_PUT_ELEM_DATA
printf("Modifying global database.\n");
#endif
/* if found, overwrite values */
      memcpy(elem_data, elem, sizeof(struct elem_pak));
      flag++;
      break;
      }
    list = g_slist_next(list);
    } 
/* if not found, add element to the list */
  if (!flag)
    {
#if DEBUG_PUT_ELEM_DATA
printf("Adding to global database.\n");
#endif
    elem_data = g_malloc(sizeof(struct elem_pak));
    memcpy(elem_data, elem, sizeof(struct elem_pak));
    sysenv.elements = g_slist_prepend(sysenv.elements, elem_data);
    }
  }
else
  {
#if DEBUG_PUT_ELEM_DATA
printf("Modifying local database.\n");
#endif
/* replace flag (as opposed to add new) */
ix=0;
/* alloc if non-existent */
if (!model->num_elem)
  {
  model->elements = g_malloc(sizeof(struct elem_pak));
  model->num_elem++;
  flag++;
  }
else
  {
/* search for existing data */
  for (i=model->num_elem ; i-- ; )
    {
    if (elem->number == i)
      {
      flag++;
      ix = i;
      break;
      }
    }
  }
/* TODO - a linked list would be easier... */
/* replace or re-alloc & add? */
  if (flag)
    memcpy(model->elements+ix, elem, sizeof(struct elem_pak));
  else
    {
    model->elements = g_renew(struct elem_pak, model->elements, model->num_elem+1);
    memcpy(model->elements+model->num_elem, elem, sizeof(struct elem_pak));
    model->num_elem++;
    }

/* update atom bond_cutoff */
  for (list=model->cores ; list ; list=g_slist_next(list))
    {
    core = (struct core_pak *) list->data;
    if (core->atom_code == elem->number)
      core->bond_cutoff = elem->cova;
    }
  }
}

/******************************************/
/* lookup and restore core default values */
/******************************************/
void init_elem(struct core_pak *core, gint type, struct model_pak *model)
{
struct elem_pak elem;

get_elem_data(core->atom_code, &elem, model);

switch (type)
  {
  case REFRESH_COLOUR:
    ARR3SET(core->colour, elem.colour);
    break;
  case REFRESH_BONDS:
    core->bond_cutoff = elem.cova;
    break;
  }
}

/* globals for the atom properties dialog */
GtkWidget *apd_label, *apd_type, *apd_charge, *apd_x, *apd_y, *apd_z;
struct model_pak *apd_data=NULL;
struct core_pak *apd_core=NULL;

/*************************************************************/
/* get the nth colour in a sequence (eg sample RGB spectrum) */
/*************************************************************/
#define MAX_COLOUR 65535
#define RAD_CYCLE 4
#define RAD_GRANULARITY 2.88
#define THETA_GRANULARITY 2.88
#define DEBUG_GET_COLOUR 0

void get_colour(gdouble *colour, gint n)
{
gint i;
gdouble rad, ra, theta, ct, st;

theta = (gdouble) n * 2.0 * PI / THETA_GRANULARITY;

i = n / (gdouble) RAD_CYCLE;
ra = i;
ra *= 2.0*PI;
ra /= RAD_GRANULARITY;
rad = fabs(tbl_cos(ra));

#if DEBUG_GET_COLOUR
printf("[%2d] (%f,%f) ", n, rad, theta);
#endif

ct = rad*tbl_cos(theta);
st = rad*tbl_sin(theta);

VEC3SET(colour, 0.0, 0.0, 0.0);

/* pairwise rgb */
if (st < 0.0)
  {
/* blue is major */
  colour[2] = 0.5 * (1.0 - st);
  if (ct < 0.0)
    {
/* mix green */
    colour[1] = 1.0 - fabs(0.866 + ct);
/* red is minor */
    if (ct > -0.5)
      colour[0] = -0.5 * ct;
    else
      colour[0] = 0.0;
    }
  else
    {
/* mix red */
    colour[0] = 1.0 - fabs(0.866 - ct);
/* green is minor */
    if (ct < 0.5)
      colour[1] = 0.5 * ct;
    else
      colour[1] = 0.0;
    }
  }
else
  {
/* FIXME - real max and min at ct = +/- 0.866... */
  colour[0] = 0.5 * (1.0 + ct);
  colour[1] = 0.5 * (1.0 - ct);
/* no blue */
  colour[2] = 0.0;
  }

#if DEBUG_GET_COLOUR
P3VEC(" : ", colour);
#endif

VEC3MUL(colour, 65535.0);
}

/****************************************/
/* change the colour scheme for an atom */
/****************************************/
void atom_colour_scheme(gint type, struct core_pak *core, struct model_pak *model)
{
struct elem_pak edata;

switch (type)
  {
  case ELEM:
    get_elem_data(core->atom_code, &edata, model);
    break;

  case MOL:
/* get the sequence number for the colour type */
/* NB: core->molecule is deprec. */
    get_colour(edata.colour, core->molecule);
    break;

  case REGION:
/* get the sequence number for the colour type */
    get_colour(edata.colour, core->region);
    break;
  }

/* set core colour */
ARR3SET(core->colour, edata.colour);
}

/****************************************/
/* change the colour scheme for a model */
/****************************************/
void model_colour_scheme(gint type, struct model_pak *model)
{
GSList *list;
struct core_pak *core;

model->colour_scheme = type;
for (list=model->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  atom_colour_scheme(type, core, model);
  }
}

/***********************************************/
/* change the properties of all selected atoms */
/***********************************************/
/* NB: this is a bit dangerous as you change everything in the */
/* selection to the specified value - even (eg) incompatible elements */
void selection_properties_change(gint type)
{
gint n;
gdouble charge;
const gchar *text;
GSList *list;
struct elem_pak edata;
struct model_pak *model;
struct core_pak *core;

model = sysenv.active_model;
if (!model)
  return;
if (!model->selection)
  return;

switch (type)
  {
  case NAME:
    text = gtk_entry_get_text(GTK_ENTRY(apd_label));
    n = elem_symbol_test(text);
    if (n)
      {
      get_elem_data(n, &edata, apd_data);

/* make sure we alow enough space for the string and the \0 */
      n = (LABEL_SIZE-1 > strlen(text)) ? strlen(text) : LABEL_SIZE-1;
      for (list=model->selection ; list ; list=g_slist_next(list))
        {
        core = (struct core_pak *) list->data;
/* core updates */
        g_snprintf(core->label,n+1,"%s",text);
/* NEW - don't update element specific data if the element type was not */
/* changed - ie the user has just made a labelling change (eg C -> C1) */
        if (edata.number != core->atom_code)
          {
          core->atom_code = edata.number;
          core->bond_cutoff = edata.cova;
          }
        init_atom_colour(core, model);
        init_atom_charge(core, model);
        }
/* model updates */
      g_slist_free(model->unique_atom_list);
      model->unique_atom_list = find_unique(ELEMENT, model);
      calc_emp(model);
      redraw_canvas(SINGLE);
      }
    break;

  case CHARGE:
    charge = str_to_float(gtk_entry_get_text(GTK_ENTRY(apd_charge)));

/*
printf("set selection charge: %f\n", charge);
*/

    for (list=model->selection ; list ; list=g_slist_next(list))
      {
      core = (struct core_pak *) list->data;
/* core updates */
      core->charge = charge;
      core->lookup_charge = FALSE;
      }
    calc_emp(model);
    redraw_canvas(SINGLE);
    break;
  }
}

/*****************************/
/* commit changes to an atom */
/*****************************/
void atom_properties_change(GtkWidget *w, gint type)
{
gint n;
const gchar *text;
struct elem_pak edata;

if (!apd_data)
  {
/* NEW - allow changes if more than one atom is selected */
  selection_properties_change(type);
  return;
  }

g_return_if_fail(apd_core != NULL);

switch(type)
  {
  case NAME:
    text = gtk_entry_get_text(GTK_ENTRY(apd_label));
/* make sure we alow enough space for the string and the \0 */
    n = (LABEL_SIZE-1 > strlen(text)) ? strlen(text) : LABEL_SIZE-1;
    g_snprintf(apd_core->label,n+1,"%s",text);
    n = elem_symbol_test(text);

/* if recognized -> update */
    if (n)
      {
      get_elem_data(n, &edata, apd_data);

/* NEW - don't update element specific data if the element type was not */
/* changed - ie the user has just made a labelling change (eg C -> C1) */
      if (n != apd_core->atom_code)
        {
        apd_core->atom_code = n;
        apd_core->bond_cutoff = edata.cova;
        }
      init_atom_colour(apd_core, apd_data);
      init_atom_charge(apd_core, apd_data);

      g_slist_free(apd_data->unique_atom_list);
      apd_data->unique_atom_list = find_unique(ELEMENT, apd_data);
      calc_emp(apd_data);
      atom_properties_update(apd_core, apd_data);
      redraw_canvas(SINGLE);
      }
    break;

/* FIXME - not the best name, it's really the FF type */
  case LABEL:
    text = gtk_entry_get_text(GTK_ENTRY(apd_type));
    if (apd_core->atom_type)
      g_free(apd_core->atom_type);
    apd_core->atom_type = g_strdup(text);
    redraw_canvas(SINGLE);
    break;

  case CHARGE:
    text = gtk_entry_get_text(GTK_ENTRY(apd_charge));
    apd_core->charge = str_to_float(text);
    apd_core->lookup_charge = FALSE;
    calc_emp(apd_data);
    if (apd_data->show_charge || apd_data->show_atom_charges)
      redraw_canvas(SINGLE);
    break;

  case COORD_X:
    text = gtk_entry_get_text(GTK_ENTRY(apd_x));
    apd_core->x[0] = str_to_float(text);
    calc_coords(REFRESH, apd_data);
    redraw_canvas(SINGLE);
    break;

  case COORD_Y:
    text = gtk_entry_get_text(GTK_ENTRY(apd_y));
    apd_core->x[1] = str_to_float(text);
    calc_coords(REFRESH, apd_data);
    redraw_canvas(SINGLE);
    break;

  case COORD_Z:
    text = gtk_entry_get_text(GTK_ENTRY(apd_z));
    apd_core->x[2] = str_to_float(text);
    calc_coords(REFRESH, apd_data);
    redraw_canvas(SINGLE);
    break;

  default:
    g_assert_not_reached();
  }
}

/*************************************/
/* updates the dialog for a new atom */
/*************************************/
void atom_properties_update(struct core_pak *core, struct model_pak *model)
{
gdouble q;
gchar *element, *label, *type, *charge, *x, *y, *z;
struct shel_pak *shell;

if (core && model)
  {
/* data available */
  element = g_strdup(elements[core->atom_code].symbol);
  label = g_strdup(core->label);

  if (core->atom_type)
    type = g_strdup(core->atom_type);
  else
    type = g_strdup("?");

  q = core->charge;
  if (core->shell)
    {
    shell = (struct shel_pak *) core->shell;
    q += shell->charge;
    }
  charge = g_strdup_printf("%9.4f", q);

  x = g_strdup_printf("%9.4f", core->x[0]);
  y = g_strdup_printf("%9.4f", core->x[1]);
  z = g_strdup_printf("%9.4f", core->x[2]);

  apd_core = core;
  }
else
  {
/* otherwise defaults */
  element = g_strdup("");
  label = g_strdup("");
  type = g_strdup("");
  charge = g_strdup("");
  x = g_strdup("");
  y = g_strdup("");
  z = g_strdup("");
  }

/* prevent changes from messing up the atom_properties_change() callback */
apd_data = NULL;

/* entry updates */
gtk_entry_set_text(GTK_ENTRY(apd_label), label);
gtk_entry_set_text(GTK_ENTRY(apd_type), type);
gtk_entry_set_text(GTK_ENTRY(apd_charge), charge);
gtk_entry_set_text(GTK_ENTRY(apd_x), x);
gtk_entry_set_text(GTK_ENTRY(apd_y), y);
gtk_entry_set_text(GTK_ENTRY(apd_z), z);

/* TODO : colour? SOF? */

apd_data = model;

/* cleanup */
g_free(element);
g_free(label);
g_free(type);
g_free(charge);
g_free(x);
g_free(y);
g_free(z);
}

/****************************************/
/* edit the properties of a single atom */
/****************************************/
void atom_properties_box(GtkWidget *box)
{
GtkWidget *hbox, *vbox, *entry;

/* checks */
g_return_if_fail(box != NULL);

/* two column element data display */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(box), hbox, TRUE, TRUE, 0);

/* left vbox - titles */
vbox = gtk_vbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

/* TODO - put in a for loop? */

entry = gtk_entry_new();
gtk_entry_set_text(GTK_ENTRY(entry), "Label");
gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);

entry = gtk_entry_new();
gtk_entry_set_text(GTK_ENTRY(entry), "FF Type");
gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);

entry = gtk_entry_new();
gtk_entry_set_text(GTK_ENTRY(entry), "Charge");
gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);

entry = gtk_entry_new();
gtk_entry_set_text(GTK_ENTRY(entry), "X");
gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);

entry = gtk_entry_new();
gtk_entry_set_text(GTK_ENTRY(entry), "Y");
gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);

entry = gtk_entry_new();
gtk_entry_set_text(GTK_ENTRY(entry), "Z");
gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);

/*
label = gtk_label_new("Colour:");
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
*/

/* right vbox - data */
vbox = gtk_vbox_new(TRUE, 0);
gtk_box_pack_end(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

apd_label = gtk_entry_new();
gtk_box_pack_start(GTK_BOX(vbox), apd_label, FALSE, FALSE, 0);

apd_type = gtk_entry_new();
gtk_box_pack_start(GTK_BOX(vbox), apd_type, FALSE, FALSE, 0);

apd_charge = gtk_entry_new();
gtk_box_pack_start(GTK_BOX(vbox), apd_charge, FALSE, FALSE, 0);

apd_x = gtk_entry_new();
gtk_box_pack_start(GTK_BOX(vbox), apd_x, FALSE, FALSE, 0);

apd_y = gtk_entry_new();
gtk_box_pack_start(GTK_BOX(vbox), apd_y, FALSE, FALSE, 0);

apd_z = gtk_entry_new();
gtk_box_pack_start(GTK_BOX(vbox), apd_z, FALSE, FALSE, 0);

/* set boxes */
atom_properties_update(NULL, NULL);

/* attach callbacks (NB: set initial data first) */
g_signal_connect(GTK_OBJECT(apd_label), "activate",
                 GTK_SIGNAL_FUNC(atom_properties_change), GINT_TO_POINTER(NAME));
g_signal_connect(GTK_OBJECT(apd_type), "activate",
                 GTK_SIGNAL_FUNC(atom_properties_change), GINT_TO_POINTER(LABEL));
g_signal_connect(GTK_OBJECT(apd_charge), "activate",
                 GTK_SIGNAL_FUNC(atom_properties_change), GINT_TO_POINTER(CHARGE));
g_signal_connect(GTK_OBJECT(apd_x), "activate",
                 GTK_SIGNAL_FUNC(atom_properties_change), GINT_TO_POINTER(COORD_X));
g_signal_connect(GTK_OBJECT(apd_y), "activate",
                 GTK_SIGNAL_FUNC(atom_properties_change), GINT_TO_POINTER(COORD_Y));
g_signal_connect(GTK_OBJECT(apd_z), "activate",
                 GTK_SIGNAL_FUNC(atom_properties_change), GINT_TO_POINTER(COORD_Z));
}

