/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#include "sginfo.h"
#include "gdis.h"

/* main structure */
extern struct sysenv_pak sysenv;

/* values for directions */
GtkWidget *image[MAX_DIALOGS][6];

/**********/
/* sginfo */
/**********/
static int str_ibegin(const char *s1, const char *s2) /* string ignore-case */
{                                                     /* begin              */
char u1, u2;

while (*s1 && *s2)
  {
  u1 = toupper(*s1++);
  u2 = toupper(*s2++);
  if (u1 < u2)
    return -1;
  else if (u1 > u2)
    return  1;
  }
if (*s2)
  return -1;
return 0;
}

/**********/
/* sginfo */
/**********/
/* TODO - we know what the input will be - trim */
gint BuildSgInfo(T_SgInfo *SgInfo, const gchar *SgName)
{
gint                VolLetter;
const T_TabSgName  *tsgn;


/* look for "VolA", "VolI", or "Hall" */

while (*SgName && isspace((int) *SgName)) SgName++;

VolLetter = -1;

if      (isdigit((int) *SgName))
  VolLetter = 'A';
else if (str_ibegin(SgName, "VolA") == 0)
  {
  VolLetter = 'A';
  SgName += 4;
  }
else if (str_ibegin(SgName, "VolI") == 0 || str_ibegin(SgName, "Vol1") == 0)
  {
  VolLetter = 'I';
  SgName += 4;
  }
else if (str_ibegin(SgName, "Hall") == 0)
  {
  VolLetter = 0;
  SgName += 4;
  }

while (*SgName && isspace((int) *SgName)) 
  SgName++;

/* default is "VolA" */

if (VolLetter == -1)
  VolLetter = 'A';

/* if we don't have a Hall symbol do a table look-up */

tsgn = NULL;

if (VolLetter)
  {
  tsgn = FindTabSgNameEntry(SgName, VolLetter);
  if (tsgn == NULL) return -1; /* no matching table entry */
  SgName = tsgn->HallSymbol;
  }

/* Allocate memory for the list of Seitz matrices and
   a supporting list which holds the characteristics of
   the rotation parts of the Seitz matrices
*/

SgInfo->MaxList = 192; /* absolute maximum number of symops */

SgInfo->ListSeitzMx
    = g_malloc(SgInfo->MaxList * sizeof (*SgInfo->ListSeitzMx));

if (SgInfo->ListSeitzMx == NULL) 
  {
  SetSgError("Not enough core");
  return -1;
  }

SgInfo->ListRotMxInfo
    = g_malloc(SgInfo->MaxList * sizeof (*SgInfo->ListRotMxInfo));

if (SgInfo->ListRotMxInfo == NULL) 
  {
  SetSgError("Not enough core");
  return -1;
  }

/* Initialize the SgInfo structure */

InitSgInfo(SgInfo);
SgInfo->TabSgName = tsgn; /* in case we know the table entry */

/* Translate the Hall symbol and generate the whole group */

ParseHallSymbol(SgName, SgInfo);
if (SgError != NULL)
  return -1;

  /* Do some book-keeping and derive crystal system, point group,
     and - if not already set - find the entry in the internal
     table of space group symbols
   */

return CompleteSgInfo(SgInfo);
}

/*****************************************/
/* generate positions of a complete cell */
/*****************************************/
#define DEBUG_GEN_POS 0
gint genpos(struct model_pak *data)
{
gint i, j, k, f, nt;
gint m, na, ns, primary_atoms, primary_shells;
gint s, smin, new_flag;
gint iList, nTrV, iTrV, nLoopInv, iLoopInv, iMatrix;
gchar *cmd, *label;
gfloat x[3];
const T_RTMx  *lsmx;
const gint *r, *t, *TrV;
T_RTMx SMx;
T_SgInfo SgInfo;

/* request info via number or name */
/* cell choice (TODO - neaten this ugly code) */
if (data->sginfo.cellchoice)
  {
  i = data->sginfo.cellchoice;
  if (data->sginfo.spacenum > 0)
    cmd = g_strdup_printf("%3d:%d",data->sginfo.spacenum,i);
  else
    cmd = g_strdup_printf("%s:%d",g_strstrip(data->sginfo.spacename),i);
  }
else
  {
  if (data->sginfo.spacenum > 0)
    cmd = g_strdup_printf("%3d",data->sginfo.spacenum);
  else
    cmd = g_strdup_printf("%s",g_strstrip(data->sginfo.spacename));
  }

#if DEBUG_GEN_POS
printf("retrieving as [%s]\n",cmd);
#endif

/* main call */
if (BuildSgInfo(&SgInfo, cmd) != 0)
  return(1);
g_free(cmd);

/* process returned space group name */
label = strdup(SgInfo.TabSgName->SgLabels);
for (i=0 ; i<strlen(label) ; i++)
  if (*(label+i) == '_')
    *(label+i) = ' ';

/* copy label to avoid any funny GULP characters */
i=strlen(label);
while(i>0)
  {
  if (*(label+i) == '=')
    {
    i++;
    break;
    }
  i--;
  }

/* fill in space group name */
g_free(data->sginfo.spacename);
data->sginfo.spacename = g_strdup_printf("%s",label+i);

/* fill in number (if an invalid value was supplied) */
if (data->sginfo.spacenum < 1)
  data->sginfo.spacenum = SgInfo.TabSgName->SgNumber;

/* crystal lattice type */
data->sginfo.lattice = SgInfo.XtalSystem;
g_free(data->sginfo.latticename);
switch(SgInfo.XtalSystem)
  {
  case XS_CUBIC:
    data->sginfo.latticename = g_strdup("Cubic");
    break;
  case XS_TETRAGONAL:
    data->sginfo.latticename = g_strdup("Tetragonal");
    break;
  case XS_ORTHOROMBIC:
    data->sginfo.latticename = g_strdup("Orthorombic");
    break;
  case XS_HEXAGONAL:
    data->sginfo.latticename = g_strdup("Hexagonal");
    break;
  case XS_TRIGONAL:
    data->sginfo.latticename = g_strdup("Trigonal");
    break;
  case XS_MONOCLINIC:
    data->sginfo.latticename = g_strdup("Monoclinic");
    break;
  case XS_TRICLINIC:
    data->sginfo.latticename = g_strdup("Triclinic");
    break;
  default:
    data->sginfo.latticename = g_strdup("Unknown");
  }

/* point group */
data->sginfo.pointgroup = SgInfo.PointGroup;

#if DEBUG_GEN_POS
printf("   Space group name: %s\n",data->sginfo.spacename);
printf(" Space group number: %d\n",data->sginfo.spacenum);
printf("       Lattice type: %c\n",SgInfo.LatticeInfo->Code);
printf("              Order: %d\n",SgInfo.OrderL);
printf("   Inversion center: ");
if (SgInfo.Centric == -1)
  printf("yes\n");
else
  printf("no\n");
#endif

/* OrderP/OrderL - which one??? */
data->sginfo.order = SgInfo.OrderL;
/* allocate for order number of pointers (to matrices) */
data->sginfo.matrix = (gfloat **) g_malloc(SgInfo.OrderL*sizeof(gfloat *));
data->sginfo.offset = (gfloat **) g_malloc(SgInfo.OrderL*sizeof(gfloat *));

iMatrix = 0;

nLoopInv = Sg_nLoopInv(&SgInfo);

nTrV = SgInfo.LatticeInfo->nTrVector;
 TrV = SgInfo.LatticeInfo->TrVector;

/* matrix counter */
m=0;
for (iTrV = 0; iTrV < nTrV; iTrV++, TrV += 3)
  {
  for (iLoopInv = 0; iLoopInv < nLoopInv; iLoopInv++)
    {
    if (iLoopInv == 0)
      f =  1;
    else
      f = -1;

    lsmx = SgInfo.ListSeitzMx;

/* loop over all matrices (order of the group) */
    for (iList = 0; iList < SgInfo.nList; iList++, lsmx++)
      {
      for (i = 0; i < 9; i++)
        SMx.s.R[i] = f * lsmx->s.R[i];
      for (i = 0; i < 3; i++)
        SMx.s.T[i] = iModPositive(f * lsmx->s.T[i] + TrV[i], STBF);

      r = SMx.s.R;
      t = SMx.s.T;

      *(data->sginfo.matrix+m) = (gfloat *) g_malloc(9*sizeof(gfloat));
      *(data->sginfo.offset+m) = (gfloat *) g_malloc(3*sizeof(gfloat));

      for (i = 0; i < 3; i++, t++)
        {
        for (j = 0; j < 3; j++, r++)
          {
/* mth general position */
          *(*(data->sginfo.matrix+m)+3*i+j) = (gfloat) *r;
          }
        nt = iModPositive(*t, STBF);
        if (nt >  STBF / 2)
          nt -= STBF;
/* offset matrix (3x1) */
        *(*(data->sginfo.offset+m)+i) = (gfloat) nt / (gfloat) STBF;
        }
      m++;
      }
    }
  }

/* number of matrices matches the order? */
if (m != data->sginfo.order)
  printf("Serious error in genpos()\n");

#if DEBUG_GEN_POS
printf("Constructed %d matrices.\n",m);
for (i=0 ; i<m ; i++)
  {
  printf("matrix %d :",i);
  for (j=0 ; j<9 ; j++)
    printf(" %4.1f",*(*(data->sginfo.matrix+i)+j));
  printf("\n");

  printf("offset %d :",i);
  for (j=0 ; j<3 ; j++)
    printf(" %4.1f",*(*(data->sginfo.offset+i)+j));
  printf("\n");
  }
#endif

/* for models that aren't 3D periodic (ie marvin surfaces) */
/* don't try to generate a (non existent) full cell */
if (data->periodic < 3)
  return(0);

/* set up core & shell counters */
na = primary_atoms = data->num_atoms;
ns = primary_shells = data->num_shells;

/* setup for site count (1 since we omit the trivial x,y,z) */
for (i=0 ; i<primary_atoms ; i++)
  (data->atoms+i)->sitecount = 1;

/* include an inversion (ie mult by -1) operation? */
if (SgInfo.Centric == -1)
  smin = -2;
else
  smin = 0;

/* s = +1 or -1 for inversion of matrix elements */
for (s=1 ; s>smin ; s-=2)
  {
/* loop over group operations */
/* j=0 is trivial (x,y,z) */
  for (j=1 ; j<data->sginfo.order ; j++)
    {
/* loop over atoms */
    for (i=0 ; i<primary_atoms ; i++)
      {
/* get current (ith) coords */
      x[0] = s*(data->atoms+i)->x;
      x[1] = s*(data->atoms+i)->y;
      x[2] = s*(data->atoms+i)->z;
/* FIXME - use order of the operation, not order of group */
/* should still work - it's just inefficient */
      for (k=0 ; k<data->sginfo.order ; k++)
        {
/* generate new position */
        vecmat(*(data->sginfo.matrix+j),x);
        x[0] += *(*(data->sginfo.offset+j)+0);
        x[1] += *(*(data->sginfo.offset+j)+1);
        x[2] += *(*(data->sginfo.offset+j)+2);
/* new position generated */
        (data->atoms+i)->sitecount++;
        new_flag=1;
/* TODO - somehow mark if in general or special position (via site count?) */
        if (na >= data->atom_limit)
          mem_grow(data, ATOM, 10);
        (data->atoms+na)->x = x[0]; 
        (data->atoms+na)->y = x[1]; 
        (data->atoms+na)->z = x[2]; 
        (data->atoms+na)->status = NORMAL;
        (data->atoms+na)->primary = FALSE;      /* not part of assym unit */
        (data->atoms+na)->orig = TRUE;        /* part of whole cell, though */
        (data->atoms+na)->sitecount = 0;
        (data->atoms+na)->sof = (data->atoms+i)->sof;
        (data->atoms+na)->offx = 0;
        (data->atoms+na)->offy = 0;
        strcpy((data->atoms+na)->label,(data->atoms+i)->label);
        strcpy((data->atoms+na)->element,(data->atoms+i)->element);
        strcpy((data->atoms+na)->tail,(data->atoms+i)->tail);
/* gets code & colour */
        elem_seek(na, ATOM, data);
/* next core */
        na++;
        }       /* operation loop on current primary atom */
      }         /* asym atom */

/* loop over shells */
    for (i=0 ; i<primary_shells ; i++)
      {
/* get orig shell's coords & apply generator */
      x[0] = (data->shells+i)->x;
      x[1] = (data->shells+i)->y;
      x[2] = (data->shells+i)->z;
/* FIXME - use order of the operation, not order of group */
/* should still work - it's just inefficient */
      for (k=0 ; k<data->sginfo.order ; k++)
        {
/* generate new position */
        vecmat(*(data->sginfo.matrix+j),x);
        x[0] += *(*(data->sginfo.offset+j)+0);
        x[1] += *(*(data->sginfo.offset+j)+1);
        x[2] += *(*(data->sginfo.offset+j)+2);
/* save new shell */
        if (ns >= data->shell_limit)
          mem_grow(data, SHELL, 10);
        (data->shells+ns)->x = x[0]; 
        (data->shells+ns)->y = x[1]; 
        (data->shells+ns)->z = x[2]; 
        (data->shells+ns)->status = NORMAL;
        (data->shells+ns)->primary = FALSE;      /* not part of assym unit */
        (data->shells+ns)->orig = TRUE;        /* part of whole cell, though */
        (data->shells+ns)->sitecount = 0;
        (data->shells+ns)->offx = 0;
        (data->shells+ns)->offy = 0;
        strcpy((data->shells+ns)->label,(data->shells+i)->label);
        strcpy((data->shells+ns)->element,(data->shells+i)->element);
        strcpy((data->shells+ns)->tail,(data->shells+i)->tail);
/* setup */
        elem_seek(ns, SHELL, data);
/* next shell */
        ns++;
        }       /* operation loop on current primary atom */
      }         /* asym shell */
    }  /* loop over order of group */
  }    /* possible inversion center */

/* all done */
data->num_orig = data->num_atoms = na;
data->num_shells = ns;

/* NEW - constrain AND delete repetitions in one fell swoop */
/* only if not P1, as P1 may have atoms that were intended */
/* to be outside the cell (ie for viewing purposes) */
if (data->sginfo.spacenum > 1)
  pbc_constrain_atoms(data);

/* unfortunately sitecount is not the site symmetry, */
/* since we would need to redo the loop that goes over */
/* the asym unit to go over the whole cell */ 
#if DEBUG_GEN_POS
printf("Atoms in assymetric unit: %d\n",primary_atoms);
printf("Atoms in whole unit: %d\n",data->num_atoms);
printf("Shells in whole unit: %d\n",data->num_shells);
for (i=0 ; i<primary_atoms ; i++)
  printf("%3s multiplicity: %d\n",(data->atoms+i)->label
                                 ,(data->atoms+i)->sitecount);
#endif

/* NEW - store the raw SgInfo structure for later calls (eg hkl absences) */
g_free(data->sginfo.raw);
data->sginfo.raw = g_malloc(sizeof(SgInfo));
memcpy(data->sginfo.raw, &SgInfo, sizeof(SgInfo));

return(0);
}

/************************/
/* force symmetry to P1 */
/************************/
void make_p1(struct model_pak *data)
{
gint i;
gfloat v[3], m[3];

/* set name & number */
data->sginfo.spacenum = 1;
g_free(data->sginfo.spacename);
data->sginfo.spacename = g_strdup("P1");
/* all (non-deleted) atoms are now primary & non fractional */
mem_shrink(data);
for (i=data->num_atoms ; i-- ; )
  {
  (data->atoms+i)->orig = TRUE;
  (data->atoms+i)->primary = TRUE;
  }
/* and the shells */
for (i=data->num_shells ; i-- ; )
  {
  (data->shells+i)->orig = TRUE;
  (data->shells+i)->primary = TRUE;
  }

/* get multiple of lattice vectors desired */
m[0] = (gfloat) (data->image_limit[1] + data->image_limit[0]);
m[1] = (gfloat) (data->image_limit[3] + data->image_limit[2]);
m[2] = (gfloat) (data->image_limit[5] + data->image_limit[4]);

/* scale the fractional coordinates down */
for (i=data->num_atoms ; i-- ; )
  {
  (data->atoms+i)->x /= m[0];
  (data->atoms+i)->y /= m[1];
  (data->atoms+i)->z /= m[2];
  }
for (i=data->num_shells ; i-- ; )
  {
  (data->shells+i)->x /= m[0];
  (data->shells+i)->y /= m[1];
  (data->shells+i)->z /= m[2];
  }

/* get current lattice vectors and scale up */
for (i=0 ; i<3 ; i++)
  {
  v[0] = data->latmat[i];
  v[1] = data->latmat[i+3];
  v[2] = data->latmat[i+6];
  VEC3MUL(v, m[i]);
  data->latmat[i] = v[0];
  data->latmat[i+3] = v[1];
  data->latmat[i+6] = v[2];
  }

/* update cell param related data */
data->construct_pbc = TRUE;
make_xlat(data);
make_cell(data);

/* reset */
data->image_limit[0] = 0;
data->image_limit[1] = 1;
data->image_limit[2] = 0;
data->image_limit[3] = 1;
data->image_limit[4] = 0;
data->image_limit[5] = 1;

/* generate symmetry data */
genpos(data);

/* NB: mem_shrink() -> redo connectivity */
init_objs(REDO_COORDS, data);
calc_bonds(data);
calc_mols(data);
}

/**************************************/
/* periodicity dialog hook to make_p1 */
/**************************************/
void remove_symmetry(gint id)
{
gint model;
struct model_pak *data;

/* get model ptr assoc. with this dialog */
model = sysenv.dialog[id].model;
data = model_ptr(model, RECALL);
g_return_if_fail(data != NULL);

/* make P1 and update */
make_p1(data);
update_images(model, id, INITIAL);
redraw_canvas(SINGLE);
}

/***********************************/
/* periodic image creation routine */
/***********************************/
#define DEBUG_UPDATE_IMAGES 0
void update_images(gint model, gint dialog, gint mode)
{
gint a, b, c, i, j, k, n, num_cells, num_images, extra;
gfloat vec[3];
static gint current=-7;
struct model_pak *data;
struct atom_pak *dest;
struct shell_pak *sdest;

/* TODO - allow 0 in the +ve direction (as long as -ve has more than 0) */

/* checks */
data = model_ptr(model, RECALL);
g_return_if_fail(data != NULL);

/* NEW - prevent multiple updates due to subsequent spinner updates */
if (current == dialog)
  {
#if DEBUG_UPDATE_IMAGES
printf("already active.\n");
#endif
  return;
  }
current = dialog;

#if DEBUG_UPDATE_IMAGES
for (i=0 ; i<sysenv.num_models ; i++)
  {
  printf("---------------------------\n");
  printf("model : %d\n",i);
  printf("a : %d - %d\n",data->image_limit[0],data->image_limit[1]);
  printf("b : %d - %d\n",data->image_limit[2],data->image_limit[3]);
  printf("c : %d - %d\n",data->image_limit[4],data->image_limit[5]);
  }
printf("---------------------------\n");
#endif

num_images = 0;
/* check mode */
switch(mode)
  {
  case RESTORE:
/* model switching - update the spin buttons */
    for (i=0 ; i<2*data->periodic ; i++)
      {
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(image[dialog][i]), 
                                data->image_limit[i]);
      gtk_signal_emit_by_name (GTK_OBJECT(image[dialog][i]), "changed");
      }
    return;
  case REFRESH:
/* get images from the spin button values */
    for (i=0 ; i<2*data->periodic ; i++)
      data->image_limit[i] = 
        gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(image[dialog][i]));
  case CREATE:
/* from here - create images according to image_limit[] */
/* total number of displayed cells */
    num_cells = (data->image_limit[0]+data->image_limit[1])
              * (data->image_limit[2]+data->image_limit[3])
              * (data->image_limit[4]+data->image_limit[5]);
/* number of images is minus the default cell */
    num_images = num_cells-1;

/* update */
    for (i=data->num_atoms ; i-- ; )
      if (!(data->atoms+i)->orig)
        (data->atoms+i)->status |= DELETED;
    for (i=data->num_shells ; i-- ; )
      if (!(data->shells+i)->orig)
        (data->shells+i)->status |= DELETED;
    mem_shrink(data);

/* allocate for atoms */
    extra = num_cells*data->num_atoms - data->atom_limit;
    if (extra > 0)
      mem_grow(data, ATOM, extra);

#if DEBUG_UPDATE_IMAGES
/* FIXME - unfrag/pbc manip may screw num_orig up
printf("orig atoms = %d\n",data->num_orig);
*/
printf("orig atoms = %d\n",data->num_atoms);
printf("req images = %d\n",num_images);
printf("old alloc (atoms) = %d\n",data->atom_limit-extra);
printf("new alloc (atoms) = %d\n",data->atom_limit);
#endif

/* allocate for shells */
    extra = num_cells*data->num_shells - data->shell_limit;
    if (extra > 0)
      mem_grow(data, SHELL, extra);

#if DEBUG_UPDATE_IMAGES
printf("old alloc (shells) = %d\n",data->shell_limit-extra);
printf("new alloc (shells) = %d\n",data->shell_limit);
#endif

/* update */
    dest = data->atoms + data->num_atoms;
    sdest = data->shells + data->num_shells;
    a = -data->image_limit[0];
    b = -data->image_limit[2];
    c = -data->image_limit[4];
    j = data->num_atoms;
    k = data->num_shells;
    for (i=num_cells ; i-- ; )
      {
/* image increment */
      if (a == data->image_limit[1])
        {
        a = -data->image_limit[0];
        b++;
        if (b == data->image_limit[3])
          {
          b = -data->image_limit[2];
          c++;
          if (c == data->image_limit[5])
            break;
          }
        }
/* don't duplicate the original cell */
      if (a || b || c)
        {
#if DEBUG_UPDATE_IMAGES
printf("Creating %d,%d,%d\n",a,b,c);
#endif
/* create core images */
        memcpy(dest, data->atoms, data->num_atoms*sizeof(struct atom_pak));
        j += data->num_atoms;
/* init atoms for this image */
        for (n=data->num_atoms ; n-- ; )
          {
          (dest+n)->status = NORMAL;
          (dest+n)->primary = FALSE;
          (dest+n)->orig = FALSE;
/* init fractional coords */
          (dest+n)->x += (gfloat) a;
          (dest+n)->y += (gfloat) b;
          if (data->periodic == 3)
            (dest+n)->z += (gfloat) c;
/* init stored initial cartesian coords */
          vec[0] = (dest+n)->x;
          vec[1] = (dest+n)->y;
          vec[2] = (dest+n)->z;
          vecmat(data->latmat, vec);
          (dest+n)->cx = vec[0];
          (dest+n)->cy = vec[1];
          (dest+n)->cz = vec[2];
          }
/* next image */
        dest += data->num_atoms;

/* create shell images */
        memcpy(sdest, data->shells, data->num_shells*sizeof(struct shell_pak));
        k += data->num_shells;
/* TODO - update cartesian coords (ie cx,cy,cx) */
/* init shells for this image */
        for (n=data->num_shells ; n-- ; )
          {
          (sdest+n)->status = NORMAL;
          (sdest+n)->primary = FALSE;
/* init fractional coords */
          (sdest+n)->x += (gfloat) a;
          (sdest+n)->y += (gfloat) b;
          if (data->periodic == 3)
            (sdest+n)->z += (gfloat) c;
          (sdest+n)->orig = FALSE;
/* init stored initial cartesian coords */
          vec[0] = (sdest+n)->x;
          vec[1] = (sdest+n)->y;
          vec[2] = (sdest+n)->z;
          vecmat(data->latmat, vec);
          (sdest+n)->cx = vec[0];
          (sdest+n)->cy = vec[1];
          (sdest+n)->cz = vec[2];
          }
/* next image */
        sdest += data->num_shells;
        }
      a++;
      }

#if DEBUG_UPDATE_IMAGES
printf("Wrote %d image atoms.\n",j-data->num_atoms);
printf("Wrote %d image shells.\n",k-data->num_shells);
#endif

/* mis-alloc'd? */
    if (j > data->atom_limit)
      printf("update_images(): serious memory allocation error.\n");
    else
      data->num_atoms = j;
/* mis-alloc'd? */
    if (k > data->shell_limit)
      printf("update_images(): serious memory allocation error.\n");
    else
      data->num_shells = k;

    break;
  case INITIAL:
    for (i=data->num_atoms ; i-- ; )
      if (!(data->atoms+i)->orig)
        (data->atoms+i)->status |= DELETED;
      else
        (data->atoms+i)->status = NORMAL;
    for (i=data->num_shells ; i-- ; )
      if (!(data->shells+i)->orig)
        (data->shells+i)->status |= DELETED;
      else
        (data->shells+i)->status = NORMAL;
    mem_shrink(data);

/* reset values to default */
    for (i=0 ; i<data->periodic ; i++)
      {
      data->image_limit[2*i] = 0;
      data->image_limit[2*i+1] = 1;
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(image[dialog][2*i]), 0);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(image[dialog][2*i+1]), 1);
      }
/* update the dialogue box */
    for (i=0 ; i<2*data->periodic ; i++)
      gtk_signal_emit_by_name (GTK_OBJECT(image[dialog][i]), "changed");
    break;
  }
current = -7;
}

/*********************/
/* spin button hooks */
/*********************/
/* TODO - combine/eliminate redundancy */
void gtk_refresh_images(GtkWidget *w, gint id)
{
gint model;
struct model_pak *data;

model = sysenv.dialog[id].model;
data = model_ptr(model, RECALL);
g_return_if_fail(data != NULL);

/* atom colour/display updates */
update_images(model, id, REFRESH);
init_objs(REFRESH_COLOUR, data);
if (data->mode == REGION_LITE)
  init_objs(REFRESH_LITE, data);
/* main coord updates */
init_objs(CENT_COORDS, data);
calc_bonds(data);
calc_mols(data);
pick_model(model);
}

void gtk_reset_images(GtkWidget *w, gint id)
{
gint model;
struct model_pak *data;

model = sysenv.dialog[id].model;
data = model_ptr(model, RECALL);
g_return_if_fail(data != NULL);

/* atom colour/display updates */
update_images(model, id, INITIAL);
init_objs(REFRESH_COLOUR, data);
if (data->mode == REGION_LITE)
  init_objs(REFRESH_LITE, data);
/* main coord updates */
init_objs(CENT_COORDS, data);
calc_bonds(data);
calc_mols(data);
pick_model(model);
}

/****************************/
/* display property updates */
/****************************/
void space_action(gint model, gint command)
{
gint i;
struct model_pak *data;

/* the usual */
data = model_ptr(model, RECALL);
if (!data)
  return;

switch(command)
  {
  case PBC_CONFINE_ATOMS:
    if (data->periodic)
      pbc_constrain_atoms(data);
    break;
  case PBC_CONFINE_MOLS:
    if (data->periodic)
      pbc_constrain_mols(data);
    break;
  case ASYM_TOGGLE:
    if (data->periodic)
      {
      data->asym_on ^= 1;
      if (data->asym_on)
        {
        for (i=0 ; i<data->num_atoms ; i++)
          if (!(data->atoms+i)->primary)
            (data->atoms+i)->status |= HIDDEN;
        for (i=0 ; i<data->num_shells ; i++)
          if (!(data->shells+i)->primary)
            (data->shells+i)->status |= HIDDEN;
        }
      else
        {
        for (i=0 ; i<data->num_atoms ; i++)
          if (!(data->atoms+i)->primary)
            (data->atoms+i)->status &= ~HIDDEN;
        for (i=0 ; i<data->num_shells ; i++)
          if (!(data->shells+i)->primary)
            (data->shells+i)->status &= ~HIDDEN;
        }
      }
    break;
  default:
    printf("space_update(): unknown action requested.\n");
  }

/* common update */
init_objs(CENT_COORDS, data);
calc_bonds(data);
calc_mols(data);
redraw_canvas(SINGLE);
}

/*****************************************************/
/* determine if a translated molecule will be bonded */
/*****************************************************/
#define DEBUG_IS_FRAG 0
gint is_fragment(struct model_pak *data, gint mol, gfloat *xlat)
{
gint a, b, i, j, k, move;
gfloat ra, r2;
gfloat va[3], vb[3]; 
struct elem_pak elem;

/* loop through atoms in molecule to test */
for (i=(data->mols+mol)->num_atoms ; i-- ; )
  {
  a = *((data->mols+mol)->atom+i);
  get_elem_data((data->atoms+a)->atom_code, &elem, data);
  ra = elem.cova;
/* get lattice space coordinates */
  va[0] = (data->atoms+a)->x;
  va[1] = (data->atoms+a)->y;
  va[2] = (data->atoms+a)->z;
/* translate */
  ARR3ADD(va, xlat);
/* get cartesian coords */
  vecmat(data->latmat, va);
/* is this molecule bonded to any other (except itself) */
/* don't search smaller (ie do only the off-diagonal compares) */
  j = mol+1;
  while (j<data->num_mols)
    {
/* loop through all atoms in molecule */
    for (k=(data->mols+j)->num_atoms ; k-- ; )
      {
/* calculate cutoff */
      b = *((data->mols+j)->atom+k);
      get_elem_data((data->atoms+b)->atom_code, &elem, data);
      r2 = ra + elem.cova;
      r2 += BOND_FUDGE*r2;
      r2 *= r2;
/* get cartesian coords */
      vb[0] = (data->atoms+b)->x;
      vb[1] = (data->atoms+b)->y;
      vb[2] = (data->atoms+b)->z;
      vecmat(data->latmat, vb);
/* a & b bonded? */
      ARR3SUB(vb, va);
      if (VEC3MAGSQ(vb) < r2)
        {
#if DEBUG_IS_FRAG
printf("bonded atoms: %d - %d\n", a, b);
#endif
/* move the smaller (or j if equal in size) */
        if ((data->mols+mol)->num_atoms >= (data->mols+j)->num_atoms)
          {
          move = j;
          VEC3MUL(xlat, -1.0);
          }
        else
          move = mol;
        goto bond_found;
        }
      }
    j++;
    }
  }
return(FALSE);
bond_found:;

#if DEBUG_IS_FRAG
printf("connected frag: %d - %d\n",mol,j);
#endif

/* NEW - bonds then & leave pbc until after */
/* update */
for (i=(data->mols+move)->num_atoms ; i-- ; )
  {
  a = *((data->mols+move)->atom+i);
/* get lattice space coordinates */
  va[0] = (data->atoms+a)->x;
  va[1] = (data->atoms+a)->y;
  va[2] = (data->atoms+a)->z;
/* translate */
  ARR3ADD(va, xlat);
/* save */
  (data->atoms+a)->x = va[0];
  (data->atoms+a)->y = va[1];
  (data->atoms+a)->z = va[2];
/* TODO - shells */

  if ((data->atoms+a)->has_shell)
    {
    b = (data->atoms+a)->idx_shell;
/* get lattice space coordinates */
    va[0] = (data->shells+b)->x;
    va[1] = (data->shells+b)->y;
    va[2] = (data->shells+b)->z;
/* translate */
    ARR3ADD(va, xlat);
/* save */
    (data->shells+b)->x = va[0];
    (data->shells+b)->y = va[1];
    (data->shells+b)->z = va[2];
    }
  }

return(TRUE);
}

/****************************************/
/* the new, more efficient unfragmenter */
/****************************************/
#define DEBUG_UNFRAGMENT 0
void unfragment(struct model_pak *data)
{
gint i, j, k, n, m, l, frag_found, mol;
gint *mol_flag;
gfloat xlat[3];

/* check */
if (!data)
  return;

#ifdef TIMER
start_timer("unfragment");
#endif

/* number of neighbour cells */
l = pow(3.0, (double) data->periodic);

/* constituents (it's too slow to recalc mols) */
/* end for mols - recalc mols (TODO - only add newly created mols) */
/* repeat until no changes */
n=0;
do 
  {
  frag_found=0;
/* for all mols */
  mol_flag = g_malloc0(data->num_mols*sizeof(gint));
/* DON'T REVERSE THIS LOOP ORDER */
  for (mol=0 ; mol<data->num_mols-1 ; mol++)
    {
/* ignore if it has been shifted this cycle */
    if (*(mol_flag+mol))
      continue;
/* exists an xlat that forms a new bond? */
/* quicker looping method (main gain is the 2D v 3D case) */
    for (m=0 ; m<l ; m++)
      {
      k = m / 9;
      j = (m % 9) / 3;
      i = m % 3;
      i--;
      j--;
      k--;
/* TODO - if 2D don't need one of these */
/* don't do the 0 translation */
      if (i || j || k)
        {
        xlat[0] = (gfloat) i;
        xlat[1] = (gfloat) j;
/* hack for 2D */
        if (data->periodic == 3)
          xlat[2] = (gfloat) k;
        else
          xlat[2] = 0.0;
/* test if this translation creates a new linkage */
        if (is_fragment(data, mol, xlat))
          {
/* flag as being shifted */
          (*(mol_flag+mol))++;
          frag_found++;
          goto next_mol;
          }
        }
      }       /* m */
next_mol:;
    }

/* get rid of old list */
  g_free(mol_flag);

/* update connectivity */
  if (frag_found)
    {
/* TODO - more speed ups could be got by reducing the need */
/* for so much updating here (eg add/sub mols rather than recalc) */
    init_objs(REDO_COORDS, data);
    calc_bonds(data);
    calc_mols(data);
    pbc_constrain_mols(data);
    }
/* prevent inf. loops - not terribly elegant */
  if (n++ > 5)
    {
    printf("Warning: forced exit from unfrag2().\n");
    break;
    }
  }
while(frag_found);

#if DEBUG_UNFRAGMENT
printf("unfragment loops: %d\n", n);
#endif

/* redraw */
redraw_canvas(SINGLE);

#ifdef TIMER
stop_timer("unfragment");
#endif
}

/****************/
/* button hooks */
/****************/
void pbc_confine_atoms(gint id)
{
space_action(sysenv.dialog[id].model, PBC_CONFINE_ATOMS);
}
void pbc_confine_mols(gint id)
{
space_action(sysenv.dialog[id].model, PBC_CONFINE_MOLS);
}
void asym_toggle(gint id)
{
space_action(sysenv.dialog[id].model, ASYM_TOGGLE);
}
#define DEBUG_COMPLETE_MOLS 0
void complete_mols(gint id)
{
gint model;
struct model_pak *data;

/* init */
model = sysenv.dialog[id].model;
data = model_ptr(model, RECALL);
g_return_if_fail(data != NULL);

/* the all new fast unfragger */
unfragment(data);

/* update & redraw */
/* atom colour/display updates */
init_objs(REFRESH_COLOUR, data);
if (data->mode == REGION_LITE)
  init_objs(REFRESH_LITE, data);
/* main coord updates */
init_objs(CENT_COORDS, data);
calc_bonds(data);
calc_mols(data);

#if DEBUG_COMPLETE_MOLS
printf("num atoms = %d\n",data->num_atoms);
printf("num mols = %d\n",data->num_mols);
#endif

/* draw - model is not nec = sysenv.active -> update all */
/* NEW - model may not be active - make it so (& update) */
pick_model(model);
}

/***************************/
/* Space Group info dialog */
/***************************/
void space_info(GtkWidget *w, struct model_pak *data)
{
gint i, j, k, l, m, rows, cols, need_sign, id;
gfloat *mat, value;
GString *info, *item;
GtkWidget *vbox, *hbox, *vbox1, *vbox2, *table, *frame, *button, *label;
GtkAdjustment *adj;
struct dialog_pak *sgid;

item = g_string_new(NULL);
info = g_string_new(NULL);

/* model checks */
g_return_if_fail(data != NULL);
g_return_if_fail(data->id != NODATA);
g_return_if_fail(data->periodic > 1);

/* retrieve a suitable dialog */
if ((id = request_dialog(data->number, SGINFO)) < 0)
  return;
sgid = &sysenv.dialog[id];

/* setup the dialog */
sgid->win = gtk_dialog_new();
gtk_window_set_title (GTK_WINDOW (sgid->win), "Space Group Info");
gtk_signal_connect(GTK_OBJECT(sgid->win), "destroy",
                   GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

/* dual pane layout */
hbox = gtk_hbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(GTK_BOX(GTK_DIALOG(sgid->win)->vbox)),hbox);

/* left & right vboxes */
vbox1 = gtk_vbox_new (FALSE, 0);
vbox2 = gtk_vbox_new (FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, FALSE, FALSE, 0);

/* LEFT FRAME - MAIN INFO */
g_string_sprintf(info,"%s",data->basename);

frame = gtk_frame_new (info->str);
gtk_box_pack_end (GTK_BOX (vbox1), frame, TRUE, TRUE, 0);
gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
vbox = gtk_vbox_new (FALSE, 5);
gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
gtk_container_add (GTK_CONTAINER(frame),vbox);

/* table for nice spacing */
table = gtk_table_new(2, 6, FALSE);
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox)),table);

label = gtk_label_new ("Space Group name:       ");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,0,1);
gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.0f);
g_string_sprintf(info,"%s",data->sginfo.spacename);
label = gtk_label_new (g_strstrip(info->str));
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,0,1);
gtk_misc_set_alignment(GTK_MISC(label), 0.5f, 0.0f);

label = gtk_label_new ("Space Group number:       ");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,1,2);
gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.0f);

/* NEW - include cell choice */
if (data->sginfo.cellchoice)
  g_string_sprintf(info,"%d : %d",data->sginfo.spacenum
                                 ,data->sginfo.cellchoice);
else
  g_string_sprintf(info,"%d",data->sginfo.spacenum);
label = gtk_label_new (g_strstrip(info->str));
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,1,2);
gtk_misc_set_alignment(GTK_MISC(label), 0.5f, 0.0f);

label = gtk_label_new ("Crystal system:       ");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,2,3);
gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.0f);
g_string_sprintf(info,"%s",g_strstrip(data->sginfo.latticename));
label = gtk_label_new (g_strstrip(info->str));
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,2,3);
gtk_misc_set_alignment(GTK_MISC(label), 0.5f, 0.0f);

label = gtk_label_new ("Point group:       ");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,3,4);
gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.0f);
i = PG_Index(data->sginfo.pointgroup);
g_string_sprintf(info,"%s",PG_Names[i]);
label = gtk_label_new (g_strstrip(info->str));
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,3,4);
gtk_misc_set_alignment(GTK_MISC(label), 0.5f, 0.0f);

label = gtk_label_new ("Atoms in asymmetric cell:       ");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,4,5);
gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.0f);
g_string_sprintf(info,"%d",data->num_asym);
label = gtk_label_new (g_strstrip(info->str));
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,4,5);
gtk_misc_set_alignment(GTK_MISC(label), 0.5f, 0.0f);

label = gtk_label_new ("Atoms in full cell:       ");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,5,6);
gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.0f);
g_string_sprintf(info,"%d",data->num_orig);
label = gtk_label_new (g_strstrip(info->str));
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,5,6);
gtk_misc_set_alignment(GTK_MISC(label), 0.5f, 0.0f);


/* NEW - changed layout */
hbox = gtk_hbox_new(TRUE, 5);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
g_string_sprintf(info,"lengths: %6.2f  %6.2f  %6.2f", data->pbc[0],
                              data->pbc[1], data->pbc[2]);
label = gtk_label_new(info->str);
gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);

hbox = gtk_hbox_new(TRUE, 5);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
g_string_sprintf(info," angles: %6.2f  %6.2f  %6.2f", 180*data->pbc[3]/PI,
                       180*data->pbc[4]/PI, 180*data->pbc[5]/PI);
label = gtk_label_new(info->str);
gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);

frame = gtk_frame_new("Lattice matrix");
gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);

vbox = gtk_vbox_new(TRUE, 5);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(frame), 5);

/* TODO - put stuff below in a table */
label = gtk_label_new("   a       b       c  ");
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

g_string_sprintf(info,"%6.2f  %6.2f  %6.2f", data->latmat[0],
                            data->latmat[1], data->latmat[2]);
label = gtk_label_new(info->str);
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

g_string_sprintf(info,"%6.2f  %6.2f  %6.2f", data->latmat[3],
                            data->latmat[4], data->latmat[5]);
label = gtk_label_new(info->str);
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

g_string_sprintf(info,"%6.2f  %6.2f  %6.2f", data->latmat[6],
                            data->latmat[7], data->latmat[8]);
label = gtk_label_new(info->str);
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

/* BOTTOM FRAME */
frame = gtk_frame_new ("General positions");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(sgid->win)->vbox),frame,TRUE,TRUE,0);
gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
vbox = gtk_vbox_new (FALSE, 5);
gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
gtk_container_add (GTK_CONTAINER(frame),vbox);
/* table for nice spacing */
/* more intelligent dimension choice? */
cols = 4;
rows = data->sginfo.order / cols;
table = gtk_table_new(cols, rows, FALSE);
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox)),table);

/* order of the group (ie mth matrix) */
m=0;
/* loop over rows */
while (m<data->sginfo.order)
  {
/* get row(i) and column(j) */
  i = m / cols;
  j = m % cols;
/* construct the three generator elements */
  mat = *(data->sginfo.matrix+m);
  g_string_sprintf(info," ");
  for (k=0 ; k<3 ; k++)
    {
/* offset elements */
    need_sign = FALSE;
    value = *(*(data->sginfo.offset+m)+k);
    if (fabs(2.0*value-1.0) < FRACTION_TOLERANCE)
      {
      g_string_append(info,"1/2");
      need_sign = TRUE;
      }
    if (fabs(3.0*value-1.0) < FRACTION_TOLERANCE)
      {
      g_string_append(info,"1/3");
      need_sign = TRUE;
      }
    if (fabs(3.0*value+1.0) < FRACTION_TOLERANCE)
      {
      g_string_append(info,"-1/3");
      need_sign = TRUE;
      }
    if (fabs(4.0*value-1.0) < FRACTION_TOLERANCE)
      {
      g_string_append(info,"1/4");
      need_sign = TRUE;
      }
    if (fabs(4.0*value+1.0) < FRACTION_TOLERANCE)
      {
      g_string_append(info,"-1/4");
      need_sign = TRUE;
      }
/* matrix elements */
    for (l='x' ; l<='z' ; l++)
      {
      if (*mat > 0.0)
        {
        if (l == 'z')
          {
          if (need_sign)
            g_string_sprintf(item,"+%c",l);
          else
            g_string_sprintf(item,"%c",l);
          }
        else
          {
          if (need_sign)
            g_string_sprintf(item,"+%c,",l);
          else
            g_string_sprintf(item,"%c,",l);
          }
        g_string_append(info,item->str);
        }
      if (*mat < 0.0)
        {
        if (l == 'z')
          g_string_sprintf(item,"-%c",l);
        else
          g_string_sprintf(item,"-%c,",l);
        g_string_append(info,item->str);
        }
      mat++;
      }
    }
/* write the generator to the table */
  label = gtk_label_new(info->str);
  gtk_table_attach_defaults(GTK_TABLE(table),label,j,j+1,i,i+1);
  m++;
  }

/* RIGHT FRAME - PERIODIC IMAGE CONSTRUCTION */
frame = gtk_frame_new ("Periodic image construction");
gtk_box_pack_start (GTK_BOX (vbox2), frame, FALSE, TRUE, 0);

gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
vbox = gtk_vbox_new (FALSE, 5);
gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
gtk_container_add (GTK_CONTAINER (frame), vbox);
/* use table for nice spacing */
table = gtk_table_new(3, 4, FALSE);
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox)),table);

/* image direction labels */
label = gtk_label_new ("-");
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,0,1);
label = gtk_label_new ("+");
gtk_table_attach_defaults(GTK_TABLE(table),label,2,3,0,1);

/* x direction */
label = gtk_label_new ("a");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,1,2);

adj = (GtkAdjustment *) gtk_adjustment_new
                           (data->image_limit[0], 0, 5, 1, 1, 1);
image[id][0] = gtk_spin_button_new (adj, 0, 0);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(image[id][0]), GTK_SHADOW_OUT);
gtk_table_attach_defaults(GTK_TABLE(table), image[id][0], 1, 2, 1, 2);
gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
                    GTK_SIGNAL_FUNC (gtk_refresh_images), (gpointer) id);

adj = (GtkAdjustment *) gtk_adjustment_new
                           (data->image_limit[1], 1, 5, 1, 1, 1);
image[id][1] = gtk_spin_button_new (adj, 0, 0);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(image[id][1]), GTK_SHADOW_OUT);
gtk_table_attach_defaults(GTK_TABLE(table), image[id][1], 2, 3, 1, 2);
gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
                    GTK_SIGNAL_FUNC (gtk_refresh_images), (gpointer) id);

/* y direction */
label = gtk_label_new ("b");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,2,3);

adj = (GtkAdjustment *) gtk_adjustment_new
                           (data->image_limit[2], 0, 5, 1, 1, 1);
image[id][2] = gtk_spin_button_new (adj, 0, 0);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(image[id][2]), GTK_SHADOW_OUT);
gtk_table_attach_defaults(GTK_TABLE(table), image[id][2], 1, 2, 2, 3);
gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
                    GTK_SIGNAL_FUNC (gtk_refresh_images), (gpointer) id);

adj = (GtkAdjustment *) gtk_adjustment_new
                           (data->image_limit[3], 1, 5, 1, 1, 1);
image[id][3] = gtk_spin_button_new (adj, 0, 0);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(image[id][3]), GTK_SHADOW_OUT);
gtk_table_attach_defaults(GTK_TABLE(table), image[id][3], 2, 3, 2, 3);
gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
                    GTK_SIGNAL_FUNC (gtk_refresh_images), (gpointer) id);

/* z direction */
if (data->periodic == 3)
  {
label = gtk_label_new ("c");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,3,4);

adj = (GtkAdjustment *) gtk_adjustment_new
                           (data->image_limit[4], 0, 5, 1, 1, 1);
image[id][4] = gtk_spin_button_new (adj, 0, 0);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(image[id][4]), GTK_SHADOW_OUT);
gtk_table_attach_defaults(GTK_TABLE(table), image[id][4], 1, 2, 3, 4);
gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
                    GTK_SIGNAL_FUNC (gtk_refresh_images), (gpointer) id);

adj = (GtkAdjustment *) gtk_adjustment_new
                           (data->image_limit[5], 1, 5, 1, 1, 1);
image[id][5] = gtk_spin_button_new (adj, 0, 0);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(image[id][5]), GTK_SHADOW_OUT);
gtk_table_attach_defaults(GTK_TABLE(table), image[id][5], 2, 3, 3, 4);
gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
                    GTK_SIGNAL_FUNC (gtk_refresh_images), (gpointer) id);
  }

/* reset button */
hbox = gtk_hbox_new(FALSE,0);
gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
button = gtk_button_new_with_label ("  Reset  ");
gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, FALSE, 0);
gtk_signal_connect (GTK_OBJECT (button), "clicked",
                    GTK_SIGNAL_FUNC (gtk_reset_images), (gpointer) id);

/* RIGHT FRAME - VIEWING TOGGLES */
frame = gtk_frame_new (NULL);
gtk_box_pack_end (GTK_BOX (vbox2), frame, TRUE, TRUE, 0);
gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
vbox = gtk_vbox_new (TRUE, 5);
gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
gtk_container_add (GTK_CONTAINER (frame), vbox);

/* view only asym */
hbox = gtk_hbox_new(FALSE,0);
gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new ("Asymmetric/full cell");
gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
button = gtk_button_new_with_label (" X ");
gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (asym_toggle),
                           (gpointer) id);

/* FIXME (broken) constrain atoms to pbc */
hbox = gtk_hbox_new(FALSE,0);
gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new (" Constrain atoms to PBC ");
gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
button = gtk_button_new_with_label (" X ");
gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (pbc_confine_atoms),
                           (gpointer) id);

/* complete 'broken' mols */
hbox = gtk_hbox_new(FALSE,0);
gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new (" Constrain mols to PBC ");
gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
button = gtk_button_new_with_label (" X ");
gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (pbc_confine_mols),
                           (gpointer) id);

/* complete 'broken' mols */
hbox = gtk_hbox_new(FALSE,0);
gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new (" Unfragment mols ");
gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
button = gtk_button_new_with_label (" X ");
gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (complete_mols),
                           (gpointer) id);

/* force to P1 symmetry */
hbox = gtk_hbox_new(FALSE,0);
gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new (" Make supercell ");
gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
button = gtk_button_new_with_label (" X ");
gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (remove_symmetry),
                           (gpointer) id);

/* transform lattice vectors */
hbox = gtk_hbox_new(FALSE,0);
gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new (" transform latvec ");
gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
button = gtk_button_new_with_label (" X ");
gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (transform_latvec),
                           (gpointer) id);

/* Do the exit button */
button = gtk_button_new_with_label ("   Close   ");
gtk_box_pack_start (GTK_BOX (GTK_DIALOG (sgid->win)->action_area), button,
                                         TRUE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (close_dialog),
                           (gpointer) id);

/* done, draw, exit */
gtk_widget_show_all(sgid->win);
/* active this model */
pick_model(data->number);

g_string_free(info, TRUE);
g_string_free(item, TRUE);
}

