/* Functions for tiling pixbufs.
 * Copyright (C) 2004 Martin Grimme
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "utils.h"

#include <stdlib.h>
#include <string.h>

#include <gconf/gconf-client.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>


PyMODINIT_FUNC inittiling(void);

static inline void
copy_n_rows (int n, GdkPixbuf *pbuf, int row_size, int offset)
{
  guchar * const pixels = gdk_pixbuf_get_pixels (pbuf);
  memcpy (pixels + offset, pixels, n * row_size);
}



static void
make_row (const GdkPixbuf *src, GdkPixbuf *dest, int offset)
{
  int     x, y;
  guchar *in, *out;

  const int src_height     = gdk_pixbuf_get_height (src);
  const int dest_height    = gdk_pixbuf_get_height (dest);
  const int src_rowstride  = gdk_pixbuf_get_rowstride (src);
  const int dest_rowstride = gdk_pixbuf_get_rowstride (dest);

  const int q = offset / dest_rowstride;

  in  = gdk_pixbuf_get_pixels (src);
  out = gdk_pixbuf_get_pixels (dest) + offset;

  for (y = 0; (y < src_height) && (y + q < dest_height); y++) {
    for (x = 0; x < dest_rowstride; x += src_rowstride) {
      memcpy (out + x, in, MIN (src_rowstride, dest_rowstride - x) );
    }

    in  +=  src_rowstride;
    out += dest_rowstride;
  }
}



static void
tile (const GdkPixbuf *src, GdkPixbuf *dest)
{
  int row, offset;

  const int row_width  = gdk_pixbuf_get_rowstride (dest);
  const int row_height = gdk_pixbuf_get_height (src);
  const int row_size   = row_width * row_height;
  const int dest_size  = row_width * gdk_pixbuf_get_height (dest);
  const int max	       = gdk_pixbuf_get_height (dest) / row_height;

  /* first iteration unrolled */
  offset = 0;
  row    = 0;
  make_row (src, dest, offset);
  row++;
  offset += row_size;

  while (offset < dest_size && row < max) {
      const int n = MIN (row, max - row);
      copy_n_rows (n, dest, row_size, offset);
      row += n;
      offset += row_size * n;
  }

  /* last iteration unrolled */
  make_row (src, dest, offset);
}



static PyObject* tile_on_image(PyObject* self, PyObject* args)
{
  GtkImage *image;
  const GdkPixbuf *pbuf;
  int width, height;

  GdkPixbuf *bg;

  if(!PyArg_ParseTuple(args, "O&O&ii",
		       parse_gtk_image,  &image,
		       parse_gdk_pixbuf, &pbuf,
		       &width, &height))
    return NULL;

  bg = gdk_pixbuf_new (GDK_COLORSPACE_RGB,
		       gdk_pixbuf_get_has_alpha (pbuf),
		       8, width, height);
  tile (pbuf, bg);
  gtk_image_set_from_pixbuf (image, bg);
  g_object_unref (G_OBJECT (bg));

  Py_INCREF(Py_None);
  return Py_None;
}



static PyObject* tile_transparency(PyObject* self, PyObject* args)
{
  GtkWidget *widget;
  long wallpaper_id;
  int x, y, width, height;

  int          pwidth, pheight, sx, sy;
  char        *value;
  GConfClient *client;
  GdkBitmap   *mask;
  GdkColormap *cmap;
  GdkPixbuf   *pbuf;
  GdkPixmap   *pix;
  GdkPixmap   *pmap;
  GdkWindow   *rootwin;
  GtkStyle    *style;

  if(!PyArg_ParseTuple(args, "O&liiii",
		       parse_gtk_widget, &widget,
		       &wallpaper_id, &x, &y, &width, &height))
    return NULL;

  pmap = gdk_pixmap_foreign_new ((GdkNativeWindow) wallpaper_id);

  gdk_drawable_get_size (GDK_DRAWABLE (pmap), &pwidth, &pheight);

  /* assume solid colored desktop if the bg pixmap is very small */
  if (pwidth * pheight < 100) {
    pwidth  = 100;
    pheight = 100;
    client = gconf_client_get_default ();
    value  = gconf_client_get_string (client,
				      "/desktop/gnome/background/primary_color",
				      NULL);
    g_object_unref (G_OBJECT (client));
    pbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, 100, 100);
    gdk_pixbuf_fill (pbuf, strtol (value + 1, NULL, 16) << 8);
    g_free (value);
  } else {
    pbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, width, height);
  }

  rootwin = gdk_get_default_root_window ();
  cmap = gdk_drawable_get_colormap (GDK_DRAWABLE (rootwin));

  /* tile wallpaper over pixbuf */
  sx = - (x % pwidth);
  sy = - (y % pheight);
  for (x = sx; x < width; x += pwidth) {
    for (y = sy; y < height; y += pheight) {
      int dstx = MAX (0, x);
      int dsty = MAX (0, y);
      int srcx = dstx - x;
      int srcy = dsty - y;

      int w = MIN (pwidth - srcx, width - dstx);
      int h = MIN (pheight - srcy, height - dsty);

      gdk_pixbuf_get_from_drawable (pbuf, pmap, cmap, srcx, srcy,
				    dstx, dsty, w, h);
    }
  }

  /* put background onto the widget */
  gdk_pixbuf_render_pixmap_and_mask (pbuf, &pix, &mask, 127);

  style = gtk_style_new ();
  style->bg_pixmap[GTK_STATE_NORMAL] = pix;
  gtk_widget_set_style (widget, style);

  g_object_unref (G_OBJECT (pbuf));
  g_object_unref (G_OBJECT (pmap));
  g_object_unref (G_OBJECT (style));

  Py_INCREF(Py_None);
  return Py_None;
}



PyMODINIT_FUNC
inittiling(void)
{
  static const PyMethodDef methods[] =
    {
      {"tile_on_image", tile_on_image, METH_VARARGS, NULL},
      {"tile_transparency", tile_transparency, METH_VARARGS, NULL},
      {NULL, NULL, 0, NULL}
    };

  if(!gdesklets_get_pygobject_type())
    return;

  Py_InitModule("tiling", (PyMethodDef*) methods);
}
