#include <gtk/gtk.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#if USE_PIXBUF
#include <gdk-pixbuf/gdk-pixbuf.h>
#endif

#include "gdancer.h"
#include "audioprocessing.h"

/* Include the default images */
#include "../pics/bass.xpm"	// *bassimage[]
#include "../pics/midrange.xpm" // *midrangeimage[]
#include "../pics/neutral.xpm"	// *neutralimage[]
#include "../pics/treble.xpm"	// *trebleimage[]

// Clears the pixmaps from memory
void clear_images_sub (frame *frame)
{
#if USE_PIXBUF
	if (frame->pixbuf != NULL) {
		gdk_pixbuf_unref (frame->pixbuf);
		frame->pixbuf = NULL;
	}
#endif
	if (frame->pixmap != NULL) {
		gdk_pixmap_unref (frame->pixmap);
		frame->pixmap = NULL;
	}
	if (frame->mask != NULL) {
		gdk_bitmap_unref (frame->mask);
		frame->mask = NULL;
	}
}
void clear_images (GDancer *dancer)
{
	//FIXME: need to add crap here
	clear_images_sub (&(dancer->bass));
	clear_images_sub (&(dancer->midrange));
	clear_images_sub (&(dancer->neutral));
	clear_images_sub (&(dancer->treble));
	clear_images_sub (&(dancer->bn));
	clear_images_sub (&(dancer->nt));
	clear_images_sub (&(dancer->mn));
	clear_images_sub (&(dancer->bt));
	clear_images_sub (&(dancer->bm));
	clear_images_sub (&(dancer->mt));
}

// Make the widget to hold the image (right now I'm using gtk_pixmap)
GtkWidget * make_image_widget (GDancer *dancer)
{
	// Make the pixmap widget
	return(gtk_pixmap_new(dancer->neutral.pixmap,
			dancer->neutral.mask));
}

gchar *getfilename (gchar *imagename, gchar *extension, GDancer *dancer) {
	gchar *file;
	file = g_strconcat (g_get_home_dir(), "/.xmms/gdancer_themes/", dancer->theme, "/", imagename, extension, NULL);
	return file;
}

void loadimage (gchar *imagename, frame *frame, GDancer *dancer) {
	gchar *file;
	// check for older .xpm files since they are most common
	file = getfilename (imagename, ".xpm", dancer);
#if DEBUGGING
	printf ("Loading image for %s from %s\n", imagename, file);
#endif
#if USE_PIXBUF
	frame->pixbuf = gdk_pixbuf_new_from_file (file);
#else
	frame->pixmap = gdk_pixmap_create_from_xpm (
			dancer->window->window,
			&(frame->mask),
			NULL,
			file);
#endif
	g_free (file);
#if USE_PIXBUF
	if (frame->pixbuf == NULL) {
#else
	if (frame->pixmap == NULL) {
#endif
		// Probably is using the new format with no extension
		file = getfilename (imagename, "", dancer);
#if DEBUGGING
		printf ("Wasn't there, trying %s from %s\n", imagename, file);
#endif
#if USE_PIXBUF
		frame->pixbuf = NULL;
		frame->pixbuf = gdk_pixbuf_new_from_file (file);
#else
		frame->pixmap = NULL;
		frame->pixmap = gdk_pixmap_create_from_xpm (
				dancer->window->window,
				&(frame->mask),
				NULL,
				file);
#endif
	}

#if USE_PIXBUF
	/* I can't use gdk_pixbuf to render the image yet cause it
	 * messes up when I try applying the mask to the window. So
	 * I need to make a pixmap and mask using the pixbuf image 
	 * so I can use the regular routines for drawing the image */
	if (frame->pixbuf != NULL)
		gdk_pixbuf_render_pixmap_and_mask (frame->pixbuf,
			&(frame->pixmap),
			&(frame->mask),
			128);
#endif
}

void set_default_images (gchar *imagename, frame *frame, GDancer *dancer)
{
	gchar **xpmdata = NULL;
	// This makes sure the important images are set and if they aren't
	// it will set them to the default theme.
	// This is also how the default theme itself gets set, because all the
	// images will be unset
#if USE_PIXBUF
	if (frame->pixbuf == NULL) {
#else
	if ((frame->pixmap == NULL) ) {
#endif
#if DEBUGGING
		printf ("Loading Default theme image for %s\n", imagename);
#endif
		if (imagename == "bass")
			xpmdata = bassimage;
		else if (imagename == "treble")
			xpmdata = trebleimage;
		else if (imagename == "midrange")
			xpmdata = trebleimage;
		else if (imagename == "neutral")
			xpmdata = neutralimage;
#if USE_PIXBUF
		frame->pixbuf = gdk_pixbuf_new_from_xpm_data (
			(const char **) xpmdata);
#else
		frame->pixmap = gdk_pixmap_create_from_xpm_d(
			dancer->window->window,
			&(frame->mask),
			NULL,
			xpmdata);
#endif
	}
}

gint load_images (GDancer *dancer)
{
	if (dancer->window == NULL) return; // Make sure dancer exists
	
	clear_images (dancer);

	loadimage ("bass", &(dancer->bass), dancer);
	loadimage ("treble", &(dancer->treble), dancer);
	loadimage ("neutral", &(dancer->neutral), dancer);
	loadimage ("midrange", &(dancer->midrange), dancer);
	loadimage ("bass-neutral", &(dancer->bn), dancer);
	loadimage ("neutral-treble", &(dancer->nt), dancer);
	loadimage ("midrange-neutral", &(dancer->mn), dancer);
	loadimage ("bass-treble", &(dancer->bt), dancer);
	loadimage ("bass-midrange", &(dancer->bm), dancer);
	loadimage ("midrange-treble", &(dancer->mt), dancer);
	// Because of spelling error in earlier versions we must
	// check to see if the theme has trebble.xpm instead
	// of treble.xpm
	if (dancer->treble.pixmap == NULL)
	{
		loadimage ("trebble", &(dancer->treble), dancer);
	}
	
	set_default_images ("bass", &(dancer->bass), dancer);
	set_default_images ("treble", &(dancer->treble), dancer);
	set_default_images ("midrange", &(dancer->midrange), dancer);
	set_default_images ("neutral", &(dancer->neutral), dancer);
	
	// Get size of pixmap
	gdk_window_get_size ((GdkWindow*) dancer->neutral.pixmap,
			&(dancer->width),
			&(dancer->height));	
	
	// Set window to the size gotten
	gtk_widget_set_usize (dancer->window, dancer->width, dancer->height);
	
	// If pixmap widget exists, set neutral image
	if (dancer->pixmap)
		gd_image_change (dancer, NEUTRAL, 1);
}

void move_dancer_random (GDancer *dancer) {
#define MOVE_AMOUNT 2
	gint random_number;
	if (dancer->move_window)
		return;
	random_number = rand();
	if (random_number > RAND_MAX/2) {
		//  move left
		dancer->x = dancer->x - MOVE_AMOUNT;
	} else {
		// move right
		dancer->x = dancer->x + MOVE_AMOUNT;
	}
	random_number = rand();
	if (random_number > RAND_MAX/2) {
		// move up
		dancer->y = dancer->y - MOVE_AMOUNT;
	} else {
		// move down
		dancer->y = dancer->y + MOVE_AMOUNT;
	}
	if (dancer->x < 0)
		dancer->x = dancer->x + (MOVE_AMOUNT*2);
	if (dancer->x > (gdk_screen_width() - dancer->width))
		dancer->x = dancer->x - (MOVE_AMOUNT*2);
	if (dancer->y < 0)
		dancer->y = dancer->y + (MOVE_AMOUNT*2);
	if (dancer->y > (gdk_screen_height() - dancer->height))
		dancer->y = dancer->y - (MOVE_AMOUNT*2);
	gd_move_window (dancer);
}

gint gd_image_change (GDancer *dancer, enum gd_freq_type newtype, gboolean always_draw)
{
	static frame *oldtype = NULL;
	frame *type;

	if (dancer->window == NULL) {
		printf ("What the hell, there is no window\n");
		return;
	}
	if (newtype == BASS) {
		switch (dancer->hadlast) {
			case BASS:
			case SEMI_BASS:
				type = &(dancer->bass);
				dancer->hadlast = BASS;
				break;
			case MIDRANGE:
			case SEMI_MIDRANGE:
				type = &(dancer->bm);
				dancer->hadlast = SEMI_BASS;
				break;
			case NEUTRAL:
			case SEMI_NEUTRAL:
				type = &(dancer->bn);
				dancer->hadlast = SEMI_BASS;
				break;
			case TREBLE:
			case SEMI_TREBLE:
				type = &(dancer->bt);
				dancer->hadlast = SEMI_BASS;
				break;
		}
	} else if (newtype == MIDRANGE) {
		switch (dancer->hadlast) {
			case BASS:
			case SEMI_BASS:
				type = &(dancer->bm);
				dancer->hadlast = SEMI_MIDRANGE;
				break;
			case MIDRANGE:
			case SEMI_MIDRANGE:
				type = &(dancer->midrange);
				dancer->hadlast = MIDRANGE;
				break;
			case NEUTRAL:
			case SEMI_NEUTRAL:
				type = &(dancer->mn);
				dancer->hadlast = SEMI_MIDRANGE;
				break;
			case TREBLE:
			case SEMI_TREBLE:
				type = &(dancer->nt);
				dancer->hadlast = SEMI_MIDRANGE;
				break;
		}
	} else if (newtype == NEUTRAL) {
		switch (dancer->hadlast) {
			case BASS:
			case SEMI_BASS:
				type = &(dancer->bn);
				dancer->hadlast = SEMI_NEUTRAL;
				break;
			case MIDRANGE:
			case SEMI_MIDRANGE:
				type = &(dancer->mn);
				dancer->hadlast = SEMI_NEUTRAL;
				break;
			case NEUTRAL:
			case SEMI_NEUTRAL:
				type = &(dancer->neutral);
				dancer->hadlast = NEUTRAL;
				break;
			case TREBLE:
			case SEMI_TREBLE:
				type = &(dancer->nt);
				dancer->hadlast = SEMI_NEUTRAL;
				break;
		}
	} else if (newtype == TREBLE) {
		switch (dancer->hadlast) {
			case BASS:
			case SEMI_BASS:
				type = &(dancer->bt);
				dancer->hadlast = SEMI_TREBLE;
				break;
			case MIDRANGE:
			case SEMI_MIDRANGE:
				type = &(dancer->mt);
				dancer->hadlast = SEMI_TREBLE;
				break;
			case NEUTRAL:
			case SEMI_NEUTRAL:
				type = &(dancer->nt);
				dancer->hadlast = SEMI_TREBLE;
				break;
			case TREBLE:
			case SEMI_TREBLE:
				type = &(dancer->treble);
				dancer->hadlast = TREBLE;
				break;
		}
	}
	else {
		printf ("Not sure what newtype is\n");
		return;
	}
					

	// if pixmap is NULL then there is no transition frame
	// so just skip the transition
	if (type->pixmap == NULL) {
		if (newtype == BASS) { // if frame is bass
			type = &(dancer->bass);
			dancer->hadlast = BASS;
		} else if (newtype == MIDRANGE) { // if frame is midrange
			type = &(dancer->midrange);
			dancer->hadlast = MIDRANGE;
		} else if (newtype == NEUTRAL) { // if frame is neutral
			type = &(dancer->neutral);
			dancer->hadlast = NEUTRAL;
		} else if (newtype == TREBLE) { // if frame is treble
			type = &(dancer->treble);
			dancer->hadlast = TREBLE;
		}
	}


	if (oldtype == type && !always_draw) {
#if DEBUGGING
		printf ("Same type as before, skipping\n");
#endif
		return; // Same as old image so leave
	}
	oldtype = type;
	
	if ((type->pixmap == NULL) || (type->mask == NULL)) {
		printf ("Ran into NULL image with hadlast %d and newtype %d\nThis should not happen, please send in a bug report\n", dancer->hadlast,newtype);
		return;
	}
	// make the window transparent where the image is transparent
		gtk_widget_shape_combine_mask (dancer->window,
				type->mask,
				0,
				0);
	// finally, draw the image
	gtk_pixmap_set (GTK_PIXMAP(dancer->pixmap),
			type->pixmap,
			type->mask);
	// move the window if enabled
	if (dancer->move_dancer)
		move_dancer_random (dancer);
}

gint gd_image_change_each (GDancer *dancer, enum gd_freq_type newtype)
{
	if (dancer->skip_count >= dancer->skip_frames) {
		gd_image_change (dancer, newtype, 0);
		dancer->skip_count = 0;
	} else {
		dancer->skip_count++;
	}
}
