package GCMainWindow;

###################################################
#
#  Copyright 2005-2006 Tian
#
#  This file is part of GCstar.
#
#  GCstar is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCstar is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCstar; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################

use utf8;
    
use strict;
use Gtk2;

{
    package GCFrame;
    use base "Gtk2::Window";

    use GCDialogs;
    use GCItemsLists;
    use GCPlugins;
    use GCModel;
    use GCGraphicComponents;
    use GCMenu;
    use GCPanel;
    use GCUtils 'glob';
    use GCOptions;
    use GCStyle;
    use GCBorrowings;
    use GCExtract;
    use GCExport;
    use GCImport;
    use GCSplash;
    use GCLang;
    use GCDisplay;
    use GCData;
    use GCMail;
    use GCBookmarks;
    #use GCGenres;

    use LWP;
    use URI::Escape;
    use File::Basename;
    use File::Spec;
    use File::Copy;
    use IO::Handle;
    use Storable qw(store_fd fd_retrieve);

    sub beforeDestroy
    {
		my $self = shift;
		$self->leave;
		return 1;
    }

    sub leave
    {
        my $self = shift;

        return if !$self->checkAndSave;

        my ($width, $height) = $self->get_size;
        $self->{options}->file('')
            if $self->{options}->noautoload;
        $self->{options}->width($width);
        $self->{options}->height($height);
        $self->{options}->split($self->{pane}->get_position) if ($self->{pane});
        
        $self->{options}->save;
        $self->{model}->save;

        $self->{items}->clean;

        if ($^O !~ /win32/i)
        {
            close $self->{searchJob}->{reader};
            close $self->{searchJob}->{writer};
            kill 9, $self->{searchJob}->{pid};
            wait;
        }
        
        $self->destroy;
    }

    sub deleteCurrentItem
    {
        my $self = shift;
		my $response = 'yes';
	    my $confirm = 0;

		if ($self->{options}->confirm)
		{
		    my  $dialog = Gtk2::MessageDialog->new($self,
							   [qw/modal destroy-with-parent/],
							   'question',
							   'cancel',
							   $self->{lang}->{RemoveConfirm});

            my $delButton = Gtk2::Button->new_from_stock('gtk-delete');
            $delButton->can_default(1);
            $dialog->add_action_widget($delButton, 'yes');
            $delButton->show_all;
            $dialog->set_default_response('yes');
            $dialog->set_position('center-on-parent');
            my $check = new Gtk2::CheckButton($self->{lang}->{OptionsDontAsk});
            $dialog->vbox->pack_start($check,0,0,5);
            $dialog->vbox->pack_start(Gtk2::HSeparator->new,0,0,5);
            $dialog->vbox->show_all;
		    $response = $dialog->run;
		    $confirm = ($check->get_active ? 0 : 1);
		    $dialog->destroy;
		}
	
		if ($response eq 'yes')
		{
            $self->markAsUpdated;
            $self->{items}->removeCurrentItem;
            $self->{options}->confirm($confirm);
		}
    }

    sub newItem
    {
			
        my ($self, $self2) = @_;
        $self = $self2 if $self2;
		
        #$self->{options}->lockPanel(0);
        # Uncomment to memorize
        #$self->{panel}->changeState($self->{panel}, 0);
        $self->{menubar}->setCollectionLock(0);
        
        my $info = $self->{model}->getInitInfo;
        $self->addItem($info, 1);
        
        $self->displayInWindow(undef, 0, 1)
            if $self->{panel}->isReadOnly;
    }

    sub duplicateItem
    {
			
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        my $needReload = 0;
        my $info = $self->{panel}->getAsHash;

        if ($info->{$self->{items}->{serieField}})
        {
            $info->{$self->{items}->{rankField}} = $self->{items}->getMaxRank($info->{$self->{items}->{serieField}}) + 1;
            $needReload = 1;
        }
        $self->addItem($info, 1);
        $self->{items}->reloadList if $needReload;
    }

    sub loadUrl
    {
        my ($self, $url) = @_;
        
        my $baseUrl;
        my $plugin;

        foreach (values %{$self->{model}->getAllPlugins})
        {
            ($baseUrl = $_->getItemUrl) =~ s/http:\/\/(w{3})?//;
            $plugin = $_;
            last if $url =~ m/$baseUrl/;
        }
        return unless $url =~ m/$baseUrl/;
        $self->setWaitCursor($self->{lang}->{StatusGetInfo});
        $self->newItem;
        my $info = $plugin->loadUrl($url);
        $self->restoreCursor;
        return if ! $info->{$self->{model}->{commonFields}->{title}};        
        $self->addItem($info, 0);
    }
    
    sub searchItemForPanel
    {
        my ($self, $panel) = @_;
    
        if ($panel)
        {
            $self->{previousPanel} = $self->{panel};
            $self->{panel} = $panel;
        }
        my $query;
        my $field;
        foreach (@{$self->{model}->getSearchFields})
        {
            $query = $self->{panel}->getValue($_);
            ($field = $_, last) if ($query);
        }
        $self->searchItem($query, 0, 0, $field);
        $self->{panel} = $self->{previousPanel} if ($panel);
    }
    
    sub searchItem
    {
        my ($self, $query, $forceask, $currentPlugin, $searchField) = @_;

        my $plugin;

        $self->{previewCache} = {};

        if (($self->{model}->{preferences}->plugin eq 'multi') && !$forceask)
        {
            my $pluginName = $self->getDialog('MultiSite')->getPlugin($currentPlugin);
            $plugin = $self->{model}->getPlugin($pluginName);
            if (!$plugin)
            {
                my $dialog = Gtk2::MessageDialog->new($self,
							   [qw/modal destroy-with-parent/],
							   'error',
							   'ok',
							   $self->{lang}->{MultiSiteEmptyError});

                $dialog->set_position('center-on-parent');
                my $response = $dialog->run;
                $dialog->destroy;
                return;
            }
        }
        elsif ($self->{model}->{preferences}->plugin eq 'multiask')
        {
            $self->getDialog('PluginsAsk')->setModel($self->{model},
                                                     $self->{model}->{preferences}->multisite);
            $self->getDialog('PluginsAsk')->query($query, $searchField);
            $self->getDialog('PluginsAsk')->show;
            $plugin = $self->getDialog('PluginsAsk')->plugin;
            return if !$plugin;
            ($query, $searchField) = $self->getDialog('PluginsAsk')->query;

        }
        elsif ($forceask || ! $self->{plugin})
        {
            $self->getDialog('Plugins')->query($query, $searchField);
            $self->getDialog('Plugins')->show;
            $plugin = $self->getDialog('Plugins')->plugin;
            return if !$plugin;
            ($query, $searchField) = $self->getDialog('Plugins')->query;
            
        }
        else
        {
            $plugin = $self->{plugin};
        }
        
        if (! $self->{options}->searchStop)
        {
            $plugin->setProxy($self->{options}->proxy);
    
            $plugin->{title} = $query;
            $plugin->{type} = 'load';
            $plugin->{urlField} = $self->{model}->{commonFields}->{url};
            $plugin->{searchField} = $searchField;
            
            $self->setWaitCursor($self->{lang}->{StatusSearch}.' ('.$plugin->getName.')');
            $plugin->load;
            $self->restoreCursor;
        }
        else
        {
            my $info = {
                            name => $plugin->getName,
                            model => $self->{model}->getName,
                            proxy => $self->{options}->proxy,
                            query => $query,
                            field => $searchField,
                            type  => 'load',
                            urlField => $self->{model}->{commonFields}->{url}
                       };
            store_fd $info, $self->{searchJob}->{command};
            my $getInfo = 0;
            my $dialog = Gtk2::MessageDialog->new($self,
                [qw/modal destroy-with-parent/],
                'info',
                'cancel',
                $self->{lang}->{StatusSearch} . "\n" . $plugin->getName);
                
            my $progress = new Gtk2::ProgressBar;
            $dialog->vbox->pack_start($progress,1,1,10);
            $progress->set_pulse_step(0.05);
            $progress->show_all;
            my $pulseTimeout = Glib::Timeout->add(50 , sub {
                return 0 if $getInfo;
                $progress->pulse;
                return 1;
            });    
                
            my $watch = Glib::IO->add_watch(fileno($self->{searchJob}->{data}),
                                'in',
                                sub {
                                    return if !$dialog;
                                    $dialog->response('cancel');
                                    $getInfo = 1;
                                    return 0;
                                });

            $dialog->set_position('center-on-parent');
            $dialog->run if !$getInfo;
            Glib::Source->remove($watch);
            Glib::Source->remove($pulseTimeout);
            $dialog->destroy;
            my $command = $self->{searchJob}->{command};
            if ($getInfo)
            {
                $plugin = fd_retrieve($self->{searchJob}->{data});
                print $command "OK\n";
            }
            else
            {
                print $command "STOP\n";
                return;
            }
        }
            
		my $itemNumber = $plugin->getItemsNumber();
	
		if ($itemNumber == 0)
		{
		    my $force = 0;
		    my $idx = 0;
		    if (!$forceask && ($self->{model}->{preferences}->plugin eq 'multi')
		     && (($currentPlugin + 1) < $self->getDialog('MultiSite')->getPluginsNumber))
		    {
		      $idx = $currentPlugin + 1;
		    }
		    else
		    {
		      my  $dialog = Gtk2::MessageDialog->new($self,
							   [qw/modal destroy-with-parent/],
							   'error',
							   'yes-no',
							   $self->{lang}->{NoItemFound});

              $dialog->set_position('center-on-parent');
		      my $response = $dialog->run;
		      $dialog->destroy;
		      return if $response ne 'yes';
		      $force = 1;
		    }
		    
            $self->searchItem($query, $force, $idx, $searchField);
		}
		elsif ($itemNumber == 1)
		{
		    $self->getItemInfo($plugin, 0);
		}
		else
		{
		    my $withNext = 0;
            $withNext = 1 if ($self->{model}->{preferences}->plugin eq 'multi')
		        && (($currentPlugin + 1) < $self->getDialog('MultiSite')->getPluginsNumber);
		    
		    my @items = $plugin->getItems();
		    my $resultsDialog = $self->getDialog('Results');
		    $resultsDialog->setModel($self->{model}, $self->{model}->{fieldsInfo});
            $resultsDialog->setWithNext($withNext);
            $resultsDialog->setSearchPlugin($plugin);
		    $resultsDialog->setList('', @items);
		    my $next = $resultsDialog->show;
		    if ($resultsDialog->{validated})
		    {
				my $idx = $resultsDialog->{itemIndex};
				$self->getItemInfo($plugin, $idx);
		    }
		    elsif ($next)
		    {
		      $self->searchItem($query, 0, $currentPlugin + 1, $searchField);
		    }
		}
    }

    sub getItemInfo
    {
        my ($self, $plugin, $idx, $withPreview) = @_;

        my $info;
        
        # Gets from cache if we already previewed it
        if ($self->{previewCache}->{$idx})
        {
            $info = $self->{previewCache}->{$idx};
        }
        else
        {
            $plugin->{wantedIdx} = $idx;
            $plugin->{type} = 'info';
            
            if (! $self->{options}->searchStop)
            {
                $self->setWaitCursor($self->{lang}->{StatusGetInfo});
                $info = $plugin->getItemInfo;
                $self->restoreCursor;
            }
            else
            {
                store_fd $plugin, $self->{searchJob}->{command};
                my $getInfo = 0;
                my $dialogGet = Gtk2::MessageDialog->new($self,
                    [qw/modal destroy-with-parent/],
                    'info',
                    'cancel',
                    $self->{lang}->{StatusGetInfo});
                    
                my $progress = new Gtk2::ProgressBar;
                $dialogGet->vbox->pack_start($progress,1,1,10);
                $progress->set_pulse_step(0.05);
                $progress->show_all;
                my $pulseTimeout = Glib::Timeout->add(50 , sub {
                    return 0 if $getInfo;
                    $progress->pulse;
                    return 1;
                });
                    
                my $watch = Glib::IO->add_watch(fileno($self->{searchJob}->{data}),
                                    'in',
                                    sub {
                                        return if !$dialogGet;
                                        $dialogGet->response('cancel');
                                        $getInfo = 1;
                                        return 0;
                                    });
    
                $dialogGet->set_position('center-on-parent');
                $dialogGet->run if !$getInfo;
                Glib::Source->remove($watch);
                Glib::Source->remove($pulseTimeout);
                $dialogGet->destroy;
                my $command = $self->{searchJob}->{command};
                if ($getInfo)
                {
                    $info = fd_retrieve($self->{searchJob}->{data});
                    print $command "OK\n";
                }
                else
                {
                    print $command "STOP\n";
                    return;
                }
            }
        }
        #$info->{original} = $info->{title} if (!$info->{original}) && ($self->{options}->alwaysOriginal);
        #$info->{image} = '' if $info->{image} eq 'empty';        
        
        if ($withPreview)
        {
            # Stores in cache
            $self->{previewCache}->{$idx} = $info;
            $self->getDialog('ImportFields')->info($info);
            $self->getDialog('ImportFields')->setReadOnly(1);
            $self->getDialog('ImportFields')->show;
        }
        else
        {
            if ($self->{options}->askImport)
            {
                $self->getDialog('ImportFields')->info($info);
                $self->getDialog('ImportFields')->setReadOnly(0);
                return if ! $self->getDialog('ImportFields')->show;
                $info = $self->getDialog('ImportFields')->info;
            }
            $self->addItem($info, 0);
        }
    }
    
    sub getUniqueImageFileName
    {
        my ($self, $suffix, $itemTitle, $imagesDir) = @_;
    
        $imagesDir ||= $self->{options}->images;
        $imagesDir =~ s+([^/])$+$1/+;
        my $imagePrefix;
        if ($self->{options}->useTitleForPics)
        {
            $imagePrefix = GCUtils::getSafeFileName($itemTitle);
            $imagePrefix .= '_';
        }
        else
        {
            $imagePrefix = $self->{imagePrefix};
        }

        if ( ! -e $imagesDir)
        {
            mkdir($imagesDir);
        }

        my $filePrefix = $imagesDir.$imagePrefix;
        my @tmp_filenames;
        @tmp_filenames = glob $filePrefix.'*.*';
        my $sysPrefix = $filePrefix;
        $sysPrefix =~ s/\\/\\\\/g if ($^O =~ /win32/i);
        my @numbers = sort {$b <=> $a} map {
            /$sysPrefix([0-9]*)\./ && $1;
        } @tmp_filenames;
        my $mostRecent = $numbers[0] || 0;

        my $picture = $filePrefix.$mostRecent.$suffix;

        while (-e $picture)
        {
            $mostRecent++;
            $picture = $filePrefix.$mostRecent.$suffix;
        }
        return $picture;
    }
    
#    sub changeInfo
#    {
#        my ($self, $info) = @_;
#
#        my @genres = split /,/, $info->{type};
#        my $newGenres = '';
#        
#        foreach (@genres)
#        {
#            $newGenres .= $self->getDialog('GenresGroups')->{convertor}->convert($_).',';
#        }
#        $newGenres =~ s/.$//;
#        
#        $info->{type} = $newGenres;
#    }
    
    sub addItem
    {
        my ($self, $info, $newItem) = @_;

        if ($newItem)
        {
            $self->{items}->updateCurrent;
            $self->markAsUpdated;
        }
        
        #$self->changeInfo($info);

        my $ignore = $self->{ignoreString};
        my $previous;
        
        if ($newItem)
        {
            for my $field (@{$self->{model}->{fieldsNames}})
            {
                $self->{panel}->$field($info->{$field});
            }
            $self->{panel}->selectTitle;
            my $id = $self->{items}->addItem($info);
            my $idField = $self->{model}->{commonFields}->{id};
            $self->{panel}->$idField($id);
        }
        else
        {
            $previous = $self->{panel}->getAsHash;
            my @picFields = ();
            foreach my $field(@{$self->{model}->{fieldsNames}})
            {
                if ($self->{model}->{fieldsInfo}->{$field}->{type} eq 'image')
                {
                    push @picFields, $field;
                    next;
                }
                #next if ($field eq $cover);
                if (($self->{model}->{fieldsInfo}->{$field}->{imported} ne 'true')
                 || ($info->{$field} eq $ignore))
                {
                    $info->{$field} = $previous->{$field};
                }
                else
                {
                    $self->{panel}->$field($info->{$field});
                }
            }
            my $title = $info->{$self->{model}->{commonFields}->{title}};
            my $imagePrefix = $self->{imagePrefix};
            foreach my $pic(@picFields)
            {
                if ($info->{$pic} && ($info->{$pic} ne $ignore))
                {
                    $self->{items}->markToBeRemoved($self->{panel}->$pic)
                        if ($self->{panel}->$pic =~ /(\/|\\)$imagePrefix[0-9]*\./);                    
                    
                    ($info->{$pic}, my $picture) = $self->downloadPicture($info->{$pic}, $title);                
                    $self->{panel}->$pic($info->{$pic});
                    $self->restoreCursor;
                }
                $info->{$pic} = $previous->{$pic} if $info->{$pic} eq $ignore;
            }
                        
            $self->{items}->updateCurrent;
            $self->{itemsList}->changeCurrent($previous, $info) if ($self->{panel}->{autoUpdate});
            $self->{itemsList}->showCurrent;
        }
        $self->{panel}->dataChanged;
    }
    
    sub downloadPicture
    {
        my ($self, $pictureUrl, $title) = @_;
    
        my ($name,$path,$suffix) = fileparse($pictureUrl, "\.gif", "\.jpg", "\.jpeg", "\.png");
        my $picture = $self->getUniqueImageFileName($suffix, $title);

        if ($pictureUrl =~ m|^http://|)
        {
            $self->setWaitCursor($self->{lang}->{StatusGetImage});
            my $browser = LWP::UserAgent->new;
            $browser->proxy(['http'], $self->{options}->proxy);
            $browser->agent($self->{agent});
    
            $browser->get($pictureUrl, ':content_file' => $picture);
            $self->restoreCursor;
        }
        else
        {
            copy $pictureUrl, $picture;
            unlink $pictureUrl;
        }
        $self->{items}->markToBeAdded($picture);
        return ($self->transformPicturePath($picture), $picture);
    }

    sub transformPicturePath
    {
        my ($self, $path, $file) = @_;

        return '' if !$path;
        $file ||= $self->{options}->file;
        $path = GCUtils::getDisplayedImage($path, '', $file);
        my $dir = undef;
        $dir = dirname($file) if $file;
        return File::Spec->rel2abs($path,$dir)
            if !$self->{options}->useRelativePaths;
        return File::Spec->abs2rel($path,$dir);
    }

    sub addBookmark
    {
        my $self = shift;
        
        my $dialog = $self->getDialog('BookmarkAdder');
        $dialog->setBookmark($self->{options}->file, $self->{items}->{information}->{name});
        $dialog->setBookmarksFolders($self->{bookmarksLoader}->{bookmarks});
        if ($dialog->show)
        {
            $self->{bookmarksLoader}->save($dialog->getBookmarks);
        }
    }

    sub editBookmark
    {
        my $self = shift;
        
        #my $dialog = $self->getDialog('BookmarksEdit');
        my $dialog = new GCBookmarksEditDialog($self);
        
        $dialog->setBookmarksFolders($self->{bookmarksLoader}->{bookmarks});
        if ($dialog->show)
        {
            $self->{bookmarksLoader}->save($dialog->getBookmarks);
        }
    }

    sub addFileHistory
    {
        my ($self, $filename) = @_;
        (my $history = $self->{options}->history) =~ s/\|?\Q$filename\E\|?/|/;
        $history =~ s/(.*?)\|[^\|]*?$/$filename\|$1/ if $history !~ /^\Q$filename\E/;
        $self->{options}->history($history);
        $self->{menubar}->{menuHistoryItem}->remove_submenu();
        $self->{menubar}->{menuHistory} = Gtk2::Menu->new();
        $self->{menubar}->addHistoryMenu();
    }
    
    sub openFile
    {
        my ($self, $filename) = @_;
        return if !$self->checkAndSave;

        $self->initProgress($self->{lang}->{StatusLoad});
        $self->{openingFile} = 1;
        if ($self->{items}->load($filename, $self, 1))
        {
            $self->addFileHistory($filename);
            $self->{options}->file($filename);
            $self->{options}->save;
            $self->selectFirst;
            $self->setTitle(basename($filename), $self->{items}->{information}->{name});
            $self->{menubar}->setLock($self->{items}->getLock);
        }
        else
        {
            my $dialog = Gtk2::MessageDialog->new($self,
                                          [qw/modal destroy-with-parent/],
                                          'error',
                                          'ok',
                                          $self->{lang}->{OpenFormatError});
            $dialog->set_position('center-on-parent');
            $dialog->run();
            $dialog->destroy ;
        }
        $self->{openingFile} = 0;
    }

    sub openList
    {
        my $self = shift;
        my $fileDialog = new GCFileChooserDialog($self->{lang}->{OpenList}, $self, 'open', 1);
        $fileDialog->set_pattern_filter(['GCstar (.gcs)', '*.gcs']);

        $fileDialog->set_filename($self->{options}->file);
        my $response = $fileDialog->run;
        if ($response eq 'ok')
        {
            my $fileName = $fileDialog->get_filename;
            $fileDialog->destroy;
            $self->openFile($fileName);
        }
        else
        {
            $fileDialog->destroy;
        }
    }

    sub newList
    {
        my ($self, $modelName, $modelAlreadySet) = @_;

        return if !$self->checkAndSave;

        if (!$modelAlreadySet)
        {
            if ($modelName)
            {
                $self->setCurrentModel(
                    $self->{modelsFactory}->getModel($modelName)
                );
            }
            else
            {
                return if ! $self->getDialog('Models')->show;
                my $model = $self->getDialog('Models')->getModel;
                if ($model->isEmpty)
                {
                    $self->{model} = $model;
                    $self->editModel;
                }
                else
                {
                    $self->setCurrentModel($model);
                }
            }
        }

        $self->{items}->clearList;
        $self->{options}->file('');
        $self->setTitle('');
    }

    sub saveAs
    {
        my $self = shift;
        my $fileDialog = new GCFileChooserDialog($self->{lang}->{SaveList}, $self, 'save', 1, 1);
        $fileDialog->set_pattern_filter(['GCstar (.gcs)', '*.gcs']);
        $fileDialog->set_filename($self->{options}->file);
            
        while (1)
        {
    	   	my $response = $fileDialog->run;
            if ($response eq 'ok')
            {
                my $filename = $fileDialog->get_filename;
                $self->addFileHistory($filename);
                my $previousFile = $self->{options}->file;
                $self->{options}->file($filename);
                if ($self->saveList())
                {
                    $self->{options}->save;
                    $self->setTitle(basename($filename), $self->{items}->{information}->{name});
                    last;
                }
                else
                {
                    $self->{options}->file($previousFile);
                }            
    		}
    		last if ($response ne 'ok')
        }
        $fileDialog->destroy;
    } 

    sub checkAndSave
    {
        my $self = shift;
        # Return value 1 means everything is OK.
        # 0 means the user clicked cancel for save confirmation and then
        # the process should be stopped.
        return 1 if !$self->{items}->{nbItems};
        if ($self->{options}->autosave)
        {
            $self->saveList;
            return 1;
        }
        $self->{items}->updateCurrent;
        return 1 if !$self->{saveIsNeeded};
 	    my $dialog = Gtk2::MessageDialog->new($self,
                                              [qw/modal destroy-with-parent/],
                                              'warning',
                                              'none',
                                              $self->{lang}->{SaveUnsavedChanges});
        my $noButton = Gtk2::Button->new($self->{lang}->{SaveDontSave});
        $dialog->add_action_widget($noButton, 'no');
        my $cancelButton = Gtk2::Button->new_from_stock('gtk-cancel');
        $dialog->add_action_widget($cancelButton, 'cancel');
        my $saveButton = Gtk2::Button->new_from_stock('gtk-save');
        $saveButton->can_default(1);
        $dialog->add_action_widget($saveButton, 'yes');
        $noButton->show_all;
        $cancelButton->show_all;
        $saveButton->show_all;
        $dialog->set_default_response('yes');
        $dialog->set_position('center-on-parent');
        my $response = $dialog->run();
        $dialog->destroy;
        ($response eq 'yes') ? $self->saveList :
        ($response eq 'cancel') ? return 0 :
        $self->removeUpdatedMark;
        return 1;
    }

    sub saveList
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;
        my $response = 'yes';

        if ($self->{options}->file)
        {
            return $self->{items}->save($self);
        }
        else
        {
	       my $dialog = Gtk2::MessageDialog->new($self,
                                                  [qw/modal destroy-with-parent/],
                                                  'info',
                                                  'ok-cancel',
                                                  $self->{lang}->{SaveCurrentList});

           $dialog->set_position('center-on-parent');
	       $response = $dialog->run();
	       $dialog->destroy ;
		   $self->saveAs if ($response eq 'ok');
        }
    }

    sub selectFirst
    {
        my $self = shift;

        $self->{items}->display($self->{items}->select(-1));        
    }

    sub removeSearch
    {
        my $self = shift;
        
        $self->setFilter('', '');
    }
    
    sub search
    {
        my ($self, $self2, $value) = @_;
        $self = $self2 if ($self2 ne 'all') && ($self2 ne 'displayed');
        my $type = 'all';
        $type = $value if ($self != $self2);

        $self->getDialog('Search')->show;
        
        my $info = $self->getDialog('Search')->search;
        return if ! $info;
        
        $self->{menubar}->selectAll if $type eq 'all';
        $self->{items}->setSearch($info);
    }
    
    sub advancedSearch
    {
        my $self = shift;
        $self->getDialog('AdvancedSearch')->show;
        my $info = $self->getDialog('AdvancedSearch')->search;
        return if ! $info;
        $self->{items}->setSearchWithTypes($info, $self->getDialog('AdvancedSearch')->getMode);
    }
    
    sub showBorrowed
    {
        my $self = shift;
        $self->{items}->updateCurrent;
        $self->getDialog('Borrowed')->setList($self->{items}->{itemArray});
        $self->getDialog('Borrowed')->show;
    }
    
    sub export
    {
        my ($self, $exporter) = @_;
        $self->getDialog('Export')->setModule($exporter);
        $self->getDialog('Export')->show;
    }

    sub import
    {
        my ($self, $importer) = @_;
        $self->getDialog('Import')->setModule($importer);
        $self->getDialog('Import')->show;
    }

    sub importWithDetect
    {
        my ($self, $file, $realPath) = @_;

        if (!$realPath)
        {
            $file =~ s/^file:\/\/(.*)\W*$/$1/;
            $file =~ s/.$//ms;
        }
        
        foreach my $importer(@GCImport::importersArray)
        {
            my $current = $importer->getSuffix;
            next if !$current;
            #if ($current eq $suffix)
            if ($file =~ /$current$/)
            {
                $self->setWaitCursor($self->{lang}->{StatusGetInfo});
                my %options;
                $options{parent} = $self;
                $options{newList} = 0;
                $options{file} = $file;
                $options{lang} = $self->{lang};
                $importer->process(\%options);
                #$self->restoreCursor;
                return 1;
            }
        }
        return 0;
    }

    sub optionsError
    {
        my ($self, $type) = shift;

        my $msg;
        if ($type eq 'open')
        {
            $msg = $self->{lang}->{OptionsOpenError};
        }
        elsif ($type eq 'create')
        {
            $msg = $self->{lang}->{OptionsCreateError};
        }
        else
        {
            $msg = $self->{lang}->{OptionsSaveError};
        }
            
        my $dialog = Gtk2::MessageDialog->new($self,
                                              [qw/modal destroy-with-parent/],
                                              'error',
                                              'ok',
                                              $msg.$self->{options}->{file});

        $dialog->set_position('center-on-parent');
        $dialog->run();
        $dialog->destroy ;

        $self->destroy;
    }

    sub checkImagesDirectory
    {
        my ($self, $withDialog) = @_;
        my $error = 0;
        if ( ! -e $self->{options}->images)
        {
            $! = '';
            mkdir($self->{options}->images);
            $error = 1 if $!;
        }
        else
        {
            $error = 1 if !( -d _ && -r _ && -w _ && -x _ )
        }
        return $error if !$withDialog;
        if ($error)
        {
            $self->{splash}->hide if $self->{splash};
            my $fileDialog = new GCFileChooserDialog($self->{lang}->{OptionsImagesTitle}, $self, 'select-folder');
            my  $errorDialog = Gtk2::MessageDialog->new($self,
						   [qw/modal destroy-with-parent/],
						   'error',
						   'ok',
						   $self->{lang}->{ImageError});

            $errorDialog->set_position('center-on-parent');
            $fileDialog->set_filename($self->{options}->images);
            my $response;
            do
            {
                $errorDialog->run();
                $errorDialog->hide();
                $response = '';
                $response = $fileDialog->run;
                exit 1 if $response ne 'ok';
                $self->{options}->images($fileDialog->get_filename);
            } while ($self->checkImagesDirectory(0));
            $errorDialog->destroy;
            $fileDialog->destroy;
        }
        return $error;
    }

    sub setFileName
    {
        my ($self, $file) = @_;
        $self->setTitle(basename($file));
    }

    sub setTitle
    {
        my ($self, $filename, $name) = @_;
        $name ||= $filename;
        $self->{windowTitle} = 'GCstar - ['.$name.']';
        $self->set_title($self->{windowTitle});
    }

    sub properties
    {
        my $self = shift;
        
        my $dialog = $self->getDialog('Properties');
        $dialog->setProperties($self->{items}->{information});
        if ($dialog->show)
        {
            $self->{items}->{information} = $dialog->getProperties;
            $self->setTitle(basename($self->{options}->file), $self->{items}->{information}->{name});
            $self->markAsUpdated;
        }
    }

    sub markAsUpdated
    {
        my $self = shift;
        $self->{saveIsNeeded} = 1;
        $self->set_title($self->{windowTitle}.' *');
        $self->{menubar}->setSaveActive(1);
        $self->{toolbar}->setSaveActive(1);
    }

    sub removeUpdatedMark
    {
        my $self = shift;
        $self->{saveIsNeeded} = 0;
        $self->set_title($self->{windowTitle});
        $self->{menubar}->setSaveActive(0);
        $self->{toolbar}->setSaveActive(0);
    }

    sub refreshFilters
    {
        my $self = shift;
        $self->{menubar}->refreshFilters;
    }

    sub setWaitCursor
    {
		my ($self, $message) = @_;
        $self->setStatus($message);
		$self->window->set_cursor(Gtk2::Gdk::Cursor->new('watch'));
		Gtk2->main_iteration while (Gtk2->events_pending);
    }
    sub restoreCursor
    {
		my $self = shift;
        $self->restoreStatus;
		$self->window->set_cursor(Gtk2::Gdk::Cursor->new('left_ptr'));
		Gtk2->main_iteration while (Gtk2->events_pending);
    }

    sub setFilter
    {
        my ($self, $filter, $parameter) = @_;
        $self->{items}->updateCurrent;
        $self->{items}->setFilter($filter, $parameter);
    }

    sub onSortClick
    {
        my ($widget, $self) = @_;
        $self->{itemsList}->getSortButton()->signal_connect('clicked', \&onSortClick, $self)
            if ($self->{itemsList}->setSortOrder($self->{items}->changeOrder));
    }

    sub getDialog
    {
        my ($self, $name) = @_;

        if ($name eq 'PluginsAsk')
        {
            $self->{PluginsAskDialog} = new GCPluginsDialog($self)
                if !$self->{PluginsAskDialog};
        }
        elsif ($name eq 'MultiSite')
        {
            $self->{MultiSiteDialog}->{$self->{model}->getName} = new GCMultiSiteDialog($self, $self->{model})
                if !$self->{MultiSiteDialog}->{$self->{model}->getName};
            return $self->{MultiSiteDialog}->{$self->{model}->getName};
        }
        elsif ($name eq 'About')
        {
            $self->{AboutDialog} = new GCAboutDialog($self, $self->{version})
                if ! $self->{AboutDialog};
        }
        elsif ($name eq 'Models')
        {
            $self->{ModelsDialog} = new GCModelsDialog($self,
                                                       $self->{modelsFactory})
                if ! $self->{ModelsDialog};
        }
        else
        {
            my $className = 'GC'.$name.'Dialog';
            if (! $self->{$name.'Dialog'})
            {
                $self->{$name.'Dialog'} = new $className($self);
                # Actually Plugins, Export and Import don't need the parameter, but
                # it has no impact and make the code more simple
                $self->{$name.'Dialog'}->setModel($self->{model})
                    if $name =~ /^(AdvancedSearch|Search|Options|QueryReplace|Plugins|Export|Import)$/;
            }
        }

        return $self->{$name.'Dialog'};
    }

    sub selectMultiPluginsList
    {
        my $self = shift;
        
        $self->getDialog('MultiSite')->show;
    }

    sub options
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        my $transform = $self->{options}->transform;
        my $articles = $self->{options}->articles;
        my $formats = $self->{options}->formats;
        my $layout = $self->{model}->{preferences}->layout;
        my $panelStyle = $self->{options}->panelStyle;
        my $status = $self->{options}->status;
        my $toolbar = $self->{options}->toolbar;
        my $toolbarPosition = $self->{options}->toolbarPosition;

        $self->getDialog('Options')->show;

        $self->checkTransform
            if ($self->{options}->articles ne $articles) || ($self->{options}->transform != $transform);
            
        $self->changePanel
            if ($self->{model}->{preferences}->layout ne $layout)
            || ($self->{options}->panelStyle ne $panelStyle);
            
        $self->checkDisplayed
            if ($self->{options}->status ne $status) || ($self->{options}->toolbar ne $toolbar);

        $self->checkToolbarPosition
            if ($self->{options}->toolbarPosition ne $toolbarPosition);
            
        $self->checkPlugin;
        $self->checkView;
    }
    
    sub displayOptions
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        my $dialog = $self->getDialog('DisplayOptions');
        my $hidden = $self->{model}->{preferences}->hidden;
        $dialog->show;
        $self->checkPanel
            if ($self->{model}->{preferences}->hidden ne $hidden);
    }

    sub setCurrentModel
    {
        my ($self, $model) = @_;

        $model = 'GCfilms' if ! $model;
        $self->{model}->save if $self->{model};
        if ((ref $model) =~ /GCModelLoader/)
        {
            $self->{model} = $model;
        }
        else
        {
            $self->{model} = $self->{modelsFactory}->getModel($model)
        }
        $self->{items}->initModel($self->{model});
        return $self->{model};
    }
    
    sub setCurrentModelFromInline
    {
        my ($self, $container) = @_;
        my $model = GCModelLoader->newFromInline($self, $container);
        $self->setCurrentModel($model);
    }

    sub setGrouping
    {
        my ($self, $field) = @_;
        $self->{options}->view(2);
        $self->{model}->{preferences}->groupBy($field);
        $self->{model}->{preferences}->masterType('generated');
        $self->setItemsList(0);
    }

    sub setLock
    {
        my ($self, $value) = @_;
    
        $self->{items}->setLock($value);
        $self->{panel}->changeState($self->{panel}, $value);
        $self->markAsUpdated;
    }

    sub notifyModelChange
    {
        my $self = shift;

        #We deactivate some updates
        my $previousInitializing = $self->{initializing};
        $self->{initializing} = 1;
        $self->setItemsList(1);
        $self->getDialog('DisplayOptions')->createContent($self->{model});
        $self->changePanel(1);
        if (%{$self->{panel}})
        {
            $self->{panel}->createContent($self->{model});
            $self->{panel}->deactivate if $self->{items}->getLock;
        }
        $self->{menubar}->setModel($self->{model});
        $self->{toolbar}->setModel($self->{model});
        $self->checkToolbarOptions;        

        $self->getDialog('Search')->setModel($self->{model})
            if $self->{SearchDialog};
        $self->getDialog('Options')->setModel($self->{model})
            if $self->{OptionsDialog};
        $self->getDialog('QueryReplace')->setModel($self->{model})
            if $self->{QueryReplaceDialog};
        $self->getDialog('Plugins')->setModel
            if $self->{PluginsDialog};
        $self->getDialog('Export')->setModel
            if $self->{ExportDialog};
        $self->getDialog('Import')->setModel
            if $self->{ImportDialog};


        $self->checkPlugin;
        
        $self->{initializing} = $previousInitializing;
    }

    sub editModel
    {
        my $self = shift;

        my $dialog = $self->getDialog('ModelsSettings');
        $dialog->setDescription($self->{model}->getDescription);
        $dialog->initFields($self->{model}->getOriginalCollection->{fields}->{field},
                            $self->{model}->getGroups,
                            $self->{model}->getCommonFields,
                            $self->{model}->{hasLending});
        $dialog->initFilters($self->{model}->{filters});
        if ($dialog->show)
        {
            $self->markAsUpdated;
            my $file;
            ($self->{model}->{collection}->{description}, $self->{model}->{collection}->{name}, $file)
             = $dialog->getName;

            my %fields = (
                id => $dialog->getIdField,
                title => $dialog->getTitleField,
                cover => $dialog->getCoverField,
                play => $dialog->getPlayField,
            );
            $self->{model}->setOptions(\%fields, 'no.png');

            #$self->{model}->setDefaultImage('no.png');
            $self->{model}->setGroups($dialog->getGroups);
            $self->{model}->setFields($dialog->{fields}, $dialog->hasLending);
            $self->{model}->setFilters($dialog->getFilters($self->{model}->{fieldsInfo}));
            if ($self->{model}->{collection}->{name})
            {
                $self->{model}->saveToFile($file);
                # This is done twice, because the 1st time the model thought it is online
                $self->{model}->loadPreferences;
            }
            #$self->notifyModelChange;
            $self->{items}->initModel($self->{model});
            $self->reloadList;
        }
    }

    sub borrowers
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        $self->getDialog('Borrowers')->show;
        $self->checkBorrowers;
    }

#    sub genresConversion
#    {
#        my ($self, $self2) = @_;
#        $self = $self2 if $self2;
#
#        $self->getDialog('GenresGroups')->show;
#    }
    
    sub queryReplace
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        if ($self->getDialog('QueryReplace')->show)
        {
            $self->{items}->queryReplace($self->getDialog('QueryReplace')->{field},
                                         $self->getDialog('QueryReplace')->{oldValue},
                                         $self->getDialog('QueryReplace')->{newValue});
        }
    }

    sub about
    {
        my $self = shift;
        
        $self->getDialog('About')->show;
    }

    sub showDependencies
    {
        my $self = shift;
        
        $self->getDialog('Dependencies')->show;
    }

    sub help
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        $self->launch('http://www.gcstar.org/doc', 'url');
    }

    sub reportBug
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        $self->launch('https://gna.org/bugs/?func=additem&group=gcstar', 'url');
    }

    sub playItem
    {
        my $self = shift;
        my $field = $self->{model}->{commonFields}->{play};
        my $format = $self->{model}->{fieldsInfo}->{$field}->{format};
        $self->launch($self->{panel}->$field, $format);
    }

    sub launch
    {
        my ($self, $file, $format, $withParams) = @_;
        
        $format ||= 'program';
        # For image, we have an internal viewer
        if ($format eq 'image')
        {
            my $dialog = new GCImageDialog($self, $file);
            $dialog->show;
            $dialog->destroy;
            $self->{itemWindow}->showMe if $self->{itemWindow};
            return;
        }
        
        # For the other ones, we use external programs
        my $command;
        $command = $self->{options}->browser if $format eq 'url';
        $command = $self->{options}->player if $format eq 'video';
        $command = $self->{options}->audio if $format eq 'audio';
        if ($^O =~ /win32/i)
        {
            #"
            if ($format eq 'program')
            {
                if ($withParams)
                {
                    $command = $file;
                }
                else
                {
                    $file =~ s/([^\\])"/$1\\"/g;
                    $command = '"'.$file.'"';
                }
                $command = '"" '.$command ;
            }
            else
            {
                $file =~ s/([^\\])"/$1\\"/g;
                $command .= ' "'.$file.'"';
            }
            system "start $command";
        }
        else
        {
            if ($format ne 'program')
            {
                $file =~ s/([^\\])"/$1\\"/g;
                #"
                $command .= ' "'.$file.'"';
            }
            else
            {
                $command = $file;
            }
            system "$command &";
        }
    }

    sub sendBorrowerEmail
    {
        my ($self, $info) = @_;

        $self->{mailer} = new GCMailer($self) if ! $self->{mailer};
        $self->{mailer}->sendBorrowerEmail($info);
    }

    sub extractInfo
    {
        my ($self, $file, $panel) = @_;
        
        my $infoExtractor = new GCVideoInfo($file);
        $self->getDialog('Extract')->setInfo($infoExtractor->getInfo, $panel);
        $self->getDialog('Extract')->show;
    }

    sub setStatus
    {
		my ($self, $status) = @_;
		$self->{status}->push(1, $status) if ($self->{status});
    }
    
    sub restoreStatus
    {
        my $self = shift;
        $self->{status}->pop(1);
    }

    sub display
    {
        my ($self, $idx) = @_;
        return $self->{items}->display($idx);
    }
    
    sub displayInWindow
    {
        my ($self, $idx, $random, $select) = @_;
        my $title = $self->{items}->getTitle($idx);

        my $window;
        if ($random)
        {
	        $window = new GCRandomItemWindow($self, $title);
        }
        else
        {
	        $window = new GCItemWindow($self, $title);			
        }

        if ($self->{previousWindowPosition})
        {
            $window->move($self->{previousWindowPosition}->{x},
                          $self->{previousWindowPosition}->{y});
        }

        $window->{panel}->setBorrowers;
        $window->{panel}->disableBorrowerChange;

        $self->{items}->displayInPanel($window->{panel}, $idx);
        $window->{panel}->selectTitle if $select;
        
        $self->{itemWindow} = $window;
        my $code = $window->show;
        $self->{itemWindow} = 0;

        if (!$random && ($code eq 'ok'))
        {
            my $previous = $self->{panel}->getAsHash;
            my $new = $window->{panel}->getAsHash;
            $self->{items}->updateCurrentFromPanel($window->{panel});
            $self->{items}->displayInPanel($self->{panel}, $idx);
            $self->{itemsList}->changeCurrent($previous, $new);
            #Init combo boxes
            foreach(@{$self->{model}->{fieldsHistory}})
            {
                $self->{panel}->{$_}->setValues($window->{panel}->getValues($_));
            }
        }

        my ($width, $height) = $window->get_size;
        $self->{options}->itemWindowWidth($width);
        $self->{options}->itemWindowHeight($height);

        ($self->{previousWindowPosition}->{x}, $self->{previousWindowPosition}->{y})
            = $window->get_position;

        $window->destroy;
        
        return $code;

    }

#    sub previewInfo
#    {
#        my ($self, $info) = @_;
#        my $title = $info->{$self->{model}->{commonFields}->{title}};
#        my $window;
#        $window = new GCItemWindow($self, $title);			
#
#        if ($self->{previousWindowPosition})
#        {
#            $window->move($self->{previousWindowPosition}->{x},
#                          $self->{previousWindowPosition}->{y});
#        }
#        for (@{$self->{model}->{fieldsNames}})
#        {
#            $window->{panel}->$_($info->{$_});
#        }
#        $window->{panel}->lock;
#        
#        #$window->{panel}->setBorrowers;
#        #$window->{panel}->disableBorrowerChange;
#        #$self->{items}->displayInPanel($window->{panel}, $idx);
#        $self->{itemWindow} = $window;
#        my $code = $window->show;
#        $self->{itemWindow} = 0;
#        my ($width, $height) = $window->get_size;
#        $self->{options}->itemWindowWidth($width);
#        $self->{options}->itemWindowHeight($height);
#        ($self->{previousWindowPosition}->{x}, $self->{previousWindowPosition}->{y})
#            = $window->get_position;
#        $window->destroy;
#        return $code;
#    }
   
    sub randomItem
    {
        my ($self) = @_;
        my @tmpArray = undef;
        $self->{items}->updateCurrent;
        
        my $message = '';
        
        #Initialize items array.
        $self->{randomPool} = [];
        my $realId = 0;
        my $filterSearch = new GCFilterSearch;
        foreach my $filter(@{$self->{model}->{random}})
        {
            $filterSearch->setFilter($filter->{field},
                                     $filter->{value},
                                     [$filter->{comparison}, $filter->{numeric}],
                                     $self->{model});
        }

        foreach (@{$self->{items}->{itemArray}})
        {
            #if (!$_->{seen})
            if ($filterSearch->test($_))
            {
                $_->{realId} = $realId;
                push @{$self->{randomPool}}, $_;
            }
            $realId++;
        }
 
        if (scalar @{$self->{randomPool}} > 0)
        {
            my $code = 'no';
            my $idx = 0;
            while ($code eq 'no')
            {
        	   $idx = int rand(scalar @{$self->{randomPool}});
        	   $realId = $self->{randomPool}->[$idx]->{realId};
        	   $code = $self->displayInWindow($realId, 1);
        	   splice @{$self->{randomPool}}, $idx, 1;
        	   last if ! @{$self->{randomPool}};
            }
            $message = $self->{lang}->{RandomEnd} if $code eq 'no';
            if ($code eq 'ok')
            {
                foreach my $filter(@{$self->{model}->{random}})
                {
                    next if !exists $filter->{after};
                    my $field = $filter->{field};
                    $self->{items}->{itemArray}[$realId]->{$field} = $filter->{after};
                    $self->{panel}->$field($filter->{after})
                        if $self->{items}->{currentItem} == $realId;
                }
            }
        }
    	else
    	{
    	   $message = $self->{lang}->{RandomError};
    	}

        if ($message)
        {
            my $dialog = Gtk2::MessageDialog->new($self->{parent},
    		  	                                 [qw/modal destroy-with-parent/],
                                                'info',
                                                  'ok',
                                                  $message);
            $dialog->set_position('center-on-parent');
    		$dialog->run();
	       	$dialog->destroy;
        }
        
        #Clean items array.
        foreach (@{$self->{items}->{itemArray}})
        {
            delete $_->{realId};
        }
    }
    
    sub initProgress
    {
        my ($self, $label) = @_;
        if ($label)
        {
            $self->setStatus($label);
            $self->{restoreNeeded} = 1;
        }
        return if $self->{openingFile};
        $self->setProgress(0.0);
        $self->{menubar}->set_sensitive(0);
        $self->{toolbar}->set_sensitive(0);
        $self->{pane}->set_sensitive(0);
    }
    
    sub endProgress
    {
        my $self = shift;
        $self->setProgress(1.0);
        if ($self->{restoreNeeded})
        {
            $self->{restoreNeeded} = 0;
            $self->restoreStatus;
        }
        Glib::Timeout->add(500 , sub {
            $self->setProgress(0.0);
            return 0;
        });
        $self->{menubar}->set_sensitive(1);
        $self->{toolbar}->set_sensitive(1);
        $self->{pane}->set_sensitive(1);
    }
    
    sub setProgress
    {
        my ($self, $current) = @_;
        $self->{progress}->set_fraction($current);
        Gtk2->main_iteration while (Gtk2->events_pending);
    }

    sub setItemsTotal
    {
        my ($self, $total) = @_;
        $self->{step} = GCUtils::round($total / 7);
        $self->{step} = 1 if $self->{step} < 1;
        $self->{total} = $total;
    }

    sub setProgressForItemsLoad
    {
        my ($self, $current) = @_;
        return if ! $self->{openingFile};
        return if ($current % $self->{step});
        if ($self->{total})
        {
            my $value = ($current / $self->{total}) / 2;
            $value = 0.5 if $value > 0.5;
            $self->{progress}->set_fraction($value);
        }
        else
        {
            $self->{progress}->set_fraction(0.3);
        }
        Gtk2->main_iteration while (Gtk2->events_pending);
    }

    sub setProgressForItemsDisplay
    {
        my ($self, $current) = @_;
        return if ($current % $self->{step});
        if ($self->{total})
        {
            my $value = ($current / $self->{total});
            $value = ($value / 2) + 0.5 if $self->{openingFile};
            $value = 1.0 if $value > 1;
            $self->{progress}->set_fraction($value);
        }
        else
        {
            $self->{progress}->set_fraction(0.7);
        }
        Gtk2->main_iteration while (Gtk2->events_pending);
    }

    sub blockListUpdates
    {
        my ($self, $value) = @_;
        
        $self->{items}->{block} = $value;
    }

    sub reloadList
    {
        my $self = shift;
        $self->{items}->reloadList($self, 1);
    }

    sub setItemsList
    {
        my ($self, $init) = @_;
        my $view = $self->{options}->view;
        my $columns = $self->{options}->columns;
        
        $self->{pane}->remove($self->{itemsList}) if $self->{pane} && $self->{itemsList};
        $self->{itemsList}->destroy if $self->{itemsList};

        $self->{itemsList} = new GCTextList($self, $self->{model}->getDisplayedText('Items'))
            if $view == 0;
        $self->{itemsList} = new GCImageList($self, $columns) if $view == 1;
        $self->{itemsList} = new GCDetailedList($self) if $view == 2;

        $self->setExpandCollapseInContext($self->{itemsList}->couldExpandAll);

        $self->{pane}->pack1($self->{itemsList},1,0) if $self->{pane};
        $self->{itemsList}->show_all;
        if ($self->{items})
        {
            $self->{items}->resetSortField if $view <= 1;
            $self->{items}->restoreSort if $view == 2;
            $self->reloadList if ! $self->{initializing};
            Glib::Timeout->add(100 ,\&showCurrent, $self);
            $self->{itemsList}->setSortOrder;
        }

        #Change corresponding item in context menu        
        $self->{ignoreContextActivation} = 1;
        $self->{context}->{menuDisplayType}->set_active($view + 1);
        $self->{context}->{menuDisplayType}->get_active->set_active(1);
        $self->{ignoreContextActivation} = 0;
        #Assign context menu to items list that will be in charge of displaying it.
        $self->{itemsList}->{context} = $self->{context};
        
        if (!$init)
        {
            $self->checkToolbarOptions;
        }
    }
    
    sub checkToolbarOptions
    {
        my $self = shift;
        $self->{toolbar}->{blocked} = 1;
        if ($self->{options}->view == 2)
        {
            $self->{toolbar}->setGroupField($self->{model}->{preferences}->groupBy);
        }
        else
        {
            $self->{toolbar}->setGroupField('')
        }
        $self->{toolbar}->{blocked} = 0;        
    }
    
    sub createContextMenu
    {
        my $self = shift;
      
        # Context menu creation. It is displayed when right clicking on a list item.
        $self->{context} = new Gtk2::Menu;
        
        $self->{context}->append(Gtk2::TearoffMenuItem->new());
        
        $self->{contextExpand}->{expand} = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{ContextExpandAll});
        $self->{contextExpand}->{expand}->signal_connect('activate', sub {
            $self->{itemsList}->expandAll;
        });
        $self->{context}->append($self->{contextExpand}->{expand});

        $self->{contextExpand}->{collapse} = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{ContextCollapseAll});
        $self->{contextExpand}->{collapse}->signal_connect('activate', sub {
            $self->{itemsList}->collapseAll;
        });
        $self->{context}->append($self->{contextExpand}->{collapse});
        
        
        #$self->{contextExpand}->{separator} = new Gtk2::SeparatorMenuItem;
        $self->{context}->append(Gtk2::SeparatorMenuItem->new);

        my $itemDuplicate = Gtk2::ImageMenuItem->new_from_stock('gtk-dnd',undef);
        $itemDuplicate->signal_connect("activate" , sub {
            $self->duplicateItem;
        });
        $self->{context}->append($itemDuplicate);

        my $itemDelete = Gtk2::ImageMenuItem->new_from_stock('gtk-delete',undef);
        $itemDelete->signal_connect("activate" , sub {
            $self->deleteCurrentItem;
        });
        $self->{context}->append($itemDelete);
        
        my $newWindow = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{MenuNewWindow});
        $newWindow->signal_connect("activate" , sub {
                $self->displayInWindow;
        });
        $self->{context}->append($newWindow);
        
        $self->{context}->append(Gtk2::SeparatorMenuItem->new);
        
        $self->{context}->{menuDisplayType} = new Gtk2::Menu;
        $self->{context}->{menuDisplayType}->append(Gtk2::TearoffMenuItem->new());
        my %views = %{$self->{lang}->{OptionsViews}};
        my $displayGroup = undef;
		foreach (0..(scalar(keys %views) - 1))
		{
		    my $item = Gtk2::RadioMenuItem->new_with_mnemonic($displayGroup, $views{$_});
		    $item->signal_connect('activate', sub {
		        my ($widget, $self) = @_;
		        return if ($self->{ignoreContextActivation});
                if ($widget->get_active)
                {
                    my $group = $widget->get_group;
                    my $i = 0;
                    $i++ while ($views{$i} ne $widget->child->get_label);
                    $self->{options}->view($i);
	                $self->setItemsList(0)
                        if ! $self->{initializing};
                    $self->checkView;
                    
                }
		    }, $self);
		    $self->{context}->{menuDisplayType}->append($item);
		    $displayGroup = $item->get_group;
		}
		
        $self->{context}->{itemDisplayType} = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{OptionsView});
        $self->{context}->{itemDisplayType}->set_submenu($self->{context}->{menuDisplayType});
        $self->{context}->append($self->{context}->{itemDisplayType});
        
        $self->{context}->append(Gtk2::SeparatorMenuItem->new);
        
        # Filters selection
                
        my $menuDisplay = Gtk2::Menu->new();
        my $displayItem = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{MenuDisplay});
        
        my $allItems = Gtk2::MenuItem->new_with_mnemonic($self->{lang}->{MenuViewAllItems});
        $allItems->signal_connect("activate" , sub {
                $self->allItems;
        });
        $menuDisplay->append($allItems);
        
        my $searchSelectedItems = Gtk2::ImageMenuItem->new_from_stock('gtk-find',undef);
        $searchSelectedItems->signal_connect("activate" , sub {
                $self->search('displayed');
        });
        $menuDisplay->append($searchSelectedItems);
        
        $displayItem->set_submenu($menuDisplay);
        $self->{context}->append($displayItem);
        
        $self->{context}->show_all;
      
    }
    
    sub setExpandCollapseInContext
    {
        my ($self, $active) = @_;
        
        foreach (keys %{$self->{contextExpand}})
        {
            $self->{contextExpand}->{$_}->set_sensitive($active);
        }
    }
    
    sub contextDisplayChange
    {
        my ($self, $widget, $menuName, $number) = @_;
        
        return if $widget && ! $widget->get_active;
        return if $self->{menubar}->{contextUpdating};

        $self->{menubar}->{contextUpdating} = 1;
        $self->{menubar}->{$menuName}->set_active($number);
        $self->{menubar}->{$menuName}->get_active->activate;
        $self->{menubar}->{contextUpdating} = 0;
    }
    
    sub showCurrent
    {
        my $self = shift;
        $self->{itemsList}->showCurrent;
    }
    
    sub allItems
    {
        my ($self, $self2) = @_;
        $self = $self2 if $self2;

        $self->{menubar}->selectAll;
        $self->reloadList;
    }
    
    sub transformTitle
    {
        my ($self, $title) = @_;

        return $title if ! $self->{options}->{options}->{transform};
        
        return $title if $title !~ $self->{articlesRegexp};
        #'
        #return $title if $title !~ /^($self->{articlesRegexp})(\s+|('))(.*)/i;
        return ucfirst($4)." ($1$3)";
    }
    
    sub new
    {
        my ($proto, $options, $version, $searchJob) = @_;

        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new('toplevel');
        bless ($self, $class);

        $options->setParent($self);
        $self->{options} = $options;
        $self->{version} = $version;
        $self->{searchJob} = $searchJob;
        $self->{itemWindow} = 0;
        
        $self->{logosDir} = $ENV{GCS_SHARE_DIR}.'/logos/';
        $self->{hasPictures} = (-f $self->{logosDir}.'splash.png');
        $self->{pluginsPicturesDir} = $ENV{GCS_SHARE_DIR}.'/plugins/';
        $self->{hasPluginsPictures} = 0;
        if (-d $self->{pluginsPicturesDir})
        {
            my @pics = glob($self->{pluginsPicturesDir}.'/*.png');
            $self->{hasPluginsPictures} = 1 if $#pics > 0;
        }

        $self->{lang} = $GCLang::langs{$self->{options}->lang};

        GCPlugins::loadPlugins;
        GCExport::loadExporters;
        GCImport::loadImporters;

        if (! $options->exists('style'))
        {
            $options->style('Gtk')     if ($^O !~ /win32/i);
            $options->style('GCstar') if ($^O =~ /win32/i);
        }
        GCStyle::initStyles;
        my $style = $GCStyle::styles{$options->style};
        
        if ((! $options->exists('itemWindowWidth'))
         || (! $options->exists('itemWindowHeight')))
        {
            $options->itemWindowWidth(600);
            $options->itemWindowHeight(500);
        }

        $self->{style} = $style;
        
        $self->{initializing} = 1;
        $self->set_sensitive(0);

        if (($self->{options}->splash) || (! $self->{options}->exists('splash')))
        {
            $self->{splash} = new GCSplashWindow($self, $self->{version});
        }
        else
        {            
            $self->init;
            $self->loadPrevious;
            $self->initEnd;
            $self->{initializing} = 0;
            $self->set_sensitive(1);
        }

        return $self;
    }

    sub createStockItems
    {
        my $self = shift;
        
        my $baseStock;
        $baseStock->{translation_domain} = 'gtk20';
        $baseStock->{keyval} = 0;
        $baseStock->{modifier} = [  ];
        
        $baseStock->{stock_id} = 'gtk-execute';
        $baseStock->{label} = $self->{lang}->{ToolbarRandom};                
        Gtk2::Stock->add($baseStock);
        
        $baseStock->{stock_id} = 'gtk-refresh';
        $baseStock->{label} = $self->{lang}->{ToolbarAll};
        Gtk2::Stock->add($baseStock);
        
        $baseStock->{stock_id} = 'gtk-convert';
        $baseStock->{label} = $self->{lang}->{MenuImport};
        Gtk2::Stock->add($baseStock);
        
        $baseStock->{stock_id} = 'gtk-revert-to-saved';
        $baseStock->{label} = $self->{lang}->{MenuExport};
        Gtk2::Stock->add($baseStock);

        $baseStock->{stock_id} = 'gtk-dnd';
        $baseStock->{label} = $self->{lang}->{MenuDuplicate};
        Gtk2::Stock->add($baseStock);

        $baseStock->{stock_id} = 'gtk-jump-to';
        $baseStock->{label} = $self->{lang}->{PanelSearchButton};
        Gtk2::Stock->add($baseStock);

        $baseStock->{stock_id} = 'gtk-help';
        #$baseStock->{keyval} = 'F1';
        #$baseStock->{modifier} = [];
        $baseStock->{label} = $self->{lang}->{MenuHelpContent};
        Gtk2::Stock->add($baseStock);

        $baseStock->{stock_id} = 'gtk-about';
        $baseStock->{label} = $self->{lang}->{MenuAbout};
        Gtk2::Stock->add($baseStock);
        
        $baseStock->{stock_id} = 'gtk-zoom-in';
        $baseStock->{label} = $self->{lang}->{ResultsPreview};
        Gtk2::Stock->add($baseStock);
        
        my $addStock = Gtk2::Stock->lookup('gtk-add');
        $addStock->{keyval} = 65;
        $addStock->{modifier} = [ 'control-mask' ];
        Gtk2::Stock->add($addStock);
    }

    sub init
    {
        my $self = shift;
        my $splash = shift;

        $self->{options}->save if $self->checkImagesDirectory(1);
        
        $self->createStockItems;

        $splash->setProgress(0.01) if $splash;

        $self->{modelsFactory} = new GCModelsCache($self);

        Gtk2::Rc->parse($self->{style}->{rcFile});
        
        $self->{agent} = 'Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.7.5) Gecko/20041111 Firefox/1.0';
        $self->{ignoreString} = 'gcf_000_ignore';
        $self->{imagePrefix} = 'gcstar_';
        if (-f $self->{logosDir}.'no.png')
        {
            $self->{defaultImage} = $self->{logosDir}.'no.png';
        }
        else
        {
            $self->{defaultImage} = $ENV{GCS_SHARE_DIR}.'/no.jpg';
        }
        
        $self->{tooltips} = Gtk2::Tooltips->new();
        if ($self->{searchJob}->{command})
        {
            $self->{searchJob}->{command}->autoflush(1);
            $self->{searchJob}->{data}->autoflush(1);
        }
        
        my $tmpTitle = $self->{options}->file ? basename($self->{options}->file) : '';
        $self->setTitle($tmpTitle, $self->{items}->{information}->{name});
        my $iconPrefix = $ENV{GCS_SHARE_DIR}.'/icons/gcstar_';
        my $pixbuf16 = Gtk2::Gdk::Pixbuf->new_from_file($iconPrefix.'16x16.png');
        my $pixbuf32 = Gtk2::Gdk::Pixbuf->new_from_file($iconPrefix.'32x32.png');
        my $pixbuf48 = Gtk2::Gdk::Pixbuf->new_from_file($iconPrefix.'48x48.png');
        my $pixbuf64 = Gtk2::Gdk::Pixbuf->new_from_file($iconPrefix.'64x64.png');
        $self->set_default_icon_list($pixbuf16, $pixbuf32, $pixbuf48, $pixbuf64);

        $self->signal_connect(delete_event => \&beforeDestroy, $self);
        $self->signal_connect(destroy => sub { Gtk2->main_quit; });
        
        #$splash->setProgress(0.02) if $splash;
        
        $self->createContextMenu;

        $splash->setProgress(0.03) if $splash;
        
        $self->{items} = new GCItems($self);
        $self->{panel} = 0;

        $self->{menubar} = new GCMenuBar($self);
        $self->{menubar}->set_name('GCMenubar');
        $self->{bookmarksLoader} = new GCBookmarksLoader($self, $self->{menubar});

        #$splash->setProgress(0.04) if $splash;

        $self->{toolbar} = GCToolBar->new($self);
        $self->{handleToolbar} = new Gtk2::HandleBox;
        
        if (-e $self->{logosDir}.'/button.png')
        {
            $self->{logoButton} = Gtk2::Button->new;
            $self->{logoButton}->set_relief('half');
            $self->{logoButton}->set_name('GCLogoButton');
            $self->{tooltips}->set_tip($self->{logoButton},
                                       $self->{lang}->{AboutTitle});
            $self->{logoButton}->signal_connect('clicked' , sub {
                $self->about;
            });
            $self->{logoOriginalPixbuf} = Gtk2::Gdk::Pixbuf->new_from_file($self->{logosDir}.'/button.png');
            $self->{logoImage} = Gtk2::Image->new;
            $self->{logoButton}->add($self->{logoImage});
        }

        $self->{mainVbox} = new Gtk2::VBox(0, 0);
        $self->{mainHbox} = new Gtk2::HBox(0, 0);
        $self->{pane} = new Gtk2::HPaned;

        $self->{pane}->set_position($self->{options}->split);

        $self->{mainVbox}->pack_start($self->{menubar}, 0, 0, 1);
        $self->{mainHbox}->pack_start($self->{pane},1,1,0);
        $self->{mainVbox}->pack_start($self->{mainHbox}, 1, 1, 0);

        $self->{status} = Gtk2::Statusbar->new;
        $self->{status}->set_has_resize_grip(1);
        $self->{progress} = new Gtk2::ProgressBar;
        $self->{progress}->set_size_request(100,-1);
        $self->{status}->pack_start($self->{progress}, 0, 0, 5);
        $self->{mainVbox}->pack_start($self->{status},0,0,0);

        $self->checkToolbarPosition;

        $self->add($self->{mainVbox});

        $splash->setProgress(0.05) if $splash;

        $self->set_default_size($self->{options}->width,$self->{options}->height);
       
        #$splash->setProgress(0.06) if $splash;

        $self->drag_dest_set('all', ['copy','private','default','move','link','ask']);

        $self->signal_connect(drag_data_received => \&drop_handler);

        my $target_list = Gtk2::TargetList->new();
        my $atom1 = Gtk2::Gdk::Atom->new('text/uri-list');
        my $atom2 = Gtk2::Gdk::Atom->new('text/plain');
        $target_list->add($atom1, 0, 0);
        $target_list->add($atom2, 0, 0);
        if ($^O =~ /win32/i)
        {
            my $atom3 = Gtk2::Gdk::Atom->new('DROPFILES_DND');
            $target_list->add($atom3, 0, 0);
        }
        
        $self->drag_dest_set_target_list($target_list);

        sub drop_handler {
            my ($widget, $context, $widget_x, $widget_y, $data, $info,$time) = @_;
            my $type = $data->type->name;
            if (($type eq 'text/uri-list')
             || ($type eq 'DROPFILES_DND'))
            {
                my @files = split /\n/, $data->data;
                my $numbers = scalar @files;
                $numbers-- if ($files[$#files] =~ /^\W*$/);
                if (($numbers == 1)
                 && ($files[0] =~ /\.gcs.?$/))
                {
                    #Special case when only one .gcs file is dropped
                    (my $fileName = $files[0]) =~ s/^file:\/\/(.*)\W*$/$1/;
                    $fileName =~ s/.$//ms;
                    $self->openFile($fileName);
                }
                else
                {
                    my $fileName = $self->{options}->file;
                    #$self->newList;
                    foreach (@files)
                    {
                        if (!$self->importWithDetect($_))
                        {
                            my $dialog = Gtk2::MessageDialog->new($self,
							    [qw/modal destroy-with-parent/],
							    'error',
							    'ok',
							    $self->{lang}->{ImportDropError});

                            $dialog->set_position('center-on-parent');
                            my $response = $dialog->run;
                            $dialog->destroy;

                            $self->openFile($fileName);
                            last;
                        }
                    }
                    $self->{items}->setStatus;
                }
            }
            elsif ((my $url = $data->data) =~ m/^http:\/\//)
            {
                $self->loadUrl($url);
            }
        }

        $splash->setProgress(0.07) if $splash;
        
        $self->show_all;

        #$splash->setProgress(0.08) if $splash;

        $self->checkDisplayed;

        $self->checkView;
        $self->checkTransform;
        
        $splash->setProgress(0.09) if $splash;
        
        $self->{options}->searchStop(0) if ($^O =~ /win32/i);

    }

    sub setPanel
    {
        my $self = shift;

        my $panelInfo;
        if (! $self->{model}->{preferences}->exists('layout') || (! $self->{model}->{preferences}->layout))
        {
            $panelInfo = $self->{model}->getDefaultPanel;
            $self->{model}->{preferences}->layout($panelInfo->{name})
        }
        else
        {
            $panelInfo = $self->{model}->{panels}->{$self->{model}->{preferences}->layout};
        }
        if ($panelInfo->{editable} eq 'true')
        {
            $self->{panel} = new GCFormPanel($self, $self->{options}, $panelInfo);
        }
        else
        {
            $self->{panel} = new GCReadOnlyPanel($self, $self->{options}, $panelInfo);
        }
    }

    sub changePanel
    {
        my ($self, $modelChanged) = @_;

        #Save previous histories
        my %savedHistories;
        if ($self->{panel})
        {
            if (!$modelChanged)
            {
                foreach(@{$self->{model}->{fieldsHistory}})
                {
                    $savedHistories{$_} = $self->{panel}->getValues($_);
                }
            
                $self->{items}->updateCurrent;
            }
            $self->{scrollPanelItem}->remove($self->{scrollPanelItem}->get_child);
            $self->{panel}->destroy;
            $self->{pane}->remove($self->{scrollPanelItem});
            $self->{scrollPanelItem}->destroy;
        }

        $self->setPanel;
        $self->{panel}->createContent($self->{model});

        $self->{scrollPanelItem} = new Gtk2::ScrolledWindow;
        $self->{scrollPanelItem}->set_policy ('automatic', 'automatic');
        $self->{scrollPanelItem}->set_shadow_type('none');
        $self->{scrollPanelItem}->add_with_viewport($self->{panel});
        $self->{pane}->pack2($self->{scrollPanelItem},1,1);
        $self->{scrollPanelItem}->show_all;
        $self->{items}->setPanel($self->{panel});
        $self->checkBorrowers;
        $self->checkPanel;

        if (%savedHistories)
        {
            #Restore histories
            foreach(@{$self->{model}->{fieldsHistory}})
            {
                $self->{panel}->{$_}->setValues($savedHistories{$_});
            }
        }
        
        $self->{items}->displayCurrent;
    }

    sub checkToolbarPosition
    {
        my $self = shift;

        #Remove previous if exists
        if ($self->{boxToolbar})
        {
            $self->{boxToolbar}->remove($self->{toolbar});
            $self->{boxToolbar}->remove($self->{logoButton});
            $self->{boxToolbar}->destroy;
            $self->{mainHbox}->remove($self->{handleToolbar})
                if ($self->{mainHbox}->child_get($self->{handleToolbar}));
            $self->{mainVbox}->remove($self->{handleToolbar})
                if ($self->{mainVbox}->child_get($self->{handleToolbar}));
            $self->{handleToolbar}->destroy;
        };

        my $position = $self->{options}->toolbarPosition;
        
        # 0 => top
        # 1 => bottom
        # 2 => left
        # 3 => right
        
        if ($position > 1) # left or right
        {
            $self->{boxToolbar} = new Gtk2::VBox(0,0);
            $self->{toolbar}->set_orientation('vertical');
            $self->{toolbar}->set_style('icons');
            $self->{mainHbox}->pack_start($self->{handleToolbar},0,0,0);
            $self->{handleToolbar}->set_handle_position('top');
            if ($position == 2) # left
            {
                $self->{handleToolbar}->set_snap_edge('left');
                $self->{mainHbox}->reorder_child($self->{handleToolbar},0);
            }
            else # right
            {
                $self->{handleToolbar}->set_snap_edge('right');
                $self->{mainHbox}->reorder_child($self->{handleToolbar},1);
            }
        }
        else # top or bottom
        {
            $self->{boxToolbar} = new Gtk2::HBox(0,0);
            $self->{toolbar}->set_orientation('horizontal');
            $self->{toolbar}->unset_style;
            $self->{mainVbox}->pack_start($self->{handleToolbar},0,0,0 );
            $self->{handleToolbar}->set_handle_position('left');
            if ($position == 1) # bottom
            {
                $self->{handleToolbar}->set_snap_edge('bottom');
                $self->{mainVbox}->reorder_child($self->{handleToolbar},2);
            }
            else # top
            {
                $self->{handleToolbar}->set_snap_edge('top');
                $self->{mainVbox}->reorder_child($self->{handleToolbar},1);
            }
        }
        $self->{boxToolbar}->pack_start($self->{toolbar},1,1,0);
        $self->{boxToolbar}->pack_start($self->{logoButton},0,0,0);
        $self->{handleToolbar}->add($self->{boxToolbar});
        $self->{handleToolbar}->show_all;
    }

    sub checkDisplayed
    {
        my $self = shift;
        
        if ($self->{options}->status)
        {
            $self->{status}->show;
        }
        else
        {
            $self->{status}->hide;
        }

        if ($self->{options}->toolbar)
        {
            $self->{handleToolbar}->show;
            $self->{toolbar}->set_icon_size('small-toolbar') if $self->{options}->toolbar == 1;
            $self->{toolbar}->set_icon_size('large-toolbar') if $self->{options}->toolbar == 2;
            
            if ($self->{logoOriginalPixbuf})
            {
                my ($width, $height) = Gtk2::IconSize->lookup($self->{toolbar}->get_icon_size);
                my $logoPixbuf = GCUtils::scaleMaxPixbuf($self->{logoOriginalPixbuf}, $width * 1.7, $height * 1.7);
                $self->{logoImage}->set_from_pixbuf($logoPixbuf);
                $self->{logoButton}->show_all;
            }
        }
        else
        {
            $self->{handleToolbar}->hide;
        }

    }

    sub checkProxy
    {
		my $self= shift;
		
		$self->{plugin}->setProxy($self->{options}->proxy) if ($self->{plugin});
    }
    
    sub checkPlugin
    {
        my $self = shift;
        $self->{plugin} = undef;
        $self->{plugin} = $self->{model}->getPlugin($self->{model}->{preferences}->plugin);

        $self->checkProxy;
    }
    
    sub checkBorrowers
    {
        my $self = shift;

        #my @borrowers = split m/\|/, $self->{options}->borrowers;
        $self->{panel}->setBorrowers;
    }
    
    sub checkView
    {
        my $self = shift;
        $self->{context}->{menuDisplayType}->set_active($self->{options}->view + 1);
        $self->{context}->{menuDisplayType}->get_active->set_active(1);
    }
    
    sub checkTransform
    {
        my $self = shift;
 
        my @array = split m/,/, $self->{options}->articles;

        my $tmpExpr = '';
        
        foreach (@array)
        {
            s/^\s*//;
            s/\s*$//;
            $tmpExpr .= "$_|";
        }
        chomp $tmpExpr;
        
        $self->{articles} = \@array;
        #$self->{articlesRegexp} = $tmpExpr;
        $self->{articlesRegexp} = qr/^($tmpExpr)(\s+|('))(.*)/i;
        #'
        $self->reloadList if ! $self->{initializing};
    }
    
    sub checkPanel
    {
        my $self = shift;

        my $hasToShow = 1;
        $hasToShow = 0 if ($self->{items}->{selectToIdArray} && (! scalar @{$self->{items}->{selectToIdArray}}));
                       
        $self->{panel}->setShowOption($self->getDialog('DisplayOptions')->{show}, $hasToShow);
    }
    
    sub loadPrevious
    {
        my $self = shift;
        my $splash = shift;
        if (! $self->{items}->setOptions($self->{options}, $splash))
        {
            $self->setTitle('');
        }
        else
        {
            $self->setTitle(basename($self->{options}->file), $self->{items}->{information}->{name});
        }
        $self->{menubar}->setLock($self->{items}->getLock);

        $self->checkPanel;
        $self->checkBorrowers;

        $self->{itemsList}->setSortOrder;

        my $splashValue = 0.91;
#        my $splashStep = 0.08 / (scalar @{$self->{model}->{fieldsHistory}});
        $splash->setProgress(0.99) if $splash;
        
        # Windows program starts iconify to hide console window.
        # Here we make only main window become visible.
        $self->deiconify if ($^O =~ /win32/i);
    }
    
    sub initEnd
    {
        my $self = shift;
        
        if ($self->{options}->{created})
        {
            $self->{options}->checkPreviousGCfilms($self);
        }
        
        if (! $self->{options}->file)
        {
            $self->newList;
        }
        
        $self->{itemsList}->done;
        $self->{items}->display($self->{items}->select(-1,undef,1));
        
        if ($ENV{GCS_PROFILING} > 2)
        {
            $self->leave;
        }
    }

}

1;
