package GCGraphicComponents;

###################################################
#
#  Copyright 2005-2006 Tian
#
#  This file is part of GCstar.
#
#  GCstar is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCstar is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCstar; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################
use utf8;
use Gtk2;
use GCBorrowings;

use strict;

{
    package GCGraphicComponent;
    
    sub expand
    {
    }
    
    sub getMainParent
    {
        my $self = shift;
        return if ! $self->{parent};
        my $tmpWidget = $self;
        $tmpWidget = $tmpWidget->{parent} while $tmpWidget && (! $tmpWidget->isa('GCFrame'));
        $self->{mainParent} = $tmpWidget;
    }
    
    sub acceptMarkup
    {
        my $self = shift;
        return 0;
    }

    sub selectAll
    {
    }
}

{
    package GCPseudoHistoryComponent;
    
    sub initHistory
    {
        my ($self, $listType) = @_;
        $self->{history} = {};

        # listType contains the type of the original field:
        #   0: No list
        #   >0: Multiple list (number of columns)
        $self->{listType} = $listType;
        
        $self->{history}->{0} = {'' => 1};
        for(my $i = 1; $i < $listType; $i++)
        {
            $self->{history}->{$i} = {'' => 1};
        }
    }

    sub addHistory
    {
        my $self = shift;

        my $value = shift;
        my $i;
        if (ref($value) eq 'HASH')
        {
            foreach (@{$value->{line}})
            {
                $i = 0;
                foreach my $item(@{$_->{col}})
                {
                    $self->{history}->{$i}->{$item} = 1;
                    $i++;                   
                }
            }
        }
        else
        {
            my @values = split m/,/, $value;
            foreach (@values)
            {
                my @items = split m/;/;
                $i = 0;
                foreach my $item(@items)
                {
                    $self->{history}->{$i}->{$item} = 1;
                    $i++;
                }
            }
        }
    }

    sub setDropDown
    {
    }

    sub getValues
    {
        my $self = shift;
        my @array;
        
        
        if ($self->{listType} < 1)
        {
            @array = sort keys %{$self->{history}->{0}};
        }
        else
        {
            foreach (sort keys %{$self->{history}})
            {
                my @tmpArray = sort keys %{$self->{history}->{$_}};
                push @array, \@tmpArray;
            }
        }
        return \@array;
    }

    sub setValues
    {
        my ($self, $values) = @_;

        if ($self->{listType} == 0)
        {
            $self->{history} = {};
            $self->{history}->{0}->{$_} = 1 foreach (@$values);
        }
        else
        {
            $self->{history} = {};
            for (my $i = 0; $i < $self->{listType}; $i++)
            {
                $self->{history}->{$i}->{$_} = 1 foreach (@{$values->[$i]});
            }
        }
    }
}

{
    package GCShortText;
    
    @GCShortText::ISA = ('Gtk2::Entry', 'GCGraphicComponent');
    
    sub new
    {
        my ($proto) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new;

        bless ($self, $class);
        return $self;
    }
    
    sub isEmpty
    {
        my $self = shift;
        
        return $self->getValue eq '';
    }
    
    sub selectAll
    {
        my $self = shift;
        $self->select_region(0, length($self->getValue));
        $self->grab_focus;
    }
    
    sub getValue
    {
        my $self = shift;
        return $self->get_text;
    }
    
    sub setValue
    {
        my ($self, $value) = @_;
        $self->set_text($value);
    }
    
    sub clear
    {
        my $self = shift;
        $self->setValue('');
    }
    
    sub setWidth
    {
        my ($self, $value) = @_;
        $self->set_width_chars($value);
    }
    
    sub lock
    {
        my ($self, $locked) = @_;
    
        $self->can_focus(!$locked);
    }
}

{
    package GCLongText;
    
    @GCLongText::ISA = ('Gtk2::ScrolledWindow', 'GCGraphicComponent');

    sub new
    {
        my ($proto) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new;
    
        $self->{text} = new Gtk2::TextView;
        $self->{text}->set_editable(1);
        $self->{text}->set_wrap_mode('word');
        $self->set_border_width(0);
        $self->set_shadow_type('in');
        $self->set_policy('automatic', 'automatic');
        #$self->set_size_request(-1,80);
        $self->add($self->{text});

        bless ($self, $class);
        return $self;
    }
    
    sub isEmpty
    {
        my $self = shift;
        
        return $self->getValue eq '';
    }
    
    sub getValue
    {
        my $self = shift;
        my $buffer = $self->{text}->get_buffer;
        my $text = $buffer->get_text($buffer->get_start_iter,
                   $buffer->get_end_iter, 1);
        #$text =~s/\n/<br\/>/g;
        return $text;
    }
    
    sub setValue
    {
        my ($self, $text) = @_;
        #$text =~s/<br\/>/\n/g;
        $self->{text}->get_buffer->set_text($text);
    }
    
    sub clear
    {
        my $self = shift;
        $self->setValue('');
    }
    
    sub lock
    {
        my ($self, $locked) = @_;
    
        $self->{text}->can_focus(!$locked);
    }
    
    sub setHeight
    {
        my ($self, $height) = @_;
        
        # TODO Change height
    }
}

{
    package GCHistoryText;
    
    @GCHistoryText::ISA = ('Gtk2::Combo', 'GCGraphicComponent');
    
    sub new
    {
        my ($proto) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new;

        $self->{history} = {'' => 1};

        bless ($self, $class);
        return $self;
    }
    
    sub isEmpty
    {
        my $self = shift;
        
        return $self->getValue eq '';
    }
    
    sub selectAll
    {
        my $self = shift;
        $self->entry->selectAll;
    }

    sub getValue
    {
        my $self = shift;
        my @children = $self->get_children;
        return $children[0]->get_text if $children[0];
    }
    
    sub setValue
    {
        my ($self, $text) = @_;
        my @children = $self->get_children;
        $children[0]->set_text($text);
    }

    sub clear
    {
        my $self = shift;
        $self->setValue('');
    }
    
    sub setWidth
    {
        my ($self, $value) = @_;
        ($self->get_children)[0]->set_width_chars($value);
    }
    
    sub lock
    {
        my ($self, $locked) = @_;
    
        ($self->get_children)[0]->can_focus(!$locked);
        ($self->get_children)[1]->set_sensitive(!$locked);
    }
    
    sub addHistory
    {
        my $self = shift;
        my $value = (scalar @_) ? shift : $self->getValue;
        my $noUpdate = shift;
        #$value = ucfirst $value;
        if (!exists $self->{history}->{$value})
        {
            $self->{history}->{$value} = 1;
            if (!$noUpdate)
            {
                $self->setDropDown(sort keys %{$self->{history}});
            }
        }
    }
    
    sub setDropDown
    {
        my $self = shift;
        my @values = (scalar @_) ? @_ : sort keys %{$self->{history}};
        my $previousValue = $self->getValue;
        $self->set_popdown_strings(@values);
        $self->setValue($previousValue);
    }
    
    sub getValues
    {
        my $self = shift;
        my @array = sort keys %{$self->{history}};
        return \@array;
    }

    sub setValues
    {
        my ($self, $values) = @_;

        $self->{history} = {};
        $self->setDropDown(@$values);
#        my $previousValue = $self->getValue;
#        $self->set_popdown_strings(@$values);
#        $self->setValue($previousValue);
        $self->{history}->{$_} = 1 foreach (@$values);
    }
}

{
    package GCNumeric;
    
    @GCNumeric::ISA = ('Gtk2::SpinButton', 'GCGraphicComponent');
    
    sub new
    {
        my ($proto, $default, $min, $max, $step) = @_;
        my $class = ref($proto) || $proto;

        $step = 1 if !$step;
        my $pageStep = 10 * $step;

        my $decimals = 0;
        $decimals = length($1) if $step =~ /\.(.*)$/;

        my $accel = 0;
        my $values = ($max - $min) / $step;
        $accel = 0.2 if $values > 100;
        $accel = 0.5 if $values > 500;
        $accel = 1.0 if $values > 2000;

        my $adj = Gtk2::Adjustment->new($default, $min, $max, $step, $pageStep, 0) ;
        my $self = $class->SUPER::new($adj, $accel, $decimals);
        $self->{default} = $default;
        $self->{step} = $step;
        $self->{pageStep} = $pageStep;
        $self->{accel} = $accel;
        $self->set_numeric(1);

        bless ($self, $class);
        return $self;
    }

    sub isEmpty
    {
        my $self = shift;
        
        return $self->get_text eq '';
    }

    sub selectAll
    {
        my $self = shift;
        $self->select_region(0, length($self->getValue));
    }
    
    sub getValue
    {
        my $self = shift;
        my $value = $self->get_text;
        $value =~ s/,/./;
        return $value;
    }

    sub setValue
    {
        my ($self, $text) = @_;
        $self->set_value($text);
    }
    
    sub clear
    {
        my $self = shift;
        $self->setValue($self->{default});
    }
    
    sub setWidth
    {
        my ($self, $value) = @_;
        $self->set_width_chars($value);
    }
    
    sub lock
    {
        my ($self, $locked) = @_;

        $self->can_focus(!$locked);
        my $step = ($locked ? 0 : $self->{step});
        $self->set_increments($step, $self->{pageStep});
    }
}

{
    package GCCheckedText;
    @GCCheckedText::ISA = ('GCShortText');
    
    sub new
    {
        my ($proto, $format) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new;
        bless ($self, $class);
        my $forbidden = qr/[^$format]/;
        $self->signal_connect('insert-text' => sub {
                # Remove forbidden characters
                $_[1] =~ s/$forbidden//g;
                () # this callback must return either 2 or 0 items.
        });

        return $self;
    }
}

{
    package GCRange;
    
    @GCRange::ISA = ('Gtk2::HBox', 'GCGraphicComponent');
    
    sub new
    {
        my ($proto, $type, $lang, $min, $max, $step) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new;
        bless ($self, $class);

        if ($type eq 'text')
        {
            $self->{from} = new GCShortText;
            $self->{to} = new GCShortText;
        }
        elsif ($type eq 'numeric text')
        {
            $self->{from} = new GCCheckedText('0-9.');
            $self->{to} = new GCCheckedText('0-9.');
        }
        else
        {
            $self->{from} = new GCNumeric($min, $min, $max, $step);
            $self->{to} = new GCNumeric($max, $min, $max, $step);
        }
        $self->pack_start(Gtk2::Label->new($lang->{PanelFrom}), 0, 0, 12);
        $self->pack_start($self->{from}, 1, 1, 0);
        $self->pack_start(Gtk2::Label->new($lang->{PanelTo}), 0, 0, 12);
        $self->pack_start($self->{to}, 1, 1, 0);
        
        return $self;
    }
    
    sub isEmpty
    {
        my $self = shift;
        
        return $self->getValue eq '';
    }
    
    sub getValue
    {
        my $self = shift;
        return $self->{from}->getValue.';'.$self->{to}->getValue;
    }
    
    sub setValue
    {
        my ($self, $value) = @_;
        my @values = split m/;/, $value;
        $self->{from}->setValue($values[0]);
        $self->{to}->setValue($values[0]);
    }
    
    sub setWidth
    {
        my ($self, $value) = @_;
        $self->{from}->setWidth($value / 2);
        $self->{to}->setWidth($value / 2);
    }
    
    sub lock
    {
        my ($self, $locked) = @_;
    
        $self->{from}->lock(!$locked);
        $self->{to}->lock(!$locked);
    }
    
    sub signal_connect
    {
        my $self = shift;
        $self->{from}->signal_connect(@_);
        $self->{to}->signal_connect(@_);
    }
    
    sub AUTOLOAD
    {
        my $self = shift;
        my $name = our $AUTOLOAD;
        return if $name =~ /::DESTROY$/;
        $name =~ s/.*?::(.*)/$1/;
        $self->{from}->$name(@_);
        $self->{to}->$name(@_);
    }
}

{
    package GCDate;
    
    @GCDate::ISA = ('Gtk2::HBox', 'GCGraphicComponent');
    
    sub selectValue
    {
        my $self = shift;
        $self->{dialog} = new GCDateSelectionDialog($self->{mainParent})
            if ! $self->{dialog};
        $self->{dialog}->date($self->{entry}->get_text);
        if ($self->{dialog}->show)
        {
            $self->{entry}->set_text($self->{dialog}->date);
        }
        $self->{parent}->showMe;
    }

    sub new
    {
        my ($proto, $parent, $lang, $reverseDate) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new;
        bless ($self, $class);
        $self->{parent} = $parent;
        $self->getMainParent;
        $self->{reverseDate} = $reverseDate;
        $self->{entry} = Gtk2::Entry->new_with_max_length(10);
        $self->{entry}->set_width_chars(12);
        $self->{button} = new Gtk2::Button($lang->{PanelDateSelect});
        $self->{button}->signal_connect('clicked' => sub {
            $self->selectValue;
        });
        $self->pack_start($self->{entry}, 1, 1, 0);
        $self->pack_start($self->{button}, 0, 0, 0);

        return $self;
    }
    
    sub isEmpty
    {
        my $self = shift;
        
        return $self->getValue eq '';
    }
    
    sub getValue
    {
        my $self = shift;
        my $value = $self->{entry}->get_text;
        return GCPreProcess::reverseDate($value) if $self->{reverseDate};
        return $value;
    }

    sub setValue
    {
        my ($self, $text) = @_;
        $text = GCPreProcess::restoreDate($text) if $self->{reverseDate};
        $self->{entry}->set_text($text);
    }
    
    sub clear
    {
        my $self = shift;
        $self->setValue('');
    }

    sub setWidth
    {
        my ($self, $value) = @_;
        $self->set_width_chars($value);
    }
    
    sub lock
    {
        my ($self, $locked) = @_;
    
        $self->{entry}->can_focus(!$locked);
        $self->{button}->set_sensitive(!$locked);
    }
}

{
    package GCFile;
    
    @GCFile::ISA = ('Gtk2::HBox', 'GCGraphicComponent', 'GCPseudoHistoryComponent');

    use File::Basename;

    sub selectValue
    {
		my $self = shift;

        my $dialog = $self->getDialog;
		$dialog->set_filename($self->getValue);
		my $response = $dialog->run;
		if ($response eq 'ok')
		{
	        $self->setValue($dialog->get_filename);
        }
        $dialog->hide;
        $self->{parent}->showMe;
    }
    
    sub setPatternFilter
    {
        my ($self, $patterns) = @_;
        
        $self->getDialog->set_pattern_filter($patterns)
            if $patterns;
    }
    
    sub getDialog
    {
        my $self = shift;
    
        if (! $self->{dialog})
        {
            $self->{dialog} = GCFileChooserDialog->new($self->{title},
                                                       $self->{mainParent},
                                                       $self->{type},
                                                       $self->{withFilter});
        }
        return $self->{dialog};
    }
    
    sub setType
    {
        my ($self, $type, $withFilter) = @_;
        
        if (($type != $self->{type}) || ($withFilter != $self->{withFilter}))
        {
            $self->{dialog}->destroy
                if $self->{dialog};
            $self->{dialog} = undef;
            $self->{type} = $type;
            $self->{withFilter} = $withFilter;
        }
    }
    
    sub new
    {
        my ($proto, $parent, $title, $type, $withFilter) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new;
        bless ($self, $class);
        $self->{parent} = $parent;
        $self->getMainParent;
 
        $title ||= $self->{parent}->{lang}->{PanelSelectFileTitle};
        $self->{title} = $title;
        $type ||= 'open';
        $self->{type} = $type;
        $withFilter = 0 if !$withFilter;
        $self->{withFilter} = $withFilter;

        $self->{entry} = Gtk2::Entry->new;
        $self->{button} = Gtk2::Button->new_from_stock('gtk-open');
        $self->{button}->signal_connect('clicked' => sub {
            $self->selectValue;
        });
        $self->pack_start($self->{entry}, 1, 1, 0);
        $self->pack_start($self->{button}, 0, 0, 0);

        return $self;
    }
    
    sub isEmpty
    {
        my $self = shift;
        
        return $self->getValue eq '';
    }
    
    sub getValue
    {
        my $self = shift;
        return $self->{entry}->get_text;
    }

    sub setValue
    {
        my ($self, $text) = @_;
        $self->{entry}->set_text($text);
    }
    
    sub clear
    {
        my $self = shift;
        $self->setValue('');
    }

    sub setWidth
    {
        my ($self, $value) = @_;
        $self->{entry}->set_width_chars($value);
    }
    
    sub lock
    {
        my ($self, $locked) = @_;
    
        $self->{entry}->can_focus(!$locked);
        $self->{button}->set_sensitive(!$locked);
    }
}

{
    package GCButton;
    
    @GCButton::ISA = ('Gtk2::Button', 'GCGraphicComponent');
    
    sub new
    {
        my ($proto, $label) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new($label);

        bless ($self, $class);
        return $self;
    }
    
    sub newFromStock
    {
        my ($proto, $stock, $nolabel, $label) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new_from_stock($stock);
        
        if ($nolabel)
        {
            my $tmpWidget = $self;
            $tmpWidget = $tmpWidget->child while ! $tmpWidget->isa('Gtk2::HBox');
            ($tmpWidget->get_children)[1]->destroy;
        }
        elsif ($label)
        {
            my $tmpWidget = $self;
            $tmpWidget = $tmpWidget->child while ! $tmpWidget->isa('Gtk2::HBox');
            ($tmpWidget->get_children)[1]->set_label($label);        
        }

        bless ($self, $class);
        return $self;
    }
    
    sub isEmpty
    {
        my $self = shift;
        
        return 0;
    }
    
    sub getValue
    {
        my $self = shift;
    }
    
    sub setValue
    {
        my ($self, $value) = @_;
        
        return $value;
    }

    sub clear
    {
        my $self = shift;
        $self->setValue('');
    }
    
    sub lock
    {
        my ($self, $locked) = @_;
    
        $self->set_sensitive(!$locked);
    }
    
    sub setWidth
    {
        my ($self, $value) = @_;
        $self->set_size_request($value, -1);
    }
}

{
    package GCUrlButton;
    
    @GCUrlButton::ISA = ('GCButton', 'GCGraphicComponent');
    
    sub new
    {
        my ($proto, $label, $opener) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new($label);

        $self->signal_connect('clicked' => sub {
            $opener->launch($self->{url}, 'url');
        });

        bless ($self, $class);
        return $self;
    }
        
    sub getValue
    {
        my $self = shift;
        
        return $self->{url};
    }
    
    sub setValue
    {
        my ($self, $value) = @_;
        
        $self->{url} = $value;
        $self->lock(!$self->{url});
    }

}

{
    package GCCheckBox;
    
    @GCCheckBox::ISA = ('Gtk2::CheckButton', 'GCGraphicComponent');
    
    sub new
    {
        my ($proto, $label) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new($label);

        bless ($self, $class);
        return $self;
    }
    
    sub isEmpty
    {
        my $self = shift;
        
        return 0;
    }
    
    sub getValue
    {
        my $self = shift;
        return 1 if ($self->get_active);
        return 0;
    }
    
    sub getValueAsText
    {
        my $self = shift;
        return 'true' if ($self->get_active);
        return 'false';
    }
    
    sub setValue
    {
        my ($self, $value) = @_;
        $self->set_active($value);
    }
    
    sub clear
    {
        my $self = shift;
        $self->setValue(0);
    }

    sub lock
    {
        my ($self, $locked) = @_;
    
        $self->set_sensitive(!$locked);
    }
}

{
    package GCCheckBoxWithIgnore;
    
    @GCCheckBoxWithIgnore::ISA = ('Gtk2::HBox', 'GCGraphicComponent');
    
    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new(0,0);
        bless ($self, $class);
        
        $self->{check}->[0] = new Gtk2::RadioButton(undef,$parent->{lang}->{CheckUndef});
        $self->{group} = $self->{check}->[0]->get_group;
        $self->{check}->[1] = new Gtk2::RadioButton($self->{group},$parent->{lang}->{CheckNo});
        $self->{check}->[2] = new Gtk2::RadioButton($self->{group},$parent->{lang}->{CheckYes});
        
        $self->pack_start($self->{check}->[0], 0, 0, 10);
        $self->pack_start($self->{check}->[1], 0, 0, 10);
        $self->pack_start($self->{check}->[2], 0, 0, 10);
        
        return $self;
    }
    
    sub isEmpty
    {
        my $self = shift;
        
        return $self->{check}->[0]->get_active;
    }
    
    sub getValue
    {
        my $self = shift;
        my $i = 0;
        foreach (@{$self->{check}})
        {
            last if $self->{check}->[$i]->get_active;
            $i++;
        }
        $i--;
        return $i;
    }
    
    sub setValue
    {
        my ($self, $value) = @_;
        $self->{check}->[$value + 1]->set_active(1);
    }
    
    sub clear
    {
        my $self = shift;
        $self->setValue(-1);
    }

    sub lock
    {
        my ($self, $locked) = @_;

        $self->set_sensitive(!$locked);
    }
}

{
    package GCMultipleList;
    
    @GCMultipleList::ISA = ('Gtk2::HBox', 'GCGraphicComponent');
    
    sub new
    {
        my ($proto, $parent, $number, $labels, $withHistory, $readonly, $useFiles) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new(0,0);

        $self->{number} = $number;
        $self->{readonly} = $readonly;
        $self->{withHistory} = $withHistory;

		my $hboxActions = new Gtk2::HBox(0,0);

        my @histories;
        my @listColumns;
        my $i;
        for $i (0..($number - 1))
        {
            push @histories, {'' => 1};
            push @listColumns, ($labels->[$i] => 'text');
            next if $readonly;
            if ($useFiles)
            {
                $self->{entries}->[$i] = GCFile->new($parent);
            }
            else
            {
                if ($withHistory)
                {
                    $self->{entries}->[$i] = GCHistoryText->new;
                }
                else
                {
                    $self->{entries}->[$i] = GCShortText->new;
                }
            }
            $self->{entries}->[$i]->setWidth(12);
            $hboxActions->pack_start($self->{entries}->[$i], 1, 1, 6);
        }

        $self->{histories} = [\@histories];

        $self->{box} = new Gtk2::VBox(0,0);
        $self->{list} = new Gtk2::SimpleList(@listColumns);
        for $i (0..($number - 1))
        {
            $self->{list}->set_column_editable($i, 1);
        }
        $self->{list}->unset_rows_drag_source;
        $self->{list}->unset_rows_drag_dest;
        #($self->{list}->get_column(0)->get_cell_renderers)[0]->set('wrap-mode' => 'word');

        for $i (0..($number - 1))
        {
            $self->{list}->get_column($i)->set_resizable(1);
        }
        my $scroll = new Gtk2::ScrolledWindow;
        $scroll->set_policy ('automatic', 'automatic');
        $scroll->set_shadow_type('etched-in');
        $scroll->set_size_request(-1, 120);
        $scroll->add($self->{list});
        $self->{box}->pack_start($scroll, 1, 1, 2);
        if (!$readonly)
        {
            $self->{addButton} = GCButton->newFromStock('gtk-add', 1);
            $self->{addButton}->signal_connect('clicked' => sub {
                $self->addValues;
            });
            $self->{removeButton} = GCButton->newFromStock('gtk-remove', 1);
            $hboxActions->pack_start($self->{addButton}, 0, 0, 6);
            $hboxActions->pack_start($self->{removeButton}, 0, 0, 6);
        }
        else
        {
            $self->{removeButton} = GCButton->newFromStock('gtk-remove', 0);
            $self->{clearButton} = GCButton->newFromStock('gtk-clear', 0);
            $self->{clearButton}->signal_connect('clicked' => sub {
                $self->clear;
            });            
            $hboxActions->pack_start($self->{removeButton}, 1, 0, 6);
            $hboxActions->pack_start($self->{clearButton}, 1, 0, 6);
        }
        $self->{box}->pack_start($hboxActions, 0, 0, 6);

        $self->{removeButton}->signal_connect('clicked' => sub {
            my @idx = $self->{list}->get_selected_indices;
            my $selected = $idx[0];
            splice @{$self->{list}->{data}}, $selected, 1;
            $selected-- if ($selected >= scalar(@{$self->{list}->{data}}));
            $selected = 0 if $selected < 0 ;
            $self->{list}->select($selected);
        });

        $self->pack_start($self->{box},1,$self->{readonly},0);

        bless ($self, $class);
        return $self;
    }

    sub expand
    {
        my $self = shift;
        $self->set_child_packing($self->{box},1,1,0,'start');
    }

    sub addValues
    {
        my ($self, @values) = @_;
        if (!$self->{readonly})
        {
            for my $i (0..($self->{number} - 1))
            {
                $values[$i] = $self->{entries}->[$i]->getValue if !$values[$i];
                $self->{entries}->[$i]->addHistory($values[$i])
                    if $self->{withHistory};
                $self->{entries}->[$i]->clear;
            }
        }
        push @{$self->{list}->{data}}, \@values;
    }

    sub isEmpty
    {
        my $self = shift;
        
        return $self->getValue eq '';
    }
    
    sub getValue
    {
        my $self = shift;
        my $result = {
                        line => []
                     };

        foreach (@{$self->{list}->{data}})
        {
            my $cols = {
                          col => []
                       };
            foreach my $value(@{$_})
            {
                push @{$cols->{col}}, $value;
            }
            push @{$result->{line}}, $cols;
        }
        
        my $formated = shift;
        if ($formated)
        {
            $result = GCPreProcess::multipleList($result, $self->{number});
        }
        
        
        return $result;
    }
    
    sub setValue
    {
        my ($self, $value) = @_;

        @{$self->{list}->{data}} = ();
        if (ref($value) eq 'HASH')
        {
            push @{$self->{list}->{data}}, $_->{col}
                foreach (@{$value->{line}})
        }
        else
        {
            my @values = split m/,/, $value;
            foreach (@values)
            {
                my @items = split m/;/;
                push @{$self->{list}->{data}}, \@items;
            }
        }
    }
    
    sub clear
    {
        my $self = shift;
        $self->setValue('');
    }
    
    sub lock
    {
        my ($self, $locked) = @_;
        return if $self->{readonly};
        $self->{addButton}->set_sensitive(!$locked);
        $self->{removeButton}->set_sensitive(!$locked);
        foreach (@{$self->{entries}})
        {
            $_->lock($locked);
        }
    }

    sub addHistory
    {
        my $self = shift;
        my $value = (scalar @_) ? shift : $self->getValue;
        my $noUpdate = shift;

        return if $self->{readonly};
        my $i;
        my $item;
        if (ref($value) eq 'HASH')
        {
            foreach (@{$value->{line}})
            {
                $i = 0;
                foreach $item(@{$_->{col}})
                {
                    $self->{entries}->[$i]->addHistory($item, $noUpdate);
                    $i++;                   
                }
            }
        }
        else
        {
            my @values = split m/,/, $value;
            foreach (@values)
            {
                my @items = split m/;/;
                $i = 0;
                foreach my $item(@items)
                {
                    $self->{entries}->[$i]->addHistory($item, $noUpdate);
                    $i++;
                }
                #push @{$self->{list}->{data}}, \@items;
            }
        }
    }

    sub setDropDown
    {
        my $self = shift;
        
        foreach (@{$self->{entries}})
        {
            $_->setDropDown;
        }
    }

    sub getValues
    {
        my $self = shift;
        return [] if $self->{readonly};
        my @array;
        foreach (@{$self->{entries}})
        {
            push @array, $_->getValues;
        }
        # = sort keys %{$self->{history}};
        return \@array;
    }
    
    sub setValues
    {
        my ($self, $values) = @_;
        return if $self->{readonly};
        my $i = 0;
        foreach (@$values)
        {
            $self->{entries}->[$i]->setValues($_);
            $i++;
        }
    }
}


{
    package GCItemImage;
    
    @GCItemImage::ISA = ('Gtk2::Image', 'GCGraphicComponent');

    use File::Spec;
    use File::Basename;
    
    sub new
    {
        my ($proto, $options, $defaultImage, $fixedSize, $width, $height) = @_;
        my $class = ref($proto) || $proto;
        my $self  = Gtk2::Image->new_from_file($defaultImage);
        $self->{options} = $options;
        $self->{defaultImage} = $defaultImage;
        $self->{displayedImage} = '';
        $self->{fixedSize} = $fixedSize;
        bless ($self, $class);
        if ($width && $height)
        {
            $self->{width} = $width;
            $self->{height} = $height;
        }
        else
        {
            $self->{width} = 120;
            $self->{height} = 160;
        }
        return $self;
    }
    
    sub isEmpty
    {
        my $self = shift;
        
        return $self->getValue eq '';
    }
    
    sub setValue
    {
        my ($self, $displayedImage) = @_;
        $self->{displayedImage} = $displayedImage;
        
        $displayedImage = GCUtils::getDisplayedImage($displayedImage,
                                                     $self->{defaultImage},
                                                     $self->{options}->file);
        my $pixbuf;
        eval
        {
            $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($displayedImage);
        };
        if ($@)
        {
            $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($self->{defaultImage});
        }
        $pixbuf = GCUtils::scaleMaxPixbuf($pixbuf, $self->{width}, $self->{height});
        $self->set_from_pixbuf($pixbuf);
        $self->set_size_request($self->{width}, $self->{height}) if $self->{fixedSize};
    }
    
    sub getValue
    {
        my $self = shift;
        return $self->{displayedImage};
    }
    
    sub clear
    {
        my $self = shift;
        $self->setValue('');
    }
    
    sub lock
    {
        my ($self, $locked) = @_;
    }
    
    sub setWidth
    {
        my ($self, $value) = @_;
        $self->{width} = $value;
    }

    sub setHeight
    {
        my ($self, $value) = @_;
        $self->{height} = $value;
    }
}

{
    package GCImageButton;

    @GCImageButton::ISA = ('Gtk2::Button', 'GCGraphicComponent');
    
    sub clearImage
    {
		my $self = shift;

        my $imagePrefix = $self->{mainParent}->{imagePrefix};
        if ($self->{imageFile} =~ /(\/|\\)$imagePrefix[0-9]*\./)
        {
            $self->{mainParent}->{items}->markToBeRemoved($self->{imageFile});
        }
        $self->setValueWithParent('');
    }

    sub changeImage
    {
		my ($self, $fileName) = @_;

        return 0 if $self->{locked};

        if (!$fileName)
        {
            if (! $self->{imageDialog})
            {
                $self->{imageDialog} = GCFileChooserDialog->new($self->{parent}->{lang}->{PanelImageTitle}, $self->{mainParent}, 'open');
                $self->{imageDialog}->setWithImagePreview(1);    
            }
        
            $self->{imageDialog}->set_filename($self->getValue);
            my $response = $self->{imageDialog}->run;
            $self->{imageDialog}->hide;
            $self->{parent}->showMe;
            if ($response eq 'ok')
            {
                $fileName = $self->{imageDialog}->get_filename;
            }
            else
            {
                return;
            }
        }

    	if ($fileName ne $self->{imageFile})
        {
            my $imagePrefix = $self->{mainParent}->{imagePrefix};
            if ($self->{imageFile} =~ /(\/|\\)$imagePrefix[0-9]*\./)
            {
                $self->{mainParent}->{items}->markToBeRemoved($self->{imageFile});
            }
        }
        my $image = $self->{mainParent}->transformPicturePath($fileName);
        $self->setValueWithParent($image);
    }
    
    sub isEmpty
    {
        my $self = shift;
        
        return $self->getValue eq '';
    }
    
    sub setValue
    {
        my ($self, $value) = @_;
        $self->{imageFile} = $value;
        $self->{img}->setValue($value);
    }

    sub setValueWithParent
    {
        my ($self, $value) = @_;
        $self->{imageFile} = $value;
        if ($self->{isCover})
        {
            my $previous = $self->{parent}->getAsHash;
            $self->{img}->setValue($value);
            my $new = $self->{parent}->getAsHash;
            $self->{mainParent}->{itemsList}->changeCurrent($previous, $new);
        }
        else
        {
            $self->{img}->setValue($value);
        }
    }

    sub getValue
    {
        my $self = shift;
        use GCUtils;
        return $self->{img}->getValue;
    }
    
    sub clear
    {
        my $self = shift;
        $self->setValue('');
    }

    sub new
    {
        my ($proto, $parent, $img, $isCover, $default) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;
        bless ($self, $class);

        $default = 'open' if !$default;

        $self->add($img);
        $self->{img} = $img;
        $self->{default} = $default;
        #$self->set_size_request(130,170);
        $self->{width} = -1;
        $self->{height} = -1;
        $self->{imageFile} = $img->getValue;
        
        # True if this is the cover used in image mode
        $self->{isCover} = $isCover;

        $self->{parent} = $parent;
        $self->getMainParent;
        $self->{tooltips} = $self->{mainParent}->{tooltips};

        my $tip = ($default eq 'open') ? $parent->{lang}->{PanelImageTipOpen} : $parent->{lang}->{PanelImageTipView};
        $tip .= $parent->{lang}->{PanelImageTipMenu};
        $self->{tooltips}->set_tip($self, $tip);
        $self->{imgContext} = new Gtk2::Menu;
        $self->{imgContext}->append(Gtk2::TearoffMenuItem->new());
        $self->{itemOpen} = Gtk2::ImageMenuItem->new_from_stock('gtk-open',undef);
        $self->{itemOpen}->signal_connect("activate" , sub {
            $self->changeImage;
        });
        $self->{imgContext}->append($self->{itemOpen});
        $self->{itemShow} = Gtk2::ImageMenuItem->new_from_stock('gtk-zoom-100',undef);
        $self->{itemShow}->signal_connect("activate" , sub {
            $self->showImage;
        });
        $self->{imgContext}->append($self->{itemShow});
        $self->{itemClear} = Gtk2::ImageMenuItem->new_from_stock('gtk-clear',undef);
        $self->{itemClear}->signal_connect("activate" , sub {
            $self->clearImage;
        });
        $self->{imgContext}->append($self->{itemClear});        
        $self->{imgContext}->show_all;
        $self->signal_connect('clicked' => sub {
            $self->changeImage if $self->{default} eq 'open';
            $self->showImage if $self->{default} eq 'view';
        });
        $self->signal_connect('button_press_event' => sub {
            my ($widget, $event) = @_;
            return 0 if $event->button ne 3;
            $self->{imgContext}->popup(undef, undef, undef, undef, $event->button, $event->time);
            return 0;
        });

        #Drag and drop a picture on a button
        $self->drag_dest_set('all', ['copy','private','default','move','link','ask']);
        my $target_list = Gtk2::TargetList->new();
        my $atom1 = Gtk2::Gdk::Atom->new('text/uri-list');
        my $atom2 = Gtk2::Gdk::Atom->new('text/plain');
        $target_list->add($atom1, 0, 0);
        $target_list->add($atom2, 0, 0);
        if ($^O =~ /win32/i)
        {
            my $atom2 = Gtk2::Gdk::Atom->new('DROPFILES_DND');
            $target_list->add($atom2, 0, 0);
        }
        $self->drag_dest_set_target_list($target_list);
        $self->signal_connect(drag_data_received => sub {
            my ($widget, $context, $widget_x, $widget_y, $data, $info,$time) = @_;
            my @files = split /\n/, $data->data;
            my $fileName = $files[0];
            if ($fileName =~ /^http/)
            {
                $fileName = $self->{mainParent}->downloadPicture($fileName);
            }
            else
            {
                $fileName =~ s/^file:\/\/?(.*)\W*$/$1/;
                $fileName =~ s/.$//ms;
            }
            $self->changeImage($fileName);
        });

        return $self;
    }
    
    sub lock
    {
        my ($self, $locked) = @_;
    
        $self->{locked} = $locked;
        for my $menu('Open', 'Clear')
        {
            $self->{'item'.$menu}->set_sensitive(!$locked);
        }		
    }
    
    sub showImage
    {
        my $self = shift;
        $self->{mainParent}->launch($self->getValue, 'image');
    }
    
    sub setWidth
    {
        my ($self, $value) = @_;
        $self->{width} = $value;
        $self->set_size_request($value, $self->{height});
        $self->{img}->{width} = $value - $GCUtils::margin;
    }

    sub setHeight
    {
        my ($self, $value) = @_;
        $self->{height} = $value;
        $self->set_size_request($self->{width}, $value);
        $self->{img}->{height} = $value - $GCUtils::margin;
    }
    
}

{
    package GCOldMenuList;

    @GCOldMenuList::ISA = ('Gtk2::OptionMenu', 'GCGraphicComponent');

    sub isEmpty
    {
        my $self = shift;
        
        my $idx = $self->get_history;
        $idx-- if $idx >= $self->{separatorPosition};
        $idx = 0 if $idx < 0;
        return $self->{'values'}->[$idx]->{displayed} eq '';
    }
    
    sub valueToDisplayed
    {
        my ($self, $value) = @_;
    
        foreach (@{$self->{'values'}})
        {
            return $_->{displayed} if $_->{value} eq $value
        }
        return '';
    }
    
    sub getValue
    {
        my $self = shift;

        my $result = 0;

        my $idx = $self->get_history;
        $idx-- if $idx >= $self->{separatorPosition};
        $result = $self->{'values'}->[$idx]->{value} if $idx > -1;
        return $result;
    }

    sub getDisplayedValue
    {
        my $self = shift;

        my $result = 0;

        my $idx = $self->get_history;
        $idx-- if $idx >= $self->{separatorPosition};
        $result = $self->{'values'}->[$idx]->{displayed} if $idx > -1;
        return $result;
    }

    sub setValue
    {
        my ($self, $value) = @_;

        $value = 0 if !$value;

        my $i = 0;
        if ($value)
        {
    		foreach (@{$self->{'values'}})
            {
                last if $_->{value} eq $value;
                $i++ if $i == $self->{separatorPosition};
                $i++;
            }
        }
        $i-- if ($self->{default} == -1) && ($i >= $self->{count});
        $self->set_history($i);
    }

    sub clear
    {
        my $self = shift;
        $self->set_history(0);
    }

    sub lock
    {
        my ($self, $locked) = @_;
        
        $self->set_sensitive(!$locked);
    }
    
    sub getValues
    {
        my $self = shift;
        
        my @values;
        return $self->{values};
    }
    
    sub setValues
    {
        my ($self, $values, $separatorPosition, $preserveValue) = @_;
        
        my $previous = $self->getValue if $preserveValue;
        
        $self->{values} = $values;
        #$self->{textValues} = [];
        $self->{separatorPosition} = $separatorPosition || 9999;
 
        if ($self->{menu})
        {
            $self->remove_menu;
            $self->{menu}->destroy;
        }
        
		$self->{menu} = new Gtk2::Menu();
        my $i = 0;
		foreach (@$values)
		{
            if ($i == $self->{separatorPosition})
            {
                $self->{menu}->append(Gtk2::SeparatorMenuItem->new);
                $i++;
            }
            #push @{$self->{textValues}}, $_->{displayed};
		    my $item = Gtk2::MenuItem->new_with_label($_->{displayed});
		    $self->{menu}->append($item);
		    $i++;
		}

        $self->{count} = $i;
        $self->set_menu($self->{menu});
        $self->{menu}->show_all;
        
        $self->setValue($previous) if $preserveValue;
    }
    
    sub setLastForDefault
    {
        my $self = shift;
        $self->{default} = -1;
    }
    
    sub new
    {
        my ($proto, $values, $separatorPosition) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;

        bless ($self, $class);

        $self->setValues($values, $separatorPosition) if $values;
        $self->{default} = 0;
        
        return $self;
    }
}

{
    package GCMenuList;

    @GCMenuList::ISA = ('Gtk2::ComboBox', 'GCGraphicComponent');

    our $separatorValue = 'GCSSeparator';

    sub isEmpty
    {
        my $self = shift;
        
        return ($self->{model}->get($self->get_active_iter))[1] eq '';
        my $idx = $self->get_history;
        $idx-- if $idx >= $self->{separatorPosition};
        $idx = 0 if $idx < 0;
        return $self->{'values'}->[$idx]->{displayed} eq '';
    }
    
    sub valueToDisplayed
    {
        my ($self, $value) = @_;
    
        foreach (@{$self->{'values'}})
        {
            return $_->{displayed} if $_->{value} eq $value
        }
        return '';
    }
    
    sub getValue
    {
        my $self = shift;
        my $iter = $self->get_active_iter;
        return ($self->{model}->get($iter))[0] if $iter;
        return '';
    }

    sub getDisplayedValue
    {
        my $self = shift;
        my $iter = $self->get_active_iter;
        return ($self->{model}->get($iter))[1] if $iter;
        return '';
    }

    sub setValue
    {
        my ($self, $value) = @_;

        $value = 0 if !$value;

        my $i = 0;
        if ($value)
        {
    		foreach (@{$self->{values}})
            {
                last if $_->{value} eq $value;
                $i++ if $i == $self->{separatorPosition};
                $i++;
            }
        }
        $i-- if ($self->{default} == -1) && ($i >= $self->{count});
        $self->set_active($i);
    }

    sub clear
    {
        my $self = shift;
        $self->set_active(0);
    }

    sub lock
    {
        my ($self, $locked) = @_;
        
        $self->set_sensitive(!$locked);
    }
    
    sub getValues
    {
        my $self = shift;
        
        my @values;
        return $self->{values};
    }
    
    sub setValues
    {
        my ($self, $values, $separatorPosition, $preserveValue) = @_;
        my $model = $self->{model};
        my $previous = $self->getValue if $preserveValue;
        $self->{values} = $values;
        $self->{separatorPosition} = $separatorPosition || 9999;

        $model->clear;
        my $i = 0;
        foreach (@$values)
        {
            if ($i == $self->{separatorPosition})
            {
                $model->set($model->append, 0 => 'GCSSeparator', 1 => '');
                $i++;
            }
            $model->set($model->append,
                        0 => $_->{value},
                        1 => $_->{displayed});
            $i++;
        }

        $self->{count} = $i;
        $self->setValue($previous) if $preserveValue;
        $self->set_active(0) if !$preserveValue;
    }
    
    sub setLastForDefault
    {
        my $self = shift;
        $self->{default} = -1;
    }
    
    sub new
    {
        my ($proto, $values, $separatorPosition) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;
        bless ($self, $class);

        $self->{model} = Gtk2::ListStore->new('Glib::String', 'Glib::String');
        $self->set_model($self->{model});
        my $renderer = Gtk2::CellRendererText->new;
        $self->pack_start($renderer, 1);
        $self->add_attribute($renderer, text => 1);
        $self->set_row_separator_func(sub {
            my ($model, $iter) = @_;
            return ($model->get($iter, 0))[0] eq $separatorValue;
        });

        $self->setValues($values, $separatorPosition) if $values;
        $self->{default} = 0;
        
        return $self;
    }
}

{
    package GCHeaderLabel;
    use base 'Gtk2::Label';
    
    sub new
    {
        my ($proto, $label) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;
        bless ($self, $class);

        $self->setText($label);
        $self->set_alignment(0,1);
        
        return $self;
    }
    
    sub setText
    {
        my ($self, $label) = @_;
        $self->set_markup('<b>'.$label.'</b>');        
    }
}

{
    package GCLabel;
    
    @GCLabel::ISA = ('Gtk2::Label', 'GCGraphicComponent');

    sub new
    {
        my ($proto, $label) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;
        bless ($self, $class);
        $self->set_markup($label);
        $self->set_alignment(0,0.5);
        return $self;
    }
}

{
    package GCColorLabel;
    
    @GCColorLabel::ISA = ('Gtk2::EventBox', 'GCGraphicComponent', 'GCPseudoHistoryComponent');

    sub new
    {
        my ($proto, $color, $listType) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;

        bless ($self, $class);

        $listType = 0 if !$listType;
        $self->modify_bg('normal', $color);
        $self->{label} = Gtk2::Label->new;
        $self->{label}->show;
        $self->{hboxFill} = new Gtk2::HBox(0,0);
        $self->{hboxFill}->pack_start($self->{label},1,1,0);
        $self->add($self->{hboxFill});

        $self->initHistory($listType);

        return $self;
    }

    sub acceptMarkup
    {
        my $self = shift;
        return 1;
    }

    sub setMarkup
    {
        my ($self, $text) = @_;
        
        $text =~ s/&/&amp;/g;
        $self->{label}->set_markup($text);
    }

    sub getValue
    {
        my $self = shift;

        (my $label = $self->{label}->get_label) =~ s/<.*?>(.*?)<\/.*?>/$1/g;
        return $label;
    }

    sub setBgColor
    {
        my ($self, $color) = @_;
        $self->modify_bg('normal', $color);
    }

    sub set_justify
    {
        my ($self, $value) = @_;
        $self->{label}->set_justify($value);
        $self->{label}->set_alignment(0.5,0) if $value eq 'center';
        $self->{label}->set_alignment(1,0) if $value eq 'right';
    }
    
    sub AUTOLOAD
    {
        my $self = shift;
        my $name = our $AUTOLOAD;
        return if $name =~ /::DESTROY$/;
        $name =~ s/.*?::(.*)/$1/;
        $self->{label}->$name(@_);
    }
}

{
    package GCColorTable;
    
    @GCColorTable::ISA = ('Gtk2::Table', 'GCGraphicComponent', 'GCPseudoHistoryComponent');

    sub new
    {
        my ($proto, $columns, $labels, $headerStyle, $contentStyle, $topHeader) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new(1, $columns);

        bless ($self, $class);

        $self->set_col_spacings(3);
        $self->set_row_spacings(3);
        $self->{number} = $columns;
        $self->{style} = $contentStyle;
        $self->{firstRow} = 0;
        if ($topHeader)
        {
            $self->{firstRow}++;
            my $top = new GCColorLabel($headerStyle->{bgColor});
            $top->set_padding(6,6);
            $top->setMarkup('<span '.$headerStyle->{style}.'>'.$topHeader.'</span>');
            $self->attach($top, 0, $columns, 0, 1, ['expand', 'fill'], 'fill', 0, 0);            
        }
        my $i;
        if ($columns > 1)
        {
            for $i(0..($columns - 1))
            {
                my $header = new GCColorLabel($headerStyle->{bgColor});
                $header->set_padding(6,6);
                $header->setMarkup('<span '.$headerStyle->{style}.'>'.$labels->[$i].'</span>');
                $self->attach($header, $i, $i + 1, $self->{firstRow}, $self->{firstRow} + 1, ['expand', 'fill'], 'fill', 0, 0);
            }
            $self->{firstRow}++;
        }
        $self->initHistory($columns);
        return $self;
    }

    sub setValue
    {
        my ($self, $value) = @_;
        $self->{value} = $value;
 
        foreach (@{$self->{labels}})
        {
            $self->remove($_);
            $_->destroy;
        }

        $self->{labels} = [];
        my @lines;
        if (ref($value) eq 'HASH')
        {
            @lines = @{$value->{line}};
        }
        else
        {
            $value =~ s/^\s*//;
            @lines = split m/,/, $value;
        }
        if ($#lines < 0)
        {
            $self->hide_all;
            return;
        }
        my $i = $self->{firstRow};
        $self->resize($#lines + 1 + $self->{firstRow}, $self->{number});
        foreach (@lines)
        {
            my @cols;
            if (ref($value) eq 'HASH')
            {
                @cols = @{$_->{col}};
            }
            else
            {
                @cols = split m/;/, $_;
            }
            my $j = 0;
            for my $col(@cols)
            {
                my $label = new GCColorLabel($self->{style}->{bgColor});
                $label->set_padding(6,6);
                $label->setMarkup('<span '.$self->{style}->{style}.'>'.$col.'</span>');
                $self->attach($label, $j, $j + 1, $i, $i + 1, ['expand', 'fill'], 'fill', 0, 0);
                push @{$self->{labels}}, $label;
                $j++;
            }
            $i++;
        }
        $self->show_all;
    }

    sub getValue
    {
        my $self = shift;
        return $self->{value};
    }
    
    sub setBgColor
    {
        my ($self, $color) = @_;
        return;
    }

    sub set_justify
    {
        my ($self, $value) = @_;
        return;
    }
    
}

{
    package GCColorText;

    @GCColorText::ISA = ('Gtk2::ScrolledWindow', 'GCGraphicComponent');

    sub new
    {
        my ($proto, $color, $height) = @_;
        my $class = ref($proto) || $proto;
    
        my $self = $class->SUPER::new;

        $self->{text} = new Gtk2::TextView;
        $self->{text}->set_editable(0);
        $self->{text}->set_wrap_mode('word');
        $self->{background} = $color;
        $self->{text}->modify_base('normal', $color);
        $self->{text}->modify_bg('normal', $color);
        $self->{text}->set_border_width($GCUtils::halfMargin);
        $self->set_border_width(0);
        $self->set_shadow_type('none');
        $self->set_policy('automatic', 'automatic');
        $self->add($self->{text});

        my $layout = $self->create_pango_layout('G');
        my (undef, $rect) = $layout->get_pixel_extents;
        $self->{em} = 1.5 * $rect->{height};        

        bless ($self, $class);
        return $self;
    }

    sub getTagFromSpan
    {
        my ($self, $desc) = @_;
        my @result = (); #('background' => $self->{background});
        my @keyvalues = split / /, $desc;
        foreach (@keyvalues)
        {
            /([^=]*)=(.*)/;
            my $key = $1;
            my $value = $2;
            $value =~ s/('|")//g;
            #"'
            next if $key =~ /=/;
            push @result, ($key, $value);
        }
        return @result;
        
    }
    
    sub acceptMarkup
    {
        my $self = shift;
        return 1;
    }

    sub setMarkup
    {
        my ($self, $text) = @_;
        
        $text =~ s/&/&amp;/g;
        if ($self->{resize})
        {
            $self->resize;
        }
        else
        {
            my $buffer = $self->{text}->get_buffer;
            $self->{text}->get_buffer->set_text('');
            $text =~ s|<span ([^>]*?)>([^<]*?)</span>|$2|;
            if ($self->{spanTag})
            {
                $buffer->get('tag-table')->remove($self->{spanTag});
            }
            $self->{spanTag} = $buffer->create_tag('span', $self->getTagFromSpan($1));
            $buffer->insert_with_tags_by_name ($buffer->get_start_iter, $text, 'span');
        }
    }

    sub getValue
    {
        my $self = shift;

        (my $label = $self->{text}) =~ s/<.*?>(.*?)<\/.*?>/$1/g;
        return $label;
    }

    sub setHeight
    {
        my ($self, $height) = @_;
        $height =~ s/^([0-9]+)em$/$1*$self->{em}/e;
        $self->set_size_request(-1, $height);
    }

    sub AUTOLOAD
    {
        my $self = shift;
        my $name = our $AUTOLOAD;
        return if $name =~ /::DESTROY$/;
        $name =~ s/.*?::(.*)/$1/;
        $self->{text}->$name(@_);
    }
}

{
    package GCDialogHeader;
    
    use base 'Gtk2::HBox';
    
    sub new
    {
        my ($proto, $text, $imageStock, $logosDir) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;

        bless ($self, $class);
        
        $self->{label} = new Gtk2::Label;
        $self->{label}->set_markup("<span size='large' weight='bold'>$text</span>");
        $self->{label}->set_alignment(0,0.5);
        
        if (-f $logosDir.$imageStock.'.png')
        {
            $self->{image} = Gtk2::Image->new_from_file($logosDir.$imageStock.'.png');
            $self->pack_start($self->{image},0,1,5);
        }
        
        $self->pack_start($self->{label},0,1,5);
        
        return $self;
    }
}

{
    package GCImageBox;
    
    use base 'Gtk2::VBox';
    
    sub new_from_file
    {
        my ($proto, $imageFile, $label) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new();

        bless ($self, $class);

        my $image = Gtk2::Image->new_from_file($imageFile);
        $self->init($image, $label);
        
        return $self;
    }
    sub new_from_stock
    {
        my ($proto, $stockId, $label) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new();

        bless ($self, $class);

        my $image = Gtk2::Image->new_from_stock($stockId, 'large-toolbar');
        $self->init($image, $label);
         
        return $self;        
    }
    
    sub init
    {
        my ($self, $image, $label) = @_;
        
        $self->{label} = new Gtk2::Label($label);
        
        $self->pack_start($image, 0, 0, 0);
        $self->pack_start($self->{label}, 0, 0, 0);
        
        $self->show_all;
    }
}

{
    package GCGroup;

    @GCGroup::ISA = ('Gtk2::Frame', 'GCGraphicComponent');
    
    sub new
    {
        my ($proto, $title) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;

        bless ($self, $class);

        $self->set_shadow_type('none');
        $self->set_border_width($GCUtils::margin);
        $self->{label} = new Gtk2::Label;
        $self->{label}->set_padding(0,0);
        #$label->set_border_width(0);
        $self->setLabel($title);
        $self->set_label_widget($self->{label});
        $self->set_label_align(0,0);

        $self->{marginBox} = new Gtk2::HBox(0,0);
        $self->{marginBox}->set_border_width($GCUtils::halfMargin);
        $self->add($self->{marginBox});

        return $self;        
    }
    
    sub setLabel
    {
        my ($self, $label) = @_;
        
        $self->{label}->set_markup('<b>'.$label.'</b>');
    }
    
    sub addWidget
    {
        my ($self, $widget, $margin) = @_;
        $margin = $GCUtils::halfMargin if !$margin;
        $self->{marginBox}->pack_start($widget, 1, 1, $margin);
    }
    
    sub setPadding
    {
        my ($self, $value) = @_;
        
        $self->{marginBox}->set_border_width($value);
    }
}

{
    package GCFieldSelector;
    @GCFieldSelector::ISA = ('GCMenuList');

    sub new
    {
        my ($proto, $withImages, $model) = @_;
        my $class = ref($proto) || $proto;
        my $self = $class->SUPER::new;
        bless($self, $class);
        $self->{withImages} = $withImages;
        $self->setModel($model) if $model;
        return $self;
     }
     
     sub setModel
     {
        my ($self, $model) = @_;
        
        my @values = ({value => '', displayed => ''});
        foreach (@{$model->{fieldsNames}})
        {
            next if ($model->{fieldsInfo}->{$_}->{type} eq 'image') && (!$self->{withImages});
            next if ! $model->{fieldsInfo}->{$_}->{displayed};
            push @values, {value => $_, 
                           displayed => $model->{fieldsInfo}->{$_}->{displayed}};
        }
        $self->setValues(\@values);
        
     }
}

{
    package GCComparisonSelector;
    @GCComparisonSelector::ISA = ('GCMenuList');

    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
        my $self = $class->SUPER::new;
        bless($self, $class);
        $self->setValues([
                          {value => 'contain', displayed => $parent->{lang}->{ModelFilterContain}},
                          {value => 'range', displayed => $parent->{lang}->{ModelFilterRange}},
                          {value => 'eq', displayed => '='},
                          {value => 'lt', displayed => '<'},
                          {value => 'le', displayed => '<='},
                          {value => 'gt', displayed => '>'},
                          {value => 'ge', displayed => '>='},
                        ]);
        return $self;
     }
}

1;
