/*
** Copyright (C) 04 Aug 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>

#include "status_text.h"
#include "verify.h"
#include "common_gtk.h"
#include "mainwindow.h"
#include "vector_commands.h"
#include "contractions.h"
#include "linebuffer.h"
#include "readers.h"
#include "modify_file_set.h"
#include "command.h"
#include "filepicker.h"
#include "locks.h"
#include "globals.h"

static GtkWidget *verify_serious_text; /* *verify_insignificant_text; */
static GtkWidget *verify_serious_scrollbar; /* *verify_insignificant_scrollbar; */
static GtkWidget *master_mount_point, *copy_mount_point, *prefs_autoscroll_text;
static GtkWidget *mounted_source, *frame, *hbox8, *progressbar;
static linebuffer diff_stdout, diff_stderr;
static char *discard_string = NULL;
static size_t discard_string_n = -1;
static GSList *source_paths = NULL;

static int new_source_paths(void) {
	int i;

	source_paths = NULL;
	if (GTK_TOGGLE_BUTTON(mounted_source)->active) {
		if (0 == strlen(gtk_entry_get_text(GTK_ENTRY(master_mount_point)))) {
			alert_user_of_error(_(" No mountpoint for master supplied "));
			return FALSE;
		}
		source_paths = g_slist_append(source_paths, gtk_entry_get_text(GTK_ENTRY(master_mount_point)));
	} else {
		if (0 == GTK_CLIST(clist)->rows) {
			alert_user_of_error(_("There are not paths added in the Data Files \n- aborting verify"));
			return FALSE;
		}
		for (i = 0; i < GTK_CLIST(clist)->rows; i++) {
			file_data *info;
			info = gtk_clist_get_row_data(GTK_CLIST(clist), i);
			g_assert (NULL != info);
			if (GTK_TOGGLE_BUTTON(mkisofs_include_dirname)->active) {
				/* in "normal" mode just add every file to the list */
				source_paths = g_slist_append(source_paths, info->realpath);
			} else {
				struct stat buf;
				int r;
	
				if (GTK_TOGGLE_BUTTON(mkisofs_follow_symlinks)->active)
					r = stat(info->realpath, &buf);
				else
					r = lstat(info->realpath, &buf);
				if (-1 == r)
					g_warning("%s::%i (l)stat returned -1 on %s",
							__FILE__, __LINE__, info->realpath);
				else if (S_ISDIR(buf.st_mode)) {
					/* we have a directory */
					struct dirent *d;
					DIR *rdir = opendir(info->realpath);
					if (NULL == rdir)
						g_warning("%s::%i opendir returned NULL on %s",
								__FILE__, __LINE__, info->realpath);
					else {
						/* go trough the whole directory and add every entry
						 * to the list */
						while (NULL != (d = readdir(rdir))) {
							char *s;
							if (strlen(info->realpath) > 0 &&
									!(strlen(info->realpath) == 1 && info->realpath[0] == '.')
									&& !(strlen(info->realpath) == 2 && info->realpath[0] == '.' && info->realpath[1] == '.')) {
								s = string_append(info->realpath, NULL);
								s = string_append(s, d->d_name);
								source_paths = g_slist_append(source_paths, s);
							}
						}
						closedir(rdir);
					}
				} else
					/* a file, treat it like in normal mode */
					source_paths = g_slist_append(source_paths, info->realpath);
			}
		}
	}
	return TRUE;
}

static cmd_v *get_diff_command(void) {
	char *path, *name, *tmp2;
	GSList *tmp;
	cmd_v *cmd;

	if (NULL == source_paths)
		return NULL;
	path = source_paths->data;
	tmp = source_paths;
	source_paths = source_paths->next;
	g_slist_free_1(tmp);

	cmd = get_new_cmd();
	add_option_to_cmd(cmd, diff_path);
	add_option_to_cmd(cmd, "-r");
	add_option_to_cmd(cmd, "--brief");
	add_option_to_cmd(cmd, path);
	name = string_append(gtk_entry_get_text(GTK_ENTRY(copy_mount_point)), NULL);
	if ('/' != name[strlen(name)])
		name = string_append(name, "/");
	tmp2 = strdup(path);
	if ('/' == tmp2[strlen(tmp2) - 1])
		tmp2[strlen(tmp2) - 1] = '\0';
	if (!GTK_TOGGLE_BUTTON(mounted_source)->active)
		name = string_append(name, my_basename(tmp2));
	add_option_to_cmd(cmd, name);
	free(name);
	free(tmp2);

	return cmd;
}


/* adds text to a texbox, scrolling down if necessary
 * discards text starting with discard_string
 * text : textbox
 * scrollbar : scrollbar of textbox
 * new_text : new text to be added
 */
static void verify_add_text(GtkWidget *text, GtkWidget *scrollbar, char *new_text) {
	if ((strlen(new_text) > discard_string_n)
		&& (0 == strncmp(new_text, discard_string, discard_string_n)))
			return;
	gtk_text_freeze(GTK_TEXT(text));
	gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL, new_text, -1);
	gtk_text_thaw(GTK_TEXT(text));

	if (GTK_TOGGLE_BUTTON(prefs_autoscroll_text)->active) {
		gtk_text_freeze(GTK_TEXT(text));
		scroll_down_text(text, scrollbar);
		gtk_text_thaw(GTK_TEXT(text));
	}
}

static int read_diff_stdout(int fd) {
	if (!read_into(&diff_stdout, fd))
		return FALSE;
	while (extract_line(&diff_stdout, "\n"))
		verify_add_text(verify_serious_text, verify_serious_scrollbar,
				diff_stdout.newline);
	return TRUE;
}

static int read_diff_stderr(int fd) {
	if (!read_into(&diff_stderr, fd))
		return FALSE;
	while (extract_line(&diff_stderr, "\n"))
		verify_add_text(verify_serious_text, verify_serious_scrollbar,
				diff_stderr.newline);
	return TRUE;
}

static int wait_diff_output(gpointer data) {
	static int diff_stdout_ok = TRUE, diff_stderr_ok = TRUE;
	int *fd = data;
	int max, result;
	fd_set rset;
	struct timeval tv;
	static int activity = 0;

	g_assert((diff_stderr_ok == TRUE) || (diff_stdout_ok == TRUE));
	gtk_progress_bar_update(GTK_PROGRESS_BAR(progressbar), (gfloat)(activity++)/100.0);
	if (activity > 100)
		activity = 0;
	tv.tv_sec = 0;
	tv.tv_usec = 0;

	FD_ZERO(&rset);
	max = -1;

	if (diff_stdout_ok) {
		FD_SET(fd[0], &rset);
		max = fd[0];
	}
	if (diff_stderr_ok) {
		FD_SET(fd[1], &rset);
		if (-1 == max)
			max = fd[1];
		else
			max = fd[0]>fd[1] ? fd[0] : fd[1];
	}
	result = select(max + 1, &rset, NULL, NULL, &tv);
	if (result < 0)
		g_error("%s::%i: select returned negative!", __FILE__, __LINE__);
/*	else if (result == 0) { nothing read
	} */
	if (diff_stdout_ok && FD_ISSET(fd[0], &rset))
		diff_stdout_ok = read_diff_stdout(fd[0]);
	if (diff_stderr_ok && FD_ISSET(fd[1], &rset))
		diff_stderr_ok = read_diff_stderr(fd[1]);

	if (!(diff_stdout_ok || diff_stderr_ok)) {
		cmd_v *cmd;
		diff_stdout_ok = diff_stderr_ok = TRUE;
		cmd = get_diff_command();
		destroy_buffer(&diff_stdout);
		destroy_buffer(&diff_stderr);
		if (NULL != cmd) {
			int *fd2;
			wait_pids();
			init_buffer(&diff_stdout);
			init_buffer(&diff_stderr);
			fd2 = popen_re_unbuffered(cmd);
			destroy_cmd(cmd);
			gtk_timeout_add(TIMEOUT_GTK, wait_diff_output, fd2);
			return FALSE;
		}
		verify_add_text(verify_serious_text, verify_serious_scrollbar,
				_("*** *** ***\nVerify finished.\n"));
		free(discard_string);
		not_running();
		gtk_widget_set_sensitive(frame, TRUE);
		gtk_widget_set_sensitive(hbox8, TRUE);
		gtk_progress_bar_update(GTK_PROGRESS_BAR(progressbar), 0.0);
		activity = 0;
		return FALSE;
	} else
		return TRUE;
}

/* verify clicked, start doing it */
static void verify_clicked(GtkWidget *widget, gpointer data) {
	cmd_v *cmd;
	int *fd;

	if (is_running()) {
		return;
	}
	if (!new_source_paths()) {
		not_running();
		return;
	}

	
	if (0 == strlen(gtk_entry_get_text(GTK_ENTRY(copy_mount_point)))) {
			alert_user_of_error(_(" No mountpoint for copy supplied "));
			not_running();
			return;
	}

	if ((GTK_TOGGLE_BUTTON(mounted_source)->active)
			&& (0 == strlen(gtk_entry_get_text(GTK_ENTRY(master_mount_point))))) {
			alert_user_of_error(_(" No mountpoint for master supplied "));
			not_running();
			return;
	}
	gtk_widget_set_sensitive(frame, FALSE);
	gtk_widget_set_sensitive(hbox8, FALSE);

	init_buffer(&diff_stdout);
	init_buffer(&diff_stderr);

	discard_string = string_append("Only in ", NULL);
	discard_string = string_append(discard_string, gtk_entry_get_text(GTK_ENTRY(copy_mount_point)));
	discard_string_n = strlen(discard_string);

	cmd = get_diff_command();
	g_assert(NULL != cmd);
	fd = popen_re_unbuffered(cmd);
	destroy_cmd(cmd);

	gtk_timeout_add(TIMEOUT_GTK, wait_diff_output, fd);
}

/* clear the textboxes */
static void delete_verify_text(GtkWidget *widget, gpointer data) {
	del_text(verify_serious_text);
/*	del_text(verify_insignificant_text); */
}

static void show_source_mountpoint(GtkWidget *widget, gpointer *data) {
	gtk_widget_set_sensitive(master_mount_point, TRUE);
}
static void hide_source_mountpoint(GtkWidget *widget, gpointer *data) {
	gtk_widget_set_sensitive(master_mount_point, FALSE);
}

/* creates and adds the verify widgets to container */
void create_verify(GtkWidget *container) {
	GtkWidget *vbox4, *vbox1;
	GtkWidget *vbox5;
	GtkWidget *hbox2;
	GtkWidget *hbox1;
	GtkWidget *button2;
	GtkWidget *hbox3;
	GtkWidget *label, *sep, *clear_text;
	GtkWidget *on_the_fly_source;
	GSList *vbox2_group = NULL;

	vbox4 = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(container), vbox4);

	vbox5 = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox4), vbox5, FALSE, TRUE, 0);

	hbox2 = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox5), hbox2, FALSE, TRUE, 0);

	frame = make_frame(_("Verify from:"), 5);
	gtk_box_pack_start(GTK_BOX(hbox2), frame, TRUE, TRUE, 0);

	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox1);

	on_the_fly_source = gtk_radio_button_new_with_label(vbox2_group, _("Data files"));
	vbox2_group = gtk_radio_button_group(GTK_RADIO_BUTTON(on_the_fly_source));
	gtk_box_pack_start(GTK_BOX(vbox1), on_the_fly_source, FALSE, FALSE, 0);
	gtk_tooltips_set_tip(tooltips, on_the_fly_source, _("Verify against the files selected "
				"in the data source window, don't use any image."), NULL);

	hbox1 = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox1, FALSE, TRUE, 0);

	mounted_source = gtk_radio_button_new_with_label(vbox2_group, _("Mounted filesystem"));
	vbox2_group = gtk_radio_button_group(GTK_RADIO_BUTTON(mounted_source));
	gtk_box_pack_start(GTK_BOX(hbox1), mounted_source, FALSE, FALSE, 0);
      gtk_tooltips_set_tip(tooltips, mounted_source, _("Verify against a mounted filesystem."),
				NULL);

	master_mount_point = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox1), master_mount_point, FALSE, TRUE, 0);
	gtk_tooltips_set_tip(tooltips, master_mount_point, _("Where the master image "
				"is mounted (for example /mnt/cdrom) - for now you must mount "
				"the master yourself."), NULL);
	gtk_widget_set_sensitive(master_mount_point, FALSE);

	gtk_signal_connect(GTK_OBJECT(on_the_fly_source),
			"button_release_event", (GtkSignalFunc) hide_source_mountpoint, NULL);
	gtk_signal_connect(GTK_OBJECT(mounted_source),
			"button_release_event", (GtkSignalFunc) show_source_mountpoint, NULL);

	hbox8 = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox5), hbox8, FALSE, TRUE, 0);

	label = gtk_label_new(_("Verify against (ie the copy):"));
	gtk_box_pack_start(GTK_BOX(hbox8), label, FALSE, TRUE, 5);
	copy_mount_point = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox8), copy_mount_point, FALSE, TRUE, 0);
	gtk_tooltips_set_tip(tooltips, copy_mount_point, _("Where the copy is mounted (for example "
				"/mnt/cdr) - for now you must mount the copy yourself"), NULL);

	button2 = gtk_button_new_with_label(_("Verify"));
	gtk_box_pack_end(GTK_BOX(hbox8), button2, FALSE, TRUE, 0);
	gtk_tooltips_set_tip(tooltips, button2, _("Start verifying "
				"- for now only a filesystem verify is available (basically "
				"a diff -r --brief master copy is done). The operation can take "
				"quite some time."), NULL);
	gtk_signal_connect(GTK_OBJECT(button2), "clicked",
			GTK_SIGNAL_FUNC(verify_clicked), 0);

	sep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox5), sep, FALSE, TRUE, 0);

	hbox3 = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox5), hbox3, TRUE, TRUE, 0);

	prefs_autoscroll_text = gtk_check_button_new_with_label(_("Autoscroll text"));
	gtk_box_pack_start(GTK_BOX(hbox3), prefs_autoscroll_text, FALSE, TRUE, 0);
	gtk_tooltips_set_tip(tooltips, prefs_autoscroll_text, _("Scroll down to "
				"the end of the text whenever something new is added"),
			NULL);
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(prefs_autoscroll_text), TRUE);

	progressbar = gtk_progress_bar_new();
	gtk_box_pack_start(GTK_BOX(hbox3), progressbar, TRUE, TRUE, 0);

	clear_text = gtk_button_new_with_label(_("Clear"));
	gtk_box_pack_end(GTK_BOX(hbox3), clear_text, FALSE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(clear_text), "clicked",
			GTK_SIGNAL_FUNC(delete_verify_text), 0);
	gtk_tooltips_set_tip(tooltips, clear_text, _("Clears the text boxes"),
			NULL);

	verify_serious_scrollbar = create_text(vbox4, &verify_serious_text,
			_("Differences between master/copy (if any) below:"));
/*	verify_insignificant_scrollbar = create_text(vbox4, &verify_insignificant_text,
			_("Insignificant errors")); */

	gtk_widget_show_all(container);
}
