/**
 * File:          $RCSfile: image_io.c,v $
 * Module:        Image file I/O functions
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.17 $
 * Last edited:   $Date: 2003/01/31 18:56:57 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include <gandalf/image/io/image_io.h>
#include <gandalf/image/io/png_io.h>
#include <gandalf/image/io/jpeg_io.h>
#include <gandalf/image/io/tiff_io.h>
#include <gandalf/image/io/pbm_io.h>
#include <gandalf/image/io/pgm_io.h>
#include <gandalf/image/io/ppm_io.h>
#include <gandalf/common/misc_error.h>

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \defgroup ImageIO Image I/O
 * \{
 */

/**
 * \brief Interprets a string as an image file format.
 * \param format_string File format string, e.g. "png"
 * \return #GAN_UNKNOWN_FORMAT on failure to interpret string, or else a
 *         successfully interpreted file format, e.g. GAN_PNG_FORMAT.
 *
 * Attempts to interpret the given string \a format_string as a file name
 * suffix or other identifying string for a specific image format known by
 * Gandalf, which is returned.
 */
Gan_ImageFileFormat
 gan_image_interpret_format_string ( const char *format_string )
{
#ifdef HAVE_PNG
   if ( !strcmp ( format_string, "PNG" ) || !strcmp ( format_string, "png" ) )
      return GAN_PNG_FORMAT;
#endif

#ifdef HAVE_JPEG
   if ( !strcmp ( format_string, "JPG" ) || !strcmp ( format_string, "jpg" ) ||
        !strcmp ( format_string, "JPEG" ) || !strcmp ( format_string, "jpeg" ))
      return GAN_JPEG_FORMAT;
#endif

#ifdef HAVE_TIFF
   if ( !strcmp ( format_string, "TIF" ) || !strcmp ( format_string, "tif" ) ||
        !strcmp ( format_string, "TIFF" ) || !strcmp ( format_string, "tiff" ))
      return GAN_TIFF_FORMAT;
#endif

   if ( !strcmp ( format_string, "PBM" ) || !strcmp ( format_string, "pbm" ) )
      return GAN_PBM_FORMAT;

   if ( !strcmp ( format_string, "PGM" ) || !strcmp ( format_string, "pgm" ) )
      return GAN_PGM_FORMAT;

   if ( !strcmp ( format_string, "PPM" ) || !strcmp ( format_string, "ppm" ) )
      return GAN_PPM_FORMAT;

   gan_err_flush_trace();
   gan_err_register ( "gan_image_interpret_format_string", GAN_ERROR_FAILURE,
                      "failed to determine format" );
   return GAN_UNKNOWN_FORMAT;
}

static Gan_ImageFileFormat
 get_file_format ( const char *filename )
{
   int i;

   /* locate suffix string following last dot in file name */
   for ( i = strlen(filename)-1; i >= 0; i-- )
      if ( filename[i] == '.' ) break;

   if ( i < 0 )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_file_format", GAN_ERROR_FAILURE, "" );
      return GAN_UNKNOWN_FORMAT;
   }
      
   return gan_image_interpret_format_string ( &filename[i+1] );
}

/**
 * \brief Reads an image file from a file stream.
 * \param infile Input file stream
 * \param file_format The format of the file, e.g. #GAN_PNG_FORMAT
 * \param image The image structure to read the image data into or \c NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads an image from the given file stream \a infile into the provided
 * \a image structure. If \a image is \c NULL a new image is dynamically
 * allocated; otherwise the already allocated \a image structure is reused.
 *
 * If the file format is known, it should be specified by \a file_format;
 * otherwise pass \a file_format as #GAN_UNKNOWN_FORMAT and the function will
 * try to determine the file format.
 *
 * \sa gan_write_image_stream().
 */
Gan_Image *
 gan_image_read_stream ( FILE *infile, Gan_ImageFileFormat file_format,
                         Gan_Image *image )
{
   switch ( file_format )
   {
#ifdef HAVE_PNG
      case GAN_PNG_FORMAT:
        image = gan_read_png_image_stream ( infile, image, 1.0 );
        break;
#endif

#ifdef HAVE_JPEG
      case GAN_JPEG_FORMAT:
        image = gan_read_jpeg_image_stream ( infile, image );
        break;
#endif

      case GAN_PBM_FORMAT:
        image = gan_read_pbm_image_stream ( infile, image );
        break;

      case GAN_PGM_FORMAT:
        image = gan_read_pgm_image_stream ( infile, image );
        break;

      case GAN_PPM_FORMAT:
        image = gan_read_ppm_image_stream ( infile, image );
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_read_stream", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        return NULL;
   }

   /* check that image was read successfully */
   if ( image == NULL )
   {
      gan_err_register ( "gan_image_read_stream", GAN_ERROR_FAILURE, "" );
      return NULL;
   }

   /* success */
   return image;
}

/**
 * \brief Reads an image file.
 * \param filename The name of the image file
 * \param file_format The format of the file, e.g. GAN_PNG_FORMAT
 * \param image The image structure to read the image data into or NULL
 * \return Pointer to \a image structure, or \c NULL on failure.
 *
 * Reads an image from the given file \a filename into the provided \a image
 * structure. If \a image is \c NULL a new image is dynamically allocated;
 * otherwise the already allocated \a image structure is reused.
 *
 * If the file format is known, it should be specified in \a file_format;
 * otherwise pass \a file_format as #GAN_UNKNOWN_FORMAT and the function will
 * try to determine the file format.
 *
 * \sa gan_write_image().
 */
Gan_Image *
 gan_image_read ( const char *filename, Gan_ImageFileFormat file_format,
                  Gan_Image *image )
{
   /* determine actual file format if necessary */
   if ( file_format == GAN_UNKNOWN_FORMAT )
      file_format = get_file_format(filename);

   if ( file_format == GAN_UNKNOWN_FORMAT )
   {
      /* can't work out the image format */
      gan_err_register ( "gan_image_read", GAN_ERROR_FAILURE, "" );
      return NULL;
   }

   switch ( file_format )
   {
#ifdef HAVE_PNG
      case GAN_PNG_FORMAT:
        image = gan_read_png_image ( filename, image, 1.0 );
        break;
#endif

#ifdef HAVE_JPEG
      case GAN_JPEG_FORMAT:
        image = gan_read_jpeg_image ( filename, image );
        break;
#endif

#ifdef HAVE_TIFF
      case GAN_TIFF_FORMAT:
        image = gan_read_tiff_image ( filename, image );
        break;
#endif

      case GAN_PBM_FORMAT:
        image = gan_read_pbm_image ( filename, image );
        break;

      case GAN_PGM_FORMAT:
        image = gan_read_pgm_image ( filename, image );
        break;

      case GAN_PPM_FORMAT:
        image = gan_read_ppm_image ( filename, image );
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_read", GAN_ERROR_ILLEGAL_TYPE, "" );
        return NULL;
   }

   /* check that image was read successfully */
   if ( image == NULL )
   {
      gan_err_register ( "gan_image_read", GAN_ERROR_FAILURE, "" );
      return NULL;
   }

   /* success */
   return image;
}

/**
 * \brief Writes an image file to a file stream.
 * \param outfile Output file stream
 * \param file_format The format of the file, e.g. #GAN_PNG_FORMAT
 * \param image The image structure to be written to the file
 * \return #GAN_TRUE on successful write operation, #GAN_FALSE on failure.
 *
 * Writes an image from the provided image structure into the given file
 * stream.
 *
 * The given file format \a file_format should support the format and type of
 * the \a image; otherwise an error is reported and #GAN_FALSE is returned.
 *
 * \sa gan_read_image().
 */
Gan_Bool
 gan_image_write_stream ( FILE *outfile, Gan_ImageFileFormat file_format,
                          Gan_Image *image )
{
   Gan_Bool result;
   switch ( file_format )
   {
#ifdef HAVE_PNG
      case GAN_PNG_FORMAT:
        result = gan_write_png_image_stream ( outfile, image, 1.0 );
        break;
#endif

#ifdef HAVE_JPEG
      case GAN_JPEG_FORMAT:
        result = gan_write_jpeg_image_stream ( outfile, image, 50 );
        break;
#endif

      case GAN_PBM_FORMAT:
        result = gan_write_pbm_image_stream ( outfile, image );
        break;

      case GAN_PGM_FORMAT:
        result = gan_write_pgm_image_stream ( outfile, image );
        break;

      case GAN_PPM_FORMAT:
        result = gan_write_ppm_image_stream ( outfile, image );
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_write_stream", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        return GAN_FALSE;
   }

   if ( !result )
   {
      gan_err_register ( "gan_image_write_stream", GAN_ERROR_FAILURE, "" );
      return GAN_FALSE;
   }

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Writes an image file.
 * \param filename The name of the image file
 * \param file_format The format of the file, e.g. #GAN_PNG_FORMAT
 * \param image The image structure to be written to the file
 * \return #GAN_TRUE on successful write operation, #GAN_FALSE on failure.
 *
 * Writes an image from the provided \a image structure into the given file.
 *
 * The given file format \a file_format should support the format and type
 * of the image; otherwise an error is reported and #GAN_FALSE is returned.
 *
 * \sa gan_read_image().
 */
Gan_Bool
 gan_image_write ( const char *filename, Gan_ImageFileFormat file_format,
                   Gan_Image *image )
{
   Gan_Bool result;
   switch ( file_format )
   {
#ifdef HAVE_PNG
      case GAN_PNG_FORMAT:
        result = gan_write_png_image ( filename, image, 1.0 );
        break;
#endif

#ifdef HAVE_JPEG
      case GAN_JPEG_FORMAT:
        result = gan_write_jpeg_image ( filename, image, 50 );
        break;
#endif

#ifdef HAVE_TIFF
      case GAN_TIFF_FORMAT:
        result = gan_write_tiff_image ( filename, image );
        break;
#endif

      case GAN_PBM_FORMAT:
        result = gan_write_pbm_image ( filename, image );
        break;

      case GAN_PGM_FORMAT:
        result = gan_write_pgm_image ( filename, image );
        break;

      case GAN_PPM_FORMAT:
        result = gan_write_ppm_image ( filename, image );
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_image_write", GAN_ERROR_ILLEGAL_TYPE, "" );
        return GAN_FALSE;
   }

   if ( !result )
   {
      gan_err_register ( "gan_image_write", GAN_ERROR_FAILURE, "" );
      return GAN_FALSE;
   }

   /* success */
   return GAN_TRUE;
}
