/***************************************************************************

  CDatabase.c

  (c) 2000-2004 Benot Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

#define __CDATABASE_C

#include "main.h"

#include "CDatabase.h"


static int valid_database(CDATABASE *_object)
{
  return (THIS->conn->handle == NULL);
}

static bool check_database(CCONNECTION *conn, const char *name, bool must_exist)
{
  bool exist = conn->driver->Database.Exist(conn->handle, (char *)name);

  if (must_exist)
  {
    if (!exist)
    {
      GB.Error("Unknown database: &1", name);
      return TRUE;
    }
  }
  else
  {
    if (exist)
    {
      GB.Error("Database already exists: &1", name);
      return TRUE;
    }
  }

  return FALSE;
}


void *CDATABASE_get(CCONNECTION *conn, const char *name)
{
  CDATABASE *_object;
  
  if (check_database(conn, name, TRUE))
    return NULL;
  
  GB.New((void **)&_object, GB.FindClass("Database"), NULL, NULL);
  THIS->conn = conn;
  GB.Ref(conn);
  THIS->driver = conn->driver;
  GB.NewString(&THIS->name, name, 0);
  return THIS;
}


int CDATABASE_exist(CCONNECTION *conn, const char *name)
{
  return conn->driver->Database.Exist(conn->handle, (char *)name);
}


void CDATABASE_list(CCONNECTION *conn, char ***list)
{
  conn->driver->Database.List(conn->handle, list);
}


/***************************************************************************

  Database

***************************************************************************/

BEGIN_METHOD_VOID(CDATABASE_free)

  if (!valid_database(THIS))
    GB.SubCollection.Remove(THIS->conn->databases, THIS->name, 0);
  GB.Unref((void **)&THIS->conn);
  GB.FreeString(&THIS->name);

END_METHOD


BEGIN_PROPERTY(CDATABASE_name)

  GB.ReturnString(THIS->name);

END_PROPERTY


BEGIN_PROPERTY(CDATABASE_system)

  GB.ReturnBoolean(THIS->driver->Database.IsSystem(THIS->conn->handle, THIS->name));

END_PROPERTY


BEGIN_METHOD_VOID(CDATABASE_delete)

  THIS->conn->driver->Database.Delete(THIS->conn->handle, THIS->name);

END_METHOD


BEGIN_PROPERTY(CDATABASE_connection)

  GB.ReturnObject(THIS->conn);

END_PROPERTY




GB_DESC CDatabaseDesc[] =
{
  GB_DECLARE("Database", sizeof(CDATABASE)),
  GB_NOT_CREATABLE(),
  GB_HOOK_CHECK(valid_database),

  GB_METHOD("_free", NULL, CDATABASE_free, NULL),
  
  GB_METHOD("Delete", NULL, CDATABASE_delete, NULL),
  
  GB_PROPERTY_READ("Name", "s", CDATABASE_name),
  GB_PROPERTY_READ("System", "b", CDATABASE_system),
  GB_PROPERTY_READ("Connection", "Connection", CDATABASE_connection),

  GB_END_DECLARE
};



/***************************************************************************

  .ConnectionDatabases

***************************************************************************/

#undef THIS
#define THIS ((GB_SUBCOLLECTION)_object)

BEGIN_METHOD(CDATABASE_add, GB_STRING name)

  CCONNECTION *conn = GB.SubCollection.Container(THIS);
  char *name = GB.ToZeroString(ARG(name));

  if (DB_CheckName(name, "database"))
    return;

  if (check_database(conn, name, FALSE))
    return;

  conn->driver->Database.Create(conn->handle, name);

END_METHOD


BEGIN_METHOD(CDATABASE_remove, GB_STRING name)

  CCONNECTION *conn = GB.SubCollection.Container(THIS);
  char *name = GB.ToZeroString(ARG(name));

  if (check_database(conn, name, TRUE))
    return;

  GB.SubCollection.Remove(THIS, STRING(name), LENGTH(name));
  conn->driver->Database.Delete(conn->handle, name);

END_METHOD



GB_DESC CConnectionDatabasesDesc[] =
{
  GB_DECLARE(".ConnectionDatabases", 0), GB_INHERITS(".SubCollection"),

  GB_METHOD("Add", NULL, CDATABASE_add, "(Name)s"),
  GB_METHOD("Remove", NULL, CDATABASE_remove, "(Name)s"),

  GB_END_DECLARE
};


