/**
 * @file service.c Galago Service message processor
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "filter.h"
#include "meta-service.h"
#include "person-list.h"
#include "utils.h"

static DBusHandlerResult
service_add_account(DBusConnection *dbus_conn, DBusMessage *message,
					GalagodCoCo *coco, GalagoService *service)
{
	GalagoAccount *account;
	GalagodAccountList *account_list;
	GalagodMetaService *meta_service;
	GalagoPerson *person;
	DBusMessageIter iter;
	DBusMessage *reply;
	char *obj_path;

	dbus_message_iter_init(message, &iter);
	account = galago_dbus_message_iter_get_object(&iter, GALAGO_TYPE_ACCOUNT);
	g_object_set_data(G_OBJECT(account), "coco", coco);

	meta_service = GALAGOD_META_SERVICE(service);
	account_list = galagod_meta_service_get_account_list(meta_service);

	/* Send the reply containing the object path. */
	reply = dbus_message_new_method_return(message);
	dbus_message_iter_init_append(reply, &iter);

	obj_path = galagod_account_generate_path(account);
	galago_object_set_dbus_path(GALAGO_OBJECT(account), obj_path);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &obj_path);

	dbus_connection_send(dbus_conn, reply, NULL);
	dbus_message_unref(reply);

	g_free(obj_path);

	galagod_account_list_add(account_list, account);
	person = galago_account_get_person(account);
	galagod_people_account_added(person);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
service_remove_account(DBusConnection *dbus_conn, DBusMessage *message,
					   GalagodCoCo *coco, GalagoService *service)
{
	GalagoAccount *account;
	GalagodAccountList *account_list;
	GalagodMetaService *meta_service;
	DBusMessageIter iter;

	dbus_message_iter_init(message, &iter);
	account = galago_dbus_message_iter_get_object(&iter, GALAGO_TYPE_ACCOUNT);

	meta_service = GALAGOD_META_SERVICE(service);
	account_list = galagod_meta_service_get_account_list(meta_service);

	galagod_account_list_remove(account_list, account);

	g_object_unref(account);

	galagod_dbus_send_empty_reply(dbus_conn, message);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
service_get_account(DBusConnection *dbus_conn, DBusMessage *message,
					GalagodCoCo *coco, GalagoService *service)
{
	DBusMessage *reply;
	DBusMessageIter iter;
	GalagodAccountList *account_list;
	GalagodMetaService *meta_service;
	GalagodMetaAccount *meta_account;
	GalagoAccount *account = NULL;
	const char *username;

	dbus_message_iter_init(message, &iter);
	dbus_message_iter_get_basic(&iter, &username);

	meta_service = GALAGOD_META_SERVICE(service);
	account_list = galagod_meta_service_get_account_list(meta_service);
	meta_account = galagod_account_list_get_meta_account(account_list,
														 username);

	if (meta_account != NULL)
		account = galagod_meta_account_get_account(meta_account);

	if (account == NULL)
	{
		reply = galagod_object_not_found_error_new(message,
			"The account '%s' on service '%s' could not be found.",
			username, galago_service_get_id(service));
	}
	else
	{
		reply = dbus_message_new_method_return(message);
		dbus_message_iter_init_append(reply, &iter);
		galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(account));
	}

	dbus_connection_send(dbus_conn, reply, NULL);
	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
service_get_accounts(DBusConnection *dbus_conn, DBusMessage *message,
					 GalagodCoCo *coco, GalagoService *service)
{
	const GList *l;
	DBusMessage *reply;
	DBusMessageIter iter, array_iter;
	GalagodAccountList *account_list;
	GalagodMetaService *meta_service;
	GalagodMetaAccount *meta_account;
	GalagoAccount *account;
	size_t account_count;

	meta_service  = GALAGOD_META_SERVICE(service);
	account_list  = galagod_meta_service_get_account_list(meta_service);
	account_count = galagod_account_list_get_meta_account_count(account_list);

	reply = dbus_message_new_method_return(message);

	dbus_message_iter_init_append(reply, &iter);
	dbus_message_iter_open_container(&iter, DBUS_TYPE_ARRAY,
		galago_object_type_get_dbus_signature(GALAGO_TYPE_ACCOUNT),
		&array_iter);

	for (l = galagod_account_list_get_meta_accounts(account_list);
		 l != NULL;
		 l = l->next)
	{
		meta_account = (GalagodMetaAccount *)l->data;
		account      = galagod_meta_account_get_account(meta_account);

		galago_dbus_message_iter_append_object(&array_iter,
											   GALAGO_OBJECT(account));
	}

	dbus_message_iter_close_container(&iter, &array_iter);

	dbus_connection_send(dbus_conn, reply, NULL);
	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
service_get_accounts_count(DBusConnection *dbus_conn, DBusMessage *message,
						   GalagodCoCo *coco, GalagoService *service)
{
	DBusMessage *reply;
	DBusMessageIter iter;
	GalagodAccountList *account_list;
	GalagodMetaService *meta_service;
	size_t count;

	meta_service  = GALAGOD_META_SERVICE(service);
	account_list  = galagod_meta_service_get_account_list(meta_service);

	count = galagod_account_list_get_meta_account_count(account_list);

	reply = dbus_message_new_method_return(message);
	dbus_message_iter_init_append(reply, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_UINT32, &count);
	dbus_connection_send(dbus_conn, reply, NULL);
	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static const GalagodCommand commands[] =
{
	GALAGOD_COMMAND("AddAccount",       service_add_account),
	GALAGOD_COMMAND("RemoveAccount",    service_remove_account),
	GALAGOD_COMMAND("GetAccount",       service_get_account),
	GALAGOD_COMMAND("GetAccounts",      service_get_accounts),
	GALAGOD_COMMAND("GetAccountsCount", service_get_accounts_count),
	GALAGOD_COMMAND_LAST
};

char *
galagod_service_generate_path(GalagoService *service)
{
	GalagodCoCo *coco = g_object_get_data(G_OBJECT(service), "coco");

	return g_strdup_printf("%s/services/%s",
		(coco == NULL
		 ? galago_context_get_obj_path_prefix()
		 : galago_object_get_dbus_path(GALAGO_OBJECT(coco))),
		galagod_dbus_normalize_name(galago_service_get_id(service)));
}

void
galagod_service_add_filter(GalagoService *service)
{
	galagod_filters_add(service, commands, GALAGO_DBUS_SERVICE_INTERFACE);
}

void
galagod_service_remove_filter(GalagoService *service)
{
	galagod_filters_remove(service);
}
