/**
 * @file account-list.c Account list
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "account-list.h"
#include "coco.h"
#include "utils.h"

static void
_galagod_dbus_accounts_emit_account_added(GalagoAccount *account)
{
	GalagoService *service;
	DBusMessage *message;
	DBusMessageIter iter;

	g_return_if_fail(account != NULL);

	service = galago_account_get_service(account);

	message = dbus_message_new_signal(
		galago_object_get_dbus_path(GALAGO_OBJECT(service)),
		GALAGO_DBUS_SERVICE_INTERFACE,
		"AccountAdded");

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(account));

	dbus_connection_send(galago_get_dbus_conn(), message, NULL);
	dbus_message_unref(message);
}

GalagodAccountList *
galagod_account_list_new(GalagodMetaService *meta_service)
{
	GalagodAccountList *list;

	list = g_new0(GalagodAccountList, 1);

	list->meta_service = meta_service;
	list->meta_accounts_table = g_hash_table_new_full(g_str_hash, g_str_equal,
													  g_free, NULL);

	return list;
}

void
galagod_account_list_destroy(GalagodAccountList *list)
{
	g_return_if_fail(list != NULL);

	while (list->meta_accounts != NULL)
	{
		galagod_account_list_remove_meta_account(list,
			(GalagodMetaAccount *)list->meta_accounts->data);
	}

	g_hash_table_destroy(list->meta_accounts_table);

	g_free(list);
}

void
galagod_account_list_add(GalagodAccountList *list, GalagoAccount *account)
{
	GalagodMetaAccount *meta_account;
	const char *username;

	g_return_if_fail(list    != NULL);
	g_return_if_fail(account != NULL);

	username = galago_account_get_username(account);

	meta_account = galagod_account_list_get_meta_account(list, username);

	if (meta_account == NULL)
	{
		meta_account = galagod_meta_account_new(
			GALAGOD_META_SERVICE(galago_account_get_service(account)),
			GALAGOD_META_PERSON(galago_account_get_person(account)),
			username);
	}

	galagod_meta_account_add_account(meta_account, account);
}

void
galagod_account_list_remove(GalagodAccountList *list, GalagoAccount *account)
{
	GalagodMetaAccount *meta_account;

	g_return_if_fail(list    != NULL);
	g_return_if_fail(account != NULL);

	meta_account = GALAGOD_META_ACCOUNT(account);

	g_return_if_fail(meta_account != NULL);

	galagod_meta_account_remove_account(meta_account, account);

	if (galagod_meta_account_get_accounts(meta_account) == NULL)
		galagod_meta_account_destroy(meta_account);
}

void
galagod_account_list_remove_with_service(GalagodAccountList *list,
										 GalagoService *service)
{
	const GList *l;

	g_return_if_fail(list    != NULL);
	g_return_if_fail(service != NULL);

	for (l = galago_service_get_accounts(service, FALSE);
		 l != NULL;
		 l = l->next)
	{
		GalagoAccount *account = (GalagoAccount *)l->data;

		if (GALAGOD_META_ACCOUNT(account) != NULL)
			galagod_account_list_remove(list, account);
	}
}

void
galagod_account_list_add_meta_account(GalagodAccountList *list,
									  GalagodMetaAccount *meta_account)
{
	GalagoService *service;
	GalagoAccount *main_account;
	const char *username;

	g_return_if_fail(list         != NULL);
	g_return_if_fail(meta_account != NULL);

	main_account = galagod_meta_account_get_account(meta_account);
	username     = galago_account_get_username(main_account);
	service      = galago_account_get_service(main_account);

	list->meta_accounts = g_list_append(list->meta_accounts, meta_account);
	g_hash_table_insert(list->meta_accounts_table,
						g_strdup(galago_service_normalize(service, username)),
						meta_account);

	list->meta_account_count++;

	_galagod_dbus_accounts_emit_account_added(
		galagod_meta_account_get_account(meta_account));
}

void
galagod_account_list_remove_meta_account(GalagodAccountList *list,
										 GalagodMetaAccount *meta_account)
{
	GalagoAccount *main_account;
	GalagoService *service;
	const char *username;

	g_return_if_fail(list         != NULL);
	g_return_if_fail(meta_account != NULL);

	main_account = galagod_meta_account_get_account(meta_account);
	service      = galago_account_get_service(main_account);
	username     = galago_account_get_username(main_account);

	g_hash_table_remove(list->meta_accounts_table,
						galago_service_normalize(service, username));
	list->meta_accounts = g_list_remove(list->meta_accounts, meta_account);

	list->meta_account_count--;
}

const GList *
galagod_account_list_get_meta_accounts(const GalagodAccountList *list)
{
	g_return_val_if_fail(list != NULL, NULL);

	return list->meta_accounts;
}

size_t
galagod_account_list_get_meta_account_count(const GalagodAccountList *list)
{
	g_return_val_if_fail(list != NULL, 0);

	return list->meta_account_count;
}

GalagodMetaAccount *
galagod_account_list_get_meta_account(const GalagodAccountList *list,
									  const char *id)
{
	GalagoService *service;
	GalagodMetaAccount *meta_account;

	g_return_val_if_fail(list != NULL, NULL);
	g_return_val_if_fail(id   != NULL, NULL);

	service = galagod_meta_service_get_service(list->meta_service);

	meta_account = g_hash_table_lookup(list->meta_accounts_table,
									   galago_service_normalize(service, id));

	return meta_account;
}
