package freenet.node.states.request;

import freenet.*;
import freenet.node.*;
import freenet.node.states.data.*;
import freenet.node.ds.KeyCollisionException;
import freenet.message.*;
import freenet.support.Logger;
import java.util.Vector;
import java.io.IOException;

/**
 * The state a request is in when transferring data from upstream
 * to the requesting peer, and caching it.
 */
public class TransferReply extends DataPending {

    // message queue
    private Vector mq = new Vector();

    TransferReply(Pending ancestor) {
        super(ancestor);
        this.receivingData = ancestor.receivingData;
        this.sendingData   = ancestor.sendingData;
        this.storeData     = ancestor.storeData;
        this.accepted      = ancestor.accepted;
    }

    public final String getName() {
        return "Transferring Reply";
    }

    private final State transition(State s, boolean doQueue) throws StateTransition {
        NodeMessageObject[] q = new NodeMessageObject[mq.size()];
        mq.copyInto(q);
        throw new StateTransition(s, q, doQueue);
    }

    /**
     * Sends CB_RESTARTED down the send pipe and causes the receive pipe
     * to go into a data-eating state.  Unless it was too late, they will
     * not schedule their DataStateReply (tiny chance of a harmless BSE
     * in the log, no doubt users will report it as a bug..)
     */
    private final void cancelStreams() {
        receivingData.cancel();                         // shut
        sendingData.abort(Presentation.CB_RESTARTED);   // up
    }


    //=== message handling =====================================================
    
    public State receivedMessage(Node n, QueryRestarted qr) throws StateException {
        if (!fromLastPeer(qr)) {
            throw new BadStateException("QueryRestarted from the wrong peer!");
        }
        cancelStreams();
        mq.addElement(qr);
        return transition(new DataPending(this), true);
    }

    public State receivedMessage(Node n, QueryRejected qr) throws StateException {
        if (!fromLastPeer(qr)) {
            throw new BadStateException("QueryRejected from the wrong peer!");
        }
        cancelStreams();
        mq.addElement(qr);
        return transition(new DataPending(this), true);
    }

    // could belong after a restart
    public State receivedMessage(Node n, DataNotFound dnf) {
        mq.addElement(dnf);
        return this;
    }
    
    // could belong after a restart
    public State receivedMessage(Node n, DataReply dr) {
        mq.addElement(dr);
        return this;
    }
    
    public State receivedMessage(Node n, Accepted a) {
        // who cares ;)
        return this;
    }

    public State receivedMessage(Node n, StoreData sd) throws StateException {
        super.receivedStoreData(n, sd);
        return this;
    }

    public State receivedMessage(Node n, DataRequest dr) {
        super.receivedRequest(n, dr);
        return this;
    }
    

    private State checkTransition(Node n) throws StateTransition {
        if (dataReceived == null || dataSent == null) {
            return this;
        }
        if (dataReceived.getCB() == Presentation.CB_OK) {
            try {
                receivingData.commit();  // make the key available
            }
            catch (KeyCollisionException e) {
                // this is a little bit of a hack.  we jump into a
                // DataPending state and then handle a restart which
                // makes us check for the data in the store again
                n.logger.log(this, "Going to DataPending after key collision",
                             Logger.MINOR);
                scheduleRestart(n, 0);
                transition(new DataPending(this), false);
            }
            catch (IOException e) {
                fail(n, "Cache failed");
                n.logger.log(this, "Cache failed on commit", e, Logger.ERROR);
                transition(new RequestDone(this), false);
            }
            
            State awsd;
            if (storeData == null) {
                NoStoreData nosd = new NoStoreData(this);
                n.schedule(Core.hopTime(2), nosd);
                awsd = new AwaitingStoreData(this, nosd);
            }
            else {
                mq.addElement(storeData);
                awsd = new AwaitingStoreData(this, null);
            }
            return transition(awsd, true);
        }
        else return transition(new RequestDone(this), false);
    }
    
    
    public State receivedMessage(Node n, DataReceived dr) throws StateException {
        if (receivingData != dr.source()) {
            throw new BadStateException("Not my DataReceived: "+dr);
        }
        dataReceived = dr;
        int cb = dr.getCB();
        switch (cb) {
            
            case Presentation.CB_OK:
                n.logger.log(this, "Data received successfully!", Logger.MINOR);
                break;

            case Presentation.CB_CACHE_FAILED:
                n.logger.log(this, "Cache failed while receiving data!", Logger.ERROR);
                // the repercussions will strike in the DataSent..
                break;

            case Presentation.CB_BAD_DATA:
                n.logger.log(this, "Upstream node sent bad data!", Logger.NORMAL);
                // the null check is not really needed but I hate NPEs..
                if (lastPeer != null)
                    routes.verityFailed();
                // do the restart with the DataSent
                break;

            case Presentation.CB_RESTARTED:
                // pick it up with the DataSent
                break;

            default:
                if (lastPeer != null)
                    routes.transferFailed();
                n.logger.log(this,
                    "Failed to receive data with CB "+Presentation.getCBdescription(cb)
                    +", on chain "+Long.toHexString(id),
                    Logger.MINOR);
        }

        return checkTransition(n);
    }
    
    public State receivedMessage(Node n, DataSent ds) throws StateException {
        if (sendingData != ds.source()) {
            throw new BadStateException("Not my DataSent: "+ds);
        }
        dataSent = ds;
        int cb = ds.getCB();
        switch (cb) {
            
            case Presentation.CB_OK:
                n.logger.log(this, "Data sent successfully!", n.logger.MINOR);
                break;

            case Presentation.CB_RESTARTED:
                n.logger.log(this, "Send failed, stream restarted", Logger.MINOR);
                // we are expecting a QueryRestarted from the next node
                // but we schedule a restart here in case it never comes
                scheduleRestart(n, Core.hopTime(2)); 
                // do Queue starting from pending
                transition(new DataPending(this), true);

            case Presentation.CB_CACHE_FAILED:
                n.logger.log(this, "Send failed, cache broken!", Logger.ERROR);
                fail(n, "Cache failed");
                transition(new RequestDone(this), false);
                
            case Presentation.CB_SEND_CONN_DIED:
                n.logger.log(this, "Send failed, connection died", Logger.MINOR);
                // well, let's hang on for the StoreData anyway
                transition(new ReceivingReply(this), false);
                break;              // we only care about the DataReceived now..

            default:
                n.logger.log(this,
                    "Failed to send data with CB "+Presentation.getCBdescription(cb)
                    +", on chain "+Long.toHexString(id),
                    Logger.MINOR);
                // restart immediately, ignoring the queue
                scheduleRestart(n, 0);
                transition(new DataPending(this), false);
        }

        return checkTransition(n);
    }
}



