package freenet.node.states.request;

import freenet.*;
import freenet.node.*;
import freenet.node.ds.*;
import freenet.message.*;
import freenet.support.io.*;
import freenet.node.states.data.*;
import freenet.support.Logger;
import java.io.*;

/**
 * The state an insert is in once the DataInsert has been received,
 * but while we are still waiting for the Accepted from upstream to
 * begin transferring the insert.  This is the state that inserts
 * restarted from TransferInsert return to.
 */
public class TransferInsertPending extends InsertPending {

    // hold it to keep circular buffer from lapping too soon
    private KeyInputStream doc;

    
    TransferInsertPending(InsertPending ancestor) {
        super(ancestor);
        dim = ancestor.dim;
        dimRecvTime   = ancestor.dimRecvTime;
        receivingData = ancestor.receivingData;
        dataReceived  = ancestor.dataReceived; 
    }

    TransferInsertPending(InsertPending ancestor, KeyInputStream doc) {
        this(ancestor);
        this.doc = doc;
    }


    public String getName() {
        return "InsertRequest Pending Transfer";
    }

    
    //=== message handling =====================================================
    
    // these two cause the state to be recreated
    
    public State receivedMessage(Node n, QueryRejected qr) throws StateException {
        try {
            super.receivedQueryRejected(n, qr);
        }
        catch (EndOfRouteException e) {
            return endRoute(n);
        }
        catch (RequestAbortException rae) {
            // we are either going to SendingReply or RequestDone with no route found
            receivingData.cancel();
            return rae.state;
        }
        return new TransferInsertPending(this);
    }

    public State receivedMessage(Node n, RequestInitiator ri) throws StateException {
        try {
            super.receivedRequestInitiator(n, ri);
        }
        catch (EndOfRouteException e) {
            return endRoute(n);
        }
        catch (RequestAbortException rae) {
            // this is going to RequestDone with no route found
            receivingData.cancel();
            return rae.state;
        }
        return new TransferInsertPending(this);
    }

    // the rest don't

    public State receivedMessage(Node n, InsertRequest ir) {
        super.receivedRequest(n, ir);
        return this;
    }

    public State receivedMessage(Node n, QueryRestarted qr) throws StateException {
        try {
            super.receivedQueryRestarted(n, qr);
        }
        catch (RequestAbortException rae) {
            // going to RequestDone with SendFailedException
            receivingData.cancel();
            return rae.state;
        }
        return this;
    }

    // if we didn't make it to TransferInsert yet...
    public State receivedMessage(Node n, DataReceived dr) throws StateException {
        if (receivingData != dr.source()) {
            throw new BadStateException("Not my DataReceived: "+dr);
        }
        dataReceived = dr;
        int cb = dr.getCB();
        switch (cb) {
            case Presentation.CB_OK:
                n.logger.log(this, "Insert data received successfully!", Logger.MINOR);
                break;

            case Presentation.CB_CACHE_FAILED:
            case Presentation.CB_BAD_DATA:
                fail(n, cb == Presentation.CB_CACHE_FAILED ? "Cache failed"
                                                           : "You sent bad data");
                // fall through
            default:
                n.logger.log(this,
                    "Failed to receive insert with CB "+Presentation.getCBdescription(cb)
                    +", on chain "+Long.toHexString(id),
                    (cb == Presentation.CB_CACHE_FAILED ? Logger.ERROR : Logger.MINOR));
                queryAborted(n);
                return new RequestDone(this);
        }
        return this;  // still waiting for Accepted
    }

    public State receivedMessage(Node n, QueryAborted qf) throws StateException {
        if (!fromOrigPeer(qf)) {
            throw new BadStateException("QueryAborted from the wrong peer!");
        }
        queryAborted(n, qf);
        receivingData.cancel();
        return new RequestDone(this);
    }

    public State receivedMessage(Node n, DataReply dr) throws StateException {
        ReceiveData recv = this.receivingData;
        State ret = super.receivedDataReply(n, dr);  // could BSE out
        recv.cancel();  // terminating insert chain
        return ret;
    }

    public State receivedMessage(Node n, Accepted a) throws StateException {
        super.receivedAccepted(n, a);
        try {
            n.logger.log(this,
                "Got Accepted " + ((System.currentTimeMillis() - dimRecvTime) / 1000)
                +"s after DataInsert", Logger.DEBUG);
            if (doc == null)
                relayInsert(n);
            else
                relayInsert(n, doc);
        }
        catch (RequestAbortException rae) {
            // immediate restart, or failure
            return rae.state;
        }
        return new TransferInsert(this);
    }

    public State receivedMessage(Node n, InsertReply ir) throws StateException {
        if (!fromLastPeer(ir)) {
            throw new BadStateException("InsertReply from wrong peer!");
        }
        if (!approved) {
            cancelRestart();
            accepted = true;
            approved = true;
            try {
                n.logger.log(this,
                    "Got InsertReply " + ((System.currentTimeMillis() - dimRecvTime) / 1000)
                    +"s after DataInsert", Logger.DEBUG);
                insertReply(n);
                if (doc == null)
                    relayInsert(n);
                else
                    relayInsert(n, doc);
            }
            catch (RequestAbortException rae) {
                return rae.state;
            }
        }
        return new TransferInsert(this);
    }

    private final State endRoute(Node n) throws StateTransition {
        try {
            insertReply(n);
        }
        catch (RequestAbortException rae) {
            return rae.state;
        }
        State s = new ReceivingInsert(this);
        if (dataReceived != null)
            throw new StateTransition(s, dataReceived, true);
        else
            return s;
    }
}    



