package freenet.message;
import freenet.*;
import freenet.node.*;
import freenet.support.Logger;
import freenet.support.Fields;

/**
 * This message is sent after the DataSend has been completed and
 * causes a routing table reference to be added for the dataSource.
 *
 * @author oskar
 **/

public class StoreData extends NodeMessage {

    public static final String messageName = "StoreData";

    private final NodeReference dataSource;

    private long requestsPerHour;
    
    public StoreData(long idnum) {
        this(idnum, null, 0);
    }

    public StoreData(long idnum, 
                     NodeReference dataSource,
                     long requestsPerHour) {
        super(idnum, null);
        this.dataSource = dataSource;
        this.requestsPerHour = requestsPerHour;
    }

    public StoreData(ConnectionHandler source, RawMessage raw) 
        throws InvalidMessageException {
        
        super(source, raw);
        try {
            FieldSet dsource = otherFields.getSet("DataSource");
            dataSource = (dsource == null ? 
                          null : 
                          new NodeReference(dsource));
            
            requestsPerHour = -1;
            String rateAsString = otherFields.get("RequestRate");
            if (rateAsString != null) {
                requestsPerHour = Fields.stringToLong(rateAsString);
            }
            
            otherFields.remove("DataSource");
            otherFields.remove("RequestRate");
        }
        catch (BadReferenceException e) {
            throw new InvalidMessageException("Bad Reference");
        }
        catch (NumberFormatException nfe) {
            throw new InvalidMessageException("Failed to read number " + nfe);
        }
    }

    public RawMessage toRawMessage(Presentation t) {
        RawMessage raw = super.toRawMessage(t);
        if (dataSource != null) {
            raw.fs.add("DataSource", dataSource.getFieldSet());
        }
        if (requestsPerHour != -1) {
            raw.fs.add("RequestRate",Fields.longToString(requestsPerHour));
        }
        return raw;
    }

    public final NodeReference dataSource() {
        return dataSource;
    }

    // for now -1, means unknown
    public final long requestsPerHour() {
        return requestsPerHour;
    }
    
    public final String getMessageName() {
        return messageName;
    }
}




