// REDFLAG: use some other name besides "context" for stateful subapps
// REDFLAG: support for refreshInterval par

package freenet.client.http;

import freenet.*;
import freenet.client.*;
import freenet.client.events.*;
import freenet.client.listeners.*;
import freenet.client.metadata.*;
import freenet.client.http.filter.*;
import freenet.config.*;
import freenet.keys.*;
import freenet.support.*;
import freenet.support.mime.*;
import freenet.support.servlet.http.*;
import freenet.node.Node;

import java.io.*;
import java.net.*;
import java.util.*;
import javax.servlet.*;
import javax.servlet.http.*;


/*
  This code is part of fproxy, an HTTP proxy server for Freenet.
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/


/**
 * Servlet to handle incoming HttpServletRequest's for Freenet keys
 * This class supersedes HttpHandlerServlet
 * <p>
 * This servlet recognizes the following parameters:
 * <ul>
 *     <li>key<br>
 *         overrides URI in GET request
 *     <li>htl<br>
 *         overrides default htl
 *     <li>mime<br>
 *          overrides mime type of requested document.
 *     <li>force<br>
 *         sets magic cookie used by anonymity filter
 * </ul>  
 * <p>
 * @author <a href="http://www.doc.ic.ac.uk/~twh1/">Theodore Hong</a><br>
 * @author <a href="mailto:giannijohansson@mediaone.net">Gianni Johansson</a>
 **/
public class FproxyServlet extends HttpServlet {
    // exported constants
    // this should actually be defined on the servlet dispatcher side
    // when it sets the user role information
    public static final String SUPERUSER = "superuser";
    protected boolean isLocalConnection = false;
    protected static boolean newBuildWarningSent = false;

    protected static final String gatewayFile = "gateway.html";
    protected static int bufSize = 65536;

    private ServletContext context;
    protected Logger logger;
    protected static ClientFactory factory;
    protected String tmpDir = null;

    protected boolean runFilter = true;
    protected String passThroughMimeTypes = "text/plain,image/jpeg,image/gif,image/png";

    protected int requestHtl = 15;
    protected int insertHtl = 10;
    protected int splitFileThreads = 5;
    protected int splitFileRetries = 1;
    protected int splitFileRetryHtlIncrement = 5;
    protected int splitFileUIMinSize = 0;
    protected int splitFileUIRefreshIntervalSecs = 30;
    protected boolean splitFileUIForceSave = true;
 
    protected boolean pollForDroppedConnection = true;

    protected ContentFilter filter = null;
    private static Random random = new Random();
    private static Hashtable randkeys = new Hashtable();
    private static Object lastForceKey = null;
    private static Object firstForceKey = null;
    private final static int MAX_FORCE_KEYS = 100;

    private final static String PREFIX = "/__INTERNAL";
    
    private static FECFactory fecFactory = null;

    private static BucketFactory bucketFactory = null;

    // Make a single TempBucketFactory to be shared
    // by all instances.  Can't just construct it 
    // statically because non-static inner classes
    // are used.
    private void makeOnlyBucketFactory() {
        synchronized (FproxyServlet.class) {
            if (bucketFactory == null) {
                bucketFactory = new TempBucketFactory();
            }
        }
    }

    private final static Reaper reaper = new Reaper(60000);
    static {
        Thread reaperThread = new Thread(reaper, "Fproxy insert request cleanup thread.");
        reaperThread.setDaemon(true);
        reaperThread.start();
    }

    private final static ContextManager contextManager = new ContextManager();

    private final void loadFECSupport() {
        if (fecFactory == null) {
            fecFactory = new FECFactory();

            // Encoders
            int count = 0;
            for(;;) {
                String prefix = "FECEncoder_" + Integer.toString(count) + "_";
                String key = prefix + "name";
                
                String name = 
                    getInitParameter(key);

                if ((name == null) || name.trim().equals("")) {
                    break;
                }
                key = prefix +"class";
                String className = getInitParameter(key);

                if ((className == null) || className.trim().equals("")) {
                    break;
                }

                if (fecFactory.registerEncoder(name, className)) {
                    logger.log(this, "Loaded FECEncoder [" + name + 
                               "]: " + className, Logger.DEBUGGING);
                    System.err.println("Loaded FECEncoder [" + name + 
                                       "]: " + className);
                }
                else {
                    logger.log(this, "FAILED to Load FECEncoder [" + name + 
                               "]: " + className, Logger.DEBUGGING);
                }
                count++;
            }

            // Decoders
            count = 0;
            for(;;) {
                String prefix = "FECDecoder_" + Integer.toString(count) + "_";
                String key = prefix + "name";
                
                String name = 
                    getInitParameter(key);

                if ((name == null) || name.trim().equals("")) {
                    break;
                }
                key = prefix +"class";
                String className = getInitParameter(key);

                if ((className == null) || className.trim().equals("")) {
                    break;
                }

                if (fecFactory.registerDecoder(name, className)) {
                    logger.log(this, "Loaded FECDecoder [" + name + 
                               "]: " + className, Logger.DEBUGGING);

                    System.err.println("Loaded FECDecoder [" + name + 
                               "]: " + className);
                
                }
                else {
                    logger.log(this, "FAILED to Load FECDecoder [" + name + 
                               "]: " + className, Logger.DEBUGGING);
                }
                count++;
            }
        }
    }

    
    private final static void initContexts() {
        SplitFileRequestContext.contextManager = contextManager;
        SplitFileRequestContext.reaper = reaper;
        SplitFileRequestContext.factory = factory;
        SplitFileRequestContext.bucketFactory = bucketFactory;
        SplitFileRequestContext.fecFactory = fecFactory;
    }


    public void init() {
        makeOnlyBucketFactory();

        // System attributes come out of context.
        context = getServletContext();
        factory = (ClientFactory)context.getAttribute("freenet.client.ClientFactory");

        //Logger parentLog = (Logger)context.getAttribute("freenet.support.Logger");
        // REDFLAG: fix... Don't want to use crappy Servlet logging calls though.
        Logger parentLog = Node.logger;

        String logFile = getInitParameter("logFile");
        int logLevel = Logger.DEBUGGING;
            
        if ((logFile == null) && (parentLog != null)) {
            // Use the logger owned by whatever object
            // created the HttpContainerImpl instance.
            logger = parentLog;
        }
        else {
            // REDFLAG: support full log parameters? 
            // Create our own logger
            if (getInitParameter("logLevel") != null) {
                logLevel = Logger.priorityOf(getInitParameter("logLevel"));
            }

            if (logger == null)
                logger = new Logger(logLevel);

            LoggerHook lh = null;
            if (logFile != null) {
                try { 
                    System.err.println("LOGFILE: " + logFile);
                    if (logFile.toLowerCase().equals("no")) {
                        lh = new FileLoggerHook(System.err,
                                                null, null,
                                                logLevel); 
                        
                    }
                    else {      
                        lh = new FileLoggerHook(new PrintStream(new FileOutputStream(logFile)),
                                                null, null,
                                                logLevel); 
                    }
                }
                catch (Exception e) {
                }
            }
            if (lh == null) {
                lh = new FileLoggerHook(System.out, null, null, logLevel);
            }
            logger.addHook(lh);
        }

        runFilter = readBoolean(this, logger, "filter", runFilter);
        if (getInitParameter("passThroughMimeTypes") != null) {
            passThroughMimeTypes = getInitParameter("passThroughMimeTypes");
        }
        
        insertHtl = readInt(this, logger, "insertHtl", insertHtl, 0, 100);
        requestHtl = readInt(this, logger, "requestHtl", requestHtl, 0, 100);
        splitFileThreads = readInt(this, logger, "splitFileThreads", splitFileThreads, 0, 100);
        splitFileRetries = readInt(this, logger, "splitFileRetries", splitFileRetries, 0, 10);
        splitFileRetryHtlIncrement = 
            readInt(this, logger, "splitFileRetryHtlIncrement", splitFileRetryHtlIncrement, 0, 100);

        // The SplitFile UI isn't displayed for SplitFiles
        // smaller than this. Set to 0 to enable the UI 
        // on all files.
        splitFileUIMinSize = 
            readInt(this, logger, 
                    "splitFileUIMinSize", 
                    splitFileUIMinSize, 0, Integer.MAX_VALUE);

        // Refresh interval for the download status frame.
        // Set to 0 to disable client-pull updating.
        splitFileUIRefreshIntervalSecs = 
            readInt(this, logger, 
                    "splitFileUIRefreshIntervalSecs", 
                    splitFileUIRefreshIntervalSecs, 0, 1000);

        // Set true to make the SplitFile UI throw up a
        // the save dialog instead of trying to render 
        // SplitFile contents in the browser.
        splitFileUIForceSave = 
            readBoolean(this, logger, 
                    "splitFileUIForceSave", 
                    splitFileUIForceSave);

        pollForDroppedConnection = 
            readBoolean(this, logger, "pollForDroppedConnection", pollForDroppedConnection);

        if ((getInitParameter("tempDir") != null) &&
            (!getInitParameter("tempDir").trim().equals(""))) {
            try {
                FileBucket.setTempDir(getInitParameter("tempDir"));
                tmpDir = FileBucket.getTempDir();
            }
            catch (IllegalArgumentException ia) {
                logger.log(this, "WARNING: Couldn't set fproxy tempDir: " + 
                           getInitParameter("tempDir")
                           , Logger.ERROR);
            }
        }

        logger.log(this, "New FproxyServlet created", Logger.MINOR);
        logger.log(this, "   insertHtl = " + insertHtl, Logger.DEBUGGING);
        logger.log(this, "   requestHtl = " + requestHtl, Logger.DEBUGGING);
        logger.log(this, "   filter = " + runFilter, Logger.DEBUGGING);
        logger.log(this, "   passThroughMimeTypes = " + passThroughMimeTypes, Logger.DEBUGGING);
        logger.log(this, "   splitFileThreads = " + splitFileThreads, Logger.DEBUGGING);
        logger.log(this, "   splitFileRetries = " + splitFileRetries, Logger.DEBUGGING);
        logger.log(this, "   splitFileRetryHtlIncrement = " + splitFileRetryHtlIncrement, Logger.DEBUGGING);
        logger.log(this, "   splitFileUIMinSize = " + splitFileUIMinSize, Logger.DEBUGGING);
        logger.log(this, "   splitFileUIRefreshIntervalSecs = " + splitFileUIRefreshIntervalSecs,
                   Logger.DEBUGGING);
        logger.log(this, "   splitFileUIForceSave = " + splitFileUIForceSave, Logger.DEBUGGING);
        logger.log(this, "   pollForDroppedConnection = " + pollForDroppedConnection, Logger.DEBUGGING);
        logger.log(this, "   logFile = " + logFile, Logger.DEBUGGING);
        logger.log(this, "   logLevel = " + logLevel, Logger.DEBUGGING);
        logger.log(this, "   tmpDir = " + tmpDir, Logger.DEBUGGING);
        
        // uses logger
        loadFECSupport();
        initContexts();
    }

    /**
     * Dispatch an HTTP transaction
     **/
    protected void service(HttpServletRequest req, HttpServletResponse resp)
        throws ServletException, IOException {
        // test if local connection
        isLocalConnection = req.isUserInRole(SUPERUSER);

        // dispatch
        super.service(req, resp);
    }


    // /__INTERNAL__??/<context_id>/foo/bar.txt
    private final static String extractContextID(String uriValue) {
        if (!uriValue.startsWith(PREFIX)) {
            return null;
        }

        int startPos = uriValue.indexOf("/", PREFIX.length());
        if ((startPos == -1) || (startPos == uriValue.length() - 1)) {
            return null;
        }
        
        int endPos = uriValue.indexOf("/", startPos + 1);
        if (endPos <= startPos) {
            return null;
        }
        
        final String ret = uriValue.substring(startPos + 1, endPos).trim();
        if (ret.equals("")) {
            return null;
        }
        return ret;
    }

    // Routes requests to stateful sub-applications inside fproxy
    // e.g. inserting and SplitFile requesting.
    private boolean handleContexts(HttpServletRequest req, HttpServletResponse resp)
        throws ServletException, IOException {

        // hoist out???
        String urlValue = null; 
        try {
            urlValue = freenet.support.URLDecoder.decode(req.getRequestURI());
        }
        catch (URLEncodedFormatException  fe) {
            throw new IOException(fe.toString()); // hmmm...
        }


        if (urlValue.startsWith(InsertContext.PREFIX)) {
            final InsertContext context = 
                (InsertContext)contextManager.lookup(extractContextID(urlValue));
            InsertContext.handle(context, 
                                 urlValue,
                                 req,
                                 resp);
            return true;
        }
        else if (urlValue.startsWith(SplitFileRequestContext.PREFIX)) {
            final SplitFileRequestContext context = 
                (SplitFileRequestContext)contextManager.lookup(extractContextID(urlValue));
            SplitFileRequestContext.handle(context, 
                                           urlValue,
                                           req,
                                           resp);
            return true;
        }

        return false;
    }


    /**
     * Initiate a Freenet request (GET method)
     **/
    protected void doGet(HttpServletRequest req, HttpServletResponse resp)
        throws ServletException, IOException {

        SplitFileDownloader splitFileDownloader = null;

        try {
            logger.log(this, "Got GET request", Logger.DEBUGGING);

            // Support checked jumps out of Freenet. 
            // REDFLAG: query parameters not supported? no req.getRequestURL???
            if (handleCheckedJump(req.getRequestURI(), resp )) {
                return;
            }

            // Inserting and SplitFile request status.
            if (handleContexts(req, resp)) {
                return;
            }

            // Query parameters
            String key = null;
            String queryKey = null;
            String queryForce = null;
            String queryHtl = null;
            String queryMime = null;
	   int htlUsed = requestHtl;

	   try {
	      
	      boolean bwParam = true;
	      if (getInitParameter("showNewBuildWarning") == null)
		  bwParam = true;
	      else
		  bwParam = getInitParameter("showNewBuildWarning").equals("true");
	      if ((Core.highestSeenBuild > Version.buildNumber) && !(newBuildWarningSent) && bwParam)  
		  {
		      newBuildWarningSent=true;
		      PrintWriter pw = resp.getWriter();
		      pw.println("<html><head><title>New Build Available</title></head>");
		      pw.println("<body bgcolor=\"#ffffff\"><h1>New Build Available: "+
				 Core.highestSeenBuild+"</h1>");
		      pw.println("A node has been observed which claims to be a more recent build ("+
				 Core.highestSeenBuild);
		      pw.println(") than your node ("+Version.buildNumber+"). ");
		      pw.println("You should download an updated version of Freenet, you can do this ");
		      pw.println("by running the ./update.sh script in Unix/Linux and restarting Freenet, ");
		      pw.println("or select the Update option from the Freenet section of the Start menu ");
		      pw.println("in Windows.<p>");
		      pw.println("This warning will be shown the first time you view the FProxy gateway page. ");
		      pw.println("You can disable the option by adding:<br>");
		      pw.println("<pre>fproxy.params.showNewBuildWarning=false</pre><br>");
		      pw.println("..to your freenet.conf or freenet.ini file.<p>");
		      pw.println("Click <a href=\""+req.getRequestURI()+
				 ((req.getQueryString()!=null) ? ("?"+req.getQueryString()) : "")+
				 "\">here</a> to continue.");
		      
		      pw.println("</body></html>");
		      pw.flush();
		      return;
		  }
	      
	      queryKey = req.getParameter("key");
	      if (queryKey != null) {
                    queryKey = freenet.support.URLDecoder.decode(queryKey);
                    // chop leading /
                    if (queryKey != null && queryKey.startsWith("/")) {
                        queryKey = queryKey.substring(1);
                    }
                    logger.log(this, "Read key from query: " + queryKey, Logger.DEBUGGING);
                }

                if (queryKey == null) {
                    key = freenet.support.URLDecoder.decode(req.getRequestURI());
                    
                    // chop leading /
                    if (key != null && key.startsWith("/")) {
                        key = key.substring(1);
                    }
                }
                
                queryForce = req.getParameter("force");
                if (queryForce != null) {
                    queryForce = freenet.support.URLDecoder.decode(queryForce);
                    logger.log(this, "Read force from query: " + queryForce, Logger.DEBUGGING);
                }

                queryHtl = req.getParameter("htl");
                if (queryHtl != null) {
                    try {
                        htlUsed = Integer.parseInt(freenet.support.URLDecoder.decode(queryHtl));
                        logger.log(this, "Read htl from query: " + htlUsed, Logger.DEBUGGING);
                    }
                    catch (NumberFormatException e) {
                        logger.log(this, "Couldn't parse htl from query, using: " + htlUsed, 
                                   Logger.DEBUGGING);
                    }
                }

                queryMime = req.getParameter("mime");
                if (queryMime != null) {
                    queryMime = freenet.support.URLDecoder.decode(queryMime);
                    logger.log(this, "Read mime from query: " + queryMime, Logger.DEBUGGING);
                }
            }
            catch (Exception e) {
                logger.log(this, "Error while parsing URI", e,
                           Logger.ERROR);
                sendError(resp, HttpServletResponse.SC_BAD_REQUEST,
                          "Couldn't parse URI: " + req.getRequestURI());
                return;
            }

            if (queryKey != null) {
                logger.log(this, "Redirecting to: " + queryKey, Logger.DEBUGGING);
            
                // Reconstruct the request w/o the key field.
                queryKey = reconstructRequest(queryKey, queryForce, queryHtl, queryMime);
            
                resp.sendRedirect("/" + queryKey);
                return;
            }

            // check for special keys
            logger.log(this, "Key is: " + key, Logger.DEBUGGING);
            if (key.equals("")) {
	       sendGateway(resp);
	       return;
            }
            else if (key.equals("robots.txt")) {
                sendRobots(resp);
                return;
            }
        
            // Do request, following redirects as necessary.
            FileBucket data = new FileBucket();
            AutoRequester r = new AutoRequester(factory);
            FreenetURI uri;
            try {
                uri = new FreenetURI(key);
            } catch (MalformedURLException e) {
                writeErrorMessage(e, resp, null, key, htlUsed);
                return;
            }
            logger.log(this, "Starting request process with htl " + htlUsed,
                       Logger.DEBUGGING);

            FailureListener listener = new FailureListener();
            r.addEventListener(listener);

            if (!r.doGet(uri, data, htlUsed)) {
                writeErrorMessage(listener.getException(), resp, null, key, htlUsed);
                return;
            }

            // Find content type, in descending order of preference:
            // x 1. specified in query parameters
            // x 2. specified in metadata
            // ? 3. guessed from data a la file(1)
            // x 4. guessed from key
            String mimeType = null;

            // User specified mime type
            if (queryMime != null) {
                mimeType = queryMime;
            }

            // Try to read the mime type out of the metadata
            if ((mimeType == null) && (r.getMetadata() != null)) {
                mimeType = r.getMetadata().getMimeType(null);
            }

            // If that doesn't work guess it from the extension
            // on the key name.
            if (mimeType == null) {
                mimeType = MimeTypeUtils.getExtType(key);
            }

            // If all else fails, fall back to octet-stream
            // so the user can download the file.
            if (mimeType == null) {
                mimeType = "application/octet-stream";
            }

            SplitFile splitFile = r.getMetadata().getSplitFile();
            if (splitFile != null) {
                logger.log(this, splitFile.toString(), Logger.DEBUGGING);
                logger.log(this, "Key is a SplitFile, mimeType: " + mimeType, Logger.DEBUGGING);
            }

            // Filter if nesc. and send data
            // Note: This doesn't check split files.
            boolean forced = false;
            if ((queryForce != null) && checkForceKey(queryForce)) {
                forced = true;
            }
            InputStream in = null;
            OutputStream out = null;
            boolean flushed = false;
            try {
                if (runFilter) {
                    filter = ContentFilterFactory.newInstance(passThroughMimeTypes);
                    filter.setAllowSecurityWarnings(forced);
                    filter.setAllowSecurityErrors(forced);
                    
                    InputStream filterIn = null;
                    try {
                        filterIn = data.getInputStream();
                        filter.run(filterIn, mimeType);
                    }
                    finally {
                        if (filterIn != null) {
                            try { filterIn.close(); } catch (Exception e) {}
                        }
                    }
                }

                // Warn that we don't run the anonymity filter 
                // over html SplitFiles.
                // LATER: check html split files.
                if (runFilter && (!forced) && 
                    (splitFile != null) && (mimeType.equals("text/html"))) {
                    throw new FilterException("Html documents not allowed in SplitFiles",
                                              "The anonymity filter doesn't check html SplitFiles.", null);
                    // REDFLAG: test this code path.
                }

                // Get InputStream.
                if (splitFile == null) {
                    // Simple data request.
                    // Set response headers, 200
                    resp.setStatus(HttpServletResponse.SC_OK);
                    resp.setContentType(mimeType);
                    resp.setContentLength((int)data.size());
                    in = data.getInputStream();
                    // set up for binary output
                    // can't do this earlier in case we need resp.getWriter()
                    out = resp.getOutputStream();
                
                    // REDFLAG: Is this an issue any more? If not delete this.
                    // This following line is a workaround for an IE bug
                    // if (mimeType.equals("application/octet-stream")) {
                    //  out.write("Content-Disposition: attachment; ".getBytes());
                    //  out.write(("filename=" + key + "\015\012\015\012").getBytes());
                    // }
                    
                    copy(in, out);
                    in.close();
                    in = null;
                }
                else {
                    // Note: contexts magically register themselves 
                    //       with the ContextManager.
                    // SplitFile request.
                    SplitFileRequestContext context =
                        new SplitFileRequestContext(splitFile,
                                                    key,
                                                    mimeType,
                                                    htlUsed,
                                                    splitFileRetryHtlIncrement,
                                                    splitFileRetries,
                                                    splitFileThreads,
                                                    splitFile.getSize() > splitFileUIMinSize,
                                                    splitFileUIForceSave, 
                                                    pollForDroppedConnection,
                                                    splitFileUIRefreshIntervalSecs,
                                                    logger);
                    context.updateParameters(req);
                    if (context.useUI()) {
                        // Send a redirect to the download UI
                        resp.sendRedirect(context.getRedirectURL());
                        flushed = true;
                    }
                    else {
                        // Send back the data on this thread.
                        context.sendData(req,resp);
                        flushed = true;
                    }
                }
            }
            catch (Exception e) {
                //e.printStackTrace();
                logger.log(this, "Error sending data to browser: " + e,
                           Logger.ERROR);
                writeErrorMessage(e, resp, out, key, htlUsed);
            }
            finally {
                if (in != null) {
                    // Don't leak file handles.
                    try {in.close();} catch (IOException e) {}
                }
                // commit response
                if (!flushed) {
                    resp.flushBuffer();
                }
            }
        }
        catch (Exception e) {
            e.printStackTrace();
            logger.log(this, "Unexpected Exception in FproxyServlet.doGet -- " + e,
                       Logger.ERROR);
            //sendError(resp, HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
            //        "Unexpected Exception in FproxyServlet.doGet -- " + e);
        }
    }

    private final String getPrefix(int n) {
        if (n == 0) {
            return "?";
        }
        return "&";
    }

    private String reconstructRequest(String queryKey, String queryForce, 
                                      String queryHtl, String queryMime) {
        int count = 0;
        if (queryForce != null) {
            queryKey += getPrefix(count++) + "force=" + queryForce;
        }
        if (queryHtl != null) {
            queryKey += getPrefix(count++) + "htl=" + queryHtl;
        }
        if (queryMime != null) {
            queryKey += getPrefix(count++) + "mime=" + queryMime;
        }
        
        return queryKey;
    }
   
    final static void copy(InputStream in, OutputStream out) 
        throws IOException {
        // stream bucket to output
        byte[] buf = new byte[bufSize];
        int bytes = 0;
        while ((bytes = in.read(buf)) > 0) {
            // REDFLAG: Should we yield here for massive downloads?
            //          e.g. a big splitfile that is already mostly
            //          in the data store. How do you yield() in Java?

            out.write(buf, 0, bytes);
        }
        out.flush();
    }

    private final static String encode(String URL) {
	final String safeURLCharacters = "*-./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz";
	StringBuffer enc = new StringBuffer(URL.length());
	for (int i = 0; i < URL.length(); ++i) {
	    char c = URL.charAt(i);
	    if (safeURLCharacters.indexOf(c) >= 0)
		enc.append(c);
	    else {
                // Too harsh.
		// if (c < 0 || c > 255)
		//    throw new RuntimeException("illegal code "+c+" of char '"+URL.charAt(i)+"'");
		// else 

                // Just keep lsb like:
                // http://java.sun.com/j2se/1.3/docs/api/java/net/URLEncoder.html
                c = (char)(c & '\u00ff');
                if (c < 16)
		    enc.append("%0");
		else
		    enc.append("%");
		enc.append(Integer.toHexString(c));
	    }
	}
	return enc.toString();
    }
    
    // REDFLAG: suspect error reporting. Get someone who knows more about servlets to
    //          look at this.
    // What if an error occurs while you have after you have already starting writing a 
    // non html document?
    //
    // This is no worse than the original fproxy implementation, but no better :-(
    //
    private final void writeErrorMessage(Exception e, HttpServletResponse resp, OutputStream out, 
                                         String key, int htl) {
        try {
            PrintWriter pw = null;

            if (e instanceof FilterException) {
                resp.setStatus(HttpServletResponse.SC_FORBIDDEN);
            }
            else {
                resp.setStatus(HttpServletResponse.SC_NOT_FOUND);
            }
            resp.setContentType("text/html");
            
            if (out == null) {
                pw = resp.getWriter(); // OK, we didn't write any binary data.
            }
            else {
                // Will this work?
                pw = new PrintWriter(new OutputStreamWriter(out)); // REDFLAG: encoding?
            }
            pw.println("<html>");
            pw.println("<head>");
            pw.println("<title> Couldn't retrieve " + key + "</title>");
            pw.println("</head>");
            pw.println("");
            pw.println("<body bgcolor=\"#ffffff\">");
            if (e instanceof FilterException) {
                FilterException fe = (FilterException)e;
                pw.println("<H1>Warning: "+e.getMessage()+"</H1>");
                pw.println(((FilterException) e).explanation);
                // Why do we have to encode the key here?  It's because this
                // HTML is generated by the gateway, and we cannot trust the
                // URL.  It might have characters which will break the security
                // of the page (e.g. close quotes).
                //
                // So we encode it, and pass it as part of the query string.
                // If the user clicks on the "Retrieve anyway", we will get the
                // URL as a form argument, decode it and redirect to it.  The
                // redirect is hopefully secure against funny characters...
                //
                String encKey = encode(key);
                String forceKey = makeForceKey();
                pw.println("<p><a href=\"/?key=" + encKey + "&force=" + 
                           forceKey + "\">Retrieve anyway</A>, see the <a href=\"/?key=" +
                           encKey + 
                           "&mime=text/plain\">source</A> or <A HREF=\"/\">return</A> to gateway page");
                if (fe.analysis != null) {
                    Enumeration errors = fe.analysis.getDisallowedElements();
                    if (errors != null) {
                        pw.println("<H3>Disallowed elements</H3>");
                        while (errors.hasMoreElements()) {
                            pw.println("<BR> - " + errors.nextElement());
                        }
                    }
                    Enumeration warnings = fe.analysis.getWarningElements();
                    if (warnings != null) {
                        pw.println("<H3>External links</H3>");
                        while (warnings.hasMoreElements()) {
                            pw.println("<BR> - " + warnings.nextElement());
                        }
                    }
                }
            }
            else if (e instanceof RequestFailedException &&
                     (!((RequestFailedException)e).getMessage().equals(""))) {

                pw.println("<H1>Network Error</H1>");
                pw.println("<p>Couldn't retrieve key: <b>" + key + 
                           "</b> <br>Hops To Live: <b>" + htl + "</b><br>");

                // Display extra feedback info.
                pw.println(e.getMessage());

                String encKey = encode(key);
                pw.println("<form name=\"changeHTL\" action=\"/" + 
                           encKey + "\"> <p>Change Hops To Live <input type=\"text\" " + 
                           "size=\"3\" name=\"htl\" value=\"" +
                           htl + "\"/> <input type=\"submit\" value=\"Retry\"/> </form>");
            }


            else {
                pw.println("<H1>Network Error</H1>");
                pw.println("<p>Couldn't retrieve key: <b>" + key + 
                           "</b> <br>Hops To Live: <b>" + htl + "</b><br>");
                String encKey = encode(key);
                pw.println("<form name=\"changeHTL\" action=\"/" + 
                           encKey + "\"> <p>Change Hops To Live <input type=\"text\" " + 
                           "size=\"3\" name=\"htl\" value=\"" +
                           htl + "\"/> <input type=\"submit\" value=\"Retry\"/> </form>");
            }
            pw.println("</body>");
            pw.println("</html>");
            pw.flush();
        }
        catch (Exception e1) {
            logger.log(this, "Couldn't report error to browser: " + e1,
                       Logger.ERROR);
            
        }
    }

    /**
     * Send back gateway.html
     **/
    protected void sendGateway(HttpServletResponse resp) throws IOException {
        logger.log(this, "No key, loading " + gatewayFile, Logger.DEBUGGING);

        // set response headers
        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/html");

        // load gateway file into memory, so we can get its length
        BufferedReader br = new BufferedReader(new 
            InputStreamReader(FproxyServlet.class.getResourceAsStream(gatewayFile)));
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        PrintWriter pw = new PrintWriter(baos);
        String s = br.readLine();
        while (s != null) {
            pw.println(s);
            s = br.readLine();
        }
        pw.flush();

        // copy to output
        byte[] buf = baos.toByteArray();
        resp.setContentLength(buf.length);
        resp.getOutputStream().write(buf);

        // commit response
        resp.flushBuffer();
    }

    /**
     * Send back robots.txt
     **/
    protected void sendRobots(HttpServletResponse resp) throws IOException {
        logger.log(this, "Sending robots.txt", Logger.DEBUGGING);
        OutputStream out = resp.getOutputStream();
        
        if (!isLocalConnection) {
            // disallow all robots
            resp.setStatus(HttpServletResponse.SC_OK);
            resp.setContentType("text/plain");
            byte[] buf = "Disallow: *".getBytes();
            resp.setContentLength(buf.length);
            out.write(buf);
        } else {
            // no robots.txt (i.e. full access)
            resp.setStatus(HttpServletResponse.SC_NOT_FOUND);
            resp.setContentType("text/plain");
            byte[] buf = "Okay, little robot, give me your best shot!".getBytes();
            resp.setContentLength(buf.length);
            out.write(buf);
        }

        // commit response
        resp.flushBuffer();
    }

    /**
     * Send back error status
     **/
    protected void sendError(HttpServletResponse resp, int status,
                             String detailMessage)
        throws IOException {

        // get status string
        String statusString = status + " " +
            HttpServletResponseImpl.getNameForStatus(status);

        // show it
        logger.log(this, "Sending HTTP error: " + statusString,
                   Logger.DEBUGGING);
        PrintWriter pw = resp.getWriter();
        resp.setStatus(status);
        resp.setContentType("text/html");
        pw.println("<html>");
        pw.println("<head><title>" + statusString + "</title></head>");
        pw.println("<body bgcolor=\"#ffffff\">");
        pw.println("<h1>" + statusString + "</h1>");
        pw.println("<p>" + detailMessage);
        pw.println("</body>");
        pw.println("</html>");
        resp.flushBuffer(); 
    }

    // BUG work around.
    //
    // The browser drops the connection and prints a nasty
    // error message when I try to send a page back from
    // doPost without setting the content length.
    //
    // Bug seen on Netscape Navigator 4.76, KDE 2.1 Konqueror
    // rh7.1, IBM JDK1.3
    private final void writeResponse(StringBuffer html, int statusCode, 
                                     HttpServletResponse resp) throws IOException {
        resp.setStatus(statusCode);
        resp.setContentType("text/html");
        resp.setContentLength(html.length());
        PrintWriter pw = resp.getWriter();
        pw.print(html.toString());
        resp.flushBuffer();
    }

    private final void writeDataErrorResponse(String msg, String baseURL, HttpServletResponse resp) 
        throws IOException {
        StringBuffer responseMsg = new StringBuffer();
        responseMsg.append("<html><head><title>Freenet insert error</title></head>");
        responseMsg.append("<body bgcolor=\"#ffffff\">");
        responseMsg.append("<p> " +  msg + " </p> \n");
        responseMsg.append("<p><a href=\"" + baseURL + "\">Try again</a>.</p> \n");
        responseMsg.append("</body> </html> \n");
        writeResponse(responseMsg, HttpServletResponse.SC_BAD_REQUEST, resp);
    }

    private final void writeExceptionResponse(String msg, String baseURL, Exception e,
                                              HttpServletResponse resp) 
        throws IOException {
        StringBuffer responseMsg = new StringBuffer();
        responseMsg.append("<html><head><title>Freenet insert error</title></head>");
        responseMsg.append("<body bgcolor=\"#ffffff\">");
        responseMsg.append("<p> " + msg + "</p> \n");
        CharArrayWriter stackTrace = new CharArrayWriter();
        e.printStackTrace(new PrintWriter(stackTrace));
        responseMsg.append("<pre> \n");
        responseMsg.append(stackTrace.toString() + "\n");
        responseMsg.append("</pre> \n");
        responseMsg.append("<p><a href=\"" + baseURL + "\">Done</a>.</p> \n");
        responseMsg.append("</body> </html> \n");
        writeResponse(responseMsg, HttpServletResponse.SC_BAD_REQUEST, resp);
        return;
    }

    // made non-static so I can use logger
    // hmmmm are we sure that the parts we want
    //       will always come back as MIME_binary?
    // order key, filename, htl, content-type
    private final MIME_binary[] extractParts(MIME_multipart formData) {
        MIME_binary[] parts = new MIME_binary[4];

        for (int i = 0; i < formData.getPartCount(); i++) {
            // REDFLAG: Remove.
            logger.log(this, i + " content-type: "
                       + formData.getPart(i).getHeader().getContent_Type() +
                       " name: " + formData.getPart(i).getHeader().
                       getContent_DispositionParameter("name"),
                       Logger.DEBUGGING);

            String name = 
                formData.getPart(i).getHeader().getContent_DispositionParameter("name");

            if (name == null) {
                logger.log(this, " Skipped mime part with no name!",
                           Logger.DEBUGGING);
                continue;
            }

            if (name.equals("key")) {
                parts[0] = (MIME_binary)formData.getPart(i);
            }
            else if (name.equals("filename")) {
                parts[1] = (MIME_binary)formData.getPart(i);
            }
            else if (name.equals("htl")) {
                parts[2] = (MIME_binary)formData.getPart(i);
            }
            else if (name.equals("content-type")) {
                parts[3] = (MIME_binary)formData.getPart(i);
            }
            else {
                freePart(formData.getPart(i));
            }
        }
        return parts;
    } 

    private final void freePart(MIME part) {
        if (part == null) {
            return;
        }
        try {
            ((MIME_binary)part).freeBody();
        }
        catch (IOException ioe) {
            logger.log(this, "Ignored exception freeing part: " + ioe, Logger.DEBUGGING);
        }
    }

    private final void freeParts(MIME_binary[] parts) {
        if (parts == null) {
            return;
        }

        for (int i=0; i < parts.length; i++) {
            freePart(parts[i]);
            parts[i] = null;
        }
    }

    /**
     * Perform a Freenet insert (POST method)
     **/
    protected void doPost(HttpServletRequest req, HttpServletResponse resp)
        throws ServletException, IOException {

        logger.log(this, "Got POST request", Logger.DEBUGGING);

        // TODO: figure out some way to restore base parameters
        // between requests
        //Params params = (Params) baseParams.clone();
        ///Params params = baseParams;
        // REDFLAG: TEST. Why did theo make this absolute?
        //String baseUrl = "http://localhost:" + params.getInt("listenPort") + "/";

        String baseURL = "/";

        String key = null;
        MIME_binary filePart = null;
        int htlUsed = insertHtl;
        String mimeType = null;

        // read and validate POST data
        InputStream in = req.getInputStream();
        MIME_binary parts[] = null;
        try {
            MIME_multipart formData = new MIME_multipart(in, req, bucketFactory);

            // Frees unknown MIME_binary parts.
            parts = extractParts(formData);            

            if (formData.getPartCount() < 4) {
                writeDataErrorResponse("The form data must have at least 3 parts!",
                                       baseURL, resp);
                freeParts(parts);
                return;
            }

            // key
            if (parts[0] == null) {
                writeDataErrorResponse("The form must POST a 'key' field!",
                                       baseURL, resp);
                freeParts(parts);
                return;
            }
            key = parts[0].getBodyAsString();
            
            // file data
            if (parts[1] == null) {
                writeDataErrorResponse("The form must POST a 'filename' field!",
                                       baseURL, resp);
                freeParts(parts);
                return;
            }
            else {
                if (parts[1].getBody().size() == 0) {
                    writeDataErrorResponse("No data was sent!",
                                           baseURL, resp);
                    freeParts(parts);
                    return;
                }
            }
            filePart = parts[1];

            // htl
            if (parts[2] != null) {
                try {
                    htlUsed = Integer.parseInt(parts[2].getBodyAsString());
                }
                catch (NumberFormatException nfe) {
                    writeDataErrorResponse("Couldn't read an integer out of the 'htl' field!",
                                           baseURL, resp);
                    freeParts(parts);
                    return;
                }
            }

            // MIME type
            if (parts[3] != null) {
                mimeType = parts[3].getBodyAsString();
            }

            // Detect content-type if not specified.
            if (mimeType == null || mimeType.equalsIgnoreCase("auto")) {
                mimeType = parts[1].getHeader().getContent_Type();
                if (mimeType != null && mimeType.equalsIgnoreCase("application/unspecified")) {
                    mimeType = null;
                }
            }

            // keep freeParts() from releasing filePart
            parts[1] = null; 

            freeParts(parts);

            // REDFLAG: ??? This code didn't have any effect in
            //          Theo's original implementation... but the comment
            //          looks too scary to ignore...
            //
            // this should be MIME_text by default, but Netscape seems
            // to send unknown binary without content-types, so we
            // need to use application/octet-stream as the default
            // typeExpected = "application/octet-stream";
            //

            logger.log(this, "Inserting key     : [" + key +"] " + Integer.toString(key.length()) ,
                       Logger.DEBUGGING);
            logger.log(this, "Inserting htl     : " + htlUsed,
                       Logger.DEBUGGING);
            logger.log(this, "Inserting mimeType: " + mimeType ,
                       Logger.DEBUGGING);
            logger.log(this, "Data size         : " + filePart.getBody().size() ,
                       Logger.DEBUGGING);

            // REDFLAG: params for size and default encoder name.
            String encoder = null;
            if (filePart.getBody().size() > 1024*1024) {
                encoder = "default_encoder"; // REDFLAG:
                if (!fecFactory.isRegistered(encoder, true)) {
                    
                    String msg = "<h1>Configuration Error</h1><p> \n" +
                        "Couldn't load the FEC encoder: [" + encoder + "]" +
                        "<p> " +
                        "A FEC encoder is required to insert redundant SplitFiles. <p> " +
                        "Check your configuration file (freenet.conf / freenet.ini) to " +
                        "make sure a default encoder implementation is specified. ";

                    StringBuffer responseMsg = new StringBuffer();
                    responseMsg.append("<html><head><title>Configuration Error</title></head>");
                    responseMsg.append("<body bgcolor=\"#ffffff\">");
                    responseMsg.append("<p> " +  msg + " </p> \n");
                    responseMsg.append("<p><a href=\"" + baseURL + "\">Done</a>.</p> \n");
                    responseMsg.append("</body> </html> \n");
                    writeResponse(responseMsg, HttpServletResponse.SC_INTERNAL_SERVER_ERROR, resp);
                    return;
                }
            }

            InsertContext context = 
                new InsertContext(reaper, factory, bucketFactory, fecFactory, contextManager,
                                  key, filePart, htlUsed, splitFileThreads, mimeType, encoder);
            
            // Force a redirect to work around the post w/o 
            // content-length bug.  See comments for writeResponse.
            resp.sendRedirect(InsertContext.PREFIX + context.getId() + "/");
            
            // Start insert on another thread.
            Thread worker = new Thread(context, "Fproxy Insert: " + key);
            worker.start();
        }
        catch (MIMEFormatException e) {
            writeExceptionResponse("There was an error parsing the POSTed data.",
                                   baseURL, e, resp);
        }
        catch (Exception e) {
            writeExceptionResponse("Unexpected Error while inserting:",
                                   baseURL, e, resp);
        }
        finally {
            freeParts(parts);
        }
    }

    

    ////////////////////////////////////////////////////////////


    ////////////////////////////////////////////////////////////
    // Support checked jumps out of Freenet.
    protected final boolean handleCheckedJump(String url, HttpServletResponse resp) throws IOException {

        String decodedURL = getCheckedJumpURL(url);
        if (decodedURL == null) {
            return false;
        }
        
        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/html");

        PrintWriter pw = resp.getWriter();
        pw.println("<html>");
        pw.println("<head>");

        pw.println("<title>");
        pw.println("External link " + decodedURL );
        pw.println("</title>");
        pw.println("</head>");

        pw.println("<h1>Warning: External link " + decodedURL + "</h1>");
        pw.println("<p>");
        pw.println("Browsing external links <em>is not</em> anonymous.");
        pw.println("<p>");
        pw.println("Click on the link below to continue or hit the");
        pw.println("back button on your browser to abort.");
        pw.println("<p>");
        pw.println("<a href=\"");
        pw.println( decodedURL );
        pw.println("\">" + decodedURL + "</a>");
        pw.println("</body>");
        pw.println("</html>");
        pw.flush();
        return true;
    }
    
    protected final static String MSG_BADURL =  "Couldn't decode checked jump url.";

    protected final static String ESCAPED_HTTP = "/__CHECKED_HTTP__";
    protected final static String UNESCAPED_HTTP = "http://";

    protected final static String ESCAPED_FTP = "/__CHECKED_FTP__";
    protected final static String UNESCAPED_FTP = "ftp://";

    protected final static String getCheckedJumpURL(String url) {
        String ret = null;

        int p = url.indexOf(ESCAPED_HTTP);
        if (p != -1) {
            if (url.length() - p < ESCAPED_HTTP.length() + 1) {
                throw new IllegalArgumentException(MSG_BADURL);
            }
            ret = UNESCAPED_HTTP + url.substring(ESCAPED_HTTP.length() + p);
        }

        p = url.indexOf(ESCAPED_FTP);
        if (p != -1) {
            if (url.length() - p < ESCAPED_FTP.length() + 1) {
                throw new IllegalArgumentException(MSG_BADURL);
            }

            ret = UNESCAPED_FTP + url.substring(ESCAPED_FTP.length() + p);
        }

        return ret;
    }
    ////////////////////////////////////////////////////////////

    private static synchronized boolean checkForceKey(Object key) {
        return randkeys.get(key) != null;
    }

    private static synchronized String makeForceKey() {
        synchronized (randkeys) {
            String forceKey = null;
            do {
                forceKey = Integer.toHexString(random.nextInt());
            }
            while (randkeys.containsKey(forceKey));

            randkeys.put(forceKey, forceKey);

            if (lastForceKey != null) {
                randkeys.remove(lastForceKey);
                randkeys.put(lastForceKey, forceKey);
            }
            else {
                firstForceKey = forceKey;
            }

            lastForceKey = forceKey;
            if (randkeys.size() > MAX_FORCE_KEYS) {
                Object newForceKey = randkeys.get(firstForceKey);
                randkeys.remove(firstForceKey);
                firstForceKey = newForceKey;
            }
            return forceKey;
        }
    }

    // REDFLAG: remove or comment out
    private static synchronized void dumpForceKeys() {
        Enumeration keys = randkeys.keys();
        while (keys.hasMoreElements()) {
            Object key = keys.nextElement();
            System.out.println("   " + key + " -> " + randkeys.get(key));
        }
    }

    ////////////////////////////////////////////////////////////
    // Listen for RNF and DNFs so we can make informative
    // 404 messages.
    static class FailureListener implements ClientEventListener {
	public void receive(ClientEvent ce) {
	    if (ce instanceof RouteNotFoundEvent) {
                rnf = (RouteNotFoundEvent)ce;
	    }
            else if (ce instanceof DataNotFoundEvent) {
                dnf = (DataNotFoundEvent)ce;
            }
	}

        public final RouteNotFoundEvent getRNF() { return rnf; }
        public final RouteNotFoundEvent getDNF() { return rnf; }
        
        public final Exception getException() {
            if (rnf != null) {
                int total = rnf.getUnreachable() + 
                    rnf.getRestarted() + 
                    rnf.getRejected();
                
                String warning = "";
                if (total == rnf.getUnreachable()) {
                    warning = "<p> \n" +
                        "<font color=\"red\"> \n" +
                        "The request couldn't even make it off of your node. \n" +
                        "Try again, perhaps with <a href=\"/KSK@gpl.txt\">gpl.txt</a> to \n" +
                        "help your node learn about others. The publicly available \n" +
                        "seed nodes have been <b>very</b> busy lately.  If possible \n" +
                        "try to get a friend to give you a reference to their \n" +
                        "node instead. \n" +
                        "</font> \n";
                }

                String msg = "Error: <b>Route not Found</b> <br>\n" +
                    "<blockquote> \n" +
                    "Attempts were made to contact " + total + " nodes." +
                    "<ul> \n" +
                    "    <li> " + rnf.getUnreachable() + " were totally unreachable. \n" +
                    "    <li> " + rnf.getRestarted() + " restarted. \n" +
                    "    <li> " + rnf.getRejected() + " cleanly rejected. \n" +
                    "</ul> \n" +
                    warning + 
                    "</blockquote> \n";

                return new RequestFailedException(msg);
            }
            else if (dnf != null) {
                String msg = "Error: <b> Data not found </b> <br>\n";
                return new RequestFailedException(msg);
            }
            else {
                return new Exception("404");
            }
        }

        private RouteNotFoundEvent rnf = null;
        private DataNotFoundEvent dnf = null;
    }

    // Tag type.
    static class RequestFailedException extends Exception {
        public RequestFailedException(String msg) {
            super(msg);
        }
    }


    ////////////////////////////////////////////////////////////
    // Parameter parsing helper functions
    final static int readInt(HttpServletRequest req, Logger lg, String name, 
                             int defaultVal, int min, int max) {
        String valueAsString = req.getParameter(name);
        int ret = defaultVal;
        if (valueAsString != null) {
            try {
                ret = Integer.parseInt(freenet.support.URLDecoder.decode(valueAsString));
                lg.log(FproxyServlet.class, "Read " + name + " from query: " + ret, Logger.DEBUGGING);
            }
            catch (Exception e) {
                lg.log(FproxyServlet.class, "Couldn't parse " + name + " from query, using: " + ret, 
                           Logger.ERROR);
            }
        }

        return enforceLimits(lg, name, min, max, ret);
    }

    final static boolean readBoolean(HttpServletRequest req, Logger lg, String name, boolean defaultVal) {
        String valueAsString = req.getParameter(name);
        boolean ret = defaultVal;
        if (valueAsString != null) {
            try {
                final String cleanValue = freenet.support.URLDecoder.decode(valueAsString).toLowerCase();
                ret = cleanValue.equals("true");
                lg.log(FproxyServlet.class, "Read " + name + " from query: " + ret, Logger.DEBUGGING);
            }
            catch (Exception e) {
                lg.log(FproxyServlet.class, "Couldn't parse " + name + " from query, using: " + ret, 
                       Logger.ERROR);
            }
        }
        return ret;
    }

    private final static int enforceLimits(Logger lg, String name, int min, int max, int value) {
        if (value > max) {
            int oldValue = value;
            value = max;
            lg.log(FproxyServlet.class, name + "=" + oldValue + " too big, using:  " + value, 
                       Logger.ERROR);
        }
        if (value < min) {
            int oldValue = value;
            value = min;
            lg.log(FproxyServlet.class, name + "=" + oldValue + " too small, using:  " + value, 
                       Logger.ERROR);
        }
        return value;
    }

    private final static int readInt(HttpServlet servlet, Logger lg, String name, 
                             int defaultVal, int min, int max) {
        String valueAsString = servlet.getInitParameter(name);
        int ret = defaultVal;
        if (valueAsString != null) {
            try {
                ret = Integer.parseInt(valueAsString);
                lg.log(FproxyServlet.class, "Read " + name + " from initParameters: " + ret, 
                       Logger.DEBUGGING);
            }
            catch (Exception e) {
                lg.log(FproxyServlet.class, "Couldn't parse " + name + " from initParameters, using: " 
                       + ret, 
                       Logger.ERROR);
            }
        }

        return enforceLimits(lg, name, min, max, ret);
    }

    private final static boolean readBoolean(HttpServlet servlet, 
                                             Logger lg, String name, boolean defaultVal) {
        String valueAsString = servlet.getInitParameter(name);
        boolean ret = defaultVal;
        if (valueAsString != null) {
            try {
                final String cleanValue = valueAsString.toLowerCase();
                ret = cleanValue.equals("true");
                lg.log(FproxyServlet.class, "Read " + name + " from initParameters: " + ret,
                       Logger.DEBUGGING);
            }
            catch (Exception e) {
                lg.log(FproxyServlet.class, "Couldn't parse " + name + 
                       " from initParameters, using: " + ret, 
                       Logger.ERROR);
            }
        }
        return ret;
    }



    ////////////////////////////////////////////////////////////
    // Temp file handling

    class TempFileBucket extends FileBucket {
        protected TempFileBucket(File f) {
            super(f);
            //  System.err.println("FproxyServlet.TempFileBucket -- created: " + 
            //         f.getAbsolutePath());
        }

        InputStream getRealInputStream() throws IOException {
            return super.getInputStream();
        }

        OutputStream getRealOutputStream() throws IOException {
            return super.getOutputStream();
        }

        // Wrap non-const members so we can tell
        // when code touches the Bucket after it
        // has been released.
        public synchronized InputStream getInputStream() throws IOException {
            InputStream newIn = new SpyInputStream(this, file.getAbsolutePath());
            streams.addElement(newIn);
            return newIn;
        }

        public synchronized OutputStream getOutputStream() throws IOException {
            OutputStream newOut = new SpyOutputStream(this, file.getAbsolutePath());
            streams.addElement(newOut);
            return newOut;
        }

        public synchronized void resetWrite() {
            if (isReleased()) {
                throw new RuntimeException("Attempt to use a released TempFileBucket: " + getName() );
            }
            super.resetWrite();
        }

        public synchronized boolean release() {
            //System.err.println("FproxyServlet.TempFileBucket -- release: " +                                           //                      file.getAbsolutePath());

            //System.err.println("CALL STACK: ");
            //(new Exception()).printStackTrace();

            // Force all open streams closed. 
            // Windows won't let us delete the file unless we
            // do this.
            for (int i =0; i < streams.size(); i++) {
                try {
                    if (streams.elementAt(i) instanceof InputStream) {
                        ((InputStream)streams.elementAt(i)).close();

                        logger.log(this, "closed open InputStream !: " + 
                                   file.getAbsolutePath(), Logger.DEBUGGING);
                    }
                    else if (streams.elementAt(i) instanceof OutputStream) {
                        ((OutputStream)streams.elementAt(i)).close();
                        logger.log(this, "closed open OutputStream !: " + 
                                   file.getAbsolutePath(), Logger.DEBUGGING);
                    }
                }
                catch (IOException ioe) {
                }
            }

            released = true;
            if(file.exists()) {
                Core.logger.log(this, "Deleting bucket "+file.getName(), Logger.DEBUGGING);
                file.delete();
                if (file.exists()) {
                    Core.logger.log(this, "Delete failed on bucket "+file.getName(), Logger.NORMAL);
                    return false;
                }
            }
            return true;
        }

        public synchronized final boolean isReleased() { return released; }

        public void finalize() throws Throwable {
            super.finalize();
        }

        private Vector streams = new Vector();
        private boolean released;
    }

    ////////////////////////////////////////////////////////////
    // The purpose of all of this gobbledeygook is to
    // keep rude FCPClient implementations from writing
    // to temp files after the requests that
    // own them have been canceled.
    //
    // REDFLAG: Remove once FCPClient implementation 
    //          deficiencies have been corrected.
    //

    private final static boolean vociferous = false;

    class SpyInputStream extends java.io.FilterInputStream  {
        String prefix = "";
        TempFileBucket tfb = null;

        private final void println(String text) {
            if (vociferous) {
                logger.log(this, text, Logger.DEBUGGING);
            }
        }
        
        private final void checkValid() throws IOException {
            if (tfb.isReleased()) {
                throw new IOException("Attempt to use a released TempFileBucket: " + prefix);
                //throw new TrackingIOE("Attempt to use a released TempFileBucket: " + prefix);
            }
        }

        public SpyInputStream(TempFileBucket tfb, String prefix) throws IOException {
            super(null);
            InputStream tmpIn = null;
            try {
                this.prefix = prefix;
                this.tfb = tfb;
                checkValid();
                tmpIn = tfb.getRealInputStream();
                in = tmpIn;
            }
            catch (IOException ioe) {
                try {
                    if (tmpIn != null) {
                        tmpIn.close();
                    }
                }
                catch(Exception e) {
                    // NOP
                }
                throw ioe;
            }
            println("Created new InputStream");
        }
        
        ////////////////////////////////////////////////////////////
        // FilterInputStream implementation

        public int read() throws java.io.IOException {
            synchronized (tfb) {
                println(".read()");
                checkValid();
                return in.read();
            }
        }
        
        public int read(byte[] bytes) throws java.io.IOException {
            synchronized (tfb) {
                println(".read(byte[])");
                checkValid();
                return in.read(bytes);
            }
        }
        
        public int read(byte[] bytes, int a, int b) throws java.io.IOException {
            synchronized (tfb) {
                println(".read(byte[], int, int)");
                checkValid();
                return in.read(bytes, a, b);
            }
        }
        
        public long skip(long a) throws java.io.IOException {
            synchronized (tfb) {
                println(".skip(long)");
                checkValid();
                return in.skip(a);
            }
        }
        
        public int available() throws java.io.IOException {
            synchronized (tfb) {
                println(".available()");
                checkValid();
                return in.available();
            }
        }
        
        public void close() throws java.io.IOException {
            synchronized (tfb) {
                println(".close()");
                checkValid();       
                in.close();
                if (tfb.streams.contains(in)) {
                    tfb.streams.removeElement(in);
                }
            }
        }
        
        public  void mark(int a) {
            synchronized (tfb) {
                println(".mark(int)");
                in.mark(a);
            }
        }

        public void reset() throws java.io.IOException {
            synchronized (tfb) {
                println(".reset()");
                checkValid();
                in.reset();
            }
        }
        
        public boolean markSupported() {
            synchronized (tfb) {
                println(".markSupported()");
                return in.markSupported();
            }
        }
    }

    /*
    // REDFLAG: remove debugging hack
    class TrackingIOE extends IOException {

        TrackingIOE(String msg) {
            super(msg);
        }

        void calledFrom() {
            System.err.println("---CALLED FROM------------------------------------");
            (new Exception()).printStackTrace();
            System.err.println("--------------------------------------------------");
        }

        public String toString() {
            calledFrom();
            return super.toString();
        }
        
    }
    */

    public class SpyOutputStream extends FilterOutputStream {
        String prefix = "";
        TempFileBucket tfb = null;

        private void println(String text) {
            if (vociferous) {
                logger.log(this, text, Logger.DEBUGGING);
            }
        }

        private final void checkValid() throws IOException {
            if (tfb.isReleased()) {
                throw new IOException("Attempt to use a released TempFileBucket: " + prefix);
                //throw new TrackingIOE("Attempt to use a released TempFileBucket: " + prefix);
            }
        }
    
        public SpyOutputStream(TempFileBucket tfb, String pref) throws IOException {
            super(null);
            OutputStream tmpOut = null;
            try {
                this.prefix = pref;
                this.tfb = tfb;
                checkValid();
                tmpOut = tfb.getRealOutputStream();
                out = tmpOut;
            }
            catch (IOException ioe) {
                //ioe.printStackTrace();
                println("SpyOutputStream ctr failed!: " + ioe.toString());
                try {
                    if (tmpOut != null) {
                        tmpOut.close();
                    }
                }
                catch(Exception e0) {
                    // NOP
                }
                println("SpyOutputStream ctr failed!: " + ioe.toString());
                throw ioe;
            }
            println("Created new OutputStream");
        }
    
        ////////////////////////////////////////////////////////////
        // FilterOutputStream implementation
        public void write(int b) throws IOException {
            synchronized (tfb) {
                println(".write(b)");
                checkValid();
                out.write(b);
            }
        }
    
        public void write(byte[] buf) throws IOException {
            synchronized (tfb) {
                println(".write(buf)");
                checkValid();
                out.write(buf);
            }
        }
    
        public void write(byte[] buf, int off, int len) throws IOException {
            synchronized (tfb) {
                println(".write(buf,off,len)");
                checkValid();
                out.write(buf, off, len);
            }
        }
    
        public void flush() throws IOException {
            synchronized (tfb) {
                println(".flush()");
                checkValid();
                out.flush();
            }
        }
    
        public void close() throws IOException {
            synchronized (tfb) {
                println(".close()");
                checkValid();
                out.close();
                if (tfb.streams.contains(out)) {
                    tfb.streams.removeElement(out);
                }
            }
        }
    }
    ////////////////////////////////////////////////////////////
    class TempBucketFactory implements BucketFactory {
        public Bucket makeBucket(long size) throws IOException {
            File f = null;
            do {
                if (tmpDir != null) {
                    f = new File(tmpDir, "tbf_" + 
                                 Long.toHexString(Math.abs(Core.randSource.nextInt())));
                }
                else {
                    f = new File("tbf_" + 
                                 Long.toHexString(Math.abs(Core.randSource.nextInt())));
                }
            } while (f.exists());


            return new TempFileBucket(f);
        }

        public void freeBucket(Bucket b) throws IOException {
            if (b instanceof TempFileBucket) {
                ((TempFileBucket)b).release();
            }
        }
    }
}

/*
  ############################################################
  # NOTE: You shouldn't need to use this but I am leaving
  #       it here for reference. The params
  #       have been added to Node.java. Just run:
  #
  # java freenet.node.Main --config.
  ############################################################

  #
  # append this to your freenet.conf file to start fproxy
  # in the same JVM as your node.
  #

  ############################################################
  # Services
  ############################################################

  ####################
  # fproxy
  fproxy.class=freenet.client.http.FproxyServlet
  fproxy.port=8888
  fproxy.insertHtl=25
  fproxy.requestHtl=25
  fproxy.passThroughMimeTypes=text/plain,image/jpeg,image/gif,image/png,foo/bar
  fproxy.filter=true

  ####################
  # Services run in the same JVM as the node
  services=fproxy

  ####################
  # Services run in a separate JVM from the node
  #externalServices=fproxy

  # To run fproxy externally, uncomment the line above and
  # comment out the services= line. The run
  # java freenet.client.http.HttpServletRunner in the same directory
  # as your freenet.conf file.
*/






