//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Sphere.hpp,v 1.3 2004/02/08 18:44:19 delpinux Exp $

// This class provides a toolkit to manipulate Spheres. It inherits from the
// Shape class.

#ifndef SPHERE_HPP
#define SPHERE_HPP

#include <Shape.hpp>
/*!
  \class Sphere
  This is the class which defines a Virtual Reality Sphere.

  \author Stphane Del Pino.
*/
class Sphere
  : public Shape
{
private:
  /*! The center of the Sphere.
    \warning could use a TinyVector<3>
  */
  Vertex __center;

  //! The radius of the Sphere.
  real_t __radius;

  //! The square of the \a radius.
  real_t __radius2;

  //! Returns \p true if the point \a P is inside the Sphere.
  inline bool inShape(const TinyVector<3, real_t>& P) const;

  //! Prints the Sphere informations using the stream \a s.
  std::ostream& put(std::ostream& s) const;
public:

  //! Copies the Sphere \a S.
  const Sphere& operator = (const Sphere& S);

  //! Default constructor.
  Sphere();

  //! Constructs the sphere centered in c(x,y,z) and of radius r
  Sphere(const Vertex& c, const real_t& r);

  //! Copy constructor.
  Sphere(const Sphere& S);

  //! Destructor.
  ~Sphere()
  {
    ;
  }
};

//! Returns true if the point \a p is inside the Sphere.
inline bool Sphere::inShape(const TinyVector<3, real_t>& p) const
{
  return (__center.distance2(p) <= __radius2);
}

#endif // SPHERE_HPP

