//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Vector.hpp,v 1.4 2004/01/24 13:33:39 delpinux Exp $


#ifndef _VECTOR_HPP_
#define _VECTOR_HPP_

// Include STD definitions to get same environment ...
#include <stddef.h>
#include <cassert>
#include <cstdlib>

#include <ZoneCopy.hpp>

#include <iostream>

#include <BaseVector.hpp>
#include <TermToTerm.hpp>

/*!
  class Vector
  This class defines a template Vector class to store huge quantity of datas.
  The stored elements type is the template argument.

  \todo Use expression template to provide a more efficient code.
  \author Stphane Del Pino.
*/

template <typename T>
class Vector
  : public BaseVector
{
public:
  //! T ValueType Used to be able to access to T using \a typename.
  typedef T ValueType;
protected:
  //! The number of elements of the vector.
  size_t s;

  //! The elements of the vector.
  T* __restrict element;

public:
  //! Read-only access to the size of the vector.
  inline const size_t& size() const
  {
    return s;
  }

  //! Finds the number of an element of the vector.
  inline const size_t number(T const& e) const
  {
    assert ((&e >= element)&&(static_cast<size_t>(&e-element) < s));
    return (&e-element);
  }

  /** 
   * fills the vector with random values
   * 
   */  
  void randomize(real_t max = 1.) {
    const real_t scale = max/RAND_MAX;
    for (size_t i=0; i<this->size(); ++i) {
      element[i] = random() * scale;
    }
  }

  //! writes a vector in a stream.
  friend std::ostream& operator << (std::ostream& os, const Vector<T>& V)
  {
    for (size_t i=0; i<V.size(); ++i) {
      os << V[i] << '\n';
    }
    return os;
  }

  //! Read-only access to the ith element of the vector.
  inline const T& operator[](const size_t& i) const
  {
    assert ((i>=0)&&(i<s));
    return element[i];
  }

  //! Access to the ith element of the vector.
  inline T& operator[](const size_t& i)
  {
    assert ((i>=0)&&(i<s));
    return element[i];
  }

  /*!
    Read-only access to the ith element of the vector.
    Used to provide a Blitz-like interface.
  */
  inline const T& operator()(const size_t& i) const
  {
    assert ((i>=0)&&(i<s));
    return element[i];
  }

  /*!
    Access to the ith element of the vector.
    Used to provide a Blitz-like interface.
  */
  inline T& operator()(const size_t& i)
  {
    assert ((i>=0)&&(i<s));
    return element[i];
  }

  /*!
    This function is to resize the vector.
    Used to provide a Blitz-like interface, the elements are lost if the size
    is changed.
   */
  void resize(const size_t& newSize)
  {
    assert (newSize>=0);

    if (newSize != s) {
      if (s!=0)	{		// Destroy previous vector
	delete [] element;
      }
      s = newSize;
      element = new ValueType[s];
    } else {
      ;				// else do nothing
    }
  }

  //! \c operator= overloading. Initializes every element to the value \a t.
  inline const Vector<T>& operator=(const T t)
  {
    for (size_t i=0; i<s; ++i)
      element[i] = t;
    return (*this);
  }

  //! \c operator= overloading. Copies the vector \a V.
  inline const Vector<T>& operator=(const Vector<T>& V)
  {
    assert (s == V.s);
    ZoneCopy(element,V.element,s);

    return (*this);
  }


  /*!
    \c operator* overloading. Returns \a t * \a this.
    \warning a temporary is created!
   */
  inline const Vector<T> operator*(const T& t) const
  {
    Vector<T> prod(*this);
    TermToTermProduct<T> tttp;
    for (size_t i=0; i<s; ++i)
      prod.element[i] = tttp(element[i],t);
    return prod;
  }


  /*!
    \c operator* overloading (friend version). Returns \a t * \a this.
    \warning a temporary is created!
   */
  friend const Vector<T> operator*(const T& t, const Vector<T>& V)
  {
    Vector<T> sum(V.s);
    TermToTermProduct<T> tttp;
    for (size_t i=0; i<V.s; ++i)
      sum.element[i] = tttp(V.element[i],t);
    return sum;
  }


  //! \c operator*= overloading. The vector is multiplied by \a t.
  template <class T2>
  inline const Vector<T>& operator*=(const T2& t)
  {
    TermToTermProduct<T> tttp;
    for (size_t i=0; i<s; ++i)
      element[i] = tttp(element[i],t);
    return (*this);
  }

  /*!
    \c operator+ overloading. Computes sum with \a V.
    \warning a temporary is created!
  */
  inline const Vector<T> operator+(const Vector<T>& V) const
  {
    assert(s==V.s);
    Vector<T> W(s);
    for (size_t i=0; i<s; ++i)
      W.element[i] = element[i] + V.element[i];
    return (W);
  }

  /*!
    \c operator- overloading. Computes difference with \a V.
    \warning a temporary is created!
  */
  inline const Vector<T> operator-(const Vector<T>& V) const
  {
    assert(s==V.s);
    Vector<T> W(s);
    for (size_t i=0; i<s; ++i) {
      W.element[i] = element[i] - V.element[i];
    }
    return W;
  }


  //! \c operator+= overloading. Addes \a V to the vector.
  inline const Vector<T>& operator+=(const Vector<T>& V)
  {
    assert(s==V.s);
    for (size_t i=0; i<s; ++i)
      element[i] += V.element[i];
    return (*this);
  }

  //! \c operator+= overloading. Addes \a V to the vector.
  inline const Vector<T>& operator+=(const T& t)
  {
    for (size_t i=0; i<s; ++i)
      element[i] += t;
    return (*this);
  }

  //! \c operator-= overloading. Substractes \a V to the vector.
  inline const Vector<T>& operator-=(const Vector<T>& V)
  {
    assert(s==V.s);
    for (size_t i=0; i<s; ++i)
      element[i] -= V.element[i];
    return (*this);
  }
 
  //! \c operator* overloading. Scalar product.
  inline const real_t operator*(const Vector<T>& V) const
  {
    assert(s==V.s);
    real_t t = 0;
    for (size_t i=0; i<s; ++i)
      t += element[i] * V.element[i];
    return t;
  }

  /*!
    Computes the  norm  of a given vector \a v. The result is a T!
    If T is of scalar type it returns the usual norm.
    \warning only works if the template type \a T allows scalar product.
    \todo Name might be change?
  */
  inline friend const real_t Norm(const Vector<T>& v)
  {
    return sqrt(v*v);
  }


  //! Default constructor initializes a dimension 0 vector.
  Vector()
    : s(0)
  {
    ;
  }

  /*!
    Constructs a vector which size is \a givenSize.
    element values are garbage.
  */
  Vector(size_t givenSize)
    : s(givenSize)
  {
    assert(s>=0);
    if (s>0)
      element = new T[givenSize];
  }

  //! Copy constructor.
  Vector(const Vector<T>& V)
    : s(V.s)
  {
    if (s>0)
      element = new T[s];
    ZoneCopy(element,V.element,s);
  }

  //! Destructor.
  ~Vector()
  {
    if (s > 0)
      delete[] element;
  }
};

#endif // _VECTOR_HPP_

