/* C Mode */

/* network.c
   Implements network utility functions for FramerD
      The core code for DType client/server interactions is here.
   Originally implemented by Ken Haase in the Machine Understanding Group
     at the MIT Media Laboratory.

   Copyright (C) 1994-2001 Massachusetts Institute of Technology
   Copyright (C) 2001-2002 beingmeta, inc. (A Delaware Corporation)

   This program comes with absolutely NO WARRANTY, including implied
   warranties of merchantability or fitness for any particular purpose.

    Use, modification, and redistribution of this program is permitted
    under the terms of either (at the developer's discretion) the GNU
    General Public License (GPL) Version 2, the GNU Lesser General Public
    License.

    This program is based on the FramerD library released in Fall 2001 by
    MIT under both the GPL and the LGPL licenses, both of which accompany
    this distribution.  Subsequent modifications by beingmeta, inc. are
    also released under both the GPL and LGPL licenses (at the developer's
    discretion).

*/ 

static char vcid[] = "$Id: network.c,v 1.28 2002/04/15 02:14:08 haase Exp $";

/** Lots of OS specific stuff **/
/** Initial declarations **/
/** Support for Windows sockets **/
/** Network database functions **/
/** Connecting TO servers **/
/** Restarting connections **/
/** Making and closing connections **/
/** Non STDIO access to sockets **/
/** dtype_eval for stdio sockets **/
/** dtype eval for non stdio sockets **/
/** fd_dtype_eval **/
/** TT (Touch-tone) encoding **/
/** Server socket utils **/
/** Server (select) loop **/
/** Threaded servers **/
/** Server utility functions **/
/** Sending SMTP mail **/
/** Accessing URLS **/
/** Posting to URLs **/
/** Initialization **/

#include "dtypes.h"
#include "signal.h"


/** Lots of OS specific stuff **/

#include <time.h>
#include <assert.h>
#include <stdarg.h>

#if defined(WIN32)
# include <winsock.h>
# include <fcntl.h>
#define MSG_WAITALL 0
#elif defined(OS2)
# define BSD_SELECT 1
# include <types.h>
# define close soclose
static unsigned int sockets_initialized=0;
#else /* We're under Unix, boys... */
#if HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif
#if HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#endif
#if HAVE_SYS_UN_H
# include <sys/un.h>
#ifndef SUN_LEN
#define SUN_LEN(x) sizeof(*x)
#endif
#endif
#if HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#if HAVE_NETDB_H
# include <netdb.h>
#endif
#if HAVE_FCNTL_H
# include <fcntl.h>
#elif HAVE_SYS_FCNTL_H
# include <sys/fcntl.h>
#endif
#if HAVE_SYS_IOCTL_H
# include <sys/ioctl.h>
# define ioctlsocket ioctl
#endif
#if HAVE_SYS_FILIO_H
# include <sys/filio.h>
#endif
#endif

#if ((defined(PF_UNIX)) && (!(defined(PF_LOCAL))))
#define PF_LOCAL PF_UNIX
#endif

#if ((defined(AF_UNIX)) && (!(defined(AF_LOCAL))))
#define AF_LOCAL AF_UNIX
#endif

#if FD_USING_THREADS
static pthread_mutex_t server_lookup_lock;
#endif

fd_exception
  fd_SocketClosed=_("Socket closed"),
  fd_UnknownHost=_("Unknown host"),
  fd_BadServerSpec=_("Bad server specification"),
  fd_ConnectionFailed=_("Couldn't open connection"),
  fd_ConnectionTimeOut=_("Connection timed out"),
  fd_UnhandledURL=_("Can't handle URL type");

static char *agent_id="FramerD";

/** Initial declarations **/

#ifndef DTIO_BLOCK_SIZE
#define DTIO_BLOCK_SIZE 65536*8
#endif

#define MAX_SERVERS 100
#define MAX_HOSTNAME 100

static struct FD_SERVER servers[MAX_SERVERS];
static int server_count=0;

fd_exception fd_NetworkInitFailed=_("Couldn't initialize networking"),
  fd_NoSocket=_("Couldn't open socket for connection"),
  fd_NoConnection=_("Couldn't connect socket across network");

fd_exception fd_NetworkReadFailed=_("Network read failed"),
             fd_NetworkWriteFailed=_("Network write failed");

static int trace_dteval=0;

static char *read_string_from_file(FILE *f)
{
  char *buf=fd_xmalloc(DTIO_BLOCK_SIZE);
  int size=0, limit=DTIO_BLOCK_SIZE, c;
  while ((c=getc(f)) >= 0) {
    if (size == limit) {
      buf=fd_xrealloc(buf,limit*2); limit=limit*2;}
    buf[size++]=(char) c;}
  if (size == limit) {
    buf=fd_xrealloc(buf,limit*2); limit=limit*2;}
  buf[size++]='\0';
  return buf;
}

/** WIN32 fdopen **/

#if WIN32
static FILE *socket2file(long socket_id,char *mode)
{
  if (((socket_id = _open_osfhandle(socket_id,_O_BINARY)) < 0) ||
      ((socket_id = dup(socket_id)) < 0))
    fd_raise_exception("WIN32/OSF socket open failed");
  return fdopen(socket_id,mode);
}
#else
#define socket2file fdopen
#endif

/** Dealing with timeouts **/

static int set_noblock(int fd)
{
  int flags=fcntl(fd,F_GETFL);
  flags|=O_NONBLOCK;
  fcntl(fd,F_SETFL,flags);
}

static int wait_on_socket(int fd,int secs,int usecs,int rd,int wr,int exc)
{
  fd_set rs, ws, xs; int sresult=0; struct timeval timeout;
  FD_ZERO(&rs); FD_ZERO(&ws); FD_ZERO(&xs);
  if (rd) FD_SET(fd,&rs); if (wr) FD_SET(fd,&ws); if (exc) FD_SET(fd,&xs); 
  timeout.tv_sec=secs; timeout.tv_usec=usecs;
  while (sresult == 0) {
    sresult=select(fd+1,&read,&write,&exc,&timeout);
    if (sresult) return sresult;
    if (errno != EINPROGRESS) break;
    FD_CLEAR_ERR();
    if (rd) FD_SET(fd,&rs); if (wr) FD_SET(fd,&ws); if (exc) FD_SET(fd,&xs); 
    timeout.tv_sec=secs; timeout.tv_usec=usecs;}
  return sresult;}
  

static int timed_connect
  (int secs,int socket_id,struct sockaddr *addr,int addr_len)
{
  int wait_result=-1;
  set_noblock(socket_id);
  if (connect(socket_id,addr,addr_len) == 0) return socket_id;
  else if (wait_result=wait_on_socket(socket_id,secs,0,1,1,0)>=0) 
    return connect(socket_id,addr,addr_len);
  else if (wait_result == 0) {
    errno=ETIMEDOUT; return -1;}
  else return -1;
}

static int timed_recv(int secs,int socket,char *data,int len,int flags)
{
  int recv_result=0;
  set_noblock(socket_id);
  if (recv_result=recv(socket_id,data,len,flags) == 0) {
    int wait_result=wait_on_socket(socket_id,secs,0,1,0,0);
    if (wait_result>0) return recv(socket_id,data,len,flags);
    else if (wait_result == 0) {
      errno=ETIMEDOUT; return -1;}
    else return -1;}
  else return recv_result;
}

/** Support for Windows **/

#ifdef WIN32
#define close closesocket
static unsigned int sockets_initialized=0;
static void init_win32sockets()
{
  WORD wVersionRequested; 
  WSADATA wsaData; 
  int err; 
  int sockopt = SO_SYNCHRONOUS_NONALERT;
  
  wVersionRequested = MAKEWORD(2, 0); 
  
  err = WSAStartup(wVersionRequested, &wsaData); 
  
  /* Tell the user that we couldn't find a useable */ 
  /* winsock.dll.     */ 
  if (err != 0) fd_raise_exception(fd_NetworkInitFailed);
  /* Confirm that the Windows Sockets DLL supports 1.1.*/ 
  /* Note that if the DLL supports versions greater */ 
  /* than 1.1 in addition to 1.1, it will still return */ 
  /* 1.1 in wVersion since that is the version we */ 
  /* requested. */ 
  
  if ( LOBYTE( wsaData.wVersion ) != 2 || HIBYTE( wsaData.wVersion ) != 0 ) 
    {WSACleanup(); 
    fd_raise_exception(fd_NetworkInitFailed);}
  
  if (setsockopt(INVALID_SOCKET, SOL_SOCKET, SO_OPENTYPE,
		 (char *)&sockopt,sizeof(sockopt)) < 0) {
    perror("setsockopt:");}
}
#endif 

/** Network database functions **/

static struct hostent localhost;

/* Gets the localhost 127.0.0.1 */
static struct hostent *get_island_host()
{
  return &localhost;
}

/** Connecting TO servers **/

DTYPES_EXPORT
/* fd_open_tcp_socket:
     Arguments: a hostname (a string) and a port (an int)
                and an id (a string) and an error flag (an int)
     Returns: an open socket

     This utility function gets an open TCP socket for a host and port. */
int fd_open_tcp_socket
   (char *hostname,int port,char *id,int signal_error,char **fullname)
{
  long socket_id; int flag=1;
  struct hostent *remote_host;
  struct sockaddr_in server_address; 
  if ((strcmp(hostname,"localhost") == 0))
    remote_host=get_island_host();
  else remote_host=gethostbyname(hostname);
  if (errno) {CLEAR_ERR();}
  if (remote_host == NULL)
    fd_raise_detailed_exception(fd_UnknownHost,hostname);
  else if (fullname) *fullname=fd_strdup(remote_host->h_name);
  socket_id=socket(PF_INET,SOCK_STREAM,0);
  if (socket_id < 0) {
    char *buf=fd_xmalloc(strlen(hostname)+10);
    if (id) 
      sprintf(buf,"%s[%d@%s] (%s)",id,port,hostname,strerror(errno));
    else sprintf(buf,"%d@%s (%s)",port,hostname,strerror(errno));
    if (signal_error) fd_raise_detailed_exception(fd_NoSocket,buf);
    else return -1;}
  server_address.sin_port=htons((short)port);
  memmove((char *) &((server_address).sin_addr),(char *) remote_host->h_addr,
	  remote_host->h_length);
  server_address.sin_family=remote_host->h_addrtype;
  if ((timed_connect(8,socket_id,(struct sockaddr *)&server_address,
		     sizeof(struct sockaddr_in))) < 0) {
    char *buf=fd_xmalloc(128);
    if (id) 
      sprintf(buf,"%s[%d@%s] (%s)",id,port,hostname,strerror(errno));
    else sprintf(buf,"%d@%s (%s)",port,hostname,strerror(errno));
    if (signal_error) {
      close(socket_id);
      fd_raise_detailed_exception(fd_NoConnection,buf);}
    else return -1;}
  return socket_id;
}

#if HAVE_SYS_UN_H
DTYPES_EXPORT
/* fd_open_local_socket:
     Arguments: a filename (a string) and a server id (a string) and an error flag (an int)
     Returns: an open socket

     This utility function gets an open socket for a local filenname. */
int fd_open_local_socket(char *filename,char *id,int signal_error)
{
  long socket_id;
  struct sockaddr_un servname;
  socket_id=socket(PF_LOCAL,SOCK_STREAM,0);
  if (socket_id < 0) {
    char *buf=fd_xmalloc(strlen(filename)+10);
    if (id) 
      sprintf(buf,"%s[@%s] (%s)",id,filename,strerror(errno));
    else sprintf(buf,"@%s (%s)",filename,strerror(errno));
    if (signal_error) fd_raise_detailed_exception(fd_NoSocket,buf);
    else return -1;}
  servname.sun_family=AF_LOCAL; strcpy(servname.sun_path,filename);
  if ((timed_connect(8,socket_id,(struct sockaddr *)&servname,
		     SUN_LEN(&servname))) < 0) {
    char *buf=fd_xmalloc(128);
    if (id) 
      sprintf(buf,"%s[@%s] (%s)",id,filename,strerror(errno));
    else sprintf(buf,"@%s (%s)",filename,strerror(errno));
    if (signal_error) {
      close(socket_id);
      fd_raise_detailed_exception(fd_NoConnection,buf);}
    else return -1;}
  return socket_id;
}
#else
int fd_open_local_socket(char *filename,char *id,int signal_error)
{
  fd_raise_exception(_("No local socket support"));
  return -1;
}
#endif

DTYPES_EXPORT
/* fd_init_connection
    Arguments: a pointer to an FD_SERVER struct,
               a host (a string), a port (an int),
	       and an id (a string or NULL)
    Returns: nothing

  Initializations a TCP/IP connection structure. */
void fd_init_connection(fd_server server,char *dest,int port,char *id)
{
  int socket_id;
  if (port < 0) socket_id=fd_open_local_socket(dest,id,1);
  else socket_id=fd_open_tcp_socket(dest,port,id,1,&(server->servername));
  server->port=port;
  if (id) server->id=fd_strdup(id); else server->id=id;
  if (port<0) server->servername=fd_strdup(dest);
  server->socket=socket_id;
#if FD_USE_STDIO_SOCKETS
  server->in=socket2file(socket_id,"rb"); 
  server->out=socket2file(socket_id,"wb");
  server->closefn=NULL;
  /* Clear dangling errors from fdopen. */
  if (server->in) {CLEAR_ERR();}
#endif
  if (errno) {CLEAR_ERR();}
  server->ref_count=1; server->traced=0;
}

DTYPES_EXPORT
/* fd_open_connection
    Arguments: a name (a string), a port (an int) and an id (a string)
    Returns: a server (a pointer to a FD_SERVER struct)

  Looks for a current connection to port@hostname and creates one
if it doesn't exist. */
fd_server fd_open_connection(char *name,int port,char *id)
{
  int i=0, free_connection=-1;
  char *realname=((port<0) ? (fd_get_real_pathname(name)) : (fd_get_real_hostname(name)));
  lock_mutex(&server_lookup_lock);
  while (i < server_count)
    if ((servers[i].port == port) &&
	((strcmp(realname,servers[i].servername)) == 0)) {
      servers[i].ref_count++;
      unlock_mutex(&server_lookup_lock);
      fd_xfree(realname);
      return &servers[i];}
    else if (servers[i].socket < 0) {free_connection=i; break;}
    else i++;
  fd_xfree(realname);
  if (free_connection < 0) {
    free_connection=server_count++;
    unlock_mutex(&server_lookup_lock);}
  else if (free_connection >= MAX_SERVERS) {
    unlock_mutex(&server_lookup_lock);
    fd_raise_exception(_("Too many open connections"));}
  (void) fd_init_connection(&servers[free_connection],name,port,id);
  servers[free_connection].ref_count=1; servers[free_connection].traced=0;
  return &servers[free_connection];
}

DTYPES_EXPORT
/* fd_get_server_count
    Arguments: none
    Returns: an integer

  Returns the number of servers you are currently connected to.
*/
int fd_get_server_count()
{
  int count=server_count, i=0;
  while (i < server_count)
    if (servers[i].port == 0) {count--; i++;} else i++;
  return count;
}

/** Getting primary host names */

DTYPES_EXPORT
char *fd_get_real_hostname(char *hostname)
{
  struct hostent *remote_host=gethostbyname(hostname);
  if (remote_host) {
    CLEAR_ERR();
    return fd_strdup(remote_host->h_name);}
  else {CLEAR_ERR(); return NULL;}
}

/** Restarting connections **/

DTYPES_EXPORT
/* fd_restart_connection
    Arguments: a server
    Returns: a server

  Restarts the connection to server, returning NULL if the restart fails
*/
#define _str(x) ((x==NULL) ? ((fd_u8char*)"") : (x))

fd_server fd_restart_connection(fd_server s)
{
  struct sockaddr_in server_address; 
  struct hostent *remote_host; int socket_id; 
  fd_notify(_("Restarting connection to %s[%d@%s]"),
	    _str(s->id),s->port,s->servername);
  close(s->socket);
  if (strcmp(s->servername,"localhost") == 0)
    remote_host=get_island_host();
  else remote_host=gethostbyname(s->servername); 
  socket_id=socket(PF_INET,SOCK_STREAM,0);
  if (socket_id < 0) {
    char buf[256], *bufptr, *err=strerror(errno), *id=_str(s->id);
    if (strlen(id)+strlen(s->servername)+strlen(err) > 200)
      bufptr=fd_xmalloc(strlen(id)+strlen(s->servername)+strlen(err)+32);
    else bufptr=buf;
    sprintf(buf,"%s[%d]@%s (%s)",
	    _str(s->id),s->port,s->servername,strerror(errno));
    fd_raise_detailed_exception(fd_NoSocket,bufptr);
    return (fd_server) 0;}
  server_address.sin_port=htons((short)s->port);
  memmove((char *) &((server_address).sin_addr),
	  (char *) remote_host->h_addr,
	  remote_host->h_length);
  server_address.sin_family=remote_host->h_addrtype;
  if ((timed_connect(32,socket_id,
		     (struct sockaddr *)&server_address,
		     sizeof(struct sockaddr_in)))
      < 0) {
    char *buf=fd_xmalloc(128);
    sprintf(buf,"%s[%d@%s] (%s)",
	    _str(s->id),s->port,s->servername,strerror(errno));
    fd_raise_detailed_exception(fd_NoConnection,buf);
    return (fd_server) 0;}
  s->socket=socket_id;
#if FD_USING_THREADS
  pthread_mutex_init(&(s->lock),pthread_mutexattr_default);
#endif
#if FD_USE_STDIO_SOCKETS
  s->in=socket2file(socket_id,"rb"); 
  s->out=socket2file(socket_id,"wb");
  if (s->in) {CLEAR_ERR();}
  sleep(5);
#endif
  return s;
}

/** Making and closing connections **/

DTYPES_EXPORT
/* fd_close_connection
    Arguments: a server
    Returns: nothing

  Closes the connection to server.
  This just does a close on the socket and nothing clever to
ensure that pending transactions are completed (though they should
throw out and restart).
*/
void fd_close_connection(fd_server s)
{
  TIDY_ERRNO("fd_close_connection preamble");
  lock_mutex(&server_lookup_lock);
  s->ref_count--; if (s->ref_count) {
    unlock_mutex(&server_lookup_lock);
    return;}
  if (s->socket < 0) {
    unlock_mutex(&server_lookup_lock);
    return;}
  if (s->closefn) s->closefn(s);
  /* Will I get into trouble for closing fclose on both the input and
     output streams and then calling close on the socket descriptor
     itself? */
#if FD_USE_STDIO_SOCKETS
  if (s->out) {
    fflush(s->out); fd_free_xfile(s->out); s->out=NULL;}
  if (s->in) {fd_free_xfile(s->in); s->in=NULL;}
#endif
  close(s->socket);
  s->socket=-1; s->port=0; 
  free(s->servername); s->servername=NULL;
  if (s->id) {free(s->id); s->id=NULL;}
  TIDY_ERRNO("fd_close_connection");
  unlock_mutex(&server_lookup_lock);
}

DTYPES_EXPORT
/* fd_close_connections
    Arguments: none
    Returns: nothing

  Closes all active connections.
*/
void fd_close_all_connections()
{
  int i=0; while (i < server_count)
    fd_close_connection(&servers[i++]);
}

DTYPES_EXPORT
/* fd_connect
    Arguments: a string identifying a server
    Returns: a server

  Makes a connection to a particular server, signalling an error
if a connection cannot be made.
  The server identification has the form port@host, where
port is either a registered service or is 'touch-tone' encoded to
a port number
*/
fd_server fd_connect(char *spec)
{
  char *host=str01-16T20:10:29GMT   	ESPERANTOB   	kontempli   rigardi   konsideri kielB&   SYNSET-CODE   VERB G@       SOURCE          	WORD-DATA	      plan      be_after       	SYNSET-ID G@   N-WORDS      
SENSE-KEYSB   plan%2:49:01::   be_after%2:49:00::   be_after%2:31:00::   be_after%2:31:00::         plan%2:31:01::   plan%2:31:01::      ,   HYPERNYM     /   HYPONYMB     /     /     /   VERB-FRAMES         GLOSS   have the will and intention to carry out some action; "He plans to be in graduate school next year"; "The rebels had planned turmoil and confusion"   PART-OF-SPEECH   VERB   WORDSB   plan   be after   ROGET-CLUSTERSB     /     /     /     &     /     /     /     &     /     /     /     /     /     /     /     /     /     /     /     +9   CYC-CONCEPTS        TRANSLATIONSB-   DEUTSCH   	entwerfen   ITALIANO   
progettare@ P O R T U G U  S   	projectar   ROMANIAN@ i n t e nc i o n a@ F R A N  A I S   	esquisser   DUTCH   plannen smeden   ITALIANO   	disegnare   	AFRIKAANS   beplan   	ESPERANTO   plani   POLISH@ p l a n o w a@ F R A N  A I S@  l a b o r e r   SUOMI   suunnitella@ E S P A  O L   redactar@ F R A N  A I S   projeter   ITALIANO   pianificare   	ESPERANTO   planizi   DEUTSCH   projektieren   ROMANIAN   	planifica@ E S P A  O L   idear   	AFRIKAANS   beraam@ E S P A  O L   trazar   	ESPERANTO   projekti@ P O R T U G U  S   projetar@ P O R T U G U  S   planejar@ E S P A  O L   
planificar   DEUTSCH   konzipieren@ F R A N  A I S   	planifier@ F R A N  A I S   dresser@ F R A N  A I S@
 b  t i r@ E S P A  O L   	proyectar@ E S P A  O L   concebir   ITALIANO   ideare   
NEDERLANDS   beramen   DANSK   planere   
NEDERLANDS   plannen   FAROESE@ r   l e g g j a   DANSK@ p l a n l  g g e@ F R A N  A I S   tracer@ E S P A  O L   planear   DEUTSCH   planen@ F R A N  A I S   	concevoir   ITALIANO   	concepire   
NEDERLANDS   	ontwerpen@ F R A N  A I S@  b a u c h e r   
NEDERLANDS   plannen smeden   
LEX-FILENO      SENSE-CATEGORY   VERB.COGNITION   OBJ-NAME   VERB.COGNITION   plan   be after   	ESPERANTOB   plani   planizi   projektiB(   SYNSET-CODE   VERB O       SOURCE          	WORD-DATA	      intend       mean       think      	SYNSET-ID O   N-WORDS      
SENSE-KEYSB	   mean%2:49:00::   think%2:49:06::   intend%2:49:00::   intend%2:31:00::   intend%2:31:00::          mean%2:31:00::   mean%2:31:00::         think%2:31:06::   think%2:31:06::         HYPERNYM     3   HYPONYMB     /     3     3   VERB-FRAMES         GLOSS   have in mind as a purpose; "I mean no harm"; "I only meant to help you"; "She didn't think to harm me"; "We thought to return early that night"   PART-OF-SPEECH   VERB   WORDSB   think   intend   mean   ROGET-CLUSTERSB     (     3               3          3          0V     3     3          3     0I     )     0J   
PORTUGUESEB   apontar   intentar   	pretender   	tencionar   ter em vista   visar a   TRANSLATIONSB@ P O R T U G U  S   visar a   
PORTUGUESE   achar   	ESPERANTO   intenci   TURKISH@ a m a  l a m a k   
PORTUGUESE   apontar   DUTCH   
van plan zijn   
PORTUGUESE   visar a   DUTCH   
overdenken   SVENSKA@
 t  n k a   DUTCH   denken   DEUTSCH   befinden@ F R A N  A I S   
penser que   	NORWEGIAN   tenke   FAROESE@  t l a   DEUTSCH   wollen   LAO@@ F R A N  A I S@ r  f l  c h i r   LATIN   cogitare@ P O R T U G U  S   opinar@ E S P A  O L   
significar   LATINA   censere   DUTCH   vinden   	ESPERANTO   pensi   DUTCH   	zinnen op@ F R A N  A I S   songer   DUTCH   	mikken op   
PORTUGUESE   opinar   DUTCH   achten   DEUTSCH   die Absicht haben   
NEDERLANDS   bedoelen   DUTCH   zinnen@ F R A N  A I S   penser@ F R A N  A I S   	envisager   SVENSKA   tycka   SUOMI   arvella   	ICELANDIC   hugsa   SUOMI   ajatella   SWEDISH@
 t  n k a   DUTCH   beogen   DANSK   sigte   DEUTSCH   meinen   DEUTSCH   vorhaben   POLISH@ m y[ l e   SVENSKA   anse   DUTCH   voornemens zijn@ P O R T U G U  S   refletir   DANSK   synes   DUTCH   bedoelen@ F R A N  A I S@  t r e   d ' a v i s   
PORTUGUESE   refletir   DUTCH   rooien   DEUTSCH@ b e d  n k e n   LATINA   opinari   DEUTSCH   zielen   LATINA   cogitare   DEUTSCH   absehen@ E S P A  O L   pensar@ P O R T U G U  S   	pretender@ E S P A  O L   querer decir@ F R A N  A I S   parler   
NEDERLANDS   zich voorstellen   
NEDERLANDS@ v a n   p l a n   z3 n   
PORTUGUESE   	pretender   DEUTSCH   
trachten nach   DUTCH   ten doel hebben@ P O R T U G U  S   ter em vista@ F R A N  A I S   viser@ F R A N  A I S   vouloir dire   LATIN   reri   
PORTUGUESE   ter em vista   ITALIANO   significare   DEUTSCH@ e r w  g e n   TURKISH@ d _  n m e k   	ESPERANTO   	voli diri   DUTCH   van mening zijn   
NEDERLANDS@ v o o r n e m e n s   z3 n   	ICELANDIC@  t l a   DUTCH   zich voorstellen   DUTCH   geloven   DUTCH   mikken   	AFRIKAANS   bedoel   	HUNGARIAN   gondol@ P O R T U G U  S@ s e r   d e   o p i n i  o   CHINESE@
 x i n g   DEUTSCH   bedenken   
PORTUGUESE@ s e r   d e   o p i n i  o   DEUTSCH@ s i c h    b e r l e g e n@ F R A N  A I S@ m  d i t e r   RUSSIAN@?>4C<0BL   FAROESE   hugsa@ P O R T U G U  S   intentar   THAI@4   ITALIANO   avere intenzione di   DUTCH   wikken   
NEDERLANDS   
voorhebben   FAROESE@ h a v a      h y g g j u   DEUTSCH@ d a f  r   h a l t e n   ITALIANO   pensare   
PORTUGUESE   intentar   POLISH@ z a m i e r z a   LATIN   censere@ P O R T U G U  S   	tencionar   DANSK   mene   DUTCH   nadenken   
PORTUGUESE   	tencionar   DEUTSCH   denken   LATINA   reri   NORSK   tenke@ F R A N  A I S   croire   SWEDISH   mena@ P O R T U G U  S   julgar@ F R A N  A I S   avoir pour but   
NEDERLANDS   denken   DEUTSCH   	bezwecken@ P O R T U G U  S   pensar   DUTCH   bedenken   SUOMI   aikoa   DUTCH   
voorhebben   NORSK   synes   
PORTUGUESE   julgar   
PORTUGUESE   pensar@ F R A N  A I S@ s e   p r  n o m m e r@ F R A N  A I S   vouloir@ P O R T U G U  S   achar   RUSSIAN@4C<0BL   DEUTSCH   erachten@ P O R T U G U  S   apontar   DANSK@
 t  n k e   DEUTSCH   
beabsichtigen   	AFRIKAANS   dink   
LEX-FILENO      SENSE-CATEGORY   VERB.COGNITION   OBJ-NAME   VERB.COGNITION   mean   intend   think   EDITSB@ E S P A  O L   ap@media.mit.edu   2001-04-24T19:30:47GMT   ITALIANO   ap@media.mit.edu   2001-04-24T19:30:47GMT   ap@media.mit.edu   2001-04-24T19:30:47GMT@ E S P A  O L   ap@media.mit.edu   2001-04-24T19:30:56GMT   ITALIANO   ap@media.mit.edu   2001-04-24T19:30:56GMT   ap@media.mit.edu   2001-04-24T19:30:56GMT@ E S P A  O L   ap@media.mit.edu   2001-04-24T19:31:05GMT   ITALIANO   ap@media.mit.edu   2001-04-24T19:31:05GMT   ap@media.mit.edu   2001-04-24T19:31:05GMT@ E S P A  O L   ap@media.mit.edu   2001-04-24T19:32:29GMT   ITALIANO   ap@media.mit.edu   2001-04-24T19:32:29GMT   ap@media.mit.edu   2001-04-24T19:32:29GMT   	ESPERANTOB   intenci   pensi   	voli diriB(   SYNSET-CODE   VERB Hb       SOURCE          	WORD-DATA	      aspire       aim       	shoot_for       	SYNSET-ID Hb   N-WORDS      
SENSE-KEYSB	   shoot_for%2:49:00::   
aim%2:49:00::   aspire%2:49:00::   
aim%2:31:00::   
aim%2:31:00::         aspire%2:31:00::   aspire%2:31:00::         shoot_for%2:31:00::   shoot_for%2:31:00::          HYPERNYM     /   HYPONYM     4   VERB-FRAMES         GLOSS   &have an ambitious plan or a lofty goal   PART-OF-SPEECH   VERB   WORDSB   aspire   aim   	shoot for   ROGET-CLUSTERSB          4     4     #[     4   
PORTUGUESEB   apontar   aspirar   desejar vivamente   	pretender   ter em vista   visar a   TRANSLATIONSB(@ F R A N  A I S   aspirer   	HUNGARIAN@ a s p i r  l   
NEDERLANDS@ a m b i  r e n   ITALIANO   
progettare   DEUTSCH   sich bewerben um   DUTCH   najagen@ P O R T U G U  S   aspirar   DUTCH@ a m b i  r e n   
PORTUGUESE   aspirar@ P O R T U G U  S   desejar vivamente   	AFRIKAANS   aansoek doen   ITALIANO   alludere   
PORTUGUESE   desejar vivamente   
NEDERLANDS   streven naar@ E S P A  O L   	apuntar a@ P O R T U G U  S   	pretender   
PORTUGUESE   	pretender   DEUTSCH   
trachten nach   DUTCH   streven naar   	ESPERANTO   aspiri@ P O R T U G U  S   planejar@ P O R T U G U  S   intencionar@ E S P A  O L   	proyectar   
NEDERLANDS   	nastreven   SUOMI@ p y r k i    DEUTSCH   sich bestreben   ITALIANO   mirare   	HUNGARIAN@ s  v  r o g   
NEDERLANDS   dingen naar@ P O R T U G U  S   ansiar@ F R A N  A I S   pointer@ P O R T U G U  S   	objetivar   DUTCH   	nastreven   	AFRIKAANS   ambieer   FAROESE@ t r  a   DUTCH   dingen naar   FAROESE   stremba   DEUTSCH   	erstreben@ P O R T U G U  S   almejar   
NEDERLANDS   najagen   
LEX-FILENO      SENSE-CATEGORY   VERB.COGNITION   OBJ-NAME   VERB.COGNITION   aim   	shoot for   aspire   EDITSB
@ E S P A  O L   jackie@media.mit.edu   2000-09-01T16:59:50GMT@ F R A N  A I S   jackie@media.mit.edu   2000-09-01T16:59:50GMT   ITALIANO   jackie@media.mit.edu   2000-09-01T16:59:50GMT@ P O R T U G U  S   jackie@media.mit.edu   2000-09-01T16:59:50GMT   jackie@media.mit.edu   2000-09-01T16:59:50GMT@ E S P A  O L   jackie@media.mit.edu   2000-09-01T17:00:19GMT@ F R A N  A I S   jackie@media.mit.edu   2000-09-01T17:00:19GMT   ITALIANO   jackie@media.mit.edu   2000-09-01T17:00:19GMT@ P O R T U G U  S   jackie@media.mit.edu   2000-09-01T17:00:19GMT   jackie@media.mit.edu   2000-09-01T17:00:19GMT   	ESPERANTO   aspiriB&   SYNSET-CODE   VERB Ib       SOURCE          	WORD-DATA	      hope       go_for       	SYNSET-ID Ib   N-WORDS      
SENSE-KEYSB   hope%2:49:00::   go_for%2:49:00::   go_for%2:31:00::   go_for%2:31:00::         hope%2:31:00::   hope%2:31:00::         HYPERNYM     /   VERB-FRAMES         GLOSS   cintend with some possibility of fulfilment; "I hope to have finished this work by tomorrow evening"   PART-OF-SPEECH   VERB   WORDSB   hope   go for   ROGET-CLUSTERSB     2     4	     2     4
     4
     4     4   CYC-CONCEPTS     u   
PORTUGUESEB@ e s p e r a n  a@ t e r   e s p e r a n  a   TRANSLATIONSB%   DEUTSCH   gelten@ F R A N  A I S   	souhaiter   ITALIANO   
essere valido@ E S P A  O L   esperar   DANSK@ h  b e   DUTCH   hopen@ F R A N  A I S@  t r e   c o n s i d  r @ F R A N  A I S@ e s p  r e r   LATINA   sperare   ROMANIAN   spera   SVENSKA   hoppas   	AFRIKAANS   hoop   FAROESE@ v  n a   	HUNGARIAN@ r e m  l n i@ E S P A  O L@ s e r   v  l i d o   TURKISH   ummak   CHINESE@ x+ w  n g   ITALIANO   sperare   NORSK@ h  p e   	ESPERANTO   esperi   POLISH@ m i e   n a d z i e j   SWEDISH   hoppas@ F R A N  A I S   se promettre   RUSSIAN@=045OBLAO@ P O R T U G U  S   esperar@ E S P A  O L   ser considerado como   LAO@   	NORWEGIAN@ h  p e@ P O R T U G U  S@ t e r   e s p e r a n  a   SUOMI   toivoa   
PORTUGUESE@ t e r   e s p e r a n  a   DEUTSCH   hoffen   ITALIANO   valere   LATIN   sperare@ F R A N  A I S   valoir   	ICELANDIC   vona   
NEDERLANDS   hopen   
LEX-FILENO      SENSE-CATEGORY   VERB.COGNITION   OBJ-NAME   VERB.COGNITION   hope   go for   	ESPERANTO   esperiB&   SYNSET-CODE   VERB BC       SOURCE          	WORD-DATA	      see      	SYNSET-ID BC   N-WORDS      HYPERNYM     /   VERB-FRAMES          GLOSS   9deliberate or decide; "See whether you can come tomorrow"   PART-OF-SPEECH   VERB   WORDS   see   ROGET-CLUSTERSB          %          ,$          ,%        
PORTUGUESEB   cuidar   preocupar-se   
presenciar@ p  r   e m   v i s t a   tutelar   ver   zelar   TRANSLATIONSB[   NORSK   se   FAROESE@ s  g g j a@ P O R T U G U  S   	verificar   
PORTUGUESE   achar   
PORTUGUESE   	encontrar   DUTCH   bezorgd zijn   
PORTUGUESE   zelar   SUOMI@
 n  h d    LAO@@ E S P A  O L   ver   DUTCH   tegemoet treden   	HUNGARIAN@ l  t   THAI@@+G   
PORTUGUESE   fazer visita   DEUTSCH   Sorge tragen@ E S P A  O L   cuidar   DEUTSCH   begegnen   SUOMI   	huolehtia   DANSK@ m  d e   SUOMI@ k  y d    v i e r a i s s a   NORSK   treffe@ F R A N  A I S   	s'occuper   RUSSIAN@C2845BL   NORSK@ m  t e   DEUTSCH   sehen   
PORTUGUESE   visitar   DANSK@ b e s  g e   	NORWEGIAN   se   LATIN   videre   	SLOVENIAN@ v i d i ee   ROMANIAN   vedea   TAGALOG@ d u m  l a w   
PORTUGUESE   
presenciar   
NEDERLANDS   zien@ F R A N  A I S@ v e i l l e r      DUTCH   treffen   SVENSKA@ m  t a@ F R A N  A I S   voir   FAROESE   hyggja eftir@ F R A N  A I S   
avoir soin de   DEUTSCH   sorgen   DUTCH   zich bekommeren   SWEDISH@
 s k  d a   DANSK   se   DUTCH   
tegenkomen   DEUTSCH   besuchen   DUTCH   zien   RUSSIAN@2845BL   DUTCH   bezoeken   SWEDISH@ s k  n j a@ P O R T U G U  S   ver   
PORTUGUESE   
fazer visitas@ E S P A  O L   hallar   TAGALOG@ m  k i t a@ E S P A  O L   	encontrar   DUTCH   zorg dragen   
PORTUGUESE   ver   NORSK@ b e s  k e@ F R A N  A I S   aviser   
PORTUGUESE   ir ver   DEUTSCH   	antreffen   DUTCH   zorgen   	AFRIKAANS   besien   LATINA   videre   
PORTUGUESE   cuidar@ F R A N  A I S   
rencontrer   DEUTSCH   ansehen   FRISIAN   sjen@ E S P A  O L   visitar   TURKISH@ g  r m e k   
PORTUGUESE   tutelar   
PORTUGUESE@ p  r   e m   v i s t a   DUTCH   opzoeken@ F R A N  A I S   regarder   
PORTUGUESE   preocupar-se   DEUTSCH   einen Besuch abstatten   DUTCH   
aantreffen@ F R A N  A I S   
se soucier de   DUTCH   	ontmoeten   	AFRIKAANS   sien   DANSK@ t r  f f e   SUOMI   tavata   TAGALOG@ t i g n  n   SWEDISH   se@ F R A N  A I S   visiter   	ESPERANTO   vidi   DUTCH   afgaan   ITALIANO   	esaminare   POLISH@ w i d z i e   
PORTUGUESE   mirar   SWEDISH   titta   
SENSE-KEYSB   
see%2:31:08::   
see%2:31:08::         
LEX-FILENO      SENSE-CATEGORY   VERB.COGNITION   OBJ-NAME   VERB.COGNITION   see   EDITSB@ E S P A  O L   jackie@media.mit.edu   2001-02-13T20:09:26GMT@ F R A N  A I S   jackie@media.mit.edu   2001-02-13T20:09:26GMT   ITALIANO   jackie@media.mit.edu   2001-02-13T20:09:26GMT@ P O R T U G U  S   jackie@media.mit.edu   2001-02-13T20:09:26GMT   jackie@media.mit.edu   2001-02-13T20:09:26GMT@ E S P A  O L   jackie@media.mit.edu   2001-02-13T20:10:07GMT@ F R A N  A I S   jackie@media.mit.edu   2001-02-13T20:10:07GMT   ITALIANO   jackie@media.mit.edu   2001-02-13T20:10:07GMT@ P O R T U G U  S   jackie@media.mit.edu   2001-02-13T20:10:07GMT   jackie@media.mit.edu   2001-02-13T20:10:07GMT@ E S P A  O L   jackie@media.mit.edu   2001-02-13T20:10:42GMT@ F R A N  A I S   jackie@media.mit.edu   2001-02-13T20:10:42GMT   ITALIANO   jackie@media.mit.edu   2001-02-13T20:10:42GMT@ P O R T U G U  S   jackie@media.mit.edu   2001-02-13T20:10:42GMT   jackie@media.mit.edu   2001-02-13T20:10:42GMT@ E S P A  O L   jackie@media.mit.edu   2001-02-13T20:10:53GMT@ F R A N  A I S   jackie@media.mit.edu   2001-02-13T20:10:53GMT   ITALIANO   jackie@media.mit.edu   2001-02-13T20:10:53GMT@ P O R T U G U  S   jackie@media.mit.edu   2001-02-13T20:10:53GMT   jackie@media.mit.edu   2001-02-13T20:10:53GMT   	ESPERANTO   vidiB   SYNSET-CODE   VERB Cm       SOURCE          	WORD-DATA	      premeditate       	SYNSET-ID Cm   N-WORDS      HYPERNYM     /   VERB-FRAMES         GLOSS   *consider beforehand; "premeditated murder"   PART-OF-SPEECH   VERB   WORDS   premeditate   ROGET-CLUSTERSB     %     3   
SENSE-KEYSB   premeditate%2:31:00::   premeditate%2:31:00::          
LEX-FILENO      SENSE-CATEGORY   VERB.COGNITION   OBJ-NAME   VERB.COGNITION   premeditateB$   SYNSET-CODE   VERB C       SOURCE          	WORD-DATA	      take_counsel       consult       	SYNSET-ID C   N-WORDS      
SENSE-KEYSB   consult%2:49:00::   take_counsel%2:49:00::   consult%2:31:00::   consult%2:31:00::          take_counsel%2:31:00::   take_counsel%2:31:00::         HYPERNYM     /   VERB-FRAMES         GLOSS   Aconsult (archaic); "The student should take counsel with himself"   PART-OF-SPEECH   VERB   WORDSB   take counsel   consult   ROGET-CLUSTERSB     3     3     3   
PORTUGUESE   	consultar   TRANSLATIONSB@ F R A N  A I S@ s o u m e t t r e   @ F R A N  A I S@ f a i r e   a p p e l      DEUTSCH   konsultieren   DEUTSCH   befragen   
NEDERLANDS   consulteren   SWEDISH   
konsultera@ F R A N  A I S   utiliser les services de   	AFRIKAANS   	raadpleeg   DEUTSCH   nachschlagen@ F R A N  A I S   	consulter   DUTCH   consulteren@ P O R T U G U  S   	consultar   
PORTUGUESE   	consultar   
NEDERLANDS   
raadplegen   DEUTSCH   zu Rate ziehen   DUTCH   
raadplegen   FRISIAN   konsultearje   SWEDISH@ r  d f r  g a   	ESPERANTO   konsulti   
LEX-FILENO      SENSE-CATEGORY   VERB.COGNITION   OBJ-NAME   VERB.COGNITION   consult   take counsel   	ESPERANTO   konsultiB"   SYNSET-CODE   VERB a       SOURCE          	WORD-DATA	      debate       	SYNSET-ID a   N-WORDS      HYPERNYM     /   VERB-FRAMES         GLOSS   Rargue with one another; "We debated the question of abortion"; "John debated Mary"   PART-OF-SPEECH   VERB   WORDS   debate   ROGET-CLUSTERSB     (     3     3     3     /   TRANSLATIONSB   ITALIANO   	discutere@ E S P A  O L   debatir   ITALIANO   	dibattere   DEUTSCH   debattieren   
SENSE-KEYSB   debate%2:31:00::   debate%2:31:00::         
LEX-FILENO      SENSE-CATEGORY   VERB.COGNITION   OBJ-NAME   VERB.COGNITION   debate   EDITSB@ E S P A  O L   ap@media.mit.edu   2000-08-23T13:16:38GMT   ITALIANO   ap@media.mit.edu   2000-08-23T13:16:38GMT   ap@media.mit.edu   2000-08-23T13:16:38GMT   ITALIANO   ap@media.mit.edu   2000-08-23T13:18:55GMT   ap@media.mit.edu   2000-08-23T13:18:55GMT   ap@media.mit.edu   2000-08-23T13:26:42GMT   ap@media.mit.edu   2000-08-23T13:28:50GMTB   SYNSET-CODE   VERB 	       SOURCE          	WORD-DATA	      scruple       	SYNSET-ID 	   N-WORDS      HYPERNYM     /   VERB-FRAMES         GLOSS   have doubts about   PART-OF-SPEECH   VERB   WORDS   scruple   ROGET-CLUSTERSB	          8     8     ?     ?     ?           ?     :   
SENSE-KEYSB   scruple%2:32:00::   scruple%2:32:00::          
LEX-FILENO       SENSE-CATEGORY   VERB.COMMUNICATION   OBJ-NAME   VERB.COMMUNICATION   scrupleB&   SYNSET-CODE   VERB         SOURCE          	WORD-DATA	      think      suppose       imagine      reckon      guess       	SYNSET-ID     N-WORDS      
SENSE-KEYSB   suppose%2:49:00::   guess%2:49:00::   imagine%2:49:01::   reckon%2:49:02::   think%2:49:03::   guess%2:31:00::   guess%2:31:00::      *   imagine%2:31:01::   imagine%2:31:01::         reckon%2:31:02::   reckon%2:31:02::         suppose%2:31:00::   suppose%2:31:00::         think%2:31:03::   think%2:31:03::         HYPERNYM     
   HYPONYMB     /     /   VERB-FRAMES         GLOSS   expect, believe, or suppose; "I imagine she earned a lot of money with her new novel"; "I thought to find her in a bad state"; "he didn't think to find her in the kitchen"   PART-OF-SPEECH   VERB   WORDSB   reckon   think   guess   imagine   suppose   ROGET-CLUSTERSB          /     
      /     /          /     /   
PORTUGUESEB
