# 25 "cil/src/logic/logic_lexer.mll"
 

  open Logic_parser
  open Lexing
  open Logic_ptree

  type state = Normal | Test

  let state_stack = Stack.create ()

  let () = Stack.push Normal state_stack

  let get_state () = try Stack.top state_stack with Stack.Empty -> Normal

  let pop_state () = try ignore (Stack.pop state_stack) with Stack.Empty -> ()

  exception Error of (int * int) * string

  let loc lexbuf = (lexeme_start lexbuf, lexeme_end lexbuf)

  let lex_error lexbuf s =
    raise (Error (loc lexbuf, "lexical error, " ^ s))

  let find_utf8 =
    let h = Hashtbl.create 97 in
    List.iter (fun (i,t) -> Hashtbl.add h i t)
      [ Utf8_logic.forall, FORALL;
        Utf8_logic.exists, EXISTS;
        Utf8_logic.eq, EQ;
        Utf8_logic.neq, NE;
        Utf8_logic.le, LE;
        Utf8_logic.ge, GE;
        Utf8_logic.implies,IMPLIES;
        Utf8_logic.iff, IFF;
        Utf8_logic.conj, AND;
        Utf8_logic.disj, OR;
        Utf8_logic.neg, NOT;
        Utf8_logic.x_or, HATHAT;
        Utf8_logic.minus, MINUS;
        Utf8_logic.boolean, BOOLEAN;
        Utf8_logic.integer, INTEGER;
        Utf8_logic.real, REAL
      ];

    fun s -> try Hashtbl.find h s
    with Not_found -> IDENTIFIER s

  let identifier, is_acsl_keyword =
    let all_kw = Hashtbl.create 37 in
    let c_kw = Hashtbl.create 37 in
    let type_kw = Hashtbl.create 3 in
    List.iter
      (fun (i,t,flag) ->
         Hashtbl.add all_kw i t;
         if flag then Hashtbl.add c_kw i t
      )
      [
        "assert", ASSERT, false;
        "assigns", ASSIGNS, false;
        "assumes", ASSUMES, false;
        "at", EXT_AT, false;(* ACSL extension for external spec file *)
        "axiom", AXIOM, false;
        "axiomatic", AXIOMATIC, false;
        "behavior", BEHAVIOR, false;
        "behaviors", BEHAVIORS, false;
        "breaks", BREAKS, false;
	"case", CASE, true;
        "char", CHAR, true;
        "complete", COMPLETE, false;
        "const", CONST, true;
        "continues", CONTINUES, false;
        "contract", CONTRACT, false;(* ACSL extension for external spec file *)
        "decreases", DECREASES, false;
        "disjoint", DISJOINT, false;
        "double", DOUBLE, true;
        "else", ELSE, true;
        "ensures", ENSURES, false ;
        "enum", ENUM, true;
        "exits", EXITS, false;
        "function", FUNCTION, false;(* ACSL extension for external spec file *)
        "float", FLOAT, true;
        "for", FOR, true;
        "global",    GLOBAL, false;
        "if", IF, true;
	"impact", IMPACT, false;
	"inductive", INDUCTIVE, false;
	"include", INCLUDE, false;(* ACSL extension for external spec file *)
        "int", INT, true;
        "invariant", INVARIANT, false;
        "label", LABEL, false;
        "lemma", LEMMA, false;
        "let", EXT_LET, false;(* ACSL extension for external spec file *)
        "logic", LOGIC, false;
        "long", LONG, true;
        "loop", LOOP, false;
        "modelfield", MODEL, false;(* ACSL extension for model fields *)
        "module", MODULE, false;(* ACSL extension for external spec file *)
        "pragma", PRAGMA, false;
        "predicate", PREDICATE, false;
        "reads", READS, false;
        "requires", REQUIRES, false;
        "returns", RETURNS, false;
        "short", SHORT, true;
        "signed", SIGNED, true;
        "sizeof", SIZEOF, true;
        "slice", SLICE, false;
        "struct", STRUCT, true;
        "terminates", TERMINATES, false;
        "type", TYPE, false;
        "union", UNION, true;
        "unsigned", UNSIGNED, true;
        "variant", VARIANT, false;
        "void", VOID, true;
        "volatile", VOLATILE, true;
        "writes", WRITES, false;
      ];
    List.iter (fun (x, y) -> Hashtbl.add type_kw x y)
      ["integer", INTEGER; "real", REAL; "boolean", BOOLEAN; ];
    (fun s ->
      try
        Hashtbl.find (if Logic_utils.is_kw_c_mode () then c_kw else all_kw) s
      with Not_found ->
        if Logic_env.typename_status s then TYPENAME s
        else
          (try
             Hashtbl.find type_kw s
           with Not_found ->
             if Logic_utils.is_rt_type_mode () then TYPENAME s
             else IDENTIFIER s)),
    (fun s -> Hashtbl.mem all_kw s || Hashtbl.mem type_kw s)

  let bs_identifier =
    let h = Hashtbl.create 97 in
    List.iter (fun (i,t) -> Hashtbl.add h i t)
      [
        "\\at", AT;
        "\\base_addr", BASE_ADDR;
        "\\block_length", BLOCK_LENGTH;
        "\\empty", EMPTY;
        "\\exists", EXISTS;
        "\\false", FALSE;
        "\\forall", FORALL;
        "\\fresh", FRESH;
        "\\from", FROM;
        "\\initialized", INITIALIZED;
        "\\inter", INTER;
        "\\lambda", LAMBDA;
        "\\let", LET;
        "\\nothing", NOTHING;
        "\\null", NULL;
        "\\old", OLD;
        "\\result", RESULT;
        "\\separated", SEPARATED;
        "\\true", TRUE;
        "\\type", BSTYPE;
        "\\typeof", TYPEOF;
        "\\union", BSUNION;
        "\\valid", VALID;
        "\\valid_index", VALID_INDEX;
        "\\valid_range", VALID_RANGE;
        "\\with", WITH;
      ];
    fun lexbuf ->
      let s = lexeme lexbuf in
      try Hashtbl.find h s with Not_found -> IDENTIFIER s


  let int_of_digit chr =
    match chr with
        '0'..'9' -> (Char.code chr) - (Char.code '0')
      | 'a'..'f' -> (Char.code chr) - (Char.code 'a') + 10
      | 'A'..'F' -> (Char.code chr) - (Char.code 'A') + 10
      | _ -> assert false

  (* Update lexer buffer. *)
  let update_newline_loc lexbuf =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
      { pos with
	Lexing.pos_lnum = pos.Lexing.pos_lnum + 1;
	Lexing.pos_bol = pos.Lexing.pos_cnum;
      }

  (* Update lexer buffer. *)
  let update_line_loc lexbuf line absolute chars =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
      { pos with
	Lexing.pos_lnum = if absolute then line else pos.Lexing.pos_lnum + line;
	Lexing.pos_bol = pos.Lexing.pos_cnum - chars;
      }

  (* Update lexer buffer. *)
  let update_file_loc lexbuf file =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
      { pos with
	Lexing.pos_fname = file;
      }

  (* Update lexer buffer. *)
  let update_bol_loc lexbuf bol =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
      { pos with
	Lexing.pos_bol = bol;
      }

# 211 "cil/src/logic/logic_lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\194\255\195\255\000\000\201\255\202\255\203\255\204\255\
    \205\255\206\255\196\000\214\255\216\255\165\000\221\255\222\255\
    \202\000\229\255\230\255\232\255\233\255\200\000\138\000\225\000\
    \220\000\223\000\205\000\241\255\001\001\223\000\004\001\252\000\
    \034\001\107\001\196\001\254\001\238\000\254\255\028\001\089\001\
    \252\255\253\255\073\002\186\002\125\002\090\001\126\002\245\255\
    \242\255\091\001\152\002\075\001\210\000\181\002\251\002\018\003\
    \081\003\219\002\151\003\246\000\055\003\189\003\119\003\222\003\
    \191\002\244\255\255\003\009\004\028\004\038\004\072\004\060\004\
    \105\004\137\004\250\000\147\004\176\004\243\255\191\004\208\004\
    \237\004\247\004\218\004\020\005\030\005\241\000\054\005\084\005\
    \094\005\064\005\224\255\226\000\240\255\198\255\200\255\244\000\
    \229\000\230\000\239\255\232\000\237\255\231\255\248\000\238\255\
    \236\255\235\255\207\255\199\255\227\255\219\255\197\255\210\255\
    \231\000\251\255\252\255\124\005\253\255\095\001\197\005\220\005\
    \188\001\254\255\026\006\251\255\206\000\252\005\254\255\255\255\
    \202\000\212\000\252\255\025\003\252\255\247\002\254\255\255\255\
    \253\255\044\003\131\001\253\255\254\255\255\255";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\059\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\043\000\255\255\255\255\037\000\255\255\255\255\
    \027\000\255\255\255\255\255\255\255\255\021\000\038\000\035\000\
    \032\000\042\000\047\000\255\255\061\000\029\000\061\000\008\000\
    \008\000\005\000\005\000\061\000\040\000\255\255\000\000\255\255\
    \255\255\255\255\004\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\011\000\255\255\009\000\007\000\255\255\255\255\
    \006\000\255\255\255\255\006\000\255\255\255\255\255\255\011\000\
    \255\255\255\255\011\000\255\255\011\000\255\255\255\255\255\255\
    \011\000\255\255\007\000\011\000\255\255\255\255\255\255\011\000\
    \255\255\011\000\255\255\011\000\255\255\030\000\011\000\255\255\
    \011\000\255\255\255\255\046\000\255\255\255\255\255\255\255\255\
    \044\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\004\000\255\255\001\000\255\255\000\000\
    \001\000\255\255\255\255\255\255\004\000\002\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\003\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_default = 
   "\003\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\255\255\000\000\000\000\
    \255\255\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\043\000\255\255\046\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\039\000\
    \000\000\000\000\255\255\043\000\046\000\046\000\046\000\000\000\
    \000\000\043\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\000\000\000\000\000\000\255\255\
    \255\255\255\255\000\000\255\255\000\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \113\000\000\000\000\000\255\255\000\000\255\255\255\255\255\255\
    \255\255\000\000\123\000\000\000\255\255\255\255\000\000\000\000\
    \255\255\255\255\000\000\132\000\000\000\137\000\000\000\000\000\
    \000\000\137\000\139\000\000\000\000\000\000\000";
  Lexing.lex_trans = 
   "\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\038\000\037\000\001\000\038\000\038\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \038\000\021\000\028\000\027\000\020\000\011\000\023\000\030\000\
    \009\000\008\000\014\000\015\000\019\000\024\000\029\000\036\000\
    \032\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\016\000\017\000\025\000\026\000\010\000\018\000\
    \038\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\034\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\005\000\035\000\004\000\013\000\033\000\
    \001\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\007\000\022\000\006\000\012\000\001\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\001\000\
    \002\000\111\000\110\000\109\000\108\000\106\000\105\000\104\000\
    \107\000\102\000\091\000\255\255\095\000\085\000\255\255\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\094\000\101\000\093\000\096\000\039\000\052\000\090\000\
    \092\000\099\000\097\000\048\000\098\000\038\000\100\000\052\000\
    \038\000\038\000\050\000\255\255\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\103\000\128\000\
    \129\000\130\000\000\000\000\000\038\000\000\000\052\000\000\000\
    \000\000\051\000\059\000\115\000\000\000\000\000\074\000\052\000\
    \052\000\000\000\000\000\059\000\000\000\000\000\000\000\074\000\
    \050\000\052\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\038\000\049\000\000\000\000\000\
    \045\000\051\000\059\000\041\000\255\255\255\255\074\000\051\000\
    \052\000\000\000\000\000\059\000\000\000\000\000\052\000\074\000\
    \000\000\052\000\000\000\000\000\000\000\000\000\076\000\052\000\
    \076\000\000\000\054\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\000\000\000\000\051\000\
    \000\000\000\000\000\000\000\000\000\000\141\000\052\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\052\000\
    \000\000\000\000\054\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\000\000\000\000\
    \000\000\000\000\033\000\000\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\043\000\114\000\
    \000\000\000\000\000\000\044\000\121\000\121\000\121\000\121\000\
    \121\000\121\000\121\000\121\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\255\255\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\000\000\
    \000\000\000\000\000\000\033\000\000\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\040\000\255\255\255\255\000\000\042\000\000\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\140\000\000\000\000\000\000\000\255\255\
    \255\255\000\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\255\255\047\000\000\000\000\000\
    \042\000\000\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\255\255\000\000\077\000\000\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\045\000\045\000\000\000\048\000\078\000\065\000\000\000\
    \000\000\000\000\000\000\050\000\065\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\000\000\051\000\000\000\000\000\078\000\065\000\000\000\
    \255\255\074\000\000\000\255\255\065\000\000\000\069\000\000\000\
    \069\000\000\000\074\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\049\000\000\000\
    \000\000\136\000\051\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\074\000\134\000\135\000\000\000\000\000\134\000\000\000\
    \000\000\055\000\074\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\255\255\000\000\000\000\
    \255\255\134\000\000\000\133\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\000\000\000\000\136\000\000\000\
    \000\000\000\000\000\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\000\000\000\000\000\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\067\000\000\000\067\000\000\000\000\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\000\000\000\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\000\000\000\000\000\000\000\000\255\255\255\255\058\000\
    \000\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \000\000\000\000\000\000\000\000\000\000\059\000\000\000\000\000\
    \000\000\057\000\064\000\000\000\064\000\000\000\059\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\000\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \000\000\000\000\255\255\000\000\000\000\059\000\000\000\000\000\
    \000\000\057\000\000\000\000\000\000\000\000\000\059\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\060\000\
    \000\000\000\000\000\000\000\000\000\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\255\255\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\000\000\000\000\000\000\060\000\
    \000\000\000\000\000\000\000\000\000\000\062\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\065\000\000\000\000\000\000\000\
    \000\000\000\000\065\000\000\000\255\255\062\000\000\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\000\000\065\000\065\000\000\000\000\000\
    \000\000\000\000\065\000\065\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \000\000\000\000\065\000\000\000\000\000\065\000\000\000\073\000\
    \065\000\073\000\000\000\065\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\000\000\000\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\065\000\000\000\000\000\000\000\000\000\000\000\
    \065\000\070\000\070\000\070\000\070\000\070\000\070\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \071\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\070\000\070\000\070\000\070\000\070\000\070\000\065\000\
    \000\000\000\000\000\000\000\000\000\000\065\000\000\000\000\000\
    \071\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\000\000\000\000\065\000\
    \000\000\000\000\000\000\000\000\000\000\065\000\000\000\000\000\
    \000\000\065\000\000\000\000\000\000\000\000\000\000\000\065\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\084\000\000\000\084\000\000\000\000\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\065\000\000\000\000\000\000\000\000\000\000\000\065\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\000\000\080\000\065\000\000\000\
    \082\000\000\000\082\000\000\000\065\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\000\000\000\000\000\000\000\000\080\000\065\000\000\000\
    \000\000\000\000\000\000\000\000\065\000\065\000\000\000\000\000\
    \000\000\000\000\000\000\065\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \000\000\000\000\065\000\000\000\000\000\065\000\000\000\000\000\
    \065\000\000\000\000\000\065\000\000\000\000\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\065\000\087\000\065\000\000\000\000\000\089\000\
    \065\000\089\000\065\000\000\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \000\000\000\000\000\000\087\000\065\000\000\000\116\000\000\000\
    \000\000\000\000\065\000\116\000\065\000\000\000\000\000\000\000\
    \000\000\000\000\065\000\000\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\116\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\065\000\000\000\000\000\000\000\
    \000\000\000\000\065\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\118\000\000\000\000\000\000\000\
    \116\000\000\000\000\000\000\000\000\000\116\000\116\000\000\000\
    \000\000\000\000\116\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\116\000\000\000\000\000\000\000\116\000\000\000\
    \116\000\000\000\116\000\000\000\118\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\126\000\127\000\000\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\000\000\000\000\
    \000\000\000\000\126\000\000\000\000\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\124\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255";
  Lexing.lex_check = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\010\000\010\000\013\000\016\000\021\000\022\000\023\000\
    \016\000\024\000\026\000\028\000\025\000\029\000\030\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\025\000\024\000\025\000\025\000\036\000\052\000\085\000\
    \091\000\095\000\096\000\028\000\097\000\038\000\099\000\052\000\
    \038\000\038\000\031\000\030\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\102\000\124\000\
    \128\000\129\000\255\255\255\255\038\000\255\255\052\000\255\255\
    \255\255\031\000\059\000\112\000\255\255\255\255\074\000\052\000\
    \031\000\255\255\255\255\059\000\255\255\255\255\255\255\074\000\
    \032\000\031\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\038\000\028\000\255\255\255\255\
    \030\000\031\000\059\000\039\000\045\000\049\000\074\000\032\000\
    \031\000\255\255\255\255\059\000\255\255\255\255\032\000\074\000\
    \255\255\031\000\255\255\255\255\255\255\255\255\051\000\032\000\
    \051\000\255\255\032\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\255\255\255\255\032\000\
    \255\255\255\255\255\255\255\255\255\255\138\000\032\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\032\000\
    \255\255\255\255\032\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\255\255\255\255\
    \255\255\255\255\033\000\255\255\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\034\000\112\000\
    \255\255\255\255\255\255\034\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\255\255\255\255\
    \255\255\028\000\255\255\255\255\030\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\255\255\
    \255\255\255\255\255\255\034\000\255\255\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\039\000\045\000\049\000\255\255\035\000\255\255\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\138\000\255\255\255\255\255\255\044\000\
    \046\000\255\255\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\044\000\046\000\255\255\255\255\
    \042\000\255\255\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\043\000\255\255\050\000\255\255\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\044\000\046\000\255\255\043\000\050\000\050\000\255\255\
    \255\255\255\255\255\255\053\000\050\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\255\255\053\000\255\255\255\255\050\000\050\000\255\255\
    \133\000\053\000\255\255\133\000\050\000\255\255\057\000\255\255\
    \057\000\255\255\053\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\255\255\043\000\255\255\
    \255\255\133\000\053\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\053\000\131\000\131\000\255\255\255\255\131\000\255\255\
    \255\255\054\000\053\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\137\000\255\255\255\255\
    \137\000\131\000\255\255\131\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\255\255\255\255\137\000\255\255\
    \255\255\255\255\255\255\055\000\055\000\055\000\055\000\055\000\
    \055\000\255\255\255\255\255\255\054\000\054\000\054\000\054\000\
    \054\000\054\000\060\000\255\255\060\000\255\255\255\255\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\255\255\255\255\055\000\055\000\055\000\055\000\055\000\
    \055\000\255\255\255\255\255\255\255\255\044\000\046\000\056\000\
    \255\255\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\056\000\056\000\056\000\056\000\056\000\056\000\
    \255\255\255\255\255\255\255\255\255\255\056\000\255\255\255\255\
    \255\255\056\000\062\000\255\255\062\000\255\255\056\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\255\255\056\000\056\000\056\000\056\000\056\000\056\000\
    \255\255\255\255\043\000\255\255\255\255\056\000\255\255\255\255\
    \255\255\056\000\255\255\255\255\255\255\255\255\056\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \058\000\058\000\058\000\058\000\058\000\058\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\058\000\
    \255\255\255\255\255\255\255\255\255\255\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\133\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\255\255\255\255\255\255\058\000\
    \255\255\255\255\255\255\255\255\255\255\061\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \255\255\131\000\255\255\255\255\255\255\255\255\061\000\061\000\
    \061\000\061\000\061\000\061\000\063\000\255\255\255\255\255\255\
    \255\255\255\255\063\000\255\255\137\000\061\000\255\255\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\255\255\063\000\066\000\255\255\255\255\
    \255\255\255\255\063\000\066\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \255\255\255\255\068\000\255\255\255\255\066\000\255\255\071\000\
    \068\000\071\000\255\255\066\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\255\255\255\255\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\068\000\255\255\255\255\255\255\255\255\255\255\
    \068\000\070\000\070\000\070\000\070\000\070\000\070\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \070\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\070\000\070\000\070\000\070\000\070\000\070\000\072\000\
    \255\255\255\255\255\255\255\255\255\255\072\000\255\255\255\255\
    \070\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\255\255\255\255\072\000\
    \255\255\255\255\255\255\255\255\255\255\072\000\255\255\255\255\
    \255\255\075\000\255\255\255\255\255\255\255\255\255\255\075\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\078\000\255\255\078\000\255\255\255\255\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\075\000\255\255\255\255\255\255\255\255\255\255\075\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\255\255\079\000\079\000\255\255\
    \080\000\255\255\080\000\255\255\079\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\255\255\255\255\255\255\255\255\079\000\079\000\255\255\
    \255\255\255\255\255\255\255\255\079\000\081\000\255\255\255\255\
    \255\255\255\255\255\255\081\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \255\255\255\255\083\000\255\255\255\255\081\000\255\255\255\255\
    \083\000\255\255\255\255\081\000\255\255\255\255\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\083\000\086\000\086\000\255\255\255\255\087\000\
    \083\000\087\000\086\000\255\255\087\000\087\000\087\000\087\000\
    \087\000\087\000\087\000\087\000\087\000\087\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \255\255\255\255\255\255\086\000\086\000\255\255\115\000\255\255\
    \255\255\255\255\086\000\115\000\088\000\255\255\255\255\255\255\
    \255\255\255\255\088\000\255\255\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\115\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\088\000\255\255\255\255\255\255\
    \255\255\255\255\088\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\115\000\255\255\255\255\255\255\
    \115\000\255\255\255\255\255\255\255\255\115\000\115\000\255\255\
    \255\255\255\255\115\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\115\000\255\255\255\255\255\255\115\000\255\255\
    \115\000\255\255\115\000\255\255\115\000\118\000\118\000\118\000\
    \118\000\118\000\118\000\118\000\118\000\118\000\118\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\118\000\118\000\
    \118\000\118\000\118\000\118\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\119\000\119\000\119\000\
    \119\000\119\000\119\000\122\000\122\000\255\255\118\000\118\000\
    \118\000\118\000\118\000\118\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\255\255\255\255\
    \255\255\255\255\122\000\255\255\255\255\119\000\119\000\119\000\
    \119\000\119\000\119\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\122\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\122\000";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\001\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\007\000\
    \010\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\001\000\001\000\000\000\000\000\000\000\004\000\
    \004\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\034\000\255\255\255\255\255\255\000\000\
    \034\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255";
  Lexing.lex_code = 
   "\255\001\255\255\002\255\255\000\002\255\000\001\255";
}

let rec token lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 3 (-1) ;   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 256 "cil/src/logic/logic_lexer.mll"
           ( token lexbuf )
# 864 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 257 "cil/src/logic/logic_lexer.mll"
         ( update_newline_loc lexbuf; token lexbuf )
# 869 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 258 "cil/src/logic/logic_lexer.mll"
                      ( update_newline_loc lexbuf; token lexbuf )
# 874 "cil/src/logic/logic_lexer.ml"

  | 3 ->
# 259 "cil/src/logic/logic_lexer.mll"
                     ( token lexbuf )
# 879 "cil/src/logic/logic_lexer.ml"

  | 4 ->
# 261 "cil/src/logic/logic_lexer.mll"
                       ( bs_identifier lexbuf )
# 884 "cil/src/logic/logic_lexer.ml"

  | 5 ->
# 262 "cil/src/logic/logic_lexer.mll"
                        ( let s = lexeme lexbuf in identifier s )
# 889 "cil/src/logic/logic_lexer.ml"

  | 6 ->
# 264 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT (IntConstant (lexeme lexbuf)) )
# 894 "cil/src/logic/logic_lexer.ml"

  | 7 ->
# 265 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT (IntConstant (lexeme lexbuf)) )
# 899 "cil/src/logic/logic_lexer.ml"

  | 8 ->
# 266 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT10 (lexeme lexbuf) )
# 904 "cil/src/logic/logic_lexer.ml"

  | 9 ->
# 267 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT (IntConstant (lexeme lexbuf)) )
# 909 "cil/src/logic/logic_lexer.ml"

  | 10 ->
let
# 268 "cil/src/logic/logic_lexer.mll"
                 prelude
# 915 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(0)
and
# 268 "cil/src/logic/logic_lexer.mll"
                                                                   content
# 920 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) (lexbuf.Lexing.lex_curr_pos + -1) in
# 269 "cil/src/logic/logic_lexer.mll"
      (
        let b = Buffer.create 5 in
        Buffer.add_string b prelude;
        let lbf = Lexing.from_string content in
        CONSTANT (IntConstant (chr b lbf ^ "'"))
      )
# 929 "cil/src/logic/logic_lexer.ml"

  | 11 ->
# 282 "cil/src/logic/logic_lexer.mll"
      ( CONSTANT (FloatConstant (lexeme lexbuf)) )
# 934 "cil/src/logic/logic_lexer.ml"

  | 12 ->
let
# 285 "cil/src/logic/logic_lexer.mll"
            n
# 940 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -2) in
# 285 "cil/src/logic/logic_lexer.mll"
                            ( lexbuf.lex_curr_pos <- lexbuf.lex_curr_pos - 2;
                              CONSTANT (IntConstant n) )
# 945 "cil/src/logic/logic_lexer.ml"

  | 13 ->
let
# 288 "cil/src/logic/logic_lexer.mll"
                prelude
# 951 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(0)
and
# 288 "cil/src/logic/logic_lexer.mll"
                                                                content
# 956 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) (lexbuf.Lexing.lex_curr_pos + -1) in
# 289 "cil/src/logic/logic_lexer.mll"
      ( STRING_LITERAL (prelude.[0] = 'L',content) )
# 960 "cil/src/logic/logic_lexer.ml"

  | 14 ->
# 290 "cil/src/logic/logic_lexer.mll"
                            ( hash lexbuf )
# 965 "cil/src/logic/logic_lexer.ml"

  | 15 ->
# 291 "cil/src/logic/logic_lexer.mll"
                            ( IMPLIES )
# 970 "cil/src/logic/logic_lexer.ml"

  | 16 ->
# 292 "cil/src/logic/logic_lexer.mll"
                            ( IFF )
# 975 "cil/src/logic/logic_lexer.ml"

  | 17 ->
# 293 "cil/src/logic/logic_lexer.mll"
                            ( BIMPLIES )
# 980 "cil/src/logic/logic_lexer.ml"

  | 18 ->
# 294 "cil/src/logic/logic_lexer.mll"
                            ( BIFF )
# 985 "cil/src/logic/logic_lexer.ml"

  | 19 ->
# 295 "cil/src/logic/logic_lexer.mll"
                            ( AND )
# 990 "cil/src/logic/logic_lexer.ml"

  | 20 ->
# 296 "cil/src/logic/logic_lexer.mll"
                            ( OR )
# 995 "cil/src/logic/logic_lexer.ml"

  | 21 ->
# 297 "cil/src/logic/logic_lexer.mll"
                            ( NOT )
# 1000 "cil/src/logic/logic_lexer.ml"

  | 22 ->
# 298 "cil/src/logic/logic_lexer.mll"
                            ( DOLLAR )
# 1005 "cil/src/logic/logic_lexer.ml"

  | 23 ->
# 299 "cil/src/logic/logic_lexer.mll"
                            ( COMMA )
# 1010 "cil/src/logic/logic_lexer.ml"

  | 24 ->
# 300 "cil/src/logic/logic_lexer.mll"
                            ( ARROW )
# 1015 "cil/src/logic/logic_lexer.ml"

  | 25 ->
# 301 "cil/src/logic/logic_lexer.mll"
                            ( Stack.push Test state_stack; QUESTION )
# 1020 "cil/src/logic/logic_lexer.ml"

  | 26 ->
# 302 "cil/src/logic/logic_lexer.mll"
                            ( SEMICOLON )
# 1025 "cil/src/logic/logic_lexer.ml"

  | 27 ->
# 303 "cil/src/logic/logic_lexer.mll"
                            ( match get_state() with
                                  Normal  -> COLON
                                | Test -> pop_state(); COLON2
                            )
# 1033 "cil/src/logic/logic_lexer.ml"

  | 28 ->
# 307 "cil/src/logic/logic_lexer.mll"
                            ( COLONCOLON )
# 1038 "cil/src/logic/logic_lexer.ml"

  | 29 ->
# 308 "cil/src/logic/logic_lexer.mll"
                            ( DOT )
# 1043 "cil/src/logic/logic_lexer.ml"

  | 30 ->
# 309 "cil/src/logic/logic_lexer.mll"
                            ( DOTDOT )
# 1048 "cil/src/logic/logic_lexer.ml"

  | 31 ->
# 310 "cil/src/logic/logic_lexer.mll"
                            ( DOTDOTDOT )
# 1053 "cil/src/logic/logic_lexer.ml"

  | 32 ->
# 311 "cil/src/logic/logic_lexer.mll"
                            ( MINUS )
# 1058 "cil/src/logic/logic_lexer.ml"

  | 33 ->
# 312 "cil/src/logic/logic_lexer.mll"
                            ( PLUS )
# 1063 "cil/src/logic/logic_lexer.ml"

  | 34 ->
# 313 "cil/src/logic/logic_lexer.mll"
                            ( STAR )
# 1068 "cil/src/logic/logic_lexer.ml"

  | 35 ->
# 314 "cil/src/logic/logic_lexer.mll"
                            ( AMP )
# 1073 "cil/src/logic/logic_lexer.ml"

  | 36 ->
# 315 "cil/src/logic/logic_lexer.mll"
                            ( HATHAT )
# 1078 "cil/src/logic/logic_lexer.ml"

  | 37 ->
# 316 "cil/src/logic/logic_lexer.mll"
                            ( HAT )
# 1083 "cil/src/logic/logic_lexer.ml"

  | 38 ->
# 317 "cil/src/logic/logic_lexer.mll"
                            ( PIPE )
# 1088 "cil/src/logic/logic_lexer.ml"

  | 39 ->
# 318 "cil/src/logic/logic_lexer.mll"
                            ( TILDE )
# 1093 "cil/src/logic/logic_lexer.ml"

  | 40 ->
# 319 "cil/src/logic/logic_lexer.mll"
                            ( SLASH )
# 1098 "cil/src/logic/logic_lexer.ml"

  | 41 ->
# 320 "cil/src/logic/logic_lexer.mll"
                            ( PERCENT )
# 1103 "cil/src/logic/logic_lexer.ml"

  | 42 ->
# 321 "cil/src/logic/logic_lexer.mll"
                            ( LT )
# 1108 "cil/src/logic/logic_lexer.ml"

  | 43 ->
# 322 "cil/src/logic/logic_lexer.mll"
                            ( GT )
# 1113 "cil/src/logic/logic_lexer.ml"

  | 44 ->
# 323 "cil/src/logic/logic_lexer.mll"
                            ( LE )
# 1118 "cil/src/logic/logic_lexer.ml"

  | 45 ->
# 324 "cil/src/logic/logic_lexer.mll"
                            ( GE )
# 1123 "cil/src/logic/logic_lexer.ml"

  | 46 ->
# 325 "cil/src/logic/logic_lexer.mll"
                            ( EQ )
# 1128 "cil/src/logic/logic_lexer.ml"

  | 47 ->
# 326 "cil/src/logic/logic_lexer.mll"
                            ( EQUAL )
# 1133 "cil/src/logic/logic_lexer.ml"

  | 48 ->
# 327 "cil/src/logic/logic_lexer.mll"
                            ( NE )
# 1138 "cil/src/logic/logic_lexer.ml"

  | 49 ->
# 328 "cil/src/logic/logic_lexer.mll"
                            ( Stack.push Normal state_stack; LPAR )
# 1143 "cil/src/logic/logic_lexer.ml"

  | 50 ->
# 329 "cil/src/logic/logic_lexer.mll"
                            ( pop_state(); RPAR )
# 1148 "cil/src/logic/logic_lexer.ml"

  | 51 ->
# 330 "cil/src/logic/logic_lexer.mll"
                            ( Stack.push Normal state_stack; LBRACE )
# 1153 "cil/src/logic/logic_lexer.ml"

  | 52 ->
# 331 "cil/src/logic/logic_lexer.mll"
                            ( pop_state(); RBRACE )
# 1158 "cil/src/logic/logic_lexer.ml"

  | 53 ->
# 332 "cil/src/logic/logic_lexer.mll"
                            ( Stack.push Normal state_stack; LSQUARE )
# 1163 "cil/src/logic/logic_lexer.ml"

  | 54 ->
# 333 "cil/src/logic/logic_lexer.mll"
                            ( pop_state(); RSQUARE )
# 1168 "cil/src/logic/logic_lexer.ml"

  | 55 ->
# 334 "cil/src/logic/logic_lexer.mll"
                            ( LTCOLON )
# 1173 "cil/src/logic/logic_lexer.ml"

  | 56 ->
# 335 "cil/src/logic/logic_lexer.mll"
                            ( COLONGT )
# 1178 "cil/src/logic/logic_lexer.ml"

  | 57 ->
# 336 "cil/src/logic/logic_lexer.mll"
                            ( LTLT )
# 1183 "cil/src/logic/logic_lexer.ml"

  | 58 ->
# 337 "cil/src/logic/logic_lexer.mll"
                            ( GTGT )
# 1188 "cil/src/logic/logic_lexer.ml"

  | 59 ->
let
# 338 "cil/src/logic/logic_lexer.mll"
                 c
# 1194 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 338 "cil/src/logic/logic_lexer.mll"
                            ( find_utf8 c )
# 1198 "cil/src/logic/logic_lexer.ml"

  | 60 ->
# 339 "cil/src/logic/logic_lexer.mll"
                            ( EOF )
# 1203 "cil/src/logic/logic_lexer.ml"

  | 61 ->
# 340 "cil/src/logic/logic_lexer.mll"
        ( lex_error lexbuf ("illegal character " ^ lexeme lexbuf) )
# 1208 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and chr buffer lexbuf =
    __ocaml_lex_chr_rec buffer lexbuf 112
and __ocaml_lex_chr_rec buffer lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 344 "cil/src/logic/logic_lexer.mll"
      ( let s = lexeme lexbuf in
        let real_s = String.sub s 2 (String.length s - 2) in
        let rec add_one_char s =
          let l = String.length s in
          if l = 0 then ()
          else
          let h = int_of_digit s.[0] in
          let c,s =
            if l = 1 then (h,"")
            else
              (16*h + int_of_digit s.[1],
               String.sub s 2 (String.length s - 2))
          in
          Buffer.add_char buffer (Char.chr c); add_one_char s
        in add_one_char real_s; chr buffer lexbuf
      )
# 1234 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 361 "cil/src/logic/logic_lexer.mll"
      ( let s = lexeme lexbuf in
        let real_s = String.sub s 1 (String.length s - 1) in
        let rec value i s =
          if s = "" then i
          else value (8*i+int_of_digit s.[0])
            (String.sub s 1 (String.length s -1))
        in let c = value 0 real_s in
        Buffer.add_char buffer (Char.chr c); chr buffer lexbuf
      )
# 1247 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 371 "cil/src/logic/logic_lexer.mll"
      ( Buffer.add_char buffer
          (match (lexeme lexbuf).[1] with
               'a' -> '\007'
             | 'b' -> '\b'
             | 'f' -> '\012'
             | 'n' -> '\n'
             | 'r' -> '\r'
             | 't' -> '\t'
             | '\'' -> '\''
             | '"' -> '"'
             | '?' -> '?'
             | '\\' -> '\\'
             | _ -> assert false
          ); chr buffer lexbuf)
# 1265 "cil/src/logic/logic_lexer.ml"

  | 3 ->
# 385 "cil/src/logic/logic_lexer.mll"
        ( Buffer.contents buffer )
# 1270 "cil/src/logic/logic_lexer.ml"

  | 4 ->
# 386 "cil/src/logic/logic_lexer.mll"
       ( Buffer.add_string buffer (lexeme lexbuf); chr buffer lexbuf )
# 1275 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_chr_rec buffer lexbuf __ocaml_lex_state

and hash lexbuf =
    __ocaml_lex_hash_rec lexbuf 122
and __ocaml_lex_hash_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 389 "cil/src/logic/logic_lexer.mll"
        ( update_newline_loc lexbuf; token lexbuf)
# 1286 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 390 "cil/src/logic/logic_lexer.mll"
             ( hash lexbuf)
# 1291 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 391 "cil/src/logic/logic_lexer.mll"
              ( (* We are seeing a line number. This is the number for the
                   * next line *)
                 let s = Lexing.lexeme lexbuf in
                 let lineno = try
                   int_of_string s
                 with Failure ("int_of_string") ->
                   (* the int is too big. *)
                   Kernel.warning ~source:lexbuf.lex_start_p
                     "Bad line number in preprocessed file: %s"  s;
                   (-1)
                 in
                 update_line_loc lexbuf (lineno - 1) true 0;
                  (* A file name may follow *)
		  file lexbuf )
# 1309 "cil/src/logic/logic_lexer.ml"

  | 3 ->
# 405 "cil/src/logic/logic_lexer.mll"
                ( hash lexbuf )
# 1314 "cil/src/logic/logic_lexer.ml"

  | 4 ->
# 406 "cil/src/logic/logic_lexer.mll"
            ( endline lexbuf)
# 1319 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_hash_rec lexbuf __ocaml_lex_state

and file lexbuf =
    __ocaml_lex_file_rec lexbuf 131
and __ocaml_lex_file_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 409 "cil/src/logic/logic_lexer.mll"
                      ( update_newline_loc lexbuf; token lexbuf)
# 1330 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 410 "cil/src/logic/logic_lexer.mll"
                  ( file lexbuf)
# 1335 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 412 "cil/src/logic/logic_lexer.mll"
            (
              let n = Lexing.lexeme lexbuf in
              let n1 = String.sub n 1
                ((String.length n) - 2) in
              update_file_loc lexbuf n1;
	      endline lexbuf
            )
# 1346 "cil/src/logic/logic_lexer.ml"

  | 3 ->
# 420 "cil/src/logic/logic_lexer.mll"
      ( endline lexbuf)
# 1351 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_file_rec lexbuf __ocaml_lex_state

and endline lexbuf =
    __ocaml_lex_endline_rec lexbuf 138
and __ocaml_lex_endline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 423 "cil/src/logic/logic_lexer.mll"
                ( update_newline_loc lexbuf; token lexbuf)
# 1362 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 424 "cil/src/logic/logic_lexer.mll"
                                ( EOF )
# 1367 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 425 "cil/src/logic/logic_lexer.mll"
      ( endline lexbuf)
# 1372 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_endline_rec lexbuf __ocaml_lex_state

;;

# 427 "cil/src/logic/logic_lexer.mll"
 

  open Format

  let copy_lexbuf dest_lexbuf src_loc =
    update_file_loc dest_lexbuf src_loc.pos_fname;
    update_line_loc dest_lexbuf src_loc.pos_lnum true 0;
    let bol = src_loc.Lexing.pos_cnum - src_loc.Lexing.pos_bol in
    update_bol_loc dest_lexbuf (-bol)

  let start_pos lexbuf =
    let pos = lexeme_start_p lexbuf in
    pos.Lexing.pos_cnum - pos.Lexing.pos_bol

  let end_pos lexbuf =
    let pos = lexeme_end_p lexbuf in
    pos.Lexing.pos_cnum - pos.Lexing.pos_bol

  let parse_from_location f (loc, s : Lexing.position * string) =
    let lb = from_string s in
    copy_lexbuf lb loc;
    try
      f token lb
    with
      | Parsing.Parse_error as _e ->
        Kernel.error
	  ~source:lb.lex_curr_p
          "unexpected token '%s'" (Lexing.lexeme lb);
        Logic_utils.exit_kw_c_mode ();
        raise Parsing.Parse_error
      | Error (_, m) ->
        Kernel.error ~source:lb.lex_curr_p "%s" m;
        Logic_utils.exit_kw_c_mode ();
        raise Parsing.Parse_error
      | Logic_utils.Not_well_formed (loc, m) ->
        Kernel.error ~source:(fst loc) "%s" m;
        Logic_utils.exit_kw_c_mode ();
        raise Parsing.Parse_error

  let lexpr = parse_from_location Logic_parser.lexpr_eof

  let annot = parse_from_location Logic_parser.annot

  let spec = parse_from_location Logic_parser.spec

  (* ACSL extension for external spec file *)
  let ext_spec = parse_from_location Logic_parser.ext_spec


# 1428 "cil/src/logic/logic_lexer.ml"
