/*
 * SDLACFG	A configuration program for Frame Relay Access Devices in
 *              Linux.  Used along with 'fradcfg'.
 *
 * Version:	@(#)sdla.c	0.20	25 Jan 1998
 *
 * Author:	Mike McLagan <mike.mclagan@linux.org>
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/socket.h>

#ifdef __GLIBC__
#define _LINUX_SOCKET_H
#endif

#include <asm/io.h>

#include <linux/if.h>
#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <linux/if_frad.h>
#include <linux/sdla.h>

#include "cfgfile.h"
#include "tags.h"
#include "sdla_sfm.h"

char *sdla_banner = "Sangoma SDLA loader, v0.20(25 Jan 1998) by mike.mclagan@linux.org.";

#define DEFAULT_BOARD		"S502E"
#define DEFAULT_FIRMWARE	"fr502.sfm"
#define DEFAULT_TESTWARE	"calib502.sfm"
#define DEFAULT_PORT		0x360
#define DEFAULT_MEM		0xD0
#define DEFAULT_IRQ		0x02

void sdla_write(char *dev, int addr, void *buf, int len)
{
   struct sdla_mem mem;
   struct ifreq       ifr;
   int                sock;

   sock = socket(AF_INET, SOCK_STREAM, 0);
   strcpy(ifr.ifr_name, dev);
   ifr.ifr_data = (void *)&mem;
   mem.addr = addr;
   mem.len = len;
   mem.data = buf;
   if (ioctl(sock, SDLA_WRITEMEM, &ifr) < 0)
   {
      fprintf(stderr, "%s: Fatal error writing memory. %s\n", dev, strerror(errno));
      exit(errno);
   }
   close(sock);
}

void sdla_read(char *dev, int addr, void *buf, int len)
{
   struct sdla_mem mem;
   struct ifreq       ifr;
   int                sock;

   sock = socket(AF_INET, SOCK_STREAM, 0);
   strcpy(ifr.ifr_name, dev);
   ifr.ifr_data = (void *)&mem;
   mem.addr = addr;
   mem.len = len;
   mem.data = buf;
   if (ioctl(sock, SDLA_READMEM, &ifr) < 0)
   {
      fprintf(stderr, "%s: Fatal error reading memory.\n", dev);
      exit(errno);
   }
   close(sock);
}

unsigned short sfm_checksum (unsigned char *buf, unsigned int len)
{
  unsigned short crc	= 0;
  unsigned	 mask, flag;

  for (; len; --len, ++buf)
  {
    for (mask = 0x80; mask; mask >>= 1)
    {
      flag = (crc & 0x8000);
      crc <<= 1;
      crc |= ((*buf & mask) ? 1 : 0);
      if (flag) crc ^= 0x1021;
    }
  }
  return crc;
}

void sdla_upload(char *dev, int type, char *fname, int *cfgsize, int *cfgaddr)
{
   struct ifreq       ifr;
   int                fd, bytes, sock, i;
   sfm_t              *buf;

   fd = open(fname, O_RDONLY);
   if (fd < 0)
   {
      fprintf(stderr, "Code module %s not found.\n", fname);
      exit(-ENOENT);
   }
 
   bytes = lseek(fd, 0, SEEK_END);
   lseek(fd, 0, SEEK_SET); 
   buf = (sfm_t *) malloc(bytes);
   if (!buf)
   {
      fprintf(stderr, "Memory allocation failure, %i bytes.\n", bytes);
      exit(-ENOMEM);
   }
   
   sock = socket(AF_INET, SOCK_STREAM, 0);
   strcpy(ifr.ifr_name, dev);
   if (ioctl(sock, SDLA_STOP, &ifr) < 0)
   {
      fprintf(stderr, "Failure to stop CPU on %s.\n", dev);
      exit(errno);
   }

   if (ioctl(sock, SDLA_CLEARMEM, &ifr) < 0)
   {
      fprintf(stderr, "Failure to clear memory on %s.\n", dev);
      exit(errno);
   }

   bytes = read(fd, buf, bytes);
   if (bytes < 0)
   {
      fprintf(stderr, "Failure to read SFM file %s.\n", fname);
      exit(EIO);
   }

   if ((strcmp(buf->signature, SFM_SIGNATURE) != 0) || 
       (buf->version != SFM_VERSION) || 
       (buf->checksum != sfm_checksum((char *) &buf->info, bytes - 0x160)) ||
       (bytes != buf->info.codesize + sizeof(sfm_t)))
   {
      fprintf(stderr, "File is not a valid SFM: %s.\n", fname);
      exit(EIO);
   }

   for(i=0;i<SFM_MAX_SDLA;i++)
      if (type == buf->info.adapter[i])
         break;

   if (i == SFM_MAX_SDLA)
   {
      fprintf(stderr, "SFM does not support this adapter\n");
      exit(EIO);
   }

   /* the cfgsize is not supplied for speed calibration */
   if (cfgsize)
   {
      switch(buf->info.codeid)
      {
         case SFID_FR508:
         case SFID_FR502:
            ifr.ifr_flags = ARPHRD_FRAD;
            break;

         case SFID_CALIB502:
         case SFID_STRM502:
         case SFID_STRM508:
         case SFID_BSC502:
         case SFID_SDLC502:
         case SFID_HDLC502:
         case SFID_X25_502:
         case SFID_X25_508:
         case SFID_PPP502:
         case SFID_PPP508:
         default:
            fprintf(stderr, "SFM for unsupported protocol, id = %i\n", buf->info.codeid);
            exit(EOPNOTSUPP);
      }

      if (ioctl(sock, SDLA_PROTOCOL, &ifr) < 0)
      {
         fprintf(stderr, "Failed to set SDLA protocol: %s.\n", strerror(errno));
         exit(errno);
      }
   }

   sdla_write(dev, buf->info.codeoffs, buf->image, buf->info.codesize);

   if (cfgsize)
   {
      *cfgsize = buf->info.datasize;
      *cfgaddr = buf->info.dataoffs;
   }

   /* clear the first 0x0100 bytes */
   memset(buf, 0, buf->info.codeoffs);
   sdla_write(dev, 0, buf, buf->info.codeoffs);

   /* z80 NMI code */
   buf->signature[0] = 0xC3;
   *(short *)(&buf->signature[1]) = buf->info.startoffs;
   sdla_write(dev, type == SDLA_S502A ? SDLA_S502A_NMIADDR : SDLA_NMIADDR, buf->signature, 3);

   close(sock);
}

void sdla_load(char *dev, char *module, char *testm, int type, int port, int mem, int irq,
                  struct frad_conf *conf, int quiet)
{
   int          cpu_speed, size, addr;
   char         *speed, *ctype;
   int          sock;
   struct ifreq ifr;
   char         buf[3 + sizeof(struct sdla_conf)];

   /* this initializes the card, and sets up the memory window */
   sock = socket(AF_INET, SOCK_STREAM, 0);
   strcpy(ifr.ifr_name, dev);
   if (ioctl(sock, SIOCGIFMAP, &ifr) < 0)
   {
      fprintf(stderr, "Error accessing %s: %s.\n", dev, strerror(errno));
      exit(errno); 
   }

   ifr.ifr_map.mem_start = mem * 0x1000;
   ifr.ifr_map.base_addr = port;
   ifr.ifr_map.irq = type == SDLA_S502A ? 0 : irq;
   if (ioctl(sock, SIOCSIFMAP, &ifr) < 0)
   {
      fprintf(stderr, "Error setting port/mem/irq for %s: %s.\n", dev, strerror(errno));
      exit(errno); 
   }

   /* check the card type against the offered type */
   if (ioctl(sock, SDLA_IDENTIFY, &ifr) < 0)
   {
      fprintf(stderr, "Error retrieving card type for %s: %s.\n", dev, strerror(errno));
      exit(errno); 
   }
  
   if (type != ifr.ifr_flags)
   {
      printf("No Sangoma card found at port 0x%04X\n", port);
      exit(-ENODEV);
   }

   switch (type)
   {
      case SDLA_S502A:
      case SDLA_S502E:
         sdla_upload(dev, type, testm, NULL, NULL); 
         if (ioctl(sock, SDLA_CPUSPEED, &ifr) < 0)
         {
            fprintf(stderr, "Error performing CPU test on %s: %s.\n", dev, strerror(errno));
            exit(errno);
         }
         cpu_speed = ifr.ifr_flags;
         break;

      case SDLA_S507:
         cpu_speed = SDLA_CPU_12M;
         break;

      case SDLA_S508:
         cpu_speed = SDLA_CPU_16M;
         break;

      default:
         cpu_speed = 0;
         break;
   }

   switch(cpu_speed)
   {
      case SDLA_CPU_16M:
         speed = "16.0";
         break;
      case SDLA_CPU_12M:
         speed = "12.288";
         break;
      case SDLA_CPU_10M:
         speed = "10.0";
         break;
      case SDLA_CPU_8M:
         speed = "8.0";
         break;
      case SDLA_CPU_7M:
         speed = "7.2";
         break;
      case SDLA_CPU_5M:
         speed = "5.0";
         break;
      case SDLA_CPU_3M:
         speed = "3.6";
         break;
      default:
         speed = "unknown";
         break;
   }

   switch (type)
   {
      case SDLA_S502A:
         ctype = KBOARD_S502A;
         break;

      case SDLA_S502E:
         cpu_speed |= 0x80;
         ctype = KBOARD_S502E;
         break;

      case SDLA_S507:
         ctype = KBOARD_S507;
         break;

      case SDLA_S508:
         ctype = KBOARD_S508;
         break;

      default:
         ctype = "unknown";
   }

   sdla_upload(dev, type, module, &size, &addr);

   if (size)
   {
      memset(buf, 0, size);
      if ((type == SDLA_S502A) || (type == SDLA_S502E))
         buf[0] = cpu_speed;

      sdla_write(dev, addr, buf, size);
   }

   ifr.ifr_data = (void *) conf;
   if (ioctl(sock, FRAD_SET_CONF, &ifr) < 0)
   {
      fprintf(stderr, "Error configuring %s: %s.\n", dev, strerror(errno));
      exit(errno); 
   }

   close(sock);

   if (!quiet)
      printf("Initialized Sangoma %s, Z80 CPU at %s MHz.\n", ctype, speed); 
}

void sdla(void *cfg_info, char *dev, struct frad_conf *conf, int quiet)
{
   char         *cmem, *cport, *cirq, *firmware, *testware, *board, *temp;
   int          mem, port, irq, type, len;

   port = DEFAULT_PORT;
   mem  = DEFAULT_MEM;
   irq  = DEFAULT_IRQ;
   type = SDLA_S502E;

   if (!quiet)
      printf("%s\n", sdla_banner);

   board = config_value(cfg_info, dev, KTAG_BOARD);
   testware = config_value(cfg_info, dev, KTAG_TESTWARE);
   firmware = config_value(cfg_info, dev, KTAG_FIRMWARE);
   cmem = config_value(cfg_info, dev, KTAG_MEM);
   cport = config_value(cfg_info, dev, KTAG_PORT);
   cirq = config_value(cfg_info, dev, KTAG_IRQ);

   if (board)
   {
      if (strcasecmp(KBOARD_S502A, board) == 0)
         type = SDLA_S502A;
      else
         if (strcasecmp(KBOARD_S502E, board) == 0)
            type = SDLA_S502E;
         else
            if (strcasecmp(KBOARD_S507, board) == 0)
               type = SDLA_S507;
            else
               if (strcasecmp(KBOARD_S508, board) == 0)
                  type = SDLA_S508;
               else
               {
                  fprintf(stderr, "Invalid board specification for type Sangoma: %s.\n", board);
                  exit(-EINVAL);
               } 
   }

   if (!testware)
      testware = DEFAULT_TESTWARE;

   if (!firmware)
      firmware = DEFAULT_FIRMWARE;

   if (cmem)
      if (ishex(cmem))
         mem = strtoul(cmem, NULL, 16);
      else
      {
         fprintf(stderr, "Specified memory window invalid: %s\n", cmem);
         exit(-EINVAL);
      }

   if (cport)
      if (ishex(cport))
         port = strtoul(cport, NULL, 16);
      else
      {
         fprintf(stderr, "Specified port invalid: %s\n", cport);
         exit(-EINVAL);
      }

   if (cirq)
      if (isnumeric(cirq))
         irq = atoi(cirq);
      else
      {
         fprintf(stderr, "Specified irq invalid: %s\n", cirq);
         exit(-EINVAL);
      }

   
   temp = config_value(cfg_info, dev, KTAG_DPORT);
   if (temp)
   {
      len = strlen(temp);
      if (strncasecmp(DPORT_V35, temp, len) == 0)
         conf->clocking |= SDLA_S508_PORT_V35;
      else
         if (strncasecmp(DPORT_RS232, temp, len) == 0)
            conf->clocking |= 0;
         else
         {
            fprintf(stderr, "Unknown data port type %s for device %s\n", temp, dev);
            exit(-EINVAL);
         }
   }

   sdla_load(dev, firmware, testware, type, port, mem, irq, conf, quiet);
}
