/***** BEGIN LICENSE BLOCK *****

    FlashGot - a Firefox extension for external download managers integration
    Copyright (C) 2004-2009 Giorgio Maone - g.maone@informaction.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
                             
***** END LICENSE BLOCK *****/

var RedirectContext = function(links, opType, dm, onfinish) {
  this.links = links;
  this.opType = opType;
  this.dm = dm;
  this.onfinish = onfinish || function() {};
  this.processedBy = {};
  this.redirects = 0;
  this.maxRedirects = 1;
  var processors = [];
  var srv = dm.service;
  for each(var p in this.processors) {
    if (srv.getPref("redir." + p.name + ".enabled", true))
      processors.push(p);
  }
  
  this.processors = processors;
};

RedirectContext.prototype = {
  prefs: CC["@mozilla.org/preferences-service;1"].getService(CI.nsIPrefService).getBranch("flashgot.redir."),
  
  print: Components.utils && Components.utils.reportError || dump,
  log: function(msg) {
    this.print("[FlashGot Redirect Processor] " + msg);
  },
  process: function(links) {
    if(!links) links = this.links;
    try {
      this.start();
      for (j = links.length; j-- > 0;) {
        this.processLink(links[j]);
      }
    } catch(e) {
      this.log(e);
    } finally {
      this.done();
    }
  },
  
  processLink: function(l) {
    const processors = this.processors;
    for (var p = 0, plen = processors.length, j; p < plen; p++) {
      try {
        processors[p](l, this);
      } catch(e) {
        this.log(processors[p].name + ": " + e + " " + e.stack);
      }
    }
  },
  
  start: function() {
    this.redirects++;
    this._progress();
  },
  
  done: function() {
    if (--this.redirects == 0) {
      this.onfinish(this.processedBy);
    }
    this._progress();
  },
  
  _progress: function() {
    if(this.redirects > this.maxRedirects) this.maxRedirects = this.redirects;
    if(this.redirects >= 0) {
      this.links.progress.update(
          40 + 30 * (this.maxRedirects - this.redirects) / this.maxRedirects);
    }
  },
  
  change: function(l, newURL, processedBy, multiReplace) {
    this.processedBy[processedBy || arguments.callee.caller.name] = true;
    if(!this.links.some(function(l) { return l.href == newURL })) {
      var nl;
      if (multiReplace) {
        nl = {};
        for(var p in l) nl[p] = l[p];
        this.links.push(nl);
        var pos = this.links.indexOf(l);
        if (pos > -1) this.links.splice(pos, 1);
      } else {
        nl = l;
      }
      nl.href = newURL;
      nl.contentType = null; // prevent spidering
      this.processLink(nl); // recursive processing
    }
  },
  createReq: function() {
    return CC["@mozilla.org/xmlextras/xmlhttprequest;1"].createInstance(CI.nsIXMLHttpRequest);
  },
  
  load: function(url, callbacks, data) {
    
    if (typeof(data) == "undefined") data = null;
    
    var req = this.createReq();
    req.open(data == null ?  "GET" :"POST", url, true);
    if (data != null) req.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
    
    if (typeof(callbacks) != "object") {
      callbacks = { ok: callbacks };
    }
    var context = this;
    callbacks.call = function(phase) {
      if (typeof(this[phase]) == "function") this[phase](req, context);
    }
    
    callbacks.call(0);
    
    req.onreadystatechange = function() {
      var phase = req.readyState;
      try {
        callbacks.call(phase);
      } catch(e) {
        context.log(e);
      }
      if (phase == 4) {
        try {
          if (req.status == 200) callbacks.call("ok");
        } catch(e) {
          context.log(e);
        } finally {
          context.done();
          context = null;
        }
      }
    };
    
    this.start();
    try {
      req.send(data); 
    } catch(e) {
      this.done();
    }
  },
  
  processors: [
    function anonym_to(l, context) { // anonym.to, anonymz.com, linkbucks.com
      var m = l.href.match(/^http:\/\/(?:[^\.\/]+\.)?(linkbucks\.com|anonym\.to|anonymz\.com)(?:\/?.*?)?\?.*?(http.*)/i);
      if (m) {
        var href = m[2];
        context.change(l, /^http%3a/i.test(href) ? unescape(href) : href, m[1].replace(".", "_"));
      }
    },
    
    function ftp2share_net(l, context) {
        if (/^https?:\/\/ftp2share\.net\//.test(l.href)) {
          var processedBy = arguments.callee.name;
          context.load(l.href, function(req) {
            var mm = req.responseText.match(/javascript:\s*go\s*\(((["']).*\2)/g);
            if (mm) {
              for(var j = 0, len = mm.length; j < len; j++) {
                try {
                  context.change(l, atob(mm[j].replace(/[^A-Za-z0-9\+\/\=]|javascript:.*?go/g, "")), processedBy, true);
                } catch(e) {
                  context.log(e);
                }
              }
            }
          }, "download=true");
          
        }
    },
    
    function depositfiles_com(l, context) {
      if (/^http:\/\/depositfiles\.com\//.test(l.href))
        context.load(l.href, function(req) {
          var m = req.responseText.match(/https?:\/\/[^"'\s]+depositfiles\.com\/auth\-[^"' ]+/);
          if (m) context.change(l, m[0]);
        });
    },
    
    function hideurl_biz(l, context) {
      if (/^http:\/\/hideurl\.biz\//.test(l.href)) {
        if (/\btype=r1/.test(l.href)) { // Rapidshare Free, skip
          context.links.splice(context.links.indexOf(l), 1);
          return;
        }
        var processedBy = arguments.callee.name;
        if (/\/link\.php/.test(l.href) && !/\btype=r2/.test(l.href)) {  // r2 = Rapidshare Premium
         if (context.sniffRedir(l.href, function(url) {
            if(url) context.change(l, url, processedBy);
            context.done();
            }))
            context.start();
        } else {
          context.load(l.href, function(req) {
            var m = req.responseText.match(/https?:\/\/(?:[^/]*\brapidshare\.|hideurl\.biz\/link\.php)[^'"]+/g);
            if (m) for each(var u in m) context.change(l, u, processedBy, true);
          });
        }
      }
    },
    
    function lix_in(l, context) {
      if (/^http:\/\/lix\.in\//.test(l.href)) {
        var processedBy = arguments.callee.name;
        context.load(l.href, function(req) {
          var m = req.responseText.match(/<iframe[^>]*src\s*=\s*['"]([^"']+).*/);
          if (m) {
           context.change(l, m[1]);
           return;
          }

          var m = req.responseText.match(/name="in" value="[^"]+/g);
          if (m) {

            for each(var s in m) {
              context.load(l.href, function(req) {
                 var m = req.responseText.match(/<iframe[^>]*src\s*=\s*['"]([^"']+).*/);
                 if (m) {
                   context.change(l, m[1], processedBy, true);
                 } else {
                   print(req.responseText);
                 }
              }, "in=" + escape(s.replace(/.*value="/, "")) + "&submit=continue");
              
            }
          }
          
        }, "tiny=" + escape(l.href.replace(/.*lix\.in\//, "")) + "&submit=continue");
      }
    },

    function link_protector_com(l, context) {
      if (!/^http:\/\/link-protector\.com\//.test(l.href)) return;
      function addRef(req) { req.setRequestHeader("Referer", context.links.referrer); }
      context.load(l.href,
        {
          0: addRef,
          1: addRef,
          ok: function(req) {
            var m = req.responseText.match(/yy\[i\]\s*-(\d+)[\S\s]+stream\(['"]([^'"]+)/);
            if (m) {
              function decode(t, x) {
                function stream(prom){var yy=new Array();for(i=0; i*4 <prom.length; i++){yy[i]=prom.substr(i*4,4);}yy.reverse();var xstream=new String;for (var i = 0; i < yy.length; i++){xstream+=String.fromCharCode(yy[i]-x);}return xstream;}
                return stream(t);
              }
              context.change(l, decode(m[2], m[1]).match(/="(https?:[^" ]+)/)[1]); 
            } else if((m = req.responseText.match(/<a href="(https?:[^" ]+)/))) {
              context.change(l, m[1]);
            }
          }
        });
    },
    
    function linkbank_eu(l, context) {
      if (!/^http:\/\/(?:[\w-]+\.)linkbank.eu\/show\.php/.test(l.href)) return;
      
      var posli = l.href.replace(/show.*/, "posli.php?match=");
      context.load(l.href, function(req) {
          var m = req.responseText.match(/posli\("\d+",\s*"\d+"\)/g);
          if (!m) return;
          for each(var sm in m) {
            sm = sm.match(/posli\("(\d+)",\s*"(\d+)"\)/);
            if(context.sniffRedir(posli + sm[1] + "&id=" + sm[2], callback))
              context.start();
          }
      });
      
      var processedBy = arguments.callee.name;
      function callback(url) {
        if (url) context.change(l, url, processedBy, true)
        context.done();
      }
    },
    
    function linkbucks_com(l, context) {
      if (/^http:\/\/(?:[\w-]+\.)linkbucks\.com\/link\/[^?]*$/.test(l.href)) {
        context.load(l.href, function(req) {
          var m = req.responseText.match(/document\.location\.href\s*=\s*"(https?:\/\/[^"]*)/);
          if (m) context.change(l, m[1]);
        });
      }
    },

    function megaupload_com(l, context) {
      if (context.links.postData ||
          !/^http:\/\/(?:[\w-]+\.)?mega(?:upload|rotic)\.com\/.*\?d=/.test(l.href)
        ) return;
      var processor = arguments.callee;
      
      if (!processor._direct) {
        processor._direct = true;
        context.megauploadQueue = [];
        context.load(l.href.replace(/^(http.*?\/\/.*?\/).*/i, '$1?c=account'), function(req) {  
          dequeue();
        }, "do=directdownloads&accountupdate=1&set_ddl=1");
      }
      
      if (context.megauploadQueue) {
        context.megauploadQueue.push(l);
        return;
      }
      
      var force = false;
      try {
        force = context.prefs.getBoolPref("megaupload_com.force");
      } catch(e) {}
      
      if (force && context.sniffRedir(l.href,
        function(url) {
          context.change(l, url, processor.name);
          context.done();
        })) {
        context.start();
      }

      function dequeue() {
        var queue = context.megauploadQueue;
        if (queue) {
          context.megauploadQueue = null;
          for each(l in queue) {
            try {
              context.start();
              processor(l, context);
            } finally {
              context.done();
            }
          }
        }
      }
    },
    
    function netload_in(l, context) {
      var m = l.href.match(/^http:\/\/netload\.in\/([^\/]+)\/.*(\.[a-z]+)$/);
      if (m) {
        context.change(l, "http://netload.in/" + m[1] + m[2]);
      }
    },
    
    function protectlinks_com(l, context) {
      if (/^https?:\/\/(?:[^\/]*\w\.)?(?:protectlinks\.com)\/\d+/i.test(l.href))
        context.load(l.href.replace(/\/(\d+)/,'/redirect.php?id=$1'), function(req) {
            var m = req.responseText.match(/<iframe[^>]*pagetext[^>]*(https?:\/\/[^"'>\s]*)/i);
            if (m) context.change(l, m[1].replace(/&#x([0-9a-f]+);/ig, 
                function($, $1) { return String.fromCharCode(parseInt("0x" + $1))}
                ).replace(/&amp;/g, '&'));
        });
    },
    
    function rapidbolt_com(l, context) {
      if (/^https?:\/\/(?:[^\/]*\w\.)?(?:rapidbolt|rsmonkey)\.com\//i.test(l.href))
        context.load(l.href, function(req) {
            var m =req.responseText.match(/\bhttps?:\/\/.*?rapidshare\.com[^"'\s]*/);
            if (m) context.change(l, m[0]);
        });
    },
    
    function relink_us(l, context) {
      if (!/^http?:\/\/(?:[^\/]*\w\.)?relink\.us\/.*\bid=/.test(l.href)) return;
      const processedBy = arguments.callee.name;
      context.load(l.href, function(req) {
        var mm = req.responseText.match(/\bhttp:\/\/(?:[^\/]*\w\.)?relink\.us\/f\/[^"]*/g);
        if (!mm) return;
        for each(var url in mm) {
          context.load(url, function(req) {
            var m = req.responseText.match(/<iframe[^>]+(http:[^'"]+)/i);
            if (m) context.change(l, m[1], processedBy, true);
          });
        }
      });
    },
    
    function rsprotect_com(l, context) {
      if (/^https?:\/\/(?:[^\/]*\w\.)?(?:rsprotect\.com|rapidsafe\.net)\//i.test(l.href))
        context.load(l.href, function(req) {
            var m = req.responseText.match(/\baction\s*=["'\s]*?(https?:\/\/.*?rapidshare\.com[^"'\s]*)/i);
            if (m) context.change(l, m[1].replace(/&#x([0-9a-f]+);/ig, 
                function($, $1) { return String.fromCharCode(parseInt("0x" + $1))}));
        });
    },
    
    function stealth_to(l, context) {

      var doc = context.links.document;
      if(!doc) return;

      var rx = /http:\/\/stealth\.to\/.*[&\?]id=/;
      if (!(rx.test(l.href) || 
          !context.stealth_to_topChecked && doc && rx.test(doc.URL))) 
        return;
      var stealth_to = arguments.callee;
      if(!context.stealth_to_topChecked) {
         context.stealth_to_topChecked = true;
         if(doc && rx.test(doc.URL) && 
            checkList(doc.documentElement.innerHTML)) 
           return;
      }
      
      var postData = context.links.postData || l.href.match(/&code=.*/) || null;
      if (postData) {
        l.href = l.href.replace(/&code=.*/, '');
        postData = postData.toString();
      }
     
      context.load(l.href, function(req) {
          checkAll(req.responseText);
        }, postData);
      
      function checkAll(html) {
        return checkCaptcha(html) || checkList(html) || checkAjax(html);
      }
    
      function checkCaptcha(html) {
        if (/<input[^>]*code/.test(html)) { // captcha page
          var docURL = l.href + "#FlashGot_Form";
          var ee, j, f;
          var renew = null;
          if(docURL == doc.URL) {
           renew = doc;
          } else {
            ee = doc.getElementsByTagName("iframe");
          
            for(j = ee.length; j-- > 0;) 
              if(ee[j].src == docURL) break;
            
            if(j >= 0) {
             f = ee[j];
             renew = f.contentDocument;
            } else {
              ee = doc.getElementsByTagName("a");
              for(var j = ee.length; j-- > 0;)
                if(ee[j].href == l.href) break;
       
              var a = j < 0 ? doc.body : ee[j];
              var f = doc.createElement("iframe");
              f.style.display = "block";
              f.style.width="350px";
              f.style.height="120px";
              f.style.borderStyle = "solid";
              f.style.borderColor = "orange";
              f.style.borderWidth = "2px";
              doc.defaultView.addEventListener("DOMFrameContentLoaded", function(ev) {
              var d = ev.target.contentDocument;
              if (!d.body) return;
              d.body.removeAttribute("onload");
              
              var f = d.getElementsByTagName("form")[0];
              d.body.insertBefore(f, d.body.firstChild);
              var ii = d.getElementsByTagName("img");
              for (var j = ii.length; j-- > 0;) {
                if (/captcha/.test(ii[j].src)) {
                  d.body.insertBefore(ii[j], f).style.display="block";
                  break;
                }
              }
              
              ii = d.getElementsByTagName("input");
              for(j = 0; j < ii.length; j++) f.appendChild(ii[j]);
              while(f.nextSibling) d.body.removeChild(f.nextSibling);
              ii = d.getElementsByTagName("link");
              for(j = 0; j < ii.length; j++) ii[j].href="data:";
              }, true);
              f.src = docURL;
              a.appendChild(f);
            }
          }
          
          if(renew) {
            // renew captcha
            ee = renew.getElementsByTagName("img");
             for(j = ee.length; j-- > 0;) 
               ee[j].src = ee[j].src + "?" + new Date().getTime();
          }
          
          context.links.splice(context.links.indexOf(l), 1); 
          return true;
        }
        return false;
      }
      
      function checkList(html) {
        var m = html.match(/\bdownload\(['"]?\d+/g);
        if (m) {
          
          var nl, args, p;
          var ids = [];
          args = [ context.links.indexOf(l), 1 ]; // Array.splice parameters
          for each (var id in m) {
            id = id.replace(/\D/g, '');
            if (ids.indexOf(id) > -1) continue;
            ids.push(id);
            // copy link;
            nl = {};
            for(p in l) nl[p] = l[p];
            nl.href = "http://stealth.to/index.php?go=download&id=" + id;  
            stealth_to(nl, context);
            args.push(nl); 
          }
          Array.prototype.splice.apply(context.links, args); // replace parent link with children
          if(/#FlashGot_Form$/.test(doc.URL))
          {
            // close iframe
            var ee = doc.defaultView.parent.document.getElementsByTagName("iframe");
            for(var j = ee.length; j-- > 0;) {
              if(ee[j].contentDocument == doc) {
                ee[j].parentNode.removeChild(ee[j]);
              }
            }
          }
          return true;
        }
        return false;
      }
      
      function checkAjax(html) {
        var parts = html.split("|||");
        if (parts.length < 2) return false;
        context.change(l, "http://" + parts[0]);
        return true;
      }
      
    },
    
    function shorten_ws(l, context) {
      if (/^http:\/\/[^\/]*shorten\.ws\//.test(l.href))
        context.load(l.href, function(req) {
          var m = req.responseText.match(/<table[^>]*shortURLTable[\s\S]*?<a[^>]*(https?:[^ ">]*)/);
          if (m) context.change(l, m[1]);
        });
    },
    
    function tube_url(l, context) {
      if (/^http:\/\/(?:[^\/]+\.)*tubeurl\.com\//.test(l.href))
        context.load(l.href, function(req) {
          var m = req.responseText.match(/<meta[^>]*refresh[^>]*(https?:[^ ">]*)/i);
          if (m) context.change(l, m[1]);
        });
    },
    
    function tinyurl_com(l, context) { // tinyurl.com or moourl.com or downloads.sourceforge.net
      var m = l.href.match(/^https?:\/\/(?:(tiny|moo)url\.com|(?:[^\/\.]+\.)?sourceforge\.net)\//);
      if (!m) return;
      var processedBy = m[1] ? m[1] + "url_com" : "sf_net";
      var limit = processedBy == "sf_net" ? 0 : 20;
      var method = processedBy == "moourl_com" ? "GET" : "HEAD";
      var callback = function(url) {
        if (url) context.change(l, url, processedBy);
        context.done();
      };
      if (context.sniffRedir(l.href, callback, method,  limit)) {
        context.start();
      }
    },
    function uploaded_to(l, context) {
      if (/^http:\/\/ul\.to\//.test(l.href)) {
        context.change(l, l.href.replace("/ul.to/", "/uploaded.to/file/"));
      }
    },
    function uploading_com(l, context) {
      if (/^http:\/\/(?:[^\/]+\.)*uploading\.com\/files\//.test(l.href)) {
        context.load(l.href, function(req) { // initial GET to set the file cookie
          context.load(l.href, { "3": function(req) { // POST with cookie
            if (req.channel.URI.spec != l.href) context.change(l, req.channel.URI.spec, "uploading_com");
            req.abort(); // otherwise we download the whole file from the browser ;)
          }}, "premium=1&x=1");
        });
      }
    },
    
    
    function zshare_net(l, context) {
      if (/^http:\/\/(?:[^\/]+\.)*zshare\.net\/[a-z]+\/[a-z0-9]+\/$/.test(l.href)) {
        context.load(l.href.replace(/^(http:\/\/(?:[^\/]+\.)*zshare\.net\/)[a-z]+(\/.*)/, '$1download$2'),
        function(req) {
          var m = req.responseText.match(/Array\('([\s\S]*?)'\)/);
          if (m) context.change(l, m[1].split(/'\s*,\s*'/).join(''));
        }, "download=1&imageField=");
      }
    },
    
    function media(l, context) {
      if (!l.contentType) return;
      switch(l.contentType) {
        // see http://gonze.com/playlists/playlist-format-survey.html
        case "audio/mpegurl":
        case "audio/x-mpegurl":
        // case "video/x-ms-asf": // we should need to differentiate asx from asf, see MediaSniffer
        case "video/x-ms-asx":
        case "video/x-ms-wax":
        case "video/x-ms-wvx":
        case "audio/vnd.rn-realaudio":
        case "audio/x-pn-realaudio":
        case "application/smil":
        case "audio/x-scpls":
        break;
        default:
        return;
      }
      context.load(l.href,
        function(req) {
          var urls = req.responseText.match(/\b[a-z]{3,6}:\/\/[^\s<"']*/g);
          if (!urls) return;
          for each(var u in urls) {
            context.change(l, u, "media", true);
          }
        });
    },
    
    function generic(l, context) {
      if (l.contentType || l.noRedir) return; // avoid jamming FlashGot Media or already redirected URLs
      
      if (typeof(context.genericRx) != "object") {
        try {
          context.genericRx = new RegExp(context.dm.service.getPref("redir.generic.rx", null), "i");
        } catch(e) {
          context.genericRx = null;
        }
      }  
      if (context.genericRx == null) return;
      var m = l.href.match(context.genericRx);
      if (m) {
        var href = m[1];
        context.change(l, /^https?%3a/i.test(href) ? unescape(href) : href, null, true); // latest arg -> add, rather than replace
      }
    },

  ],
  
  sniffRedir: function(url, callback, method, limit) {
    var ch = CC["@mozilla.org/network/io-service;1"].getService(CI.nsIIOService
        ).newChannel(url, null, null);
    if(!(ch instanceof CI.nsIHttpChannel)) return false;
    
    if (method && (method instanceof CI.nsIHttpChannel)) { // copy data from another channel
      var och = method;
      method = och.requestMethod;
      if (ch instanceof CI.nsIUploadChannel && och instanceof CI.nsIUploadChannel) {
        ch.loadFlags |= ch.LOAD_BYPASS_CACHE;
        och.uploadStream.seek(0, 0);
        ch.setUploadStream(och.uploadStream, och.getRequestHeader("Content-Type"), -1);
      } 
    }
    ch.requestMethod = method || "HEAD";
    ch.redirectionLimit = typeof(limit) == "undefined" ? 20 : limit;
    ch.asyncOpen(this.redirSniffer, {
       callback: callback,
       get wrappedJSObject() { return this; }
    });
    return true;
  },
  redirSniffer: {
    onStartRequest: function(req, ctx) {
      req.cancel(NS_BINDING_ABORTED);
    },
    onDataAvailable: function(req, ctx , stream , offset , count ) {},
    onStopRequest: function(req, ctx) {
      var url;
      if (req instanceof CI.nsIHttpChannel) {
        try {
          url = req.URI.resolve(req.getResponseHeader("Location"));
        } catch(e) {}
      }
      if (!url) {
        url = (req instanceof CI.nsIChannel) ? req.URI.spec : "";
      }
      ctx.wrappedJSObject.callback(url);
    }
  }
  
};
