/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *   
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity 
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *    
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <ferite.h>

struct {
   char *scriptname;
   int verbose;
   int showargs;
   int estring;
} opt;

void show_help()
{
   fprintf( stderr, "This is %s, (v %s)\n", FERITE_NAME, FERITE_VERSION );
   printf( "\nCopyright 1999-2001, Chris Ross\n" );
   printf( "\nUsage: ferite [OPTIONS] [FILE]\n\n" );
   printf( "\t--help    \t-h\tShow this help screen.\n" );
   printf( "\t--verbose \t-v\tBe verbose when running.\n" );
   printf( "\t--showargs\t-s\tShow all the arguments passed to ferite.\n" );
   printf( "\t--execute \t-e\tExecute the script supplied on the command line\n" );
   
   printf( "\n" );
   exit(0);
}

void parse_args( int argc, char **argv )
{
   int i, gotScript = 0;
   char *buf;
   
   opt.verbose = 0;
   opt.showargs = 0;
   opt.estring = 0;
   for(i = 1; i < argc; i++) {
      if(argv[i][0] == '-') {
		 if((!strcmp(argv[i], "--verbose")) || (!strcmp(argv[i], "-v")))
		   opt.verbose = 1;
		 if((!strcmp(argv[i], "--showargs")) || (!strcmp(argv[i], "-s")))
		   opt.showargs = 1;
		 if((!strcmp(argv[i], "--help")) || (!strcmp(argv[i], "-h"))) {
			show_help();
			exit(0);
		 }
		 if((!strcmp(argv[i], "--execute")) || (!strcmp(argv[i], "-e"))) {
			opt.scriptname = strdup( argv[++i] );
			opt.estring = 1;
		 }
      } else {
		 if( !gotScript ){
			buf = (char *)malloc(1024);
			memset(buf, '\0', 1024);
			if(argv[i][0] != '/') {
			   /* relative path */
			   getcwd(buf, 1024);
			   strcat(buf, "/");
			   strcat(buf, argv[i]);
			} else {
			   /* absolute path */
			   strcpy(buf, argv[i]);
			}	 
			gotScript = 1;
			opt.scriptname = buf;
		 }
      }
   }
   
   if(!opt.scriptname) {
      fprintf( stderr, "This is %s, (v %s)\n", FERITE_NAME, FERITE_VERSION );
	  printf( "\nCopyright 1999-2001, Chris Ross\n" );
      printf( "\nUsage: ferite [OPTIONS] [FILE]\n\n" );
      printf( "\tYou need to supply a file name. For help try ferite --help\n\n" );
      exit( -1 );
   }   
}


int main( int argc, char **argv )
{
   FeriteEnvironment *env;
   FeriteScript *script;
   
   int i = 0;
   
   if ( argc < 2 )
   {
      fprintf( stderr, "%s (v %s)\n", FERITE_NAME, FERITE_VERSION );
      printf( "\nUsage: ferite [OPTIONS] [FILE]\n\n" );
      printf( "\tYou need to supply a file name. For help try ferite --help\n\n" );
      exit( -1 );
   }
   else
   {
      parse_args( argc, argv );
      if( opt.showargs )
      {
	 printf( "Argument Count: %d\n", argc );
	 for( i = 0; i < argc; i++ )
	   printf( "Argument #%d: %s\n", i+1, argv[i] );
      }
      
      if( opt.verbose )
	printf( "--> initialising ferite\n");
      
      env = ferite_init( argc, argv );
      
      if( opt.verbose )
	printf( "--> creating script object\n" );
      
      if( opt.verbose )
      {
	 printf( "--> loading script from file\n" );	 
	 printf( "-->   `-> script file: %s\n", opt.scriptname );
      }
      
      
      if( opt.verbose )
	printf( "--> compiling script\n" );
      
      if( !opt.estring ){ 
		 script = ferite_script_compile( opt.scriptname );
      } else {
		 ferite_set_filename( "-e" );
		 script = __ferite_compile_string( opt.scriptname );
      }
      
      if( script )
      {
	 if( opt.verbose )
	   printf( "--> executing script\n" );
	 ferite_script_execute( script );

	 if( opt.verbose )
	   printf( "--> destroying script object\n" );
	 ferite_script_delete( script );
      }
      
      ferite_deinit();
      free( opt.scriptname );
      if( opt.verbose )
	printf( "--> Quiting Application\n" );
      
      exit(0);
   }
}
