/* -*- mode: c; mode: fold; -*- */
/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *   
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity 
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *    
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "ferite.h"
#include <math.h>

/*{{{ FeriteVariable *__ferite_build_object( FeriteScript *script, FeriteClass *nclass) */
FeriteVariable *__ferite_build_object( FeriteScript *script, FeriteClass *nclass)
{
   FeriteVariable *ptr = NULL;
   FeriteVariable *result, *result2;
   FeriteFunction *func;
   void *(*fptr)( FeriteScript *s, FeriteVariable **plist );
   
   FE_ENTER_FUNCTION;
   
   if( nclass != NULL )
   {
      FUD(("BUILDOBJECT: Creating an instance of %s\n", nclass->name ));
      ptr = __ferite_create_object_variable( "new_object" );
      VAO(ptr) = fmalloc( sizeof( FeriteObject ) );
      VAO(ptr)->name = fstrdup( nclass->name );
      VAO(ptr)->tmpl = nclass;

      FUD(("BUILDOBJECT: Creating a duplicate varaible hash\n"));
      VAO(ptr)->variables = __ferite_duplicate_variable_hash( script, nclass->variables );

      FUD(("BUILDOBJECT: Linking function list up\n"));
      VAO(ptr)->functions = nclass->functions;
      VAO(ptr)->child = NULL;
      VAO(ptr)->next = NULL;
      VAO(ptr)->parent = NULL;

      FUD(("BUILDOBJECT: Marking as disposable\n"));
      MARK_VARIABLE_AS_DISPOSABLE( ptr );

      VAO(ptr)->oid = nclass->id;
      VAO(ptr)->odata = NULL;

      VAO(ptr)->refcount = 1;

      __ferite_add_to_gc( VAO(ptr) );
   }
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FeriteVariable *__ferite_new_object( FeriteClass *nclass, FeriteParameterList *plist ) */
FeriteVariable *__ferite_new_object(FeriteScript *script, FeriteClass *nclass, FeriteVariable **plist)
{
   FeriteVariable *ptr = NULL;
   FeriteVariable *result, *result2;
   FeriteFunction *func;
   void *(*fptr)( FeriteScript *s, FeriteVariable **plist);

   FE_ENTER_FUNCTION;
   if(nclass != NULL) {
      ptr = __ferite_build_object(script,nclass);
      FUD(("NEWOBJECT: Searching for constructor\n" ));
      if( (func = __ferite_hash_get( script, nclass->functions, nclass->name )) != NULL )
      {
		 /* we have the constructor */
		 result = __ferite_duplicate_variable( script, ptr );
		 MARK_VARIABLE_AS_DISPOSABLE( result );
		 ffree( result->name );
		 result->name = fstrdup( "super" );
		 plist = __ferite_add_to_parameter_list( plist, result );
		 
		 result2 = __ferite_duplicate_variable( script, ptr );
		 MARK_VARIABLE_AS_DISPOSABLE( result2 );
		 ffree( result2->name );
		 result2->name = fstrdup( "self" );
		 plist = __ferite_add_to_parameter_list( plist, result2 );
		 
		 if( func->type == FNC_IS_EXTRL )
		 {
			FUD(("OPS: Calling constructor in class %s\n", nclass->name ));
			if( __ferite_check_params( script, plist, func->signature ) )
			{
			   fptr = func->fncPtr;
			   __ferite_variable_destroy( script, (fptr)( script, plist ) );
			}
			else
			{
			   ferite_error( script, "Wrong parameters for the constructor of the class %s", nclass->name );
			   __ferite_stop_execution( script );
			}
		 }
		 else
		 {
			FUD(("NEWOBJECT: Calling script constructor\n"));
			if( __ferite_check_params( script, plist, func->signature ) )
			{		   
			   __ferite_script_function_execute( script, func, plist );
			   __ferite_variable_destroy( script, func->returnt );
			} else {
			   ferite_error( script, "Wrong parameters for the constructor of the class %s", nclass->name );
			   __ferite_stop_execution( script );
			}
		 }
	  }
   }
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_UNARY_OP( left_incrc ) */
FERITE_UNARY_OP( left_incr )
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   switch( a->type )
   {
    case F_VAR_LONG:
      FUD(("OPS: Incrementing %s from %ld to ", a->name, VAI( a ) ));
      VAI( a ) += 1;
      FUD(("%ld\n", VAI( a ) ));
      DUPLICATE_LONG_VARIABLE( a, ptr );
      MARK_VARIABLE_AS_DISPOSABLE( ptr );
      FE_LEAVE_FUNCTION( ptr );
      break;
    default:
      ferite_error( script, "Can't increment variables of type %s\n", __ferite_variable_id_to_str( script,a->type) );
   }
   FE_LEAVE_FUNCTION( NULL );
}
/*}}}*/

/*{{{ FERITE_UNARY_OP( right_incr ) */
FERITE_UNARY_OP( right_incr )
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   switch( a->type )
   {
    case F_VAR_LONG:
      DUPLICATE_LONG_VARIABLE( a, ptr );
      FUD(("OPS: Incrementing %s from %ld to ", a->name, VAI( a ) ));
      VAI( a ) += 1;
      FUD(("%ld\n", VAI( a ) ));
      MARK_VARIABLE_AS_DISPOSABLE( ptr );
      FE_LEAVE_FUNCTION( ptr );
      break;
    default:
      ferite_error( script,"Can't increment variables of type %s\n", __ferite_variable_id_to_str( script,a->type) );
   }
   FE_LEAVE_FUNCTION( NULL );
}
/*}}}*/

/*{{{ FERITE_UNARY_OP( left_decr ) */
FERITE_UNARY_OP( left_decr )
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   switch( a->type )
   {
    case F_VAR_LONG:
      FUD(("OPS: Decrementing %s from %ld to ", a->name, VAI( a ) ));
      VAI( a ) -= 1;
      FUD(("%ld\n", VAI( a ) ));
      DUPLICATE_LONG_VARIABLE( a, ptr );
      MARK_VARIABLE_AS_DISPOSABLE( ptr );
      FE_LEAVE_FUNCTION( ptr );
      break;
    default:
      ferite_error( script,"Can't decrement variables of type %s\n", __ferite_variable_id_to_str( script,a->type) );
   }
   FE_LEAVE_FUNCTION( NULL );
}
/*}}}*/

/*{{{ FERITE_UNARY_OP( right_decr ) */
FERITE_UNARY_OP( right_decr )
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   switch( a->type )
   {
    case F_VAR_LONG:
      DUPLICATE_LONG_VARIABLE( a, ptr );
      MARK_VARIABLE_AS_DISPOSABLE( ptr );
      FUD(("OPS: Decrementing %s from %ld to ", a->name, VAI( a ) ));
      VAI( a ) -= 1;
      FUD(("%ld\n", VAI( a ) ));
      FE_LEAVE_FUNCTION( ptr );
      break;
    default:
      ferite_error( script, "Can't decrement variables of type %s\n", __ferite_variable_id_to_str( script,a->type) );
   }
   FE_LEAVE_FUNCTION( NULL );
}
/*}}}*/

/*{{{ FERITE_UNARY_OP( positive_var ) */
FERITE_UNARY_OP( positive_var )
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   switch( a->type )
   {
    case F_VAR_LONG:
      DUPLICATE_LONG_VARIABLE( a, ptr );
      MARK_VARIABLE_AS_DISPOSABLE( ptr );
	  if( VAI(ptr) < 0 ){
		 VAI(ptr) = 0 - VAI(ptr);
	  }
      FE_LEAVE_FUNCTION( ptr );
      break;
    case F_VAR_DOUBLE:
      DUPLICATE_DOUBLE_VARIABLE( a, ptr );
      MARK_VARIABLE_AS_DISPOSABLE( ptr );
	  if( VAF(ptr) < 0 ){
		 VAF(ptr) = 0 - VAF(ptr);
	  }
      FE_LEAVE_FUNCTION( ptr );
      break;
    default:
      ferite_error( script, "Can't positise variables of type %s\n", __ferite_variable_id_to_str( script,a->type) );
   }
   FE_LEAVE_FUNCTION( NULL );
}
/*}}}*/

/*{{{ FERITE_UNARY_OP( negative_var ) */
FERITE_UNARY_OP( negative_var )
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   switch( a->type )
   {
    case F_VAR_LONG:
      DUPLICATE_LONG_VARIABLE( a, ptr );
      MARK_VARIABLE_AS_DISPOSABLE( ptr );
	  VAI(ptr) = 0 - VAI(ptr);
      FE_LEAVE_FUNCTION( ptr );
      break;
    case F_VAR_DOUBLE:
      DUPLICATE_DOUBLE_VARIABLE( a, ptr );
      MARK_VARIABLE_AS_DISPOSABLE( ptr );
	  VAF(ptr) = 0 - VAF(ptr);
      FE_LEAVE_FUNCTION( ptr );
      break;
    default:
      ferite_error( script, "Can't negatise variables of type %s\n", __ferite_variable_id_to_str( script,a->type) );
   }
   FE_LEAVE_FUNCTION( NULL );
}
/*}}}*/

/*{{{ MACROS FOR WRITING CLEAN OPERATORS */

#define BEGIN_BLOCK( t, var ) \
 case t: \
   switch( var->type ) \
   {

#define END_BLOCK \
   } \
   break;

#define BEGIN_OP( var ) \
   FeriteVariable *retv = NULL; \
   FE_ENTER_FUNCTION; \
   switch( var->type ){

#define BLOCK_DIE( t ) \
 default: \
   __ferite_raise_script_error( script, 1, "Can't %s variables of type %s and %s", t,  \
			        __ferite_variable_id_to_str( script,a->type),  __ferite_variable_id_to_str( script,b->type) );

#define END_OP( d ) \
   BLOCK_DIE( d )\
   } \
   if( retv != NULL ) { MARK_VARIABLE_AS_DISPOSABLE( retv ); } \
   FE_LEAVE_FUNCTION( retv );

#define DO_OP( t, cv, c ) \
 case t: \
   cv; \
   c; \
   break;

#define RETVNAME( n ) "op-" n "-return-value"

#define FE_VAR_TRUE( var, op ) var = __ferite_create_number_long_variable( RETVNAME( op ), 1 )
#define FE_VAR_FALSE( var, op ) var = __ferite_create_number_long_variable( RETVNAME( op ), 0 )

#define FE_VAR_TEST( test, op ) \
	   if( test ) \
		   FE_VAR_TRUE( ptr, op ); \
       else \
	       FE_VAR_FALSE( ptr, op );

/*}}}*/

/*{{{ FERITE_UNARY_OP( not_op ) */
FERITE_UNARY_OP( not_op )
{
   FeriteVariable *ptr;
   FE_ENTER_FUNCTION;

   if( !__ferite_variable_is_false( script,a) )
     ptr = __ferite_create_number_long_variable( RETVNAME( "not_op" ), 0 );
   else
     ptr = __ferite_create_number_long_variable( RETVNAME( "not_op" ), 1 );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );

   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_UNARY_OP( eval ) */
extern int __ferite_compile_error;

FERITE_UNARY_OP( eval )
{
   FeriteVariable *ptr;
   FeriteScript *sptr;
   FeriteFunction *func;
   int error_state = 0;
   
   FE_ENTER_FUNCTION;
   
   if( a->type != F_VAR_STR )
     ferite_error( script, "Can not eval variables of type %s\n", __ferite_variable_id_to_str( script,a->type) );
   
   sptr = __ferite_compile_string( VAS(a) );
   
   if( ! __ferite_compile_error  ){
	  
	  func = __ferite_function_get( sptr, "_start" );
	  __ferite_script_eval_execute( sptr, func, script->mainns, __ferite_stack_top( script->exec_stack ) );
	  
	  ptr = func->returnt;
	  MARK_VARIABLE_AS_DISPOSABLE( ptr );
	  error_state = sptr->error_state;
	  
	  FUD(( "Script error_state: %d\n", error_state ));
	  
	  ferite_script_delete( sptr );
	  
	  if( error_state == 0 ){
		 FUD(( "return from eval: %s\n", __ferite_variable_id_to_str( script, ptr->type ) ));
		 FE_LEAVE_FUNCTION( ptr );
	  } else {
		 FUD(( "execution error on eval\n" ));
	  }
   } else {
      FUD(( "Compile error on eval\n" ));
   }
   ptr = __ferite_create_void_variable( "eval-return" );
   
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_UNARY_OP( include ) */
FERITE_UNARY_OP( include )
{
   FeriteVariable *ptr;

   FE_ENTER_FUNCTION;
   if( a->type != F_VAR_STR )
     ferite_error( script, "You must pass include a string\n" );

   __ferite_script_include( script, VAS(a) );
   
   ptr = __ferite_create_void_variable( "include-return" );

   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( add ) */
FERITE_BINARY_OP( add )
{
   double dval = 0;

   BEGIN_OP( a )
     BEGIN_BLOCK( F_VAR_DOUBLE, b )
       DO_OP( F_VAR_LONG, , retv = __ferite_create_number_double_variable( RETVNAME( "add" ), VAF(a) + VAI(b) ) )
	 DO_OP( F_VAR_DOUBLE, , retv = __ferite_create_number_double_variable( RETVNAME( "add" ), VAF(a) + VAF(b) ) )
	   BLOCK_DIE( "add" )
	     END_BLOCK
     BEGIN_BLOCK( F_VAR_STR, b )
       DO_OP( F_VAR_STR,
	     retv = __ferite_variable_alloc(); retv->type = F_VAR_STR; retv->name = fstrdup( RETVNAME( "add" ) ); VAS(retv) = fmalloc( strlen( VAS(a) ) + strlen( VAS(b) ) + 1 ); ,
	     strcpy( VAS(retv), VAS(a) ); strcat( VAS(retv), VAS(b) );FUD(( "returning (str)\"%s\"\n", VAS(retv) ));
	     )
	 DO_OP( F_VAR_LONG,
	       retv = __ferite_variable_alloc(); retv->type = F_VAR_STR; retv->name = fstrdup( RETVNAME( "add" ) ); VAS(retv) = fmalloc( strlen( VAS(a) ) +1024 ); memset( VAS(retv), '\0', strlen( VAS(a) ) + 1024 ); ,
	       sprintf( VAS(retv), "%s%ld", VAS(a), VAI(b) );FUD(( "returning (int)\"%s\"\n", VAS(retv) ));
	       )
	   DO_OP( F_VAR_DOUBLE,
		 retv = __ferite_variable_alloc(); retv->type = F_VAR_STR; retv->name = fstrdup( RETVNAME( "add" ) ); VAS(retv) = fmalloc( strlen( VAS(a) ) +1024 ); memset( VAS(retv), '\0', strlen( VAS(a) ) + 1024 ); ,
		 sprintf( VAS(retv), "%s%f", VAS(a), VAF(b));FUD(( "returning (flt)\"%s\"\n", VAS(retv) ));
		 )
        BLOCK_DIE( "add" );
     END_BLOCK
     BEGIN_BLOCK( F_VAR_LONG, b )
       DO_OP( F_VAR_LONG,
	     dval = (double)VAI(a) + (double)VAI(b);
	     if( dval > (double)LONG_MAX )
	       retv = __ferite_create_number_double_variable( RETVNAME( "add" ), dval );
	     else
	       retv = __ferite_create_number_long_variable( RETVNAME( "add" ), VAI(a) + VAI(b) ), NOWT )
       DO_OP( F_VAR_DOUBLE, NOWT, retv = __ferite_create_number_double_variable( RETVNAME( "add" ), VAI(a)+VAF(b) ); )
     BLOCK_DIE( "add" );
   END_BLOCK
 END_OP( "add" )
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( minus ) */
FERITE_BINARY_OP( minus )
{
   double dval = 0;

   BEGIN_OP( a )
     BEGIN_BLOCK( F_VAR_DOUBLE, b )
       DO_OP( F_VAR_LONG, , retv = __ferite_create_number_double_variable( RETVNAME( "minus" ), VAF(a) - VAI(b) ) )
	 DO_OP( F_VAR_DOUBLE, , retv = __ferite_create_number_double_variable( RETVNAME( "minus" ), VAF(a) - VAF(b) ) )
	   BLOCK_DIE( "minus" )
	     END_BLOCK
     BEGIN_BLOCK( F_VAR_LONG, b )
       DO_OP( F_VAR_LONG,
	     dval = (double)VAI(a) - (double)VAI(b);
	     if( dval < (double)LONG_MIN )
	     retv = __ferite_create_number_double_variable( RETVNAME( "minus" ), (double)VAI(a) - (double)VAI(b) );
	     else
	     retv = __ferite_create_number_long_variable( RETVNAME( "minus" ), VAI(a)-VAI(b) ) , NOWT )
	 DO_OP( F_VAR_DOUBLE, , retv = __ferite_create_number_double_variable( RETVNAME( "minus" ), VAI(a)-VAF(b) ); )
	   BLOCK_DIE( "minus" );
   END_BLOCK
     END_OP( "minus" )
}
/*}}}*/

/*{{{ FcERITE_BINARY_OP( mult ) */
FERITE_BINARY_OP( mult )
{
   double dval = 0;

   BEGIN_OP( a )
     BEGIN_BLOCK( F_VAR_DOUBLE, b )
       DO_OP( F_VAR_LONG, , retv = __ferite_create_number_double_variable( RETVNAME( "mult" ), VAF(a) * VAI(b) ) )
	 DO_OP( F_VAR_DOUBLE, , retv = __ferite_create_number_double_variable( RETVNAME( "mult" ), VAF(a) * VAF(b) ) )
	   BLOCK_DIE( "mult" )
	     END_BLOCK
     BEGIN_BLOCK( F_VAR_LONG, b )
       DO_OP( F_VAR_LONG,
	     dval = (double)VAI(a) * (double)VAI(b);
	     if( dval > (double)LONG_MAX )
	     retv = __ferite_create_number_double_variable( RETVNAME( "mult" ), dval );
	     else
	     retv = __ferite_create_number_long_variable( RETVNAME( "mult" ), VAI(a) * VAI(b) ); , NOWT)
	 DO_OP( F_VAR_DOUBLE, , retv = __ferite_create_number_double_variable( RETVNAME( "mult" ), VAI(a)*VAF(b) ); )
	   BLOCK_DIE( "mult" );
   END_BLOCK
     END_OP( "mult" )
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( divide ) */
FERITE_BINARY_OP( divide )
{
   BEGIN_OP( a )
     BEGIN_BLOCK( F_VAR_DOUBLE, b )
       DO_OP( F_VAR_LONG, if( VAI(b) == 0 ) ferite_error( script, "Divide By Zero Error\n" ); , retv = __ferite_create_number_double_variable( RETVNAME( "divide" ), VAF(a) / VAI(b) ) )
	 DO_OP( F_VAR_DOUBLE, if( VAF(b) == 0 ) ferite_error( script, "Divide By Zero Error\n" ); , retv = __ferite_create_number_double_variable( RETVNAME( "divide" ), VAF(a) / VAF(b) ) )
	   BLOCK_DIE( "divide" )
	     END_BLOCK
     BEGIN_BLOCK( F_VAR_LONG, b )
       DO_OP( F_VAR_LONG, if( VAI(b) == 0 ) ferite_error( script, "Divide By Zero Error\n" );, retv = __ferite_create_number_long_variable( RETVNAME( "divide" ), VAI(a)/VAI(b) ); )
	 DO_OP( F_VAR_DOUBLE, if( VAF(b) == 0 ) ferite_error( script, "Divide By Zero Error\n" );, retv = __ferite_create_number_double_variable( RETVNAME( "divide" ), VAI(a)/VAF(b) ); )
	   BLOCK_DIE( "divide" );
   END_BLOCK
     END_OP( "divide" )
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( modulus ) */
FERITE_BINARY_OP( modulus )
{
   BEGIN_OP( a )
     BEGIN_BLOCK( F_VAR_DOUBLE, b )
       DO_OP( F_VAR_LONG, if( VAI(b) == 0 ) ferite_error( script, "modulus By Zero Error\n" ); , retv = __ferite_create_number_double_variable( RETVNAME( "modulus" ), (long)VAF(a) % VAI(b) ) )
	 DO_OP( F_VAR_DOUBLE, if( VAF(b) == 0 ) ferite_error( script, "modulus By Zero Error\n" ); , retv = __ferite_create_number_double_variable( RETVNAME( "modulus" ), (long)VAF(a) % (long)VAF(b) ) )
	   BLOCK_DIE( "modulus" )
	     END_BLOCK
     BEGIN_BLOCK( F_VAR_LONG, b )
       DO_OP( F_VAR_LONG, if( VAI(b) == 0 ) ferite_error( script, "modulus By Zero Error\n" );, retv = __ferite_create_number_long_variable( RETVNAME( "modulus" ), VAI(a)%VAI(b) ); )
	 DO_OP( F_VAR_DOUBLE, if( VAF(b) == 0 ) ferite_error( script, "modulus By Zero Error\n" );, retv = __ferite_create_number_double_variable( RETVNAME( "modulus" ), VAI(a)%(long)VAF(b) ); )
	   BLOCK_DIE( "modulus" );
   END_BLOCK
     END_OP( "modulus" )
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( assign ) */
FERITE_BINARY_OP( assign )
{
   FeriteVariable *retv = NULL;

   FE_ENTER_FUNCTION;
   if( a->flags.constant == 0 || a->flags.constant == 1 )
   {
      switch( a->type )
      {
	 BEGIN_BLOCK( F_VAR_LONG, b )
	   DO_OP( F_VAR_LONG, VAI(a) = VAI(b);, retv = __ferite_create_number_long_variable( RETVNAME( "assign" ), VAI(a) ); )
	   DO_OP( F_VAR_DOUBLE, a->type = F_VAR_DOUBLE; VAF(a) = VAF(b);, retv = __ferite_create_number_double_variable( RETVNAME( "assign" ), VAF(a) ); )
	   BLOCK_DIE( "assign" )
	 END_BLOCK

         BEGIN_BLOCK( F_VAR_STR, b )
	   DO_OP( F_VAR_STR, ffree( VAS(a) ); VAS(a) = fstrdup( VAS(b) );, retv = __ferite_create_string_variable( RETVNAME( "assign" ), VAS(a) ); )
	   BLOCK_DIE( "assign" )
	 END_BLOCK

	   BEGIN_BLOCK( F_VAR_DOUBLE, b )
	     DO_OP( F_VAR_DOUBLE, VAF(a) = VAF(b);, retv = __ferite_create_number_double_variable( RETVNAME( "assign" ), VAF(a) ); )
	       DO_OP( F_VAR_LONG, a->type = F_VAR_LONG; VAI(a) = VAI(b);, retv = __ferite_create_number_long_variable( RETVNAME( "assign" ), VAI(a) ); )
		 BLOCK_DIE( "assign" )
		   END_BLOCK

	   BEGIN_BLOCK( F_VAR_OBJ, b )
	     DO_OP( F_VAR_OBJ,
		   if( VAO(a) != NULL )
			  {
				 VAO(a)->refcount--;
				 if( VAO(a)->refcount <= 0 )
				   FUD(("OPS: GC: Object \"%s\" Flagged for DELETION\n", a->name));
			  }
			  if( VAO(b) != NULL )
				VAO(b)->refcount++;
			  VAO(a) = VAO(b);,
			  retv = __ferite_create_number_long_variable( RETVNAME( "assign" ), 1 );
		   )
	       BLOCK_DIE( "assign" );
         END_BLOCK

	   BEGIN_BLOCK( F_VAR_UARRAY, b )
	     DO_OP( F_VAR_UARRAY,
		   __ferite_uarray_destroy( script, VAUA(a));
		   VAUA(a) = __ferite_uarray_dup( script, VAUA(b), (void *(*)(FeriteScript*,FeriteVariable *))__ferite_duplicate_variable );
		   VAUA(a)->size = VAUA(b)->size;,
		   retv = __ferite_duplicate_variable( script, a );
		   )
	       BLOCK_DIE( "assign" );
	 END_BLOCK

	   BEGIN_BLOCK( F_VAR_VOID, b )
	      DO_OP( F_VAR_LONG,
		     a->type = F_VAR_LONG;
		     VAI(a) = VAI(b);,
		     retv = fe_new_lng( RETVNAME( "assign" ), VAI(a) );
		  )
	      DO_OP( F_VAR_DOUBLE,
		     a->type = F_VAR_DOUBLE;
		     VAF(a) = VAF(b);,
		     retv = fe_new_dbl( RETVNAME( "assign" ), VAF(a) );
		  )
		  DO_OP( F_VAR_STR,
    	     a->type = F_VAR_STR;
	         VAS(a) = fstrdup(VAS(b));,
		     retv = fe_new_str( RETVNAME( "assign" ), VAS(a) );
		  )
		  DO_OP( F_VAR_OBJ,
			 a->type = F_VAR_OBJ;
			 VAO(b)->refcount++;
			 VAO(a) = VAO(b);,
			 retv = __ferite_create_number_long_variable( RETVNAME( "assign" ), 1 );
		  )
		  DO_OP( F_VAR_UARRAY,
			 a->type = F_VAR_UARRAY;
			 VAUA(a) = __ferite_uarray_dup( script, VAUA(b), (void *(*)(FeriteScript*,FeriteVariable *))__ferite_duplicate_variable );
			 VAUA(a)->size = VAUA(b)->size;,
			 retv = __ferite_duplicate_variable( script, a );
		  )
	    BLOCK_DIE( "assign" );
          END_BLOCK
	BLOCK_DIE( "assign" );
      }
      if( a->flags.constant == 1 )
		a->flags.constant = 2; /* set it so that the variable is constant for forever and a day */
   }
   else
   {
      ferite_error( script, "Can not assign to a constant variable.\n" );
   }
   MARK_VARIABLE_AS_DISPOSABLE( retv );
   FE_LEAVE_FUNCTION( retv );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( add_assign ) */
FERITE_BINARY_OP( add_assign )
{
   FeriteVariable *ptr, *tmp_ptr;

   FE_ENTER_FUNCTION;
   tmp_ptr = __ferite_op_add( script, a, b );
   ptr = __ferite_op_assign( script, a, tmp_ptr );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   __ferite_variable_destroy( script, tmp_ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( minus_assign ) */
FERITE_BINARY_OP( minus_assign )
{
   FeriteVariable *ptr, *tmp_ptr;

   FE_ENTER_FUNCTION;
   tmp_ptr = __ferite_op_minus( script, a, b );
   ptr = __ferite_op_assign( script, a, tmp_ptr );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   __ferite_variable_destroy( script, tmp_ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( mult_assign ) */
FERITE_BINARY_OP( mult_assign )
{
   FeriteVariable *ptr, *tmp_ptr;

   FE_ENTER_FUNCTION;
   tmp_ptr = __ferite_op_mult( script, a, b );
   ptr = __ferite_op_assign( script, a, tmp_ptr );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   __ferite_variable_destroy( script, tmp_ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( divide_assign ) */
FERITE_BINARY_OP( divide_assign )
{
   FeriteVariable *ptr, *tmp_ptr;

   FE_ENTER_FUNCTION;
   tmp_ptr = __ferite_op_divide( script, a, b );
   ptr = __ferite_op_assign( script, a, tmp_ptr );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   __ferite_variable_destroy( script, tmp_ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( logical_or ) */
FERITE_BINARY_OP( logical_or )
{
   FeriteVariable *ptr;
   
   FE_ENTER_FUNCTION;
   FE_VAR_TEST( (!__ferite_variable_is_false( script,a) || !__ferite_variable_is_false( script,b )), "logical_or" );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( logical_and ) */
FERITE_BINARY_OP( logical_and )
{
   FeriteVariable *ptr;
   
   FE_ENTER_FUNCTION;
   FE_VAR_TEST( (!__ferite_variable_is_false( script,a) && !__ferite_variable_is_false( script,b )), "logical_and" );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( equals ) */
/* binary operators that push a true or falsae value onto the stack */
FERITE_BINARY_OP( equals )
{
   FeriteVariable *ptr = NULL;

   FE_ENTER_FUNCTION;
   if( a->type != b->type )
   {
      FUD(("OPS: Variable types do not match in equals( %s, %s )\nReturning false\n", a->name, b->name ));
      ptr = __ferite_create_number_long_variable( "equals", 0 );
      MARK_VARIABLE_AS_DISPOSABLE( ptr );
      FE_LEAVE_FUNCTION( ptr );
   }
   /* we know that they are the same */
   switch( a->type )
   {
    case F_VAR_LONG:
	  FE_VAR_TEST( (VAI(a) == VAI(b)), "equals" );
      break;
    case F_VAR_DOUBLE:
      FE_VAR_TEST( (VAF(a) == VAF(b)), "equals" );
      break;
    case F_VAR_STR:
      FE_VAR_TEST( (strcmp( VAS(a), VAS(b) ) == 0), "equals" );
      break;
    case F_VAR_OBJ:
      FE_VAR_TEST( (VAO(a) == VAO(b)), "equals" );
      break;
    default:
      ferite_error( script, "EEEK: unknown type %s in equals()\n", __ferite_variable_id_to_str( script,a->type ) );
   }
   if( ptr )
     MARK_VARIABLE_AS_DISPOSABLE( ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( case ) */
/* binary operators that push a true or falsae value onto the stack */
FERITE_BINARY_OP( case )
{
   FeriteVariable *ptr = NULL;
   FeriteExecuteRec *exec = NULL;
   
   FE_ENTER_FUNCTION;
   
   /* we have to push the a value back onto the stack for the other case statements */
   ptr = __ferite_duplicate_variable( script, a );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   exec = __ferite_stack_top( script->exec_stack );
   __ferite_stack_push( exec->stack, ptr );
   ptr = NULL;
   
   if( a->type != b->type )
   {
      FUD(("OPS: Variable types do not match in equals( %s, %s )\nReturning false\n", a->name, b->name ));
      ptr = __ferite_create_number_long_variable( "equals", 0 );
      MARK_VARIABLE_AS_DISPOSABLE( ptr );
      FE_LEAVE_FUNCTION( ptr );
   }
   /* we know that they are the same */
   switch( a->type )
   {
    case F_VAR_LONG:
	  FE_VAR_TEST( (VAI(a) == VAI(b)), "equals" );
      break;
    case F_VAR_DOUBLE:
      FE_VAR_TEST( (VAF(a) == VAF(b)), "equals" );
      break;
    case F_VAR_STR:
      FE_VAR_TEST( (strcmp( VAS(a), VAS(b) ) == 0), "equals" );
      break;
    case F_VAR_OBJ:
      FE_VAR_TEST( (VAO(a) == VAO(b)), "equals" );
      break;
    default:
      ferite_error( script, "EEEK: unknown type %s in equals()\n", __ferite_variable_id_to_str( script,a->type ) );
   }
   if( ptr )
     MARK_VARIABLE_AS_DISPOSABLE( ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( notequals ) */
FERITE_BINARY_OP( notequals )
{
   FeriteVariable *ptr, *tptr;
   FE_ENTER_FUNCTION;
   tptr = __ferite_op_equals( script, a, b );
   FE_VAR_TEST( (__ferite_variable_is_false( script,tptr )), "not_equals" );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   __ferite_variable_destroy( script, tptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( less_than ) */
FERITE_BINARY_OP( less_than )
{
   FeriteVariable *ptr = NULL;
   
   FE_ENTER_FUNCTION;
   switch( a->type )
   {
    case F_VAR_LONG:
      switch( b->type )
      {
       case F_VAR_LONG:
		 FE_VAR_TEST( (VAI( a ) < VAI(b)), "less_than" );
		 break;
       case F_VAR_DOUBLE:
		 FE_VAR_TEST( (VAI( a ) < VAF( b )), "less_than" );
       default:
		 FUD(("OPS: ERROR: can't compare values of type %s with integers\n", __ferite_variable_id_to_str( script,b->type) ));
      }
      break;
    case F_VAR_DOUBLE:
      switch( b->type )
      {
       case F_VAR_LONG:
         FE_VAR_TEST( (VAF( a ) < VAI(b)), "less_than" );
		 break;
       case F_VAR_DOUBLE:
         FE_VAR_TEST( (VAF( a ) < VAF(b)), "less_than" );
		 break;
       default:
		 FUD(("OPS: ERROR: can't compare values of type %s with floats\n", __ferite_variable_id_to_str( script,b->type) ));
      }
      break;
    default:
      ferite_error( script, "ERK, can't compare items of type %s\n", __ferite_variable_id_to_str( script,a->type) );
   }
   if( ptr )
     MARK_VARIABLE_AS_DISPOSABLE( ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( less_than_equals ) */
FERITE_BINARY_OP( less_than_equals )
{
   FeriteVariable *ptr = NULL;
   FE_ENTER_FUNCTION;
   switch( a->type )
   {
    case F_VAR_LONG:
      switch( b->type )
      {
       case F_VAR_LONG:
		 FE_VAR_TEST( (VAI( a ) <= VAI(b)), "less_than_equals" );
		 break;
       case F_VAR_DOUBLE:
		 FE_VAR_TEST( (VAI( a ) <= VAF( b )), "less_than_equals" );
       default:
		 FUD(("OPS: ERROR: can't compare values of type %s with integers\n", __ferite_variable_id_to_str( script,b->type) ));
      }
      break;
    case F_VAR_DOUBLE:
      switch( b->type )
      {
       case F_VAR_LONG:
         FE_VAR_TEST( (VAF( a ) <= VAI(b)), "less_than_equals" );
		 break;
       case F_VAR_DOUBLE:
         FE_VAR_TEST( (VAF( a ) <= VAF(b)), "less_than_equals" );
		 break;
       default:
		 FUD(("OPS: ERROR: can't compare values of type %s with floats\n", __ferite_variable_id_to_str( script,b->type) ));
      }
      break;
    default:
      ferite_error( script, "ERK, can't compare items of type %s\n", __ferite_variable_id_to_str( script,a->type) );
   }
   if( ptr )
     MARK_VARIABLE_AS_DISPOSABLE( ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( greater_than ) */
FERITE_BINARY_OP( greater_than )
{
   FeriteVariable *ptr = NULL;
   FE_ENTER_FUNCTION;
   switch( a->type )
   {
    case F_VAR_LONG:
      switch( b->type )
      {
       case F_VAR_LONG:
		 FE_VAR_TEST( (VAI( a ) > VAI(b)), "greater_than" );
		 break;
       case F_VAR_DOUBLE:
		 FE_VAR_TEST( (VAI( a ) > VAF( b )), "greater_than" );
       default:
		 FUD(("OPS: ERROR: can't compare values of type %s with integers\n", __ferite_variable_id_to_str( script,b->type) ));
      }
      break;
    case F_VAR_DOUBLE:
      switch( b->type )
      {
       case F_VAR_LONG:
         FE_VAR_TEST( (VAF( a ) > VAI(b)), "greater_than" );
		 break;
       case F_VAR_DOUBLE:
         FE_VAR_TEST( (VAF( a ) > VAF(b)), "greater_than" );
		 break;
       default:
		 FUD(("OPS: ERROR: can't compare values of type %s with floats\n", __ferite_variable_id_to_str( script,b->type) ));
      }
      break;
    default:
      ferite_error( script, "ERK, can't compare items of type %s\n", __ferite_variable_id_to_str( script,a->type) );
   }
   if( ptr )
     MARK_VARIABLE_AS_DISPOSABLE( ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( greater_than_equals ) */
FERITE_BINARY_OP( greater_than_equals )
{
   FeriteVariable *ptr = NULL;
   FE_ENTER_FUNCTION;
   switch( a->type )
   {
    case F_VAR_LONG:
      switch( b->type )
      {
       case F_VAR_LONG:
		 FE_VAR_TEST( (VAI( a ) >= VAI(b)), "greater_than_equals" );
		 break;
       case F_VAR_DOUBLE:
		 FE_VAR_TEST( (VAI( a ) >= VAF( b )), "greater_than_equals" );
       default:
		 FUD(("OPS: ERROR: can't compare values of type %s with integers\n", __ferite_variable_id_to_str( script,b->type) ));
      }
      break;
    case F_VAR_DOUBLE:
      switch( b->type )
      {
       case F_VAR_LONG:
         FE_VAR_TEST( (VAF( a ) >= VAI(b)), "greater_than_equals" );
		 break;
       case F_VAR_DOUBLE:
         FE_VAR_TEST( (VAF( a ) >= VAF(b)), "greater_than_equals" );
		 break;
       default:
		 FUD(("OPS: ERROR: can't compare values of type %s with floats\n", __ferite_variable_id_to_str( script,b->type) ));
      }
      break;
    default:
      ferite_error( script, "ERK, can't compare items of type %s\n", __ferite_variable_id_to_str( script,a->type) );
   }
   if( ptr )
     MARK_VARIABLE_AS_DISPOSABLE( ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( array_index ) */
FERITE_BINARY_OP( array_index )
{
   /* a = array variable */
   /* b = index */
   FeriteVariable *ptr = NULL;
   
   FE_ENTER_FUNCTION;
   
   if( a->type != F_VAR_STR && a->type != F_VAR_UARRAY)
     ferite_error( script, "OPS: array_index: First Variable is not an Array or String\n" );
   
   switch( a->type )
   {
    case F_VAR_STR:
      {
		 switch( b->type )
		 {
		  case F_VAR_LONG:
			if( VAI(b) > strlen( VAS(a) ) )
			{
			   ferite_error( script, "String index out of range [%d]\n", VAI(b) );
			   __ferite_stop_execution( script );
			   break;
			}
			ptr = __ferite_create_number_long_variable( "array_String_return", VAS(a)[VAI(b)] );
			MARK_VARIABLE_AS_DISPOSABLE( ptr );
			break;
		  case F_VAR_DOUBLE:
			{
			   long index = (long)VAF(b);
			   if( VAI(b) > strlen( VAS(a) ) )
			   {
				  ferite_error( script, "String index out of range [%d]\n", index );
				  __ferite_stop_execution( script );
				  break;
			   }
			   ptr = __ferite_create_number_long_variable( "array_String_return", VAS(a)[index] );
			   MARK_VARIABLE_AS_DISPOSABLE( ptr );			   
			}
			break;
		  default:
			ferite_error( script, "Other index methods not implemented in strings (%s)\n", __ferite_variable_id_to_str( script, b->type ) );
			__ferite_stop_execution( script );
		 }
		 break;
      }
    case F_VAR_UARRAY:
      {
		 ptr = __ferite_uarray_op( script, VAUA(a), b, NULL);
		 break;
      }
   }
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( left_shift ) */
FERITE_BINARY_OP( left_shift )
{
   BEGIN_OP( a )
     BEGIN_BLOCK( F_VAR_LONG, b )
       DO_OP( F_VAR_LONG, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "left_shift" ), VAI(a) << VAI(b) ); )
	   DO_OP( F_VAR_DOUBLE, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "left_shift" ), VAI(a) << (long)VAF(b) ); )
	   BLOCK_DIE( "left_shift" );
     END_BLOCK
     BEGIN_BLOCK( F_VAR_DOUBLE, b )
       DO_OP( F_VAR_LONG, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "left_shift" ), (long)VAF(a) << VAI(b) ); )
	   DO_OP( F_VAR_DOUBLE, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "left_shift" ), (long)VAF(a) << (long)VAF(b) ); )
	   BLOCK_DIE( "left_shift" );
     END_BLOCK
   END_OP( "left_shift" )
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( right_shift ) */
FERITE_BINARY_OP( right_shift )
{
   BEGIN_OP( a )
     BEGIN_BLOCK( F_VAR_LONG, b )
       DO_OP( F_VAR_LONG, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "right_shift" ), VAI(a) >> VAI(b) ); )
	   DO_OP( F_VAR_DOUBLE, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "right_shift" ), VAI(a) >> (long)VAF(b) ); )
	   BLOCK_DIE( "right_shift" );
     END_BLOCK
     BEGIN_BLOCK( F_VAR_DOUBLE, b )
       DO_OP( F_VAR_LONG, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "right_shift" ), (long)VAF(a) >> VAI(b) ); )
	   DO_OP( F_VAR_DOUBLE, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "right_shift" ), (long)VAF(a) >> (long)VAF(b) ); )
	   BLOCK_DIE( "right_shift" );
     END_BLOCK
   END_OP( "right_shift" )
}
/*}}}*/
			
/*{{{ FERITE_BINARY_OP( binary_or ) */
FERITE_BINARY_OP( binary_or )
{
   BEGIN_OP( a )
     BEGIN_BLOCK( F_VAR_LONG, b )
       DO_OP( F_VAR_LONG, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "binary_or" ), VAI(a) | VAI(b) ); )
	   DO_OP( F_VAR_DOUBLE, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "binary_or" ), VAI(a) | (long)VAF(b) ); )
	   BLOCK_DIE( "binary_or" );
     END_BLOCK
     BEGIN_BLOCK( F_VAR_DOUBLE, b )
       DO_OP( F_VAR_LONG, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "binary_or" ), (long)VAF(a) | VAI(b) ); )
	   DO_OP( F_VAR_DOUBLE, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "binary_or" ), (long)VAF(a) | (long)VAF(b) ); )
	   BLOCK_DIE( "binary_or" );
     END_BLOCK
   END_OP( "binary_or" )
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( binary_and ) */
FERITE_BINARY_OP( binary_and )
{
   BEGIN_OP( a )
     BEGIN_BLOCK( F_VAR_LONG, b )
       DO_OP( F_VAR_LONG, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "binary_and" ), VAI(a) & VAI(b) ); )
	   DO_OP( F_VAR_DOUBLE, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "binary_and" ), VAI(a) & (long)VAF(b) ); )
	   BLOCK_DIE( "binary_and" );
     END_BLOCK
     BEGIN_BLOCK( F_VAR_DOUBLE, b )
       DO_OP( F_VAR_LONG, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "binary_and" ), (long)VAF(a) & VAI(b) ); )
	   DO_OP( F_VAR_DOUBLE, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "binary_and" ), (long)VAF(a) & (long)VAF(b) ); )
	   BLOCK_DIE( "binary_and" );
     END_BLOCK
   END_OP( "binary_and" )
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( binary_xor ) */
FERITE_BINARY_OP( binary_xor )
{
   BEGIN_OP( a )
     BEGIN_BLOCK( F_VAR_LONG, b )
       DO_OP( F_VAR_LONG, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "binary_xor" ), VAI(a) ^ VAI(b) ); )
	   DO_OP( F_VAR_DOUBLE, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "binary_xor" ), VAI(a) ^ (long)VAF(b) ); )
	   BLOCK_DIE( "binary_xor" );
     END_BLOCK
     BEGIN_BLOCK( F_VAR_DOUBLE, b )
       DO_OP( F_VAR_LONG, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "binary_xor" ), (long)VAF(a) ^ VAI(b) ); )
	   DO_OP( F_VAR_DOUBLE, NOWT, retv = __ferite_create_number_long_variable( RETVNAME( "binary_xor" ), (long)VAF(a) ^ (long)VAF(b) ); )
	   BLOCK_DIE( "binary_xor" );
     END_BLOCK
   END_OP( "binary_xor" )
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( left_shift_assign ) */
FERITE_BINARY_OP( left_shift_assign )
{
   FeriteVariable *ptr, *tmp_ptr;

   FE_ENTER_FUNCTION;
   tmp_ptr = __ferite_op_left_shift( script, a, b );
   ptr = __ferite_op_assign( script, a, tmp_ptr );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   __ferite_variable_destroy( script, tmp_ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( right_shift_assign ) */
FERITE_BINARY_OP( right_shift_assign )
{
   FeriteVariable *ptr, *tmp_ptr;

   FE_ENTER_FUNCTION;
   tmp_ptr = __ferite_op_right_shift( script, a, b );
   ptr = __ferite_op_assign( script, a, tmp_ptr );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   __ferite_variable_destroy( script, tmp_ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( binary_and_assign ) */
FERITE_BINARY_OP( binary_and_assign )
{
   FeriteVariable *ptr, *tmp_ptr;

   FE_ENTER_FUNCTION;
   tmp_ptr = __ferite_op_binary_and( script, a, b );
   ptr = __ferite_op_assign( script, a, tmp_ptr );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   __ferite_variable_destroy( script, tmp_ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( binary_or_assign ) */
FERITE_BINARY_OP( binary_or_assign )
{
   FeriteVariable *ptr, *tmp_ptr;

   FE_ENTER_FUNCTION;
   tmp_ptr = __ferite_op_binary_or( script, a, b );
   ptr = __ferite_op_assign( script, a, tmp_ptr );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   __ferite_variable_destroy( script, tmp_ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/

/*{{{ FERITE_BINARY_OP( binary_xor_assign ) */
FERITE_BINARY_OP( binary_xor_assign )
{
   FeriteVariable *ptr, *tmp_ptr;

   FE_ENTER_FUNCTION;
   tmp_ptr = __ferite_op_binary_xor( script, a, b );
   ptr = __ferite_op_assign( script, a, tmp_ptr );
   MARK_VARIABLE_AS_DISPOSABLE( ptr );
   __ferite_variable_destroy( script, tmp_ptr );
   FE_LEAVE_FUNCTION( ptr );
}
/*}}}*/
