/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "ferite.h"

int __ferite_hash_gen( char *key, int size )
{
   int tot;
   unsigned int h;
   char *str;

   FE_ENTER_FUNCTION;
   if( key )
   {
      str = key;
      /* hash the string */
      for( h = 0; *str != '\0'; str++ )
		h = (h*64 + *str) % size;
      tot = h;
      FUD(("HASH: Returning hash value %d for %s\n", tot, key ));
      FE_LEAVE_FUNCTION( tot );
   }
   FE_LEAVE_FUNCTION( 0 );
}

FeriteHashBucket *__ferite_create_hash_bucket( FeriteScript *script, char *key, void *value )
{
   FeriteHashBucket *ptr;

   FE_ENTER_FUNCTION;

   ptr = fmalloc( sizeof( FeriteHashBucket ) );
   ptr->id = fstrdup( key );
   ptr->hashval = 0;
   ptr->data = value;
   ptr->next = NULL;
   ptr->prev = NULL;
   FE_LEAVE_FUNCTION( ptr );
}

FeriteHash *__ferite_create_hash( FeriteScript *script, int size )
{
   FeriteHash *ptr;
   int i;

   FE_ENTER_FUNCTION;
   ptr = fmalloc( sizeof( FeriteHash ) );
   ptr->size = size;
   ptr->numitems = 0;
   ptr->hash = fmalloc( sizeof( FeriteHash ) * size );
   for( i = 0; i < ptr->size; i++ )
     ptr->hash[i] = NULL;
   FE_LEAVE_FUNCTION( ptr );
}

void __ferite_delete_hash_bucket_list( FeriteScript *script, FeriteHashBucket *b, void (*cb)(FeriteScript*,void *data) )
{
   FE_ENTER_FUNCTION;
   if( b != NULL )
   {
      if( b->next != NULL )
		__ferite_delete_hash_bucket_list( script, b->next, cb );
      FUD(("HASH: Deleting hash bucket: %s\n", b->id ));
      if(b->id != NULL)
      {
		 ffree( b->id );
      }
      if( cb != NULL )
		(cb)( script, b->data );
      ffree( b );
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_delete_hash( FeriteScript *script, FeriteHash *hash, void (*cb)(FeriteScript*,void *data) )
{
   int i;

   FE_ENTER_FUNCTION;
   if( hash != NULL )
   {
      for( i = 0; i < hash->size; i++ )
      {
		 if( hash->hash[i] != NULL )
		 {
			FUD(( "Deleting Hash List %d:%s:%p\n", i, hash->hash[i]->id, hash->hash[i] ));
			__ferite_delete_hash_bucket_list( script, hash->hash[i], cb );
			hash->hash[i] = NULL;
		 }
      }
      ffree( hash->hash );
      ffree( hash );
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_process_hash_bucket_list( FeriteScript *script, FeriteHashBucket *b, void (*cb)(FeriteScript*,void *data, char *key) )
{
   FE_ENTER_FUNCTION;
   if( b != NULL )
   {
      FUD(("Bucket id=%s\n", b->id));
      if( b->next != NULL )
		__ferite_process_hash_bucket_list( script, b->next, cb );
      if( cb != NULL )
		(cb)( script, b->data, b->id );
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_process_hash( FeriteScript *script, FeriteHash *hash, void (*cb)(FeriteScript*,void *data,char *key) )
{
   int i;

   FE_ENTER_FUNCTION;
   if( hash != NULL )
   {
      for( i = 0; i < hash->size; i++ )
      {
		 if( hash->hash[i] != NULL )
		 {
			__ferite_process_hash_bucket_list( script, hash->hash[i], cb );
		 }
      }
   }
   FE_LEAVE_FUNCTION( NOWT );
}

FeriteHashBucket *__ferite_hash_add_get( FeriteScript *script, FeriteHash *hash, char *key, void *data)
{
   int loc;
   FeriteHashBucket *ptr, *out;

   FE_ENTER_FUNCTION;

   if( hash == NULL )
     ferite_error( script, "Trying to add \"%s\" to a non-existant hash\n", key );
   if( key == NULL )
     ferite_error( script, "Trying to add a NULL key to a hash\n" );

   loc = __ferite_hash_gen( key, hash->size );

   if(hash->hash[loc] == NULL)
   {
      hash->hash[loc] = __ferite_create_hash_bucket( script, key, data );
      out = hash->hash[loc];
      FUD(("HASH: Started bucket list with \"%s\" at %d\n", key, loc ));
   }
   else
   {
      if( hash->hash != NULL )
	  {
		 for( ptr = hash->hash[loc]; ptr != NULL && ptr->next != NULL; ptr = ptr->next )
		   ;
		 ptr->next = __ferite_create_hash_bucket( script, key, data );
		 ptr->next->prev = ptr;
		 out = ptr->next;
		 FUD(("HASH: Added \"%s\" to bucket list at %d\n", key, loc ));
      }
	  else
	  {
		 out = NULL;
      }
   }

   FE_LEAVE_FUNCTION( out );
}

void __ferite_hash_add( FeriteScript *script, FeriteHash *hash, char *key, void *data )
{
   int loc;
   FeriteHashBucket *ptr;

   FE_ENTER_FUNCTION;

   if( hash == NULL )
     ferite_error( script, "Trying to add \"%s\" to a non-existant hash\n", key );
   if( key == NULL )
     ferite_error( script, "Trying to add a NULL key to a hash\n" );

   loc = __ferite_hash_gen( key, hash->size );
   FUD(( "HASH: Adding %s to key %d\n", key, loc ));

   if( hash->hash[loc] == NULL )
   {
      hash->hash[loc] = __ferite_create_hash_bucket( script, key, data );
      hash->hash[loc]->hashval = loc;
      FUD(("HASH: Started bucket list with \"%s\" at %d\n", key, loc ));
   }
   else
   {
      for( ptr = hash->hash[loc]; ptr != NULL && ptr->next != NULL; ptr = ptr->next )
		;
      ptr->next = __ferite_create_hash_bucket( script, key, data );
      ptr->next->prev = ptr;
      FUD(("HASH: Added \"%s\" to bucket list at %d\n", key, loc ));
   }

   FE_LEAVE_FUNCTION( NOWT );
}

void *__ferite_hash_get( FeriteScript *script, FeriteHash *hash, char *key )
{
   int loc;
   FeriteHashBucket *ptr;

   FE_ENTER_FUNCTION;

   if( hash == NULL )
     ferite_error( script, "Trying to search for \"%s\" in a non-existant hash\n", key );
   if( key == NULL )
     ferite_error( script, "Trying to search for a NULL key in a hash\n" );

   loc = __ferite_hash_gen( key, hash->size );
   if( hash->hash != NULL )
   {
      for( ptr = hash->hash[loc]; ptr != NULL; ptr = ptr->next )
      {
		 FUD(("HASH: Testing \"%s\" for a match on \"%s\"\n", ptr->id, key ));
		 if( strcmp( key, ptr->id ) == 0 )
		 {
			FUD(("HASH: located data\n"));
			FE_LEAVE_FUNCTION( ptr->data );
		 }
      }
   }
   FUD(("HASH: Could not find data, returning NULL\n"));
   FE_LEAVE_FUNCTION( NULL );
}

FeriteHashBucket *__ferite_hash_get_bucket( FeriteScript *script, FeriteHash *hash, char *key)
{
   int loc;
   FeriteHashBucket *ptr;

   FE_ENTER_FUNCTION;
   if( hash == NULL )
     ferite_error( script, "Trying to search for \"%s\" in a non-existant hash\n", key );
   if( key == NULL )
     ferite_error( script, "Trying to search for a NULL key in a hash\n" );

   loc = __ferite_hash_gen( key, hash->size );
   for( ptr = hash->hash[loc]; ptr != NULL; ptr = ptr->next )
   {
      FUD(("HASH: Testing \"%s\" for a match on \"%s\"\n", ptr->id, key ));
      if( strcmp( key, ptr->id ) == 0 )
      {
		 FUD(("HASH: located data\n"));
		 FE_LEAVE_FUNCTION( ptr );
      }
   }
   FUD(("HASH: Could not find data, returning NULL\n"));
   FE_LEAVE_FUNCTION( NULL );
}

void *__ferite_hash_walk(FeriteScript *script, FeriteHash *hash, FeriteIterator *iter)
{
   int i;

   FE_ENTER_FUNCTION;
   if(hash == NULL)
   {
      ferite_error( script, "Unable to walk non-existant hash\n");
   }
   if(iter == NULL)
   {
      ferite_error( script, "Unable to walk using null iterator\n");
   }
   if(iter->curbucket == NULL)
   {
      for(i = 0; i < hash->size; i++)
      {
		 if(hash->hash[i] != NULL)
		 {
			iter->curbucket = hash->hash[i];
			iter->curindex = i;
			FE_LEAVE_FUNCTION( iter->curbucket );
		 }
      }
      FE_LEAVE_FUNCTION( iter->curbucket );
   }

   if(iter->curbucket->next != NULL)
   {
      iter->curbucket = iter->curbucket->next;
      FE_LEAVE_FUNCTION( iter->curbucket );
   }
   else
   {
      iter->curindex++;
   }

   for(i = iter->curindex; i < hash->size; i++)
   {
      if(hash->hash[i] != NULL)
      {
		 iter->curbucket = hash->hash[i];
		 iter->curindex = i;
		 FE_LEAVE_FUNCTION( iter->curbucket );
      }
   }

   FE_LEAVE_FUNCTION( NULL );
}

void *__ferite_hash_get_index( FeriteScript *script, FeriteHash *hash, int index)
{
   FeriteHashBucket *ptr;
   FeriteIterator *iter;

   FE_ENTER_FUNCTION;

   if( hash == NULL)
   {
      ferite_error( script, "Trying to to search for index %d in a non-existant hash\n", index);
   }

   iter = __ferite_create_iterator( script );

   while((ptr = __ferite_hash_walk( script, hash,iter)))
   {
      if(ptr->index == index)
      {
		 FE_LEAVE_FUNCTION( ptr );
      }
   }
   FE_LEAVE_FUNCTION( NULL );
}

void __ferite_hash_delete( FeriteScript *script, FeriteHash *hash, char *key )
{
   int loc;
   FeriteHashBucket *ptr;
   FeriteHashBucket *optr;

   FE_ENTER_FUNCTION;

   if( hash == NULL )
     ferite_error( script, "Trying to search for \"%s\" in a non-existant hash\n", key );
   if( key == NULL )
     ferite_error( script, "Trying to search for a NULL key in a hash\n" );

   loc = __ferite_hash_gen( key, hash->size );
   for( ptr = hash->hash[loc]; ptr != NULL; optr = ptr, ptr = ptr->next )
   {
      FUD(("HASH: Testing \"%s\" for a match on \"%s\"\n", ptr->id, key ));
      if( strcmp( key, ptr->id ) == 0 )
      {
		 if( hash->hash[loc] == ptr )
		   hash->hash[loc] = NULL; /* remove head from array */
		 else
		   optr->next = ptr->next; /* remove buxket from list */
		 FUD(("HASH: located data\n"));
		 ffree( ptr->id );
		 ffree( ptr );
		 FE_LEAVE_FUNCTION( NOWT );
      }
   }
   FUD(("HASH: Could not find data, returning NULL\n"));
   FE_LEAVE_FUNCTION( NOWT );
}

FeriteHash       *__ferite_hash_dup( FeriteScript *script, FeriteHash *hash, void *(*ddup)( FeriteScript*,void *data ) )
{
   FeriteHash *ptr;
   FeriteHashBucket *bptr, *nbptr;
   int i;

   FE_ENTER_FUNCTION;
   ptr = __ferite_create_hash( script, hash->size );
   for( i = 0; i < hash->size; i++ )
   {
      if( hash->hash[i] != NULL )
      {
		 FUD(("HASHDUP: Duplicating bucket(%p) %s (top level::%d)\n", hash->hash[i], hash->hash[i]->id, i ));
		 ptr->hash[i] = __ferite_create_hash_bucket( script, hash->hash[i]->id, (ddup)(script,hash->hash[i]->data) );
		 nbptr = ptr->hash[i];
		 if( hash->hash[i]->next != NULL )
		 {
			for( bptr = hash->hash[i]->next; bptr != NULL; bptr = bptr->next )
			{
			   FUD(("HASHDUP: Duplicating bucket %s\n", bptr->id ));
			   nbptr->next = __ferite_create_hash_bucket( script, bptr->id, (ddup)( script, bptr->data ));
			   nbptr = nbptr->next;
			}
		 }
      }
   }

   FE_LEAVE_FUNCTION( ptr );
}

void __ferite_hash_compress( FeriteScript *script, FeriteHash *hash )
{
   /* do nothing yet :) */
}

