/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define _GNU_SOURCE

#include <assert.h>
#include <fenv.h>
#include <math.h>

#include "softfloat.h"

static void
float_start(int rc, int pc)
{
	static const int mode[4] = {
		[0b00] = FE_TONEAREST,
		[0b01] = FE_DOWNWARD,
		[0b10] = FE_UPWARD,
		[0b11] = FE_TOWARDZERO,
	};

	feclearexcept(FE_ALL_EXCEPT);
	fesetround(mode[rc]);
}

static void
float_end(
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	int s;

	fesetround(FE_TONEAREST);

	s = fetestexcept(FE_ALL_EXCEPT);

	if (s & FE_INEXACT) {
		*pep = 1;
	}
	if (s & FE_UNDERFLOW) {
		*uep = 1;
	}
	if (s & FE_OVERFLOW) {
		*oep = 1;
	}
	if (s & FE_DIVBYZERO) {
		*zep = 1;
	}
	if (s & __FE_DENORM) {
		*dep = 1;
	}
	if (s & FE_INVALID) {
		*iep = 1;
	}
}

/*
 * Exams
 */
int
float80_isneg(float80 f)
{
	return signbit(f) != 0;
}

int
float80_iszero(float80 f)
{
	return f == 0.0L;
}

int
float80_isnorm(float80 f)
{
	return fpclassify(f) == FP_NORMAL;
}

int
float80_isdenorm(float80 f)
{
	return fpclassify(f) == FP_SUBNORMAL;
}

int
float80_isinf(float80 f)
{
	return fpclassify(f) == FP_INFINITE;
}

int
float80_isnan(float80 f)
{
	return fpclassify(f) == FP_NAN;
}

/*
 * Comparisons
 */
int
float80_isunordered(float80 a, float80 b)
{
	return isunordered(a, b);
}

int
float80_isless(float80 a, float80 b)
{
	return isless(a, b);
}

/*
 * Constants
 */
float80
float80_0(void)
{
	return 0.0L;
}

float80
float80_1(void)
{
	return 1.0L;
}

float80
float80_l2t(void)
{
	return log2l(10.0L);
}

float80
float80_l2e(void)
{
	return log2l(M_E);
}

float80
float80_pi(void)
{
	return M_PI;
}

float80
float80_lg2(void)
{
	return log10l(2.0L);
}

float80
float80_ln2(void)
{
	return logl(2.0L);
}

/*
 * Conversions
 */
float80
float80_from_i16(int16_t val)
{
	return (float80) val;
}

int16_t
float80_to_i16(
	uint8_t rc,
	float80 val,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	int16_t res;

	float_start(rc, 0); /* FIXME */

	res = (int16_t) rintl(val);

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_from_i32(int32_t val)
{
	return (float80) val;
}

int32_t
float80_to_i32(
	uint8_t rc,
	float80 val,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	int32_t res;

	float_start(rc, 0); /* FIXME */

	res = (int32_t) rintl(val);

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_from_i64(int64_t val)
{
	return (float80) val;
}

int64_t
float80_to_i64(
	uint8_t rc,
	float80 val,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	int64_t res;

	float_start(rc, 0); /* FIXME */

	res = (int64_t) rintl(val);

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_from_f32(float32 val)
{
	return (float80) val;
}

float32
float80_to_f32(
	uint8_t rc,
	float80 val,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float32 res;

	float_start(rc, 0); /* FIXME */

	res = (float32) val;

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_from_f64(float64 val)
{
	return (float80) val;
}

float64
float80_to_f64(
	uint8_t rc,
	float80 val,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float64 res;

	float_start(rc, 0); /* FIXME */

	res = (float64) val;

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_from_bcd(uint8_t *src)
{
	float80 val;
	uint64_t ival;
	int i;

	ival = 0;
	for (i = 8; 0 <= i; i--) {
		int v;

		v = src[i];
		ival *= 100;
		ival += ((v >> 4) & 0xf) * 10 + ((v >> 0) & 0xf) * 1;
	}
	if (src[9] & 0x80) {
		val = -(float80) ival;
	} else {
		val = (float80) ival;
	}

	return val;
}
	
void
float80_to_bcd(
	uint8_t *dst,
	uint8_t rc,
	float80 val,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	uint64_t ival;
	int neg;
	int i;

	float_start(rc, 0); /* FIXME */

	val = rintl(val);

	float_end(pep, uep, oep, zep, dep, iep);

	if (0.0 <= val) {
		ival = (uint64_t) val;
		neg = 0;
	} else {
		ival = (uint64_t) -val;
		neg = 1;
	}

	for (i = 0; i < 9; i++) {
		int v;

		v = ival % 100;
		ival /= 100;

		dst[i] = ((v / 10) << 4) | ((v % 10) << 0);
	}

	dst[9] = neg ? 0x80 : 0x00;
}

/*
 * Unary Functions
 */
float80
float80_chs(
	int rc,
	int pc,
	float80 a,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = -a;

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_abs(
	int rc,
	int pc,
	float80 a,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = fabsl(a);

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_sqrt(
	int rc,
	int pc,
	float80 a,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = sqrtl(a);

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_tan(
	int rc,
	int pc,
	float80 a,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = tanl(a);

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_atan(
	int rc,
	int pc,
	float80 a,
	float80 b,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = atan2l(a, b);

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_sin(
	int rc,
	int pc,
	float80 a,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = sinl(a);

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_cos(
	int rc,
	int pc,
	float80 a,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = cosl(a);

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

void
float80_sincos(
	int rc,
	int pc,
	float80 *res1p,
	float80 *res2p,
	float80 a,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float_start(rc, pc);

	sincosl(a, res1p, res2p);

	float_end(pep, uep, oep, zep, dep, iep);
}

float80
float80_2xm1(
	int rc,
	int pc,
	float80 a,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = powl(2.0L, a) - 1.0L;

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_rndint(
	int rc,
	int pc,
	float80 a,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = rintl(a);

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

/*
 * Binary Functions
 */
float80
float80_scale(
	int rc,
	int pc,
	float80 a,
	float80 b,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = ldexp(a, (int) b);

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_fprem(
	int rc,
	int pc,
	float80 a,
	float80 b,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep,
	uint8_t *c0p,
	uint8_t *c1p,
	uint8_t *c2p,
	uint8_t *c3p
)
{
	float80 res;
	uint16_t ax;

	asm (
		"fprem\n"
		"fnstsw %%ax\n"
		: "=t" (res), "=a" (ax)
		: "0" (a), "u" (b)
		: "cc"
	);

	*c3p = (ax >> 14) & 1;
	*c2p = (ax >> 10) & 1;
	*c1p = (ax >> 9) & 1;
	*c0p = (ax >> 8) & 1;

	*pep = (ax >> 5) & 1;
	*uep = (ax >> 4) & 1;
	*oep = (ax >> 3) & 1;
	*zep = (ax >> 2) & 1;
	*dep = (ax >> 1) & 1;
	*iep = (ax >> 0) & 1;

	return res;
}

float80
float80_fprem1(
	int rc,
	int pc,
	float80 a,
	float80 b,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep,
	uint8_t *c0p,
	uint8_t *c1p,
	uint8_t *c2p,
	uint8_t *c3p
)
{
	float80 res;
	uint16_t ax;

	asm (
		"fprem1\n"
		"fnstsw %%ax\n"
		: "=t" (res), "=a" (ax)
		: "0" (a), "u" (b)
		: "cc"
	);

	*c3p = (ax >> 14) & 1;
	*c2p = (ax >> 10) & 1;
	*c1p = (ax >> 9) & 1;
	*c0p = (ax >> 8) & 1;

	*pep = (ax >> 5) & 1;
	*uep = (ax >> 4) & 1;
	*oep = (ax >> 3) & 1;
	*zep = (ax >> 2) & 1;
	*dep = (ax >> 1) & 1;
	*iep = (ax >> 0) & 1;

	return res;
}

float80
float80_yl2x(
	int rc,
	int pc,
	float80 a,
	float80 b,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = b * log2l(a);

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_yl2xp1(
	int rc,
	int pc,
	float80 a,
	float80 b,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = log2l(a + 1.0) * b;

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_add(
	int rc,
	int pc,
	float80 a,
	float80 b,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = a + b;

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_sub(
	int rc,
	int pc,
	float80 a,
	float80 b,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = a - b;

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_mul(
	int rc,
	int pc,
	float80 a,
	float80 b,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = a * b;

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}

float80
float80_div(
	int rc,
	int pc,
	float80 a,
	float80 b,
	uint8_t *pep,
	uint8_t *uep,
	uint8_t *oep,
	uint8_t *zep,
	uint8_t *dep,
	uint8_t *iep
)
{
	float80 res;

	float_start(rc, pc);

	res = a / b;

	float_end(pep, uep, oep, zep, dep, iep);

	return res;
}
