/*
   FALCON - The Falcon Programming Language
   FILE: uri_ext.cpp

   Falcon class reflecting URI class
   -------------------------------------------------------------------
   Author: Giancarlo Niccolai
   Begin: Wed, 27 Feb 2008 22:39:35 +0100

   -------------------------------------------------------------------
   (C) Copyright 2004: the FALCON developers (see list in AUTHORS file)

   See LICENSE file for licensing details.
*/

/*#
   @beginmodule core
*/

/** \file
   Falcon class reflecting URI class.
*/

#include <falcon/module.h>
#include <falcon/vm.h>
#include <falcon/uri.h>
#include <falcon/lineardict.h>

#include "core_module.h"
#include <falcon/eng_messages.h>
#include <falcon/crobject.h>

namespace Falcon {
namespace core {

CoreObject* UriObject::factory( const CoreClass *me, void *uri, bool )
{
   if ( uri != 0 )
      return new UriObject( me, *static_cast<URI*>( uri ) );
   else
      return new UriObject( me );
}

/*# @class URI
   @brief Interface to RFC3986 Universal Resource Indicator.
   @optparam path The URI that will be used as initial data.
   @optparam decode True if the path is URI encoded, and must be decoded (default).
   @raise ParamError in case the inital URI is malformed.

   This class offers an object oriented interface to access
   URI elements.

   Setting the properties in this class immediately reflects on the
   related fields; for example setting the value of the @b uri
   property causes a complete re-parse of the item; setting a field
   as the query string will cause the uri to change.

   Each update is subject to RFC3986 compliance checks, and will raise
   a ParseError if conformance of the URI object is broken.

   @prop scheme URI scheme.
   @prop userInfo User, password or account specification preceding '\@' host.
   @prop host Host specificator.
   @prop port Optional port specificator (following the host after a ':').
   @prop path Path specificator.
   @prop query Query string in the URI.
   @prop fragment Fragment string in the uri (following path and query after a '#').
   @prop uri Complete URI.
*/


UriObject::UriObject( const UriObject &other ):
   CoreObject( other ),
   m_uri( other.uri() )
{
}

UriObject::~UriObject()
{
}

UriObject *UriObject::clone() const
{
   return new UriObject( *this );
}


bool UriObject::getProperty( const String &prop, Item &value ) const
{
   if ( prop == "scheme" )
   {
      value = new CoreString( uri().scheme() );
   }
   else if ( prop == "userInfo" )
   {
      value = new CoreString( uri().userInfo() );
   }
   else if ( prop == "path" )
   {
      value = new CoreString( uri().path() );
   }
   else if ( prop == "host" )
   {
      value = new CoreString( uri().host() );
   }
   else if ( prop == "port" )
   {
      value = new CoreString( uri().port() );
   }
   else if ( prop == "query" )
   {
      value = new CoreString( uri().query() );
   }
   else if ( prop == "fragment" )
   {
      value = new CoreString( uri().fragment() );
   }
   else if ( prop == "uri" )
   {
      value = new CoreString( uri().get(true) );
   }
   else
   {
      return defaultProperty( prop, value );
   }

   return true;
}


bool UriObject::setProperty( const String &prop, const Item &value )
{
   if ( prop == "scheme" )
   {
      if ( ! value.isString() ) goto complain;
      uri().scheme( *value.asString() );
   }
   else if ( prop == "userInfo" )
   {
      if ( ! value.isString() ) goto complain;
      uri().userInfo( *value.asString() );
   }
   else if ( prop == "path" )
   {
      if ( ! value.isString() ) goto complain;
      uri().path( *value.asString() );
   }
   else if ( prop == "host" )
   {
      if ( ! value.isString() ) goto complain;
      uri().host( *value.asString() );
   }
   else if ( prop == "port" )
   {
      if ( ! value.isString() ) goto complain;
      uri().port( *value.asString() );
   }
   else if ( prop == "query" )
   {
      if ( ! value.isString() ) goto complain;
      uri().query( *value.asString() );
   }
   else if ( prop == "fragment" )
   {
      if ( ! value.isString() ) goto complain;
      uri().fragment( *value.asString() );
   }
   else if ( prop == "uri" )
   {
      if ( ! value.isString() ) goto complain;
      uri().parse( *value.asString() );
   }
   else
      return false;

   if ( ! uri().isValid() )
   {
      VMachine* vm = VMachine::getCurrent();
      throw new AccessError( ErrorParam( e_param_range, __LINE__)
         .origin( e_orig_runtime )
         .extra( vm != 0 ? vm->moduleString( rtl_invalid_uri ) : "" ) );
   }

   return true;

   complain:
      throw new AccessError( ErrorParam( e_param_type, __LINE__)
         .origin( e_orig_runtime )
         .extra( "S" ) );
}


FALCON_FUNC  URI_init ( ::Falcon::VMachine *vm )
{
   Item *p0 = vm->param(0);
   Item *i_parse = vm->param(1);

   // nothing to do
   if ( p0 == 0 )
      return;

   UriObject *self = dyncast<UriObject*>(vm->self().asObject());

   // take the URI generated by the factory (can be empty).
   if ( ! p0->isString() )
   {
      throw new ParamError( ErrorParam( e_inv_params, __LINE__ ).
         origin( e_orig_runtime ).extra( "[S]" ) );
   }
   else {
      self->uri().parse( *p0->asString(), false, (i_parse == 0 || i_parse->isTrue()) );
      if ( !self->uri().isValid() )
      {
         throw new ParamError( ErrorParam( e_inv_params ).
            origin( e_orig_runtime ).
            extra( vm->moduleString( rtl_invalid_uri ) ) );
      }
   }
}



/*# @method encode URI
   @brief Encode a string to URL encoding (static).
   @param string The string to be encoded.
   @return the URL/URI encoded string.
*/

FALCON_FUNC  URI_encode ( ::Falcon::VMachine *vm )
{
   Item *p0 = vm->param(0);

   if ( ( p0 == 0 ) || ( ! p0->isString() ) )
   {
      throw new ParamError( ErrorParam( e_inv_params, __LINE__ ).
         origin( e_orig_runtime ).extra( "S" ) );
      return;
   }

   CoreString *str = new CoreString;
   URI::URLEncode( *p0->asString(), *str );
   vm->retval( str );
}

/*# @method decode URI
   @brief Decode a string to from URL encoding (static).
   @param enc_string The URI/URL encoded string.
   @return The decoded string.
   @raise ParamError if the string is not a valid URI/URL encoded string.
*/
FALCON_FUNC  URI_decode ( ::Falcon::VMachine *vm )
{
   Item *p0 = vm->param(0);

   if ( ( p0 == 0 ) || ( ! p0->isString() ) )
   {
      throw new ParamError( ErrorParam( e_inv_params, __LINE__ ).
         origin( e_orig_runtime ).extra( "S" ) );
      return;
   }

   CoreString *str = new CoreString;
   if ( ! URI::URLDecode( *p0->asString(), *str ) )
   {
      throw new ParamError( ErrorParam( e_inv_params, __LINE__ ).
         origin( e_orig_runtime ).extra( vm->moduleString( rtl_invalid_uri ) ) );
      return;
   }

   vm->retval( str );
}


/*# @method toString URI
   @brief Returns a string representing this URI.
   @return An URI encoded string containing a valid URI.

   Renders the URI into a valid string representation.

   @see URI.uri
*/
FALCON_FUNC  URI_toString ( ::Falcon::VMachine *vm )
{
   UriObject *self = dyncast<UriObject*>( vm->self().asObject() );
   URI &uri = self->uri();

   vm->retval( new CoreString( uri.get( true ) ) );
}


/*# @method getFields URI
   @brief Returns fields contained in the query element into a dictionary.
   @return The fields as a dictionary of nil if the query part contains no element.
   @raise ParamError if the string is not a valid URI/URL encoded string.
*/
FALCON_FUNC  URI_getFields ( ::Falcon::VMachine *vm )
{
   UriObject *self = dyncast<UriObject*>( vm->self().asObject() );
   URI &uri = self->uri();

   if ( uri.query().size() == 0 )
   {
      vm->retnil();
      return;
   }

   if( uri.fieldCount() == 0 )
   {
      // we have a query but no fields; this means we still have to parse it.
      if ( ! uri.parseQuery( true ) )
      {
         // todo: better signalation
         throw new ParamError( ErrorParam( e_inv_params, __LINE__ ).
            origin( e_orig_runtime ).extra( vm->moduleString( rtl_invalid_uri ) ) );
         return;
      }

      // really nothing to parse?
      if ( uri.fieldCount() == 0 )
      {
         vm->retnil();
         return;
      }
   }

   // ok, build our dictionary
   uint32 count = uri.fieldCount();
   CoreDict *dict = new CoreDict( new LinearDict( count ) );
   CoreString *key = new CoreString;
   CoreString *value = new CoreString;
   uri.firstField( *key, *value );
   count--;
   dict->put( key, value );
   while( count > 0 )
   {
      key = new CoreString;
      value = new CoreString;
      uri.nextField( *key, *value );
      count --;
      dict->put( key, value );
   }

   vm->retval( dict );
}

/*# @method setFields URI
   @brief Sets query fields for this uri.
   @param fields A dictionary of fields or nil to clear the query.
   @raise ParamError if the input dictionary contains non-string values.
*/
FALCON_FUNC  URI_setFields ( ::Falcon::VMachine *vm )
{
   UriObject *self = dyncast<UriObject*>( vm->self().asObject() );
   URI &uri = self->uri();

   Item *p0 = vm->param(0);

   if ( ( p0 == 0 ) || ( ! p0->isDict() ) )
   {
      throw new ParamError( ErrorParam( e_inv_params, __LINE__ ).
         origin( e_orig_runtime ).extra( "S" ) );
      return;
   }

   CoreDict *dict = p0->asDict();
   Iterator iter( &dict->items() );

   while( iter.hasCurrent() )
   {
      if ( ( !iter.getCurrentKey().isString()) || (! iter.getCurrent().isString() ) )
      {
         throw new ParamError( ErrorParam( e_inv_params, __LINE__ ).
            origin( e_orig_runtime ).extra( "S" ) );
         return;
      }

      uri.setField( *iter.getCurrentKey().asString(), *iter.getCurrent().asString() );
      iter.next();
   }

   uri.makeQuery();
}

}
}

/* end of uri_ext.cpp */
