/* -*- c-file-style: "GNU" -*- */
/*
 * Copyright  CNRS, INRIA, Universit Bordeaux 1
 * See COPYING in top-level directory.
 */

#ifndef __EZTRACE_H__
#define __EZTRACE_H__

/* don't forget this line, otherwise, FUT_DO_PROBE0 is defined to (void) 0 */
#define CONFIG_FUT

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <stddef.h>
#include <stdint.h>
#include <dlfcn.h>
#ifdef USE_GETTID
  #include <unistd.h>
  #include <sys/syscall.h>	/* For SYS_xxx definitions */
#else
  #include <pthread.h>
#endif	/* USE_GETTID */

#ifdef EXTERNAL_FXT
#include "fxt/fxt.h"
#include "fxt/fut.h"
#else
#include "fxt.h"
#include "fut.h"
#endif

#define EZTRACE_API_VERSION 0x00000700

/* start the recording of events */
void eztrace_start();
/* stop recording events and write the trace to the disk */
void eztrace_stop();
/* change the trace filename */
void eztrace_set_filename(char* name);

void eztrace_code0(uint32_t code);
void eztrace_code1(uint32_t code, uint64_t arg1);
void eztrace_code2(uint32_t code, uint64_t arg1, uint64_t arg2);
void eztrace_code3(uint32_t code, uint64_t arg1, uint64_t arg2, uint64_t arg3);
void eztrace_code4(uint32_t code, uint64_t arg1, uint64_t arg2, uint64_t arg3, uint64_t arg4);
void eztrace_code5(uint32_t code, uint64_t arg1, uint64_t arg2, uint64_t arg3, uint64_t arg4, uint64_t arg5);

typedef void (*eztrace_function_t) (void);

/* register a callback to be called when the application calls
 * eztrace_start().
 * This function is only usefull when AUTOSTART is disabled
 */
void eztrace_register_init_routine(eztrace_function_t init_func);

#ifdef USE_PAPI
extern int record_hardware_counter;
extern void papi_record() __attribute__((weak));
#define RECORD_HW_COUNTERS()			\
  {						\
    if(record_hardware_counter && papi_record)	\
      papi_record();				\
  }

#else
#define RECORD_HW_COUNTERS() (void)0
#endif

/* current thread id */
#ifdef USE_GETTID
#define CUR_TID syscall(SYS_gettid)
#else
#define CUR_TID pthread_self()
#endif

#ifdef TID_RECORDING_ENABLED
/* create an event without any data */
#define EZTRACE_EVENT0(code) FUT_DO_PROBE0(code)
/* create an event with one value */
#define EZTRACE_EVENT1(code, arg1) FUT_DO_PROBE1(code, arg1)
/* create an event with two value */
#define EZTRACE_EVENT2(code, arg1, arg2) FUT_DO_PROBE2(code, arg1, arg2)
/* etc. */
#define EZTRACE_EVENT3(code, arg1, arg2, arg3) FUT_DO_PROBE3(code, arg1, arg2, arg3)
#define EZTRACE_EVENT4(code, arg1, arg2, arg3, arg4) FUT_DO_PROBE4(code, arg1, arg2, arg3, arg4)
#define EZTRACE_EVENT5(code, arg1, arg2, arg3, arg4, arg5) FUT_DO_PROBE5(code, arg1, arg2, arg3, arg4, arg5)

#else

/* create an event without any data */
#define EZTRACE_EVENT0(code) FUT_DO_PROBE1(code, CUR_TID)
/* create an event with one value */
#define EZTRACE_EVENT1(code, arg1) FUT_DO_PROBE2(code, CUR_TID, arg1)
/* create an event with two value */
#define EZTRACE_EVENT2(code, arg1, arg2) FUT_DO_PROBE3(code, CUR_TID, arg1, arg2)
/* etc. */
#define EZTRACE_EVENT3(code, arg1, arg2, arg3) FUT_DO_PROBE4(code, CUR_TID, arg1, arg2, arg3)
#define EZTRACE_EVENT4(code, arg1, arg2, arg3, arg4) FUT_DO_PROBE5(code, CUR_TID, arg1, arg2, arg3, arg4)
#define EZTRACE_EVENT5(code, arg1, arg2, arg3, arg4, arg5) FUT_DO_PROBE6(code, CUR_TID, arg1, arg2, arg3, arg4, arg5)

#endif

/* check wether dlsym returned successfully */
#define  TREAT_ERROR()				\
  do {						\
    char * error;				\
    if ((error = dlerror()) != NULL)  {		\
      fputs(error, stderr);			\
      return;					\
    }						\
  }while(0)

/* interception function func and store its previous value into var */
#define INTERCEPT(func, var)					\
  do {								\
    if(var) break;						\
    void *__handle = RTLD_NEXT;					\
    var = (typeof(var)) (uintptr_t) dlsym(__handle, func);	\
    TREAT_ERROR();						\
  } while(0)


/* return the offset of the field MEMBER in a structure TYPE */
#define ezt_offset_of(TYPE, MEMBER) ((size_t) &((TYPE *)0)->MEMBER)

/* Find the global structure's address
 * It needs :
 * - ptr: address of intern field
 * - type: type of the global structure
 * - member: name of the intern field of the global structure
 */
#define ezt_container_of(ptr, type, member)			\
  ((type *)((char *)(__typeof__ (&((type *)0)->member))(ptr)-	\
	    ezt_offset_of(type,member)))


/* record an event (code=FUT_CALLING_FUNCTION) with the calling function name */
void record_backtrace();

/* return 1 if an event is being recorded. */
int recursion_shield_on();

void set_recursion_shield_on();
void set_recursion_shield_off();

/* avoid infinite recursion */
#define EZTRACE_PROTECT if(! recursion_shield_on())

/* Set the recursion shield */
#define EZTRACE_PROTECT_ON() set_recursion_shield_on()
#define EZTRACE_PROTECT_OFF() set_recursion_shield_off()

#endif	/* __EZTRACE_H__ */
