/* 
 * conn.h
 * see ezbounce.cpp and LICENCE for licence details  
 */

#ifndef __conn_h
#define __conn_h

class conn;

#include "ezbounce.h"

#define ezbounce_guy "ezbounce!srv"

#include <time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>
#include "dynbuff.h"
#include "server.h"
#include "general.h"
#include "config.h"
#include "ruleset.h"
#include "objset.h"
#include "dcc.h"
#include "linkedlist.h"
#include "logfile.h"
#include "socket.h"

enum {
     NICKED                   =     0x8,
     USERED                   =    0x10,
     PWED                     =    0x20,
     BOUNCED                  =    0x40,
     ADMINED                  =    0x80,
     CONNECTING               =   0x100,
     DETACHED                 =   0x200,
     REATTACHING              =   0x400,
     FROM_RULESETS_REGISTERED =   0x800,
     TO_RULESETS_REGISTERED   =  0x1000,
     GOTSERVINFO              =  0x2000,
     GOTSERVINFO2             =  0x4000,
     REGISTERED               = (NICKED | USERED | PWED),
};


class conn 
{
private:
    class psock : public pollsocket
    {
    public:
        psock(conn * owner, int f)  
            : pollsocket::pollsocket(f, POLLIN)
        {
            this->owner = owner;
        }
    protected:
        conn * owner;
        virtual int event_handler(const struct pollfd * pfd);
        friend class conn;
    };
    friend class psock;

private:
    psock * client;
    psock * server;
    dynbuff * client_buff;                     // storage place for data from client
    dynbuff * server_buff;                     // storage place for data from serv
    
    int stat, log_options;
    unsigned short failed_admins, failed_passwords;
    unsigned long id;                         
    time_t connect_time, last_recved;         // time at which connected and last data recved
                  
    struct sockaddr_in local_saddr;           /* sin_addr = where he's from sin_port: what port he connected on */
    struct sockaddr_in client_saddr;          // where client is coming from
    struct sockaddr_in serv_saddr;            // where client is connected to
    obj_set<rule_set> rules;                  // which rule sets this obj belongs to
    obj_set<char> * loglist;                  /* List of log file search results and such */
    logfile * log;                            /* Detached logging facilities */

    class uinfo 
    {
    public:
        char * irc_nick;
        char * fulladdr;                        /* Full address of user */
        char * usercmd;                         // USER command text supplied by user
        char * server;                          // server user is connecting to
        char * ircpass;                         // optional pw supplied by user to be (sent to server on connect)
        char * detachpass;
        char * autopass;                        /* Auto-Detach pass */
        char * fake_ident;                      
        char * serverversion;                   
        char * servercreated;
        char * servermodes;
        char * server005;               
        
    
        struct in_addr iface;                   // interface to bind to
        u_short port;
        uinfo();
        ~uinfo();
    } user;
    
    /* Information used during reattach process */
    class rinfo 
    {
    public:
        char * server;
        char * nick;
        char * channels;
        char * ident;
        char * hostname;
        char * umode;
        conn * target;
      
        rinfo()
        {
            server = nick = channels = hostname = ident = umode = NULL;
            target = NULL;
        }
        ~rinfo()
        {
            delete[] server;
            delete[] nick;
            delete[] channels;
            delete[] hostname;
            delete[] ident;
            delete[] umode;
        }
    } * reattach;

    conn *pNext;
    static conn *pFirst;
    static unsigned num_active;               // how many of us are working right now
    static unsigned current_id;               // next free id

public:
    conn(int fd);
    ~conn();         

    static conn * find_by_id(unsigned);

    static conn *first()
    {
        return pFirst;
    }
    conn *next() const
    {
        return pNext;
    }
    
    static unsigned get_num_active(void)  
    {
        return num_active;
    }
    
    bool dead() const
    { 
        return (!server && !client); 
    }
    unsigned get_id() const
    { 
        return id; 
    }
    bool registered() const
    { 
        return ((stat & PWED) && (stat & NICKED) && (stat & USERED)); 
    }
    const char * addr(void) const
    {
        return user.fulladdr;
    }

    int check_server(int);
    int check_client(int); 
    
    void kill(const char *, const char *);
    bool check_idle(time_t);
    static void do_timers(time_t);

    
    static void broadcast(const char *);

protected:
    int  parse(void);
    int  parse_incoming(void);
    int  init_rulesets(void);
    int  unregister_rulesets(char);

    int  send_client(const char *, ...);
    int  send_client_multiline(const char *);
    
    int  detach(const char *, const char *);
    int  reattach_start(const char *, int);
    int  reattach_complete(void);
    int  reattach_cancel(void);
    bool do_auto_detach(void);
    
    int  do_ctcp(bool incoming, const char * ctcp, const char * source, 
                  const char * target, const char * args);
    int  do_connect(const char *, unsigned short port, const char * pass = NULL);
    bool can_connect(const char *, unsigned short);    
    bool copy_fake_ident(const char * = 0);
    
    
    int  on_server_disconnect(int = 0);
    int  on_client_disconnect(int = 0);
    int  on_server_connect(int);
    int  on_client_connect(int);

private:
    struct command_set 
    {
        const char * cmd;           /* Command string */
        int id,                     /* A unique id for this command */
            req_flags,              /* Flags required for this command to pass. (0 = None) */
            bad_flags;              /* Flags that must not be set for this command to pass */
        const char * desc;          /* Description of this command */
    };
    static const struct command_set user_commands[];
    static const struct command_set from_server[];

protected:
    int  do_command(const char *, const char *);
    int  do_admin_command(const struct command_set *, const char *);
    int  do_log_command(const char *, const char *);
    int  do_conn_command(const char *);
    const struct command_set * match_command(int, const char * cmd, bool simple = 0);

    int  show_motd(void);
    dcc *dcc_send_file(const char *, const char * = 0, struct sockaddr_in * = 0, bool = 0);
    


private:
    linkedlist<dcc_list_t> * my_dccs;
    friend class dcc_list_t;

public:
    static int  do_dcc(dcc_list_t * , int , struct sockaddr_in * );

protected:
    static void kill_dccs(void);
    void        disown_dccs(conn *);
    
  
    /* To help with use of flags and such */
    inline int checkf(int flag) const
    {
        return (stat & flag);
    }
    inline int clearf(int flag)
    {
        return (stat &= ~(flag));
    }
    inline int setf(int flag)
    {
        return (stat |= flag);
    }
};      


/*
 * Now.. I don't know if enums will always have the same value in each
 * file this is #include'd in.
 * It works properly here.
 * I guess bug reports of 'it just doesn't work' will show that
 * it ought to be changed ;)
 */
enum  {
     CONN_NOWAITING = -50,                    // no data to read
     CONN_DISCONNECTED,                       // sock got disconnected during operation
     CONN_FAILED,                             // operation didn't work
     CONN_KILLME = -30,                       // fatal error, destroy this object   
     CONN_NOTCONNECTED,      
     CONN_FULL, 
     CONN_NOLINK, 
     CONN_LINK_ACTIVATED,                        
     CONN_LINK_DEACTIVATED
};

#endif
