/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/*********************************************************************
*
*     file:        method1.c
*
*     contents:  quantities for vertices and edges
*/

#include "include.h"


/*********************************************************************

                    vertex_scalar_integral method

*********************************************************************/

/*********************************************************************
*
* function: vertex_scalar_integral()
*
* purpose:  method value
*
*/

REAL vertex_scalar_integral(v_info)
struct qinfo *v_info;
{ REAL value;
  value = eval(METH_INST[v_info->method].expr[0],v_info->x[0],v_info->id);
  return value;
}

/*********************************************************************
*
* function: vertex_scalar_integral_grad()
*
* purpose:  method gradient
*
*/

REAL vertex_scalar_integral_grad(v_info)
struct qinfo *v_info;
{ REAL value = 0.0;

  eval_all(METH_INST[v_info->method].expr[0],v_info->x[0],SDIM,&value,
     v_info->grad[0],v_info->id);

  return value;
}

/*********************************************************************
*
* function: vertex_scalar_integral_hess()
*
* purpose:  method gradient and hessian
*
*/

REAL vertex_scalar_integral_hess(v_info)
struct qinfo *v_info;
{ 
  REAL value = 0.0;

  eval_second(METH_INST[v_info->method].expr[0],v_info->x[0],SDIM,&value,
        v_info->grad[0], v_info->hess[0][0],v_info->id);
  return value;
}


/*********************************************************************

                                Edge length quantity

*********************************************************************/

/*********************************************************************
*
*  function: q_edge_tension_init()
*
*  purpose:  initialize web.total_area to 0.
*/

void q_edge_tension_init(mode,mi)
int mode; /* energy or gradient */
struct method_instance *mi;
{
  if ( everything_quantities_flag && (mode==METHOD_VALUE) ) 
    web.total_area = 0.0;
}

/*********************************************************************
*
*  function: q_edge_tension_value()
*
*  purpose:  General quantity value of edge tension.
*/

REAL q_edge_tension_value(e_info)
struct qinfo *e_info;
{ REAL energy;
  if ( web.modeltype == QUADRATIC ) return edge_length_q_value(e_info);
  if ( web.modeltype == LAGRANGE ) 
      return lagrange_edge_tension_value(e_info);
  energy = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  set_edge_length(e_info->id,energy);
  if ( quantities_only_flag )
  { 
#ifdef SHARED_MEMORY
     if ( nprocs > 1 ) 
      proc_total_area[m_get_myid()] += energy;
     else
#endif
     web.total_area += energy;
  }
  if ( basic_gen_methods[METH_INST[e_info->method].gen_method].spec_flags 
                & SPEC_USE_DENSITY )
      energy *= get_edge_density(e_info->id);
  return energy;
}


/*********************************************************************
*
*  function: q_edge_tension_gradient()
*
*  purpose:  General quantity value and gradient of edge tension.
*/

REAL q_edge_tension_gradient(e_info)
struct qinfo *e_info;
{ REAL energy;
  REAL fudge;
  int j;

  if ( web.modeltype == QUADRATIC ) 
     return edge_length_q_grad(e_info);
  if ( web.modeltype == LAGRANGE ) 
      return lagrange_edge_tension_grad(e_info);
  energy = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  fudge = 1/energy;
  if ( basic_gen_methods[METH_INST[e_info->method].gen_method].spec_flags 
                & SPEC_USE_DENSITY )
  {  REAL density = get_edge_density(e_info->id);
     energy *= density; fudge *= density;
  }
  for ( j = 0 ; j < SDIM ; j++ ) 
     { e_info->grad[0][j] = -e_info->sides[0][0][j]*fudge;
       e_info->grad[1][j] =  e_info->sides[0][0][j]*fudge;
     }
  return energy;
}

/*********************************************************************
*
*  function: q_edge_tension_hessian()
*
*  purpose:  General quantity value, gradient, and hessian of edge length.
*
*  Remark to programmers: e_info->hess[m][n][i][j] is the entry for
*     coordinate i of vertex m of the edge and coordinate j of vertex n.
*/

REAL q_edge_tension_hessian(e_info)
struct qinfo *e_info;
{ REAL energy;
  int i,j;
  REAL e1,e3,ss;
  REAL fudge,len;

  if ( web.modeltype == QUADRATIC ) return edge_length_q_hess(e_info);
  if ( web.modeltype == LAGRANGE )  return lagrange_edge_tension_hess(e_info);

  energy = len = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  fudge = 1/len;
  if ( basic_gen_methods[METH_INST[e_info->method].gen_method].spec_flags 
                & SPEC_USE_DENSITY )
  {  REAL density = get_edge_density(e_info->id);
     energy *= density; fudge *= density;
  }
  for ( j = 0 ; j < SDIM ; j++ ) 
     { e_info->grad[0][j] = -e_info->sides[0][0][j]*fudge;
       e_info->grad[1][j] =  e_info->sides[0][0][j]*fudge;
     }
  e3 = fudge/(len*len);
  e1 = fudge;
  for ( i = 0 ; i < SDIM ; i++ ) 
    for ( j = 0 ; j < SDIM ; j++ ) 
    { ss = e_info->sides[0][0][i]*e_info->sides[0][0][j]*e3;
      e_info->hess[0][0][i][j] = -ss;
      e_info->hess[1][1][i][j] = -ss;
      e_info->hess[0][1][i][j] =  ss;
      e_info->hess[1][0][i][j] =  ss;
    }
  for ( i = 0 ; i < SDIM ; i++ ) 
    { e_info->hess[0][0][i][i] += e1;
      e_info->hess[1][1][i][i] += e1;
      e_info->hess[0][1][i][i] -= e1; 
      e_info->hess[1][0][i][i] -= e1;
    }
  return energy;
}


/*********************************************************************

         quadratic edge_length method

*********************************************************************/

/*********************************************************************
*
* function: edge_length_q_value()
*
* purpose:  method value
*
*/

REAL edge_length_q_value(e_info)
struct qinfo *e_info;
{ int j,k,m;
  REAL value = 0.0;
  REAL tang[MAXCOORD];

  for ( m = 0 ; m < gauss1D_num ; m++ )
  { for ( j = 0 ; j < SDIM ; j ++ )
    { tang[j] = 0.0;
      for ( k = 0 ; k < edge_ctrl ; k++ )
         tang[j] += gauss1polyd[k][m]*e_info->x[k][j];
    }
    value += gauss1Dwt[m]*sqrt(SDIM_dot(tang,tang));
  }
  set_edge_length(e_info->id,value); 
  if ( quantities_only_flag )
  {
#ifdef SHARED_MEMORY
     if ( nprocs > 1 ) 
      proc_total_area[m_get_myid()] += value;
     else
#endif
     web.total_area += value;
  }
  if ( basic_gen_methods[METH_INST[e_info->method].gen_method].spec_flags 
                & SPEC_USE_DENSITY )
      value *= get_edge_density(e_info->id);
  return value;
}

/*********************************************************************
*
* function: edge_length_integral_q_grad()
*
* purpose:  method gradient
*
*/

REAL edge_length_q_grad(e_info)
struct qinfo *e_info;
{ int m,k,j;
  REAL value = 0.0;
  REAL len,fudge;
  REAL tang[MAXCOORD];
  REAL density;

  if ( basic_gen_methods[METH_INST[e_info->method].gen_method].spec_flags 
                & SPEC_USE_DENSITY )
     density = get_edge_density(e_info->id);
  else density = 1.0;

  for ( j = 0 ; j < SDIM ; j++ )
     for ( m = 0 ; m < edge_ctrl ; m++ )
        e_info->grad[m][j] = 0.0;
  for ( m = 0 ; m < gauss1D_num ; m++ )
  { for ( j = 0 ; j < SDIM ; j ++ )
    { tang[j] = 0.0;
      for ( k = 0 ; k < edge_ctrl ; k++ )
          tang[j] += gauss1polyd[k][m]*e_info->x[k][j];
    }
    len = sqrt(SDIM_dot(tang,tang));
    if ( len == 0.0 ) continue;
    value += gauss1Dwt[m]*len;
    fudge = density*gauss1Dwt[m]/len;
    for ( k = 0 ; k < edge_ctrl ; k++ )
      for ( j = 0 ; j < SDIM ; j++ )
         e_info->grad[k][j] += fudge*tang[j]*gauss1polyd[k][m];
  }

  return density*value;
}

/*********************************************************************
*
* function: edge_length_q_hess()
*
* purpose:  method gradient and hessian
*
*/

REAL edge_length_q_hess(e_info)
struct qinfo *e_info;
{ int m,j,jj,k,kk;
  REAL value = 0.0;
  REAL len,density,fudge;
  REAL sumgrad[2][MAXCOORD];
  REAL sumhess[2][2][MAXCOORD][MAXCOORD];
  REAL tang[MAXCOORD];

  if ( basic_gen_methods[METH_INST[e_info->method].gen_method].spec_flags 
                & SPEC_USE_DENSITY )
     density = get_edge_density(e_info->id);
  else density = 1.0;

  /* derivatives of gaussian sum part */
  memset((char*)sumgrad,0,sizeof(sumgrad));
  memset((char*)sumhess,0,sizeof(sumhess));
  for ( m = 0 ; m < gauss1D_num ; m++ )
  { for ( j = 0 ; j < SDIM ; j ++ )
    { tang[j] = 0.0;
      for ( k = 0 ; k < edge_ctrl ; k++ )
          tang[j] += gauss1polyd[k][m]*e_info->x[k][j];
    }
    len = sqrt(SDIM_dot(tang,tang));
    if ( len == 0.0 ) continue;
    value += gauss1Dwt[m]*len;
    fudge = density*gauss1Dwt[m]/len;
    for ( k = 0 ; k < edge_ctrl ; k++ )
      for ( j = 0 ; j < SDIM ; j++ )
         e_info->grad[k][j] += fudge*tang[j]*gauss1polyd[k][m];
    for ( k = 0 ; k < edge_ctrl ; k++ )
      for ( kk = 0 ; kk < edge_ctrl ; kk++ )
         for ( j = 0 ; j < SDIM ; j++ )
            for ( jj = 0 ; jj < SDIM ; jj++ )
              e_info->hess[k][kk][j][jj] += fudge*
            ( - tang[j]*tang[jj]*gauss1polyd[k][m]*gauss1polyd[kk][m]/len/len
            + ((j==jj)? gauss1polyd[k][m]*gauss1polyd[kk][m] : 0.0));
  }

  return density*value;
}


/*********************************************************************

                    edge_scalar_integral method

*********************************************************************/

/*********************************************************************
*
* function: edge_scalar_integral()
*
* purpose:  method value
*
*/

REAL edge_scalar_integral(e_info)
struct qinfo *e_info;
{ int m;
  REAL value = 0.0;

  if ( web.modeltype == QUADRATIC ) return edge_scalar_integral_q(e_info);
  if ( web.modeltype == LAGRANGE ) return edge_scalar_integral_lagr(e_info);

  for ( m = 0 ; m < gauss1D_num ; m++ )
  {  value += gauss1Dwt[m]*eval(METH_INST[e_info->method].expr[0],e_info->gauss_pt[m],
                                            e_info->id);
  }
  value *= sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  return value;
}

/*********************************************************************
*
* function: edge_scalar_integral_grad()
*
* purpose:  method gradient
*
*/

REAL edge_scalar_integral_grad(e_info)
struct qinfo *e_info;
{ int m,j;
  REAL value = 0.0;
  REAL len,val;
  REAL derivs[MAXCOORD];

  if ( web.modeltype == QUADRATIC ) return edge_scalar_integral_q_grad(e_info);
  if ( web.modeltype == LAGRANGE ) return edge_scalar_integral_lagr_grad(e_info);

  for ( j = 0 ; j < SDIM ; j++ ) 
     for ( m = 0 ; m < 2 ; m++ )
        e_info->grad[m][j] = 0.0;
  len = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  for ( m = 0 ; m < gauss1D_num ; m++ )
  { eval_all(METH_INST[e_info->method].expr[0],e_info->gauss_pt[m],SDIM,&val,
                                                   derivs,e_info->id);
    value += gauss1Dwt[m]*val;
    for ( j = 0 ; j < SDIM ; j++ )
    { e_info->grad[0][j] += gauss1Dwt[m]*gauss1poly[0][m]*derivs[j]*len;
      e_info->grad[1][j] += gauss1Dwt[m]*gauss1poly[1][m]*derivs[j]*len;
    }
  }
  for ( j = 0 ; j < SDIM ; j++ )
  { e_info->grad[0][j] -= value*e_info->sides[0][0][j]/len;
    e_info->grad[1][j] += value*e_info->sides[0][0][j]/len;
  }

  return len*value;
}

/*********************************************************************
*
* function: edge_scalar_integral_hess()
*
* purpose:  method gradient and hessian
*
*/

REAL edge_scalar_integral_hess(e_info)
struct qinfo *e_info;
{ int m,j,k,i;
  REAL value = 0.0;
  REAL len,sum,val;
  REAL derivs[MAXCOORD];
  REAL lengrad[2][MAXCOORD],sumgrad[2][MAXCOORD];
  REAL lenhess[2][2][MAXCOORD][MAXCOORD],sumhess[2][2][MAXCOORD][MAXCOORD];
  MAT2D(second,MAXCOORD,MAXCOORD);

  if ( web.modeltype == QUADRATIC ) return edge_scalar_integral_q_hess(e_info);
  if ( web.modeltype == LAGRANGE ) return edge_scalar_integral_lagr_hess(e_info);

  len = SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]);
  if ( len <= 0.0 )
  { 
     return 0.0;
  }
  len = sqrt(len);

  /* derivatives of gaussian sum part */
  sum = 0.0;
  memset((char*)sumgrad,0,sizeof(sumgrad));
  memset((char*)sumhess,0,sizeof(sumhess));
  for ( m = 0 ; m < gauss1D_num ; m++ )
  { eval_second(METH_INST[e_info->method].expr[0],e_info->gauss_pt[m],SDIM,&val,
                                                 derivs,second,e_info->id);
    sum += gauss1Dwt[m]*val;
    for ( j = 0 ; j < SDIM ; j++ )
    { sumgrad[0][j] += gauss1Dwt[m]*gauss1poly[0][m]*derivs[j];
      sumgrad[1][j] += gauss1Dwt[m]*gauss1poly[1][m]*derivs[j];
    }
    for ( j = 0 ; j < SDIM ; j++ )
      for ( k = 0 ; k < SDIM ; k++ )
      { sumhess[0][0][j][k] += gauss1Dwt[m]*gauss1poly[0][m]*gauss1poly[0][m]
                                              *second[j][k];
        sumhess[0][1][j][k] += gauss1Dwt[m]*gauss1poly[0][m]*gauss1poly[1][m]
                                              *second[j][k];
        sumhess[1][0][j][k] += gauss1Dwt[m]*gauss1poly[1][m]*gauss1poly[0][m]
                                              *second[j][k];
        sumhess[1][1][j][k] += gauss1Dwt[m]*gauss1poly[1][m]*gauss1poly[1][m]
                                              *second[j][k];
      }
  }

  /* derivatives of length part */
  for ( j = 0 ; j < SDIM ; j++ )
  { lengrad[0][j] = -e_info->sides[0][0][j]/len;
    lengrad[1][j] = e_info->sides[0][0][j]/len;
  }
  for ( j = 0 ; j < SDIM ; j++ )
    for ( k = 0 ; k < SDIM ; k++ )
      { val = -e_info->sides[0][0][j]*e_info->sides[0][0][k]/len/len/len;
        if ( j == k ) val += 1/len;
        lenhess[0][0][j][k] = lenhess[1][1][j][k] = val;
        lenhess[0][1][j][k] = lenhess[1][0][j][k] = -val;
      }

  /* final values */
  value = len*sum;
  for ( j = 0 ; j < SDIM ; j++ )
    for ( m = 0 ; m < 2 ; m++ )
      e_info->grad[m][j] = lengrad[m][j]*sum + len*sumgrad[m][j];
  for ( j = 0 ; j < SDIM ; j++ ) 
    for ( k = 0 ; k < SDIM ; k++ ) 
     for ( m = 0 ; m < 2 ; m++ )
      for ( i = 0 ; i < 2 ; i++ )
         e_info->hess[m][i][j][k] = lenhess[m][i][j][k]*sum
            + lengrad[m][j]*sumgrad[i][k] + sumgrad[m][j]*lengrad[i][k]
            + len*sumhess[m][i][j][k];
  return value;
}


/*********************************************************************

         quadratic edge_scalar_integral method

*********************************************************************/

/*********************************************************************
*
* function: edge_scalar_integral_q()
*
* purpose:  method value
*
*/

REAL edge_scalar_integral_q(e_info)
struct qinfo *e_info;
{ int j,k,m;
  REAL value = 0.0;
  REAL tang[MAXCOORD];

  for ( m = 0 ; m < gauss1D_num ; m++ )
  { for ( j = 0 ; j < SDIM ; j ++ )
      { tang[j] = 0.0;
        for ( k = 0 ; k < edge_ctrl ; k++ )
           tang[j] += gauss1polyd[k][m]*e_info->x[k][j];
      }
    value += gauss1Dwt[m]*eval(METH_INST[e_info->method].expr[0],e_info->gauss_pt[m],e_info->id)
         *sqrt(SDIM_dot(tang,tang));
  }
  return value;
}

/*********************************************************************
*
* function: edge_scalar_integral_q_grad()
*
* purpose:  method gradient
*
*/

REAL edge_scalar_integral_q_grad(e_info)
struct qinfo *e_info;
{ int m,k,j;
  REAL value = 0.0;
  REAL len,val;
  REAL derivs[MAXCOORD];
  REAL tang[MAXCOORD];

  for ( j = 0 ; j < SDIM ; j++ )
     for ( m = 0 ; m < edge_ctrl ; m++ )
        e_info->grad[m][j] = 0.0;
  for ( m = 0 ; m < gauss1D_num ; m++ )
  { eval_all(METH_INST[e_info->method].expr[0],e_info->gauss_pt[m],SDIM,&val,
                                                         derivs,e_info->id);
    for ( j = 0 ; j < SDIM ; j ++ )
    { tang[j] = 0.0;
      for ( k = 0 ; k < edge_ctrl ; k++ )
          tang[j] += gauss1polyd[k][m]*e_info->x[k][j];
    }
    len = sqrt(SDIM_dot(tang,tang));
    if ( len == 0.0 ) continue;
    value += gauss1Dwt[m]*val*len;
    for ( k = 0 ; k < edge_ctrl ; k++ )
      for ( j = 0 ; j < SDIM ; j++ )
         e_info->grad[k][j] += gauss1Dwt[m]*(gauss1poly[k][m]*derivs[j]*len
                                      + val*tang[j]/len*gauss1polyd[k][m]);
  }

  return value;
}

/*********************************************************************
*
* function: edge_scalar_integral_q_hess()
*
* purpose:  method gradient and hessian
*
*/

REAL edge_scalar_integral_q_hess(e_info)
struct qinfo *e_info;
{ int m,j,jj,k,kk;
  REAL value = 0.0;
  REAL len,val;
  REAL derivs[MAXCOORD];
  REAL sumgrad[2][MAXCOORD];
  REAL sumhess[2][2][MAXCOORD][MAXCOORD];
  MAT2D(second,MAXCOORD,MAXCOORD);
  REAL tang[MAXCOORD];

  /* derivatives of gaussian sum part */
  memset((char*)sumgrad,0,sizeof(sumgrad));
  memset((char*)sumhess,0,sizeof(sumhess));
  for ( m = 0 ; m < gauss1D_num ; m++ )
  { eval_second(METH_INST[e_info->method].expr[0],e_info->gauss_pt[m],SDIM,&val,
                                                    derivs,second,e_info->id);
    for ( j = 0 ; j < SDIM ; j ++ )
    { tang[j] = 0.0;
      for ( k = 0 ; k < edge_ctrl ; k++ )
          tang[j] += gauss1polyd[k][m]*e_info->x[k][j];
    }
    len = sqrt(SDIM_dot(tang,tang));
    if ( len == 0.0 ) continue;
    value += gauss1Dwt[m]*val*len;
    for ( k = 0 ; k < edge_ctrl ; k++ )
      for ( j = 0 ; j < SDIM ; j++ )
         e_info->grad[k][j] += gauss1Dwt[m]*(gauss1poly[k][m]*derivs[j]*len
                                      + val*tang[j]/len*gauss1polyd[k][m]);
    for ( k = 0 ; k < edge_ctrl ; k++ )
      for ( kk = 0 ; kk < edge_ctrl ; kk++ )
         for ( j = 0 ; j < SDIM ; j++ )
            for ( jj = 0 ; jj < SDIM ; jj++ )
              e_info->hess[k][kk][j][jj] += gauss1Dwt[m]*
     ( second[j][jj]*gauss1poly[k][m]*gauss1poly[kk][m]*len
     + derivs[j]*gauss1poly[k][m]*tang[jj]*gauss1polyd[kk][m]/len
     + derivs[jj]*gauss1poly[kk][m]*tang[j]*gauss1polyd[k][m]/len
     - val*tang[j]*tang[jj]*gauss1polyd[k][m]*gauss1polyd[kk][m]/len/len/len
     + ((j==jj)? val*gauss1polyd[k][m]*gauss1polyd[kk][m]/len : 0.0));
  }

  return value;
}


/*********************************************************************

         Lagrange edge_scalar_integral method

*********************************************************************/

/*********************************************************************
*
* function: edge_scalar_integral_lagr()
*
* purpose:  method value
*
*/

REAL edge_scalar_integral_lagr(e_info)
struct qinfo *e_info;
{ int m;
  REAL value = 0.0;
  struct gauss_lag *gl = &gauss_lagrange[1][web.gauss1D_order];

  for ( m = 0 ; m < gl->gnumpts ; m++ )
  { value += gl->gausswt[m]
         *eval(METH_INST[e_info->method].expr[0],e_info->gauss_pt[m],e_info->id)
         *sqrt(SDIM_dot(e_info->sides[m][0],e_info->sides[m][0]));
  }
  return value;
}

/*********************************************************************
*
* function: edge_scalar_integral_lagr_grad()
*
* purpose:  method gradient
*
*/

REAL edge_scalar_integral_lagr_grad(e_info)
struct qinfo *e_info;
{ int m,k,j;
  REAL value = 0.0;
  REAL len,val;
  REAL derivs[MAXCOORD];
  REAL *tang;
  struct gauss_lag *gl = &gauss_lagrange[1][web.gauss1D_order];

  for ( m = 0 ; m < gl->gnumpts ; m++ )
  { eval_all(METH_INST[e_info->method].expr[0],e_info->gauss_pt[m],SDIM,&val,
                                                         derivs,e_info->id);
    tang = e_info->sides[m][0];
    len = sqrt(SDIM_dot(tang,tang));
    if ( len == 0.0 ) continue;
    value += gl->gausswt[m]*val*len;
    for ( k = 0 ; k < gl->lagpts ; k++ )
      for ( j = 0 ; j < SDIM ; j++ )
         e_info->grad[k][j] += 
          gl->gausswt[m]*(gl->gpoly[m][k]*derivs[j]*len
              + val*tang[j]/len*gl->gpolypart[m][0][k]);
  }
  return value;
}

/*********************************************************************
*
* function: edge_scalar_integral_lagr_hess()
*
* purpose:  method gradient and hessian
*
*/

REAL edge_scalar_integral_lagr_hess(e_info)
struct qinfo *e_info;
{ int m,j,jj,k,kk;
  REAL value = 0.0;
  REAL len,val;
  REAL derivs[MAXCOORD];
  REAL sumgrad[2][MAXCOORD];
  REAL sumhess[2][2][MAXCOORD][MAXCOORD];
  MAT2D(second,MAXCOORD,MAXCOORD);
  REAL *tang;
  struct gauss_lag *gl = &gauss_lagrange[1][web.gauss1D_order];

  /* derivatives of gaussian sum part */
  memset((char*)sumgrad,0,sizeof(sumgrad));
  memset((char*)sumhess,0,sizeof(sumhess));
  for ( m = 0 ; m < gl->gnumpts ; m++ )
  { eval_second(METH_INST[e_info->method].expr[0],e_info->gauss_pt[m],SDIM,&val,
                                                     derivs,second,e_info->id);
    tang = e_info->sides[m][0];
    len = sqrt(SDIM_dot(tang,tang));
    if ( len == 0.0 ) continue;
    value += gl->gausswt[m]*val*len;
    for ( k = 0 ; k < gl->lagpts ; k++ )
      for ( j = 0 ; j < SDIM ; j++ )
         e_info->grad[k][j] += 
             gl->gausswt[m]*(gl->gpoly[m][k]*derivs[j]*len
                                  + val*tang[j]/len*gl->gpolypart[m][0][k]);
    for ( k = 0 ; k < gl->lagpts ; k++ )
      for ( kk = 0 ; kk < gl->lagpts ; kk++ )
         for ( j = 0 ; j < SDIM ; j++ )
            for ( jj = 0 ; jj < SDIM ; jj++ )
              e_info->hess[k][kk][j][jj] += gl->gausswt[m]*
     ( second[j][jj]*gl->gpoly[m][k]*gl->gpoly[m][kk]*len
     + derivs[j]*gl->gpoly[m][k]*tang[jj]*gl->gpolypart[m][0][kk]/len
     + derivs[jj]*gl->gpoly[m][kk]*tang[j]*gl->gpolypart[m][0][k]/len
     - val*tang[j]*tang[jj]*gl->gpolypart[m][0][k]*gl->gpolypart[m][0][kk]/len/len/len
     + ((j==jj)? val*gl->gpolypart[m][0][k]*gl->gpolypart[m][0][kk]/len : 0.0));
  }

  return value;
}


/*********************************************************************

                    edge_vector_integral method

*********************************************************************/

/*********************************************************************
*
* function: edge_vector_integral()
*
* purpose:  method value
*
*/

REAL edge_vector_integral(e_info)
struct qinfo *e_info;
{ int m,j;
  REAL value=0.0;
  if ( web.modeltype == QUADRATIC ) return edge_vector_integral_q(e_info);
  if ( web.modeltype == LAGRANGE ) return edge_vector_integral_lagrange(e_info);
  for (  m = 0 ; m < gauss1D_num ; m++ )
     for ( j = 0 ; j < SDIM ; j++ )
     { REAL green = gauss1Dwt[m]*eval(METH_INST[e_info->method].expr[j],
                       e_info->gauss_pt[m], e_info->id);
        value += e_info->sides[0][0][j]*green;
     }
  
  return (get_eattr(e_info->id) & NEGBOUNDARY) ? -value : value;
}

/*********************************************************************
*
* function: edge_vector_integral_grad()
*
* purpose:  method gradient
*
*/


REAL edge_vector_integral_grad(e_info)
struct qinfo *e_info;
{ int m,j,k;
  REAL value = 0.0;
  REAL val[MAXCOORD];
  REAL derivs[MAXCOORD][MAXCOORD];
  REAL sum;
  REAL sign = (get_eattr(e_info->id) & NEGBOUNDARY) ? -1.0 : 1.0;

  if ( web.modeltype == QUADRATIC ) return edge_vector_integral_q_grad(e_info);
  if ( web.modeltype == LAGRANGE ) 
     return edge_vector_integral_lagrange_grad(e_info);
  for ( k = 0 ; k < 2 ; k++ )
     for ( j = 0 ; j < SDIM ; j++ ) 
        e_info->grad[k][j] = 0.0;
  for ( m = 0 ; m < gauss1D_num ; m++ )
  { REAL weight = sign*gauss1Dwt[m];
    for ( j = 0 ; j < SDIM ; j++ ) 
      eval_all(METH_INST[e_info->method].expr[j],e_info->gauss_pt[m],SDIM,
        val+j, derivs[j],e_info->id);
    value += gauss1Dwt[m]*SDIM_dot(val,e_info->sides[0][0]);
    for ( k = 0 ; k < SDIM ; k++ )
    { for ( sum = 0.0, j = 0 ; j < SDIM ; j++ )
        sum += derivs[j][k]*e_info->sides[0][0][j];
      e_info->grad[0][k] += weight*(-val[k] + gauss1poly[0][m]*sum);
      e_info->grad[1][k] += weight*(val[k] + gauss1poly[1][m]*sum);
    }
  }

  return sign*value;
}

/*********************************************************************
*
* function: edge_vector_integral_hess()
*
* purpose:  method gradient and hessian
*
*/

REAL edge_vector_integral_hess(e_info)
struct qinfo *e_info;
{ int m,i,j,k;
  REAL value = 0.0;
  REAL val[MAXCOORD];
  REAL derivs[MAXCOORD][MAXCOORD];
  REAL sum;
  MAT3D(second,MAXCOORD,MAXCOORD,MAXCOORD);
  REAL sign = (get_eattr(e_info->id) & NEGBOUNDARY) ? -1.0 : 1.0;

  if ( web.modeltype == QUADRATIC ) return edge_vector_integral_q_hess(e_info);
  if ( web.modeltype == LAGRANGE ) 
     return edge_vector_integral_lagrange_hess(e_info);
  for ( m = 0 ; m < gauss1D_num ; m++ )
  { REAL weight = sign*gauss1Dwt[m];
    for ( j = 0 ; j < SDIM ; j++ ) 
      eval_second(METH_INST[e_info->method].expr[j],e_info->gauss_pt[m],SDIM,val+j,
                                              derivs[j],second[j],e_info->id);
    value += weight*SDIM_dot(val,e_info->sides[0][0]);
    for ( k = 0 ; k < SDIM ; k++ )
    { for ( sum = 0.0, j = 0 ; j < SDIM ; j++ )
          sum += derivs[j][k]*e_info->sides[0][0][j];
      e_info->grad[0][k] += weight*(-val[k] + gauss1poly[0][m]*sum);
      e_info->grad[1][k] += weight*(val[k] + gauss1poly[1][m]*sum);
    }
    for ( k = 0 ; k < SDIM ; k++ )
     for ( i = 0 ; i < SDIM ; i++ )
     { for ( sum = 0.0, j = 0 ; j < SDIM ; j++ )
         sum += second[j][k][i]*e_info->sides[0][0][j];
       e_info->hess[0][0][k][i] += weight*gauss1poly[0][m]
            *(gauss1poly[0][m]*sum - derivs[k][i] - derivs[i][k]);
       e_info->hess[0][1][k][i] += weight
            *(gauss1poly[0][m]*gauss1poly[1][m]*sum 
            - gauss1poly[1][m]*derivs[k][i] + gauss1poly[0][m]*derivs[i][k]);
       e_info->hess[1][0][k][i] += weight
            *(gauss1poly[0][m]*gauss1poly[1][m]*sum 
            + gauss1poly[0][m]*derivs[k][i] - gauss1poly[1][m]*derivs[i][k]);
       e_info->hess[1][1][k][i] += weight*gauss1poly[1][m]
            *(gauss1poly[1][m]*sum + derivs[k][i] + derivs[i][k]);
     }
  }

  return value;
}


/*********************************************************************

                  quadratic edge_vector_integral method

*********************************************************************/

/*********************************************************************
*
* function: edge_vector_integral_q()
*
* purpose:  method value
*
*/

REAL edge_vector_integral_q(e_info)
struct qinfo *e_info;
{ int m,j,k;
  REAL value=0.0;
  REAL tang[MAXCOORD];
  REAL sign = (get_eattr(e_info->id) & NEGBOUNDARY) ? -1.0 : 1.0;

  for ( m = 0 ; m < gauss1D_num ; m++ )
  { REAL weight = sign*gauss1Dwt[m];
     for ( j = 0 ; j < SDIM ; j++ )
     { tang[j] = 0.0;
        for ( k = 0 ; k < edge_ctrl ; k++ )
          tang[j] += gauss1polyd[k][m]*e_info->x[k][j];
        value += weight*tang[j]
                        *eval(METH_INST[e_info->method].expr[j],e_info->gauss_pt[m],e_info->id);
     }
  }
  return value;
}

/*********************************************************************
*
* function: edge_vector_integral_q_grad()
*
* purpose:  method gradient
*
*/


REAL edge_vector_integral_q_grad(e_info)
struct qinfo *e_info;
{ int m,j,k,i;
  REAL value = 0.0;
  REAL val[MAXCOORD];
  REAL derivs[MAXCOORD][MAXCOORD];
  REAL sum;
  REAL tang[MAXCOORD];
  REAL sign = (get_eattr(e_info->id) & NEGBOUNDARY) ? -1.0 : 1.0;

  for ( m = 0 ; m < gauss1D_num ; m++ )
     { REAL weight = sign*gauss1Dwt[m];
        for ( j = 0 ; j < SDIM ; j++ ) 
        { tang[j] = 0.0;
          for ( k = 0 ; k < edge_ctrl ; k++ )
              tang[j] += gauss1polyd[k][m]*e_info->x[k][j];
        }
        for ( j = 0 ; j < SDIM ; j++ ) 
          eval_all(METH_INST[e_info->method].expr[j],e_info->gauss_pt[m],SDIM,val+j,
                                                                         derivs[j],e_info->id);
        value += weight*SDIM_dot(val,tang);
        for ( k = 0 ; k < SDIM ; k++ )
          { for ( sum = 0.0, j = 0 ; j < SDIM ; j++ )
                 sum += derivs[j][k]*tang[j];
             for ( i = 0 ; i < edge_ctrl ; i++ )
                e_info->grad[i][k] += 
                  weight*(gauss1polyd[i][m]*val[k] + gauss1poly[i][m]*sum);
          }
     }

  return value;
}

/*********************************************************************
*
* function: edge_vector_integral_q_hess()
*
* purpose:  method gradient and hessian
*
*/


REAL edge_vector_integral_q_hess(e_info)
struct qinfo *e_info;
{ int m,i,j,k,ii,kk;
  REAL value = 0.0;
  REAL val[MAXCOORD];
  REAL derivs[MAXCOORD][MAXCOORD];
  REAL sum;
  MAT3D(second,MAXCOORD,MAXCOORD,MAXCOORD);
  REAL tang[MAXCOORD];
  REAL sign = (get_eattr(e_info->id) & NEGBOUNDARY) ? -1.0 : 1.0;

  for ( m = 0 ; m < gauss1D_num ; m++ )
     { REAL weight = sign*gauss1Dwt[m];
        for ( j = 0 ; j < SDIM ; j++ ) 
        { tang[j] = 0.0;
          for ( k = 0 ; k < edge_ctrl ; k++ )
              tang[j] += gauss1polyd[k][m]*e_info->x[k][j];
        }
        for ( j = 0 ; j < SDIM ; j++ ) 
          eval_second(METH_INST[e_info->method].expr[j],e_info->gauss_pt[m],SDIM,val+j,
                                                              derivs[j],second[j],e_info->id);
        value += weight*SDIM_dot(val,tang);
        for ( k = 0 ; k < SDIM ; k++ )
          { for ( sum = 0.0, j = 0 ; j < SDIM ; j++ )
                 sum += derivs[j][k]*tang[j];
             for ( i = 0 ; i < edge_ctrl ; i++ )
                e_info->grad[i][k] += 
                  weight*(gauss1polyd[i][m]*val[k] + gauss1poly[i][m]*sum);
          }

        for ( ii = 0 ; ii < SDIM ; ii++ )
         for ( i = 0 ; i < SDIM ; i++ )
          { for ( sum = 0.0, j = 0 ; j < SDIM ; j++ )
                 sum += second[j][ii][i]*tang[j];
             for ( k = 0 ; k < edge_ctrl ; k++ )
                for ( kk = 0 ; kk < edge_ctrl ; kk++ )
                  e_info->hess[k][kk][i][ii] += weight*
                  ( sum*gauss1poly[k][m]*gauss1poly[kk][m]
                     + gauss1polyd[k][m]*derivs[i][ii]*gauss1poly[kk][m]
                     + gauss1polyd[kk][m]*derivs[ii][i]*gauss1poly[k][m]
                  );
          }
     }

  return value;
}



/**********************************************************************
                    Linear area quantity (STRING model)
**********************************************************************/

/**********************************************************************
*
*  function: q_edge_area()
*
*  purpose: value of area integral on edge
*/

REAL q_edge_area(e_info)
struct qinfo *e_info;
{ REAL **x;
  REAL area;

  if ( web.torus_flag ) return q_edge_torus_area(e_info);
  if ( web.modeltype == QUADRATIC ) return q_edge_area_q(e_info);
  if ( web.modeltype == LAGRANGE )  return q_edge_area_lagrange(e_info);

  x = e_info->x;

  /* main integral over edge */
  area = (x[0][1]+x[1][1])*(x[0][0] - x[1][0])/2;

  return area;
}

/**********************************************************************
*
*  function: q_edge_area_grad()
*
*  purpose: value and gradient of area integral on edge
*/

REAL q_edge_area_grad(e_info)
struct qinfo *e_info;
{ REAL **x,**g;
  REAL area;

  if ( web.torus_flag ) return q_edge_torus_area_grad(e_info);
  if ( web.modeltype == QUADRATIC ) return q_edge_area_q_grad(e_info);
  if ( web.modeltype == LAGRANGE ) return q_edge_area_lagrange_grad(e_info);

  x = e_info->x;
  g = e_info->grad;

  /* main integral over edge */
  area = (x[0][1]+x[1][1])*(x[0][0] - x[1][0])/2;
  g[0][0] = (x[1][1] + x[0][1])/2;
  g[1][0] = -(x[1][1] + x[0][1])/2;
  g[0][1] = (x[0][0] - x[1][0])/2;
  g[1][1] = (x[0][0] - x[1][0])/2;
  return area;
}


/**********************************************************************
*
*  function: q_edge_area_hess()
*
*  purpose: value and gradient and hessian of area integral on edge
*/

REAL q_edge_area_hess(e_info)
struct qinfo *e_info;
{ REAL **x,**g,****h;
  REAL area;

  if ( web.torus_flag ) return q_edge_torus_area_hess(e_info);
  if ( web.modeltype == QUADRATIC ) return q_edge_area_q_hess(e_info);
  if ( web.modeltype == LAGRANGE )  return q_edge_area_lagrange_hess(e_info);

  x = e_info->x;
  g = e_info->grad;
  h = e_info->hess;

  /* main integral over edge */
  area = (x[0][1]+x[1][1])*(x[0][0] - x[1][0])/2;
  g[0][0] = (x[1][1] + x[0][1])/2;
  g[1][0] = -(x[1][1] + x[0][1])/2;
  g[0][1] = (x[0][0] - x[1][0])/2;
  g[1][1] = (x[0][0] - x[1][0])/2;
  h[0][1][0][1] += 0.5;
  h[0][0][0][1] += 0.5;
  h[1][1][0][1] -= 0.5;
  h[1][0][0][1] -= 0.5;
  h[0][0][1][0] += 0.5;
  h[0][1][1][0] -= 0.5;
  h[1][0][1][0] += 0.5;
  h[1][1][1][0] -= 0.5;

  return area;
}


/**********************************************************************
                    Quadratic area quantity (STRING model)
**********************************************************************/

/**********************************************************************
*
*  function: q_edge_area_q()
*
*  purpose: value of area integral on edge
*/

REAL q_edge_area_q(e_info)
struct qinfo *e_info;
{ REAL **x;
  REAL area;
  int i,j;

  x = e_info->x;

  /* main integral over edge */
  area = 0.0;
  for ( i = 0 ; i < edge_ctrl ; i++ )
     for ( j = 0 ; j < edge_ctrl ; j++ )
        { REAL v = scoeff[j][i];
          area += v*x[i][0]*x[j][1];
        }

  return area;
}

/**********************************************************************
*
*  function: q_edge_area_q_grad()
*
*  purpose: value and gradient of area integral on edge
*/

REAL q_edge_area_q_grad(e_info)
struct qinfo *e_info;
{ REAL **x,**g;
  REAL area;
  int i,j,k;

  x = e_info->x;
  g = e_info->grad;
  /* gradients */
  for ( k = 0 ; k < edge_ctrl ; k++ )
     for ( j = 0 ; j < SDIM ; j++ )
        g[k][j] = 0.0;

  /* main integral over edge */
  area = 0.0;
  for ( i = 0 ; i < edge_ctrl ; i++ )
     for ( j = 0 ; j < edge_ctrl ; j++ )
        { REAL v = scoeff[j][i];
          area += v*x[i][0]*x[j][1];
          g[i][0] += v*x[j][1];
          g[j][1] += v*x[i][0];
        }

  return area;
}


/**********************************************************************
*
*  function: q_edge_area_q_hess()
*
*  purpose: value and gradient and hessian of area integral on edge
*/

REAL q_edge_area_q_hess(e_info)
struct qinfo *e_info;
{ REAL **x,**g,****h;
  REAL area;
  int i,j;

  x = e_info->x;
  g = e_info->grad;
  h = e_info->hess;

  /* main integral over edge */
  area = 0.0;
  for ( i = 0 ; i < edge_ctrl ; i++ )
     for ( j = 0 ; j < edge_ctrl ; j++ )
        { REAL v = scoeff[j][i];
          if ( v == 0.0 ) continue;
          area += v*x[i][0]*x[j][1];
          g[i][0] += v*x[j][1];
          g[j][1] += v*x[i][0];
          h[i][j][0][1] += v;
          h[j][i][1][0] += v;
        }

  return area;
}


/**********************************************************************
                    Lagrange area quantity (STRING model)
**********************************************************************/

/**********************************************************************
*
*  function: q_edge_area_lagrange()
*
*  purpose: value of area integral on edge
*/

REAL q_edge_area_lagrange(e_info)
struct qinfo *e_info;
{ REAL **x;
  REAL area;
  struct gauss_lag *gl = &gauss_lagrange[web.dimension][web.gauss1D_order];
  int ctrl = web.skel[EDGE].ctrlpts;
  int m,k;

  x = e_info->x;

  /* main integral over edge */
  area = 0.0;
  for ( m = 0 ; m < gl->gnumpts ; m++ )
  { REAL y,dx;
     for ( y = 0.0, dx = 0.0, k = 0 ; k < ctrl ; k++ )
     { y += gl->gpoly[m][k]*x[k][1];
        dx += gl->gpolypart[m][0][k]*x[k][0];
     }
     area -= gl->gausswt[m]*y*dx;
  }
     
  return area;
}

/**********************************************************************
*
*  function: q_edge_area_lagrange_grad()
*
*  purpose: value and gradient of area integral on edge
*/

REAL q_edge_area_lagrange_grad(e_info)
struct qinfo *e_info;
{ REAL **x,**g;
  REAL area;
  int m,k;
  struct gauss_lag *gl = &gauss_lagrange[web.dimension][web.gauss1D_order];
  int ctrl = web.skel[EDGE].ctrlpts;

  x = e_info->x;
  g = e_info->grad;

  /* main integral over edge */
  area = 0.0;
  for ( m = 0 ; m < gl->gnumpts ; m++ )
  { REAL y,dx;
     for ( y = 0.0, dx = 0.0, k = 0 ; k < ctrl ; k++ )
     { y += gl->gpoly[m][k]*x[k][1];
        dx += gl->gpolypart[m][0][k]*x[k][0];
     }
     area -= gl->gausswt[m]*y*dx;
     for ( k = 0 ; k < ctrl ; k++ )
     { g[k][0] -= gl->gausswt[m]*y*gl->gpolypart[m][0][k];
        g[k][1] -= gl->gausswt[m]*dx*gl->gpoly[m][k];
     }
  }

  return area;
}


/**********************************************************************
*
*  function: q_edge_area_lagrange_hess()
*
*  purpose: value and gradient and hessian of area integral on edge
*/

REAL q_edge_area_lagrange_hess(e_info)
struct qinfo *e_info;
{ REAL **x,**g,****h;
  REAL area;
  int m,k,kk;
  struct gauss_lag *gl = &gauss_lagrange[web.dimension][web.gauss1D_order];
  int ctrl = web.skel[EDGE].ctrlpts;

  x = e_info->x;
  g = e_info->grad;
  h = e_info->hess;

  /* main integral over edge */
  area = 0.0;
  for ( m = 0 ; m < gl->gnumpts ; m++ )
  { REAL y,dx;
     for ( y = 0.0, dx = 0.0, k = 0 ; k < ctrl ; k++ )
     { y += gl->gpoly[m][k]*x[k][1];
        dx += gl->gpolypart[m][0][k]*x[k][0];
     }
     area -= gl->gausswt[m]*y*dx;
     for ( k = 0 ; k < ctrl ; k++ )
     { g[k][0] -= gl->gausswt[m]*y*gl->gpolypart[m][0][k];
        g[k][1] -= gl->gausswt[m]*dx*gl->gpoly[m][k];
        for ( kk = 0 ; kk < ctrl ; kk++ )
        { h[k][kk][0][1] -= gl->gausswt[m]*gl->gpoly[m][kk]*gl->gpolypart[m][0][k];
          h[k][kk][1][0] -= gl->gausswt[m]*gl->gpolypart[m][0][kk]*gl->gpoly[m][k];
        }
     }
  }

  return area;
}


/**********************************************************************
                    Linear torus area quantity (STRING model)
**********************************************************************/

/**********************************************************************
*
*  function: q_edge_torus_area()
*
*  purpose: value of area integral on edge
*/

REAL q_edge_torus_area(e_info)
struct qinfo *e_info;
{ REAL **x;
  REAL area;
  REAL **dx = web.inverse_periods;
  MAT2D(u,EDGE_VERTS,MAXCOORD); /* affine coordinates of vertices */
  int wrap;

  if ( !dx )
     kb_error(3681,"Need torus model to use edge_torus_area method.\n",RECOVERABLE);
  if ( web.modeltype == QUADRATIC ) return q_edge_torus_area_q(e_info);
  if ( web.modeltype == LAGRANGE ) 
     return q_edge_torus_area_lagrange(e_info);

  x = e_info->x;

  /* get affine coordinates of vertices */
  mat_mul_tr(x,dx,u,EDGE_VERTS,SDIM,SDIM);
  /* main integral over edge */
  area = (u[0][1]+u[1][1])*(u[0][0] - u[1][0])/2;

  /* wrap correction */
  wrap = (get_edge_wrap(e_info->id)>>TWRAPBITS) & WRAPMASK;
  switch ( wrap )
  { case NEGWRAP: area -= u[1][0]; break;
     case POSWRAP: area += u[1][0]; break;
  }
  return area*web.torusv;
}

/**********************************************************************
*
*  function: q_edge_torus_area_grad()
*
*  purpose: value and gradient of area integral on edge
*/

REAL q_edge_torus_area_grad(e_info)
struct qinfo *e_info;
{ REAL **x;
  REAL area;
  REAL **dx = web.inverse_periods;
  MAT2D(u,EDGE_VERTS,MAXCOORD); /* affine coordinates of vertices */
  MAT2D(g,EDGE_VERTS,MAXCOORD);
  int wrap;

  if ( web.modeltype == QUADRATIC ) return q_edge_torus_area_q_grad(e_info);
  if ( web.modeltype == LAGRANGE ) 
     return q_edge_torus_area_lagrange_grad(e_info);

  x = e_info->x;

  /* get affine coordinates of vertices */
  mat_mul_tr(x,dx,u,EDGE_VERTS,SDIM,SDIM);
  /* main integral over edge */
  area = (u[0][1]+u[1][1])*(u[0][0] - u[1][0])/2;
  g[0][0] = (u[1][1] + u[0][1])/2*web.torusv;
  g[1][0] = -(u[1][1] + u[0][1])/2*web.torusv;
  g[0][1] = (u[0][0] - u[1][0])/2*web.torusv;
  g[1][1] = (u[0][0] - u[1][0])/2*web.torusv;

  /* wrap correction */
  wrap = (get_edge_wrap(e_info->id)>>TWRAPBITS) & WRAPMASK;
  switch ( wrap )
  { case NEGWRAP: area -= u[1][0]; g[1][0] -= web.torusv; break;
     case POSWRAP: area += u[1][0]; g[1][0] += web.torusv; break;
  }
  mat_mult(g,dx,e_info->grad,EDGE_VERTS,SDIM,SDIM);
  return area*web.torusv;
}


/**********************************************************************
*
*  function: q_edge_torus_area_hess()
*
*  purpose: value and gradient and hessian of area integral on edge
*/

REAL q_edge_torus_area_hess(e_info)
struct qinfo *e_info;
{ REAL **x;
  REAL area;
  REAL **dx = web.inverse_periods;
  MAT2D(u,EDGE_VERTS,MAXCOORD); /* affine coordinates of vertices */
  MAT2D(g,EDGE_VERTS,MAXCOORD);
  MAT4D(h,EDGE_VERTS,EDGE_VERTS,MAXCOORD,MAXCOORD);
  MAT2D(temph,MAXCOORD,MAXCOORD);
  int i,ii,j,jj,wrap;

  if ( web.modeltype == QUADRATIC ) return q_edge_torus_area_q_hess(e_info);
  if ( web.modeltype == LAGRANGE )  
     return q_edge_torus_area_lagrange_hess(e_info);

  x = e_info->x;
  for ( i = 0 ; i < EDGE_VERTS ; i++ )
     for ( ii = 0 ; ii < EDGE_VERTS ; ii++ )
        for ( j = 0 ; j < SDIM ; j++ )
            for ( jj = 0 ; jj < SDIM ; jj++ )
              h[i][ii][j][jj] = 0.0;

  /* get affine coordinates of vertices */
  mat_mul_tr(x,dx,u,EDGE_VERTS,SDIM,SDIM);
  /* main integral over edge */
  area = (u[0][1]+u[1][1])*(u[0][0] - u[1][0])/2;
  g[0][0] = (u[1][1] + u[0][1])/2*web.torusv;
  g[1][0] = -(u[1][1] + u[0][1])/2*web.torusv;
  g[0][1] = (u[0][0] - u[1][0])/2*web.torusv;
  g[1][1] = (u[0][0] - u[1][0])/2*web.torusv;
  h[0][1][0][1] += 0.5*web.torusv;
  h[0][0][0][1] += 0.5*web.torusv;
  h[1][1][0][1] -= 0.5*web.torusv;
  h[1][0][0][1] -= 0.5*web.torusv;
  h[0][0][1][0] += 0.5*web.torusv;
  h[0][1][1][0] -= 0.5*web.torusv;
  h[1][0][1][0] += 0.5*web.torusv;
  h[1][1][1][0] -= 0.5*web.torusv;

  /* wrap correction */
  wrap = (get_edge_wrap(e_info->id)>>TWRAPBITS) & WRAPMASK;
  switch ( wrap )
  { case NEGWRAP: area -= u[1][0]; g[1][0] -= web.torusv; break;
     case POSWRAP: area += u[1][0]; g[1][0] += web.torusv; break;
  }

  /* form pullback */
  mat_mult(g,dx,e_info->grad,EDGE_VERTS,SDIM,SDIM);
  for ( i = 0 ; i < EDGE_VERTS ; i++ )
     for ( ii = 0 ; ii < EDGE_VERTS ; ii++ )
        { mat_mult(h[i][ii],dx,temph,SDIM,SDIM,SDIM);
          tr_mat_mul(dx,temph,e_info->hess[i][ii],SDIM,SDIM,SDIM);
        }

  return area*web.torusv;
}


/**********************************************************************
                    Quadratic torus area quantity (STRING model)
**********************************************************************/

/**********************************************************************
*
*  function: q_edge_torus_area_q()
*
*  purpose: value of area integral on edge
*/

REAL q_edge_torus_area_q(e_info)
struct qinfo *e_info;
{ REAL **x;
  REAL area;
  REAL **dx = web.inverse_periods;
  MAT2D(u,EDGE_CTRL,MAXCOORD); /* affine coordinates of vertices */
  int i,j;
  int wrap;

  x = e_info->x;

  /* get affine coordinates of vertices */
  mat_mul_tr(x,dx,u,edge_ctrl,SDIM,SDIM);
  /* main integral over edge */
  area = 0.0;
  for ( i = 0 ; i < edge_ctrl ; i++ )
     for ( j = 0 ; j < edge_ctrl ; j++ )
        { REAL v = scoeff[i][j];
          if ( v == 0.0 ) continue;
          area += v*u[i][1]*u[j][0];
        }

  /* wrap correction */
  wrap = (get_edge_wrap(e_info->id)>>TWRAPBITS) & WRAPMASK;
  switch ( wrap )
  { case NEGWRAP: area -= u[2][0]; break;
     case POSWRAP: area += u[2][0]; break;
  }
  return area*web.torusv;
}

/**********************************************************************
*
*  function: q_edge_torus_area_q_grad()
*
*  purpose: value and gradient of area integral on edge
*/

REAL q_edge_torus_area_q_grad(e_info)
struct qinfo *e_info;
{ REAL **x;
  REAL area;
  REAL **dx = web.inverse_periods;
  MAT2D(u,EDGE_CTRL,MAXCOORD); /* affine coordinates of vertices */
  MAT2D(g,EDGE_CTRL,MAXCOORD);
  int i,j,k;
  int wrap;

  x = e_info->x;
  for ( i = 0 ; i < edge_ctrl ; i++ )
     for ( j = 0 ; j < SDIM ; j++ ) g[i][j] = 0.0;

  /* get affine coordinates of vertices */
  mat_mul_tr(x,dx,u,edge_ctrl,SDIM,SDIM);
  /* main integral over edge */
  area = 0.0;
  for ( i = 0 ; i < edge_ctrl ; i++ )
     for ( j = 0 ; j < edge_ctrl ; j++ )
        { REAL v = scoeff[j][i];
          if ( v == 0.0 ) continue;
          area += v*u[i][0]*u[j][1];
          g[i][0] += v*u[j][1];
          g[j][1] += v*u[i][0];
        }

  /* wrap correction */
  wrap = (get_edge_wrap(e_info->id)>>TWRAPBITS) & WRAPMASK;
  switch ( wrap )
  { case NEGWRAP: area -= u[2][0]; g[2][0] -= 1.0; break;
     case POSWRAP: area += u[2][0]; g[2][0] += 1.0; break;
  }
  for ( k = 0 ; k < edge_ctrl ; k++ )
     for ( j = 0 ; j < SDIM ; j++ )
        g[k][j] *= web.torusv;
  mat_mult(g,dx,e_info->grad,edge_ctrl,SDIM,SDIM);
  return area*web.torusv;
}


/**********************************************************************
*
*  function: q_edge_torus_area_q_hess()
*
*  purpose: value and gradient and hessian of area integral on edge
*/

REAL q_edge_torus_area_q_hess(e_info)
struct qinfo *e_info;
{ REAL **x;
  REAL area;
  REAL **dx = web.inverse_periods;
  MAT2D(u,EDGE_CTRL,MAXCOORD); /* affine coordinates of vertices */
  MAT2D(g,EDGE_CTRL+1,MAXCOORD);
  REAL ****h;
  MAT2D(temph,MAXCOORD,MAXCOORD);
  int i,ii,j,k;
  int wrap;

  x = e_info->x;
  h = e_info->hess;
  for ( i = 0 ; i < edge_ctrl ; i++ )
     for ( j = 0 ; j < SDIM ; j++ ) g[i][j] = 0.0;

  /* get affine coordinates of vertices */
  mat_mul_tr(x,dx,u,edge_ctrl,SDIM,SDIM);
  /* main integral over edge */
  area = 0.0;
  for ( i = 0 ; i < edge_ctrl ; i++ )
     for ( j = 0 ; j < edge_ctrl ; j++ )
        { REAL v = scoeff[j][i];
          if ( v == 0.0 ) continue;
          area += v*u[i][0]*u[j][1];
          g[i][0] += v*u[j][1];
          g[j][1] += v*u[i][0];
          h[i][j][0][1] += v*web.torusv;
          h[j][i][1][0] += v*web.torusv;
        }

  /* wrap correction */
  wrap = (get_edge_wrap(e_info->id)>>TWRAPBITS) & WRAPMASK;
  switch ( wrap )
  { case NEGWRAP: area -= u[2][0]; g[2][0] -= 1.0; break;
     case POSWRAP: area += u[2][0]; g[2][0] += 1.0; break;
  }
  for ( k = 0 ; k < edge_ctrl ; k++ )
     for ( j = 0 ; j < SDIM ; j++ )
        g[k][j] *= web.torusv;

  /* form pullback */
  mat_mult(g,dx,e_info->grad,edge_ctrl,SDIM,SDIM);
  for ( i = 0 ; i < edge_ctrl ; i++ )
     for ( ii = 0 ; ii < edge_ctrl ; ii++ )
        { mat_mult(h[i][ii],dx,temph,SDIM,SDIM,SDIM);
          tr_mat_mul(dx,temph,e_info->hess[i][ii],SDIM,SDIM,SDIM);
        }
  return area*web.torusv;
}


/**********************************************************************
                    Lagrange torus area quantity (STRING model)
**********************************************************************/

/**********************************************************************
*
*  function: q_edge_torus_area_lagrange()
*
*  purpose: value of area integral on edge
*/

REAL q_edge_torus_area_lagrange(e_info)
struct qinfo *e_info;
{ REAL **x;
  REAL area;
  struct gauss_lag *gl = &gauss_lagrange[web.dimension][web.gauss1D_order];
  int ctrl = web.skel[EDGE].ctrlpts;
  int m,k;
  REAL **u = e_info->u;
  int wrap;

  x = e_info->x;
  /* get affine coordinates of vertices */
  mat_mul_tr(x,web.inverse_periods,u,ctrl,SDIM,SDIM);

  /* main integral over edge */
  area = 0.0;
  for ( m = 0 ; m < gl->gnumpts ; m++ )
  { REAL y,dx;
     for ( y = 0.0, dx = 0.0, k = 0 ; k < ctrl ; k++ )
     { y += gl->gpoly[m][k]*u[k][1];
        dx += gl->gpolypart[m][0][k]*u[k][0];
     }
     area -= gl->gausswt[m]*y*dx;
  }
     
  /* wrap correction */
  wrap = (get_edge_wrap(e_info->id)>>TWRAPBITS) & WRAPMASK;
  switch ( wrap )
  { case NEGWRAP: area -= u[ctrl-1][0]; break;
     case POSWRAP: area += u[ctrl-1][0]; break;
  }
  return area * web.torusv;
}

/**********************************************************************
*
*  function: q_edge_torus_area_lagrange_grad()
*
*  purpose: value and gradient of area integral on edge
*/

REAL q_edge_torus_area_lagrange_grad(e_info)
struct qinfo *e_info;
{ REAL **x;
  REAL area;
  int m,j,k,i;
  struct gauss_lag *gl = &gauss_lagrange[web.dimension][web.gauss1D_order];
  int ctrl = web.skel[EDGE].ctrlpts;
  REAL **u = e_info->u;
  int wrap;
  MAT2D(g,MAXVCOUNT,MAXCOORD);

  x = e_info->x;
  for ( i = 0 ; i < ctrl ; i++ )
     for ( j = 0 ; j < SDIM ; j++ ) g[i][j] = 0.0;

  /* get affine coordinates of vertices */
  mat_mul_tr(x,web.inverse_periods,u,edge_ctrl,SDIM,SDIM);

  /* main integral over edge */
  area = 0.0;
  for ( m = 0 ; m < gl->gnumpts ; m++ )
  { REAL y,dx;
     for ( y = 0.0, dx = 0.0, k = 0 ; k < ctrl ; k++ )
     { y += gl->gpoly[m][k]*u[k][1];
        dx += gl->gpolypart[m][0][k]*u[k][0];
     }
     area -= gl->gausswt[m]*y*dx;

     for ( k = 0 ; k < ctrl ; k++ )
     { g[k][0] -= gl->gausswt[m]*y*gl->gpolypart[m][0][k];
        g[k][1] -= gl->gausswt[m]*dx*gl->gpoly[m][k];
     }
  }

  /* wrap correction */
  wrap = (get_edge_wrap(e_info->id)>>TWRAPBITS) & WRAPMASK;
  switch ( wrap )
  { case NEGWRAP: area -= u[ctrl-1][0]; g[ctrl-1][0] -= 1.0; break;
     case POSWRAP: area += u[ctrl-1][0]; g[ctrl-1][0] += 1.0; break;
  }
  for ( k = 0 ; k < ctrl ; k++ )
     for ( j = 0 ; j < SDIM ; j++ )
        g[k][j] *= web.torusv;
  mat_mult(g,web.inverse_periods,e_info->grad,ctrl,SDIM,SDIM);
  return area*web.torusv;
}


/**********************************************************************
*
*  function: q_edge_torus_area_lagrange_hess()
*
*  purpose: value and gradient and hessian of area integral on edge
*/

REAL q_edge_torus_area_lagrange_hess(e_info)
struct qinfo *e_info;
{ REAL **x,****h;
  REAL area;
  int i,ii,j;
  int m,k,kk;
  struct gauss_lag *gl = &gauss_lagrange[web.dimension][web.gauss1D_order];
  int ctrl = web.skel[EDGE].ctrlpts;
  REAL **u = e_info->u;
  int wrap;
  MAT2D(temph,MAXCOORD,MAXCOORD);
  MAT2D(g,MAXVCOUNT,MAXCOORD);

  x = e_info->x;
  h = e_info->hess;
  for ( i = 0 ; i < ctrl ; i++ )
     for ( j = 0 ; j < SDIM ; j++ ) g[i][j] = 0.0;

  /* get affine coordinates of vertices */
  mat_mul_tr(x,web.inverse_periods,u,edge_ctrl,SDIM,SDIM);

  /* main integral over edge */
  area = 0.0;
  for ( m = 0 ; m < gl->gnumpts ; m++ )
  { REAL y,dx;
     for ( y = 0.0, dx = 0.0, k = 0 ; k < ctrl ; k++ )
     { y += gl->gpoly[m][k]*u[k][1];
        dx += gl->gpolypart[m][0][k]*u[k][0];
     }
     area -= gl->gausswt[m]*y*dx;
     for ( k = 0 ; k < ctrl ; k++ )
     { g[k][0] -= gl->gausswt[m]*y*gl->gpolypart[m][0][k];
        g[k][1] -= gl->gausswt[m]*dx*gl->gpoly[m][k];
        for ( kk = 0 ; kk < ctrl ; kk++ )
        { h[k][kk][0][1] -= gl->gausswt[m]*gl->gpoly[m][kk]*gl->gpolypart[m][0][k];
          h[k][kk][1][0] -= gl->gausswt[m]*gl->gpolypart[m][0][kk]*gl->gpoly[m][k];
        }
     }
  }

  /* wrap correction */
  wrap = (get_edge_wrap(e_info->id)>>TWRAPBITS) & WRAPMASK;
  switch ( wrap )
  { case NEGWRAP: area -= u[ctrl-1][0]; g[ctrl-1][0] -= 1.0; break;
     case POSWRAP: area += u[ctrl-1][0]; g[ctrl-1][0] += 1.0; break;
  }
  for ( k = 0 ; k < ctrl ; k++ )
     for ( j = 0 ; j < SDIM ; j++ )
        g[k][j] *= web.torusv;

  /* form pullback */
  mat_mult(g,web.inverse_periods,e_info->grad,ctrl,SDIM,SDIM);
  for ( i = 0 ; i < ctrl ; i++ )
     for ( ii = 0 ; ii < ctrl ; ii++ )
        { mat_mult(h[i][ii],web.inverse_periods,temph,SDIM,SDIM,SDIM);
          tr_mat_mul(web.inverse_periods,temph,e_info->hess[i][ii],SDIM,SDIM,SDIM);
        }
  return area*web.torusv;
}


/*******************************************************************
  
     Hooke Energy - Hooke's Law to keep edge lengths nearly 
     uniform.

********************************************************************/

#define POWER_NAME "hooke_power"
#define LENGTH_NAME "hooke_length"
static int exponent_param;
static int length_param;
static REAL hooke_length, hooke_power;

/***************************************************************
*
*  function: hooke_energy_init()
*
*  purpose: initialization for hooke_energy() and 
*              hooke_energy_gradient().
*
*    No special prep.
*/

void hooke_energy_init(mode,mi)
int mode; /* energy or gradient */
struct method_instance *mi;
{
  if ( web.modeltype != LINEAR )
     kb_error(1766,"hooke_energy only for LINEAR model.\n",RECOVERABLE);


  exponent_param = lookup_global(POWER_NAME);
  if ( exponent_param < 0 ) /* missing, so add */
        { exponent_param = add_global(POWER_NAME);
          globals[exponent_param].value.real = 2.0;  /* default */
          globals[exponent_param].flags |=  ORDINARY_PARAM;
        }
  hooke_power =  globals[exponent_param].value.real; 
  length_param = lookup_global(LENGTH_NAME);
  if ( length_param < 0 ) /* missing, so add */
        { length_param = add_global(LENGTH_NAME);
          if ( web.representation == STRING )
             globals[length_param].value.real 
                 = web.total_area/web.skel[EDGE].count; /* default */
             else globals[length_param].value.real 
                 = sqrt(2.3*web.total_area/web.skel[FACET].count); /* default */
          globals[length_param].flags |=  ORDINARY_PARAM;
        }
  hooke_length =  globals[length_param].value.real; 
}

/*******************************************************************
*
*  function: hooke_energy
*
*  purpose:  energy of one edge, deviation from set edge length
*
*/

REAL hooke_energy(e_info)
struct qinfo *e_info;
{
  REAL d,diff;

  d = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  diff = fabs(d - hooke_length);
  if ( hooke_power == 0.0 ) return -log(diff);
  return pow(diff,hooke_power);
}



/*******************************************************************
*
*  function: hooke_energy_gradient
*
*  purpose:  energy grad of one edge, deviation from set edge length
*
*/

REAL hooke_energy_gradient(e_info)
struct qinfo *e_info;
{
  REAL d,diff,coeff;
  REAL energy;
  int j;

  d = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  diff = d - hooke_length;
  if ( diff == 0.0 ) return 0.0;
  if ( hooke_power == 0.0 )
  { energy = -log(fabs(diff));
     coeff = -1/diff/d;
  }
  else
  { energy =  pow(fabs(diff),hooke_power);
     coeff = hooke_power*energy/diff/d;
  }
  for ( j = 0 ; j < SDIM ; j++ )
     {
        e_info->grad[0][j] = -coeff*e_info->sides[0][0][j];
        e_info->grad[1][j] = coeff*e_info->sides[0][0][j];
     }
  return energy;
}




/*******************************************************************
*
*  function: hooke_energy_hessian
*
*  purpose:  energy hessian of one edge, deviation from set edge length
*
*/

REAL hooke_energy_hessian(e_info)
struct qinfo *e_info;
{
  REAL d,diff,coeff;
  REAL energy;
  int j,jj;
  REAL ****h = e_info->hess;

  d = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  diff = d - hooke_length;
  if ( diff == 0.0 ) return 0.0;
  if ( hooke_power == 0.0 )
  { energy = -log(fabs(diff));
     coeff = -1/diff/d;
  }
  else
  { energy =  pow(fabs(diff),hooke_power);
     coeff = hooke_power*energy/diff/d;
  }
  for ( j = 0 ; j < SDIM ; j++ )
     {
        e_info->grad[0][j] = -coeff*e_info->sides[0][0][j];
        e_info->grad[1][j] = coeff*e_info->sides[0][0][j];
     }

  if ( hooke_power == 0.0 )
  {
     for ( j = 0 ; j < SDIM ; j++ )
     { REAL f;
        f = -1/diff/d;
        h[1][1][j][j] += f;
        h[1][0][j][j] -= f;
        h[0][1][j][j] -= f;
        h[0][0][j][j] += f;
        for ( jj = 0 ; jj < SDIM ; jj++ )
        { f = (1/diff + 1/d)/diff*e_info->sides[0][0][j]*e_info->sides[0][0][jj]/d/d;
          h[1][1][j][jj] += f;
          h[1][0][j][jj] -= f;
          h[0][1][j][jj] -= f;
          h[0][0][j][jj] += f;
        }
     }
  }
  else
  {
     for ( j = 0 ; j < SDIM ; j++ )
     { REAL f;
        f = hooke_power*energy/diff/d;
        h[1][1][j][j] += f;
        h[1][0][j][j] -= f;
        h[0][1][j][j] -= f;
        h[0][0][j][j] += f;
        for ( jj = 0 ; jj < SDIM ; jj++ )
        { f = hooke_power*energy/diff*((hooke_power-1)/diff - 1.0/d)
                     *e_info->sides[0][0][j]*e_info->sides[0][0][jj]/d/d;
          h[1][1][j][jj] += f;
          h[1][0][j][jj] -= f;
          h[0][1][j][jj] -= f;
          h[0][0][j][jj] += f;
        }
     }
  } 
  return energy;
}


/*******************************************************************
  
     Hooke Energy 2 - Hooke's Law to keep edge lengths nearly 
     uniform. Version using extra edge attribute "hooke_size"
     as base length.  Also example of using extra attributes
     in quantities.

********************************************************************/

#define POWER2_NAME "hooke2_power"
#define HOOKE2_ATTR_NAME "hooke_size"
static REAL hooke2_power;
static int hooke2_attr;  /* index number of hooke_size attribute */

/***************************************************************
*
*  function: hooke2_energy_init()
*
*  purpose: initialization for hooke2_energy() and 
*              hooke2_energy_gradient().
*
*/

void hooke2_energy_init(mode,mi)
int mode; /* energy or gradient */
struct method_instance *mi;
{ edge_id e_id;

  if ( web.modeltype != LINEAR )
     kb_error(1767,"hooke2_energy only for LINEAR model.\n",RECOVERABLE);


  exponent_param = lookup_global(POWER2_NAME);
  if ( exponent_param < 0 ) /* missing, so add */
        { exponent_param = add_global(POWER2_NAME);
          globals[exponent_param].value.real = 2.0;  /* default */
          globals[exponent_param].flags |=  ORDINARY_PARAM;
        }
  hooke2_power =  globals[exponent_param].value.real; 

  /* extra edge atribute */
  hooke2_attr = find_attribute(EDGE,HOOKE2_ATTR_NAME);
  if ( hooke2_attr < 0 ) /* not found */
  { hooke2_attr = add_attribute(EDGE,HOOKE2_ATTR_NAME,REAL_ATTR,1 /*dim*/,
          DUMP_ATTR,NULL);
     FOR_ALL_EDGES(e_id)  /* initialize to current length */
     { calc_edge(e_id);
        *((REAL*)(get_extra(e_id,hooke2_attr))) = get_edge_length(e_id);
     }
  }
}

/*******************************************************************
*
*  function: hooke2_energy
*
*  purpose:  energy of one edge, deviation from set edge length
*
*/

REAL hooke2_energy(e_info)
struct qinfo *e_info;
{
  REAL d,diff;

  d = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  diff = fabs(d - *((REAL*)get_extra(e_info->id,hooke2_attr)));
  if ( hooke2_power == 0.0 ) return -log(diff);
  return pow(diff,hooke2_power);
}



/*******************************************************************
*
*  function: hooke2_energy_gradient
*
*  purpose:  energy grad of one edge, deviation from set edge length
*
*/

REAL hooke2_energy_gradient(e_info)
struct qinfo *e_info;
{
  REAL d,diff,coeff;
  REAL energy;
  int j;

  d = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  diff = (d - *((REAL*)get_extra(e_info->id,hooke2_attr)));
  if ( diff == 0.0 ) return 0.0;
  if ( hooke2_power == 0.0 )
  { energy = -log(fabs(diff));
     coeff = -1/diff/d;
  }
  else
  { energy =  pow(fabs(diff),hooke2_power);
     coeff = hooke2_power*energy/diff/d;
  }
  for ( j = 0 ; j < SDIM ; j++ )
     {
        e_info->grad[0][j] = -coeff*e_info->sides[0][0][j];
        e_info->grad[1][j] = coeff*e_info->sides[0][0][j];
     }
  return energy;
}


/*******************************************************************
*
*  function: hooke2_energy_hessian
*
*  purpose:  energy hessian of one edge, deviation from set edge length
*
*/

REAL hooke2_energy_hessian(e_info)
struct qinfo *e_info;
{
  REAL d,diff,coeff;
  REAL energy;
  int j,jj;
  REAL ****h = e_info->hess;

  d = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  diff = (d - *((REAL*)get_extra(e_info->id,hooke2_attr)));
  if ( diff == 0.0 ) return 0.0;
  if ( hooke2_power == 0.0 )
  { energy = -log(fabs(diff));
     coeff = -1/diff/d;
  }
  else
  { energy =  pow(fabs(diff),hooke2_power);
     coeff = hooke2_power*energy/diff/d;
  }
  for ( j = 0 ; j < SDIM ; j++ )
     {
        e_info->grad[0][j] = -coeff*e_info->sides[0][0][j];
        e_info->grad[1][j] = coeff*e_info->sides[0][0][j];
     }

  if ( hooke2_power == 0.0 )
  {
     for ( j = 0 ; j < SDIM ; j++ )
     { REAL f;
        f = -1/diff/d;
        h[1][1][j][j] += f;
        h[1][0][j][j] -= f;
        h[0][1][j][j] -= f;
        h[0][0][j][j] += f;
        for ( jj = 0 ; jj < SDIM ; jj++ )
        { f = (1/diff + 1/d)/diff*e_info->sides[0][0][j]*e_info->sides[0][0][jj]/d/d;
          h[1][1][j][jj] += f;
          h[1][0][j][jj] -= f;
          h[0][1][j][jj] -= f;
          h[0][0][j][jj] += f;
        }
     }
  }
  else
  {
     for ( j = 0 ; j < SDIM ; j++ )
     { REAL f;
        f = hooke2_power*energy/diff/d;
        h[1][1][j][j] += f;
        h[1][0][j][j] -= f;
        h[0][1][j][j] -= f;
        h[0][0][j][j] += f;
        for ( jj = 0 ; jj < SDIM ; jj++ )
        { f = hooke2_power*energy/diff*((hooke2_power-1)/diff - 1.0/d)
                     *e_info->sides[0][0][j]*e_info->sides[0][0][jj]/d/d;
          h[1][1][j][jj] += f;
          h[1][0][j][jj] -= f;
          h[0][1][j][jj] -= f;
          h[0][0][j][jj] += f;
        }
     }
  } 
  return energy;
}


/*******************************************************************
  
     Hooke Energy 3 - Hooke's Law using elastic model.
     Uses "hooke_size" edge attribute as equilibrium length.
     energy = 0.5*(length-hooke_size)^2/hooke_size
    
********************************************************************/

#define POWER3_NAME "hooke3_power"
#define HOOKE3_ATTR_NAME "hooke_size"
static REAL hooke3_power;
static int hooke3_attr;  /* index number of hooke_size attribute */

static int frickenhaus_flag; /* special feature for S. Frickenhaus */

/***************************************************************
*
*  function: hooke3_energy_init()
*
*  purpose: initialization for hooke3_energy() and 
*              hooke3_energy_gradient().
*
*/

void hooke3_energy_init(mode,mi)
int mode; /* energy or gradient */
struct method_instance *mi;
{ edge_id e_id;
  int n;

  if ( web.modeltype != LINEAR )
     kb_error(1767,"hooke3_energy only for LINEAR model.\n",RECOVERABLE);


  exponent_param = lookup_global(POWER3_NAME);
  if ( exponent_param < 0 ) /* missing, so add */
        { exponent_param = add_global(POWER3_NAME);
          globals[exponent_param].value.real = 2.0;  /* default */
          globals[exponent_param].flags |=  ORDINARY_PARAM;
        }
  hooke3_power =  globals[exponent_param].value.real; 

  /* extra edge atribute */
  hooke3_attr = find_attribute(EDGE,HOOKE3_ATTR_NAME);
  if ( hooke3_attr < 0 ) /* not found */
  { hooke3_attr = add_attribute(EDGE,HOOKE3_ATTR_NAME,REAL_ATTR,1 /*dim*/,
          DUMP_ATTR,NULL);
     FOR_ALL_EDGES(e_id)  /* initialize to current length */
     { calc_edge(e_id);
        *((REAL*)(get_extra(e_id,hooke3_attr))) = get_edge_length(e_id);
     }
  }

  n = lookup_global("frickenhaus_flag");
  if ( (n >= 0) && (globals[n].value.real != 0.0) ) frickenhaus_flag = 1;
  else frickenhaus_flag = 0;
}

/*******************************************************************
*
*  function: hooke3_energy
*
*  purpose:  energy of one edge, deviation from set edge length
*
*/

REAL hooke3_energy(e_info)
struct qinfo *e_info;
{
  REAL d,diff,length;

  length = *((REAL*)get_extra(e_info->id,hooke3_attr));
  if ( length == 0.0 )
  { sprintf(errmsg,"Edge %d has %s zero.\n",ordinal(e_info->id)+1,
        HOOKE3_ATTR_NAME);
     kb_error(2011,errmsg,RECOVERABLE);
  }
  d = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  if ( frickenhaus_flag && (d < length) ) return 0.0;
  diff = fabs(d - length);
  if ( hooke3_power == 0.0 ) return -log(diff);
  return 0.5*pow(diff,hooke3_power)/length;
}



/*******************************************************************
*
*  function: hooke3_energy_gradient
*
*  purpose:  energy grad of one edge, deviation from set edge length
*
*/

REAL hooke3_energy_gradient(e_info)
struct qinfo *e_info;
{
  REAL d,diff,coeff;
  REAL energy;
  int j;
  REAL length;

  length = *((REAL*)get_extra(e_info->id,hooke3_attr));
  d = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  if ( frickenhaus_flag && (d < length) ) return 0.0;
  diff = (d - length);
  if ( diff == 0.0 ) return 0.0;
  if ( hooke3_power == 0.0 )
  { energy = -log(fabs(diff));
     coeff = -1/diff/d;
  }
  else
  { energy = 0.5*pow(fabs(diff),hooke3_power)/length;
     coeff  = hooke3_power*energy/diff/d;
  }
  for ( j = 0 ; j < SDIM ; j++ )
     {
        e_info->grad[0][j] = -coeff*e_info->sides[0][0][j];
        e_info->grad[1][j] = coeff*e_info->sides[0][0][j];
     }
  return energy;
}


/*******************************************************************
*
*  function: hooke3_energy_hessian
*
*  purpose:  energy hessian of one edge, deviation from set edge length
*
*/

REAL hooke3_energy_hessian(e_info)
struct qinfo *e_info;
{
  REAL d,diff,coeff;
  REAL energy;
  int j,jj;
  REAL ****h = e_info->hess;
  REAL length;

  length = *((REAL*)get_extra(e_info->id,hooke3_attr));
  d = sqrt(SDIM_dot(e_info->sides[0][0],e_info->sides[0][0]));
  if ( frickenhaus_flag && (d < length) ) return 0.0;
  diff = (d - length);
  if ( diff == 0.0 ) return 0.0;
  if ( hooke3_power == 0.0 )
  { energy = -log(fabs(diff));
     coeff = -1/diff/d;
  }
  else
  { energy = 0.5*pow(fabs(diff),hooke3_power)/length;
     coeff  = hooke3_power*energy/diff/d;
  }
  for ( j = 0 ; j < SDIM ; j++ )
     {
        e_info->grad[0][j] = -coeff*e_info->sides[0][0][j];
        e_info->grad[1][j] = coeff*e_info->sides[0][0][j];
     }

  if ( hooke3_power == 0.0 )
  {
     for ( j = 0 ; j < SDIM ; j++ )
     { REAL f;
        f = -1/diff/d;
        h[1][1][j][j] += f;
        h[1][0][j][j] -= f;
        h[0][1][j][j] -= f;
        h[0][0][j][j] += f;
        for ( jj = 0 ; jj < SDIM ; jj++ )
        { f = (1/diff + 1/d)/diff*e_info->sides[0][0][j]*e_info->sides[0][0][jj]/d/d;
          h[1][1][j][jj] += f;
          h[1][0][j][jj] -= f;
          h[0][1][j][jj] -= f;
          h[0][0][j][jj] += f;
        }
     }
  }
  else
  {
     for ( j = 0 ; j < SDIM ; j++ )
     { REAL f;
        f = hooke3_power*energy/diff/d;
        h[1][1][j][j] += f;
        h[1][0][j][j] -= f;
        h[0][1][j][j] -= f;
        h[0][0][j][j] += f;
        for ( jj = 0 ; jj < SDIM ; jj++ )
        { f = hooke3_power*energy/diff*((hooke3_power-1)/diff - 1.0/d)
                     *e_info->sides[0][0][j]*e_info->sides[0][0][jj]/d/d;
          h[1][1][j][jj] += f;
          h[1][0][j][jj] -= f;
          h[0][1][j][jj] -= f;
          h[0][0][j][jj] += f;
        }
     }
  } 
  return energy;
}

