/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/* matrix.c */

/* matrix routines, mostly from Numerical Recipes */

/* Note: matrix allocation routine assumes whole matrix can be 
    allocated in one call to calloc.  If not true (say for large
    matrices on an IBM PC), dmatrix() and free_matrix() will
    have to be modified */

/* Note: Matrices allocated as row-pointer structures.
    The -1 index is used to store a private copy of 
    pointer to memory, so users may swap row pointers freely.
*/

#include "include.h"

/*****************************************************************
*
* Function: nerror()
*
* Purpose: Numerical Recipes error reporting
*
*/

void nrerror(error_text)
char *error_text;
{ 
  kb_error(1204,error_text,RECOVERABLE);
}

/******************************************************************
*
* Function: matcopy()
*
* Purpose:  copies matrix b to matrix a 
*           for zero-base indexing only 
*/

void matcopy(a,b,rows,cols)
REAL **a,**b;
int rows,cols;
{
  int i;

  for ( i = 0 ; i < rows ; i++ )
     memcpy((char *)a[i],(char *)b[i],cols*sizeof(REAL));
}

/******************************************************************
*
* Function: kb_dmatrix()
*
* Purpose: Allocates zeroed 2D matrix in pointer-pointer form,
*          a REAL matrix with range [rlo..rhi][clo..chi] 
*/

#ifdef MEMSTRINGS
REAL **kb_dmatrix(rlo,rhi,clo,chi,file,line)
int rlo,rhi,clo,chi;
char *file; int line;
#else
REAL **kb_dmatrix(rlo,rhi,clo,chi)
int rlo,rhi,clo,chi;
#endif
{
  int i;
  REAL **m;

#ifdef MEMSTRINGS
  if ( memdebug)
  { sprintf(msg,"dmatrix from %s  %d.\n",file,line);
     outstring(msg);
  }
#endif
  if ( rhi-rlo+1 == 0 ) return NULL;
  m = (REAL **)mycalloc((unsigned)(rhi-rlo+1+1),sizeof(REAL *));
  if ( !m ) nrerror("dmatrix allocation error.");
  m -= rlo;
  m++; /* room for private pointer */

  m[rlo-1] = (REAL *) mycalloc((unsigned)(chi-clo+1),(rhi-rlo+1)*sizeof(REAL));
  if ( !m[rlo-1] ) nrerror("dmatrix allocation error.");
  for ( i = rlo ; i <= rhi ; i++ )
        m[i] = m[rlo-1] + (i - rlo)*(chi - clo + 1) - clo;

  return m;
}

/******************************************************************
*
* Function: kb_dmatrix3()
*
* Purpose: Allocates a zeroed 3D matrix in pointer-pointer-pointer form,
*  a REAL matrix with range [0..n1-1][0..n2-1][0..n3-1] 
*
*/

#ifdef MEMSTRINGS
REAL ***kb_dmatrix3(n1,n2,n3,file,line)
int n1,n2,n3;
char * file; int line;
#else
REAL ***kb_dmatrix3(n1,n2,n3)
int n1,n2,n3;
#endif
{
  int i,j;
  REAL ***m;

#ifdef MEMSTRINGS
  if ( memdebug )
  { sprintf(msg,"dmatrix3 from %s  %d.\n",file,line);
     outstring(msg);
  }
#endif

  if ( n1 <= 0 ) n1 = 1;
  if ( n2 <= 0 ) n2 = 1;
  /* assumes all pointers same machine size and alignment */
  m = (REAL ***)mycalloc((n2+1)*n1+1,sizeof(REAL **));
  if ( !m ) nrerror("dmatrix3 allocation error.");

  m++; /* room for private pointer to doubles */
  for ( i = 0 ; i < n1 ; i++ )
     m[i] = (REAL **)(m + n1 + i*n2);
  m[0][0] = (REAL *) mycalloc(n1*n2*n3,sizeof(REAL));
  m[-1] = (REAL **)(m[0][0]);
  if ( !m[0][0] ) nrerror("dmatrix3 allocation error.");
  for ( i = 0 ; i < n1 ; i++ )
     for ( j = 0 ; j < n2 ; j++ )
        m[i][j] = m[0][0] + i*n2*n3 + j*n3;

  return m;
}

/******************************************************************
*
* Function: kb_dmatrix4()
*
* Purpose: Allocates a zeroed 4D matrix in pointer form,
*  a REAL matrix with range [0..n1-1][0..n2-1][0..n3-1][0..n4-1] 
*/

#ifdef MEMSTRINGS
REAL ****kb_dmatrix4(n1,n2,n3,n4,file,line)
int n1,n2,n3,n4;
char * file; int line;
#else
REAL ****kb_dmatrix4(n1,n2,n3,n4)
int n1,n2,n3,n4;
#endif
{
  int i,j,k;
  REAL ****m;

#ifdef MEMSTRINGS
  if ( memdebug )
  { sprintf(msg,"dmatrix4 from %s  %d.\n",file,line);
     outstring(msg);
  }
#endif

  if ( n1 <= 0 ) n1 = 1;
  if ( n2 <= 0 ) n2 = 1;
  if ( n3 <= 0 ) n3 = 1;

  /* assumes all pointers same machine size and alignment */
  m = (REAL ****)mycalloc(1+n1+n1*n2+n1*n2*n3,sizeof(REAL ***));
  if ( !m ) nrerror("dmatrix4 allocation error.");

  m++; /* room for private pointer */
  for ( i = 0 ; i < n1 ; i++ )
     { m[i] = (REAL ***)(m + n1 + i*n2);
        for ( j = 0 ; j < n2 ; j++ )
          m[i][j] = (REAL **)(m + n1 + n1*n2 + i*n2*n3 + j*n3);
     }
  m[0][0][0] = (REAL *) mycalloc(n1*n2*n3*n4,sizeof(REAL));
  m[-1] = (REAL***)(m[0][0][0]);
  if ( !m[0][0][0] ) nrerror("dmatrix4 allocation error.");
  for ( i = 0 ; i < n1 ; i++ )
     for ( j = 0 ; j < n2 ; j++ )
        for ( k = 0 ; k < n3 ; k++ )
          m[i][j][k] = m[0][0][0] + i*n2*n3*n4 + j*n3*n4 + k*n4;

  return m;
}

/******************************************************************
*
* Function: matNd_setup()
*
* Purpose: routines for initializing matrices declared as local variables 
*          with MAT2D etc macros.
*/

REAL ** mat2d_setup(name,spacename,rows,cols)
REAL **name;
REAL *spacename;
int rows,cols;
{ REAL **spot = name;
  for ( ; rows > 0 ; rows--,spacename += cols,spot++ )
     *spot = spacename;
  return name;
}

REAL *** mat3d_setup(name,spacename,rows,cols,levels)
REAL ***name;
REAL *spacename;
int rows,cols,levels;
{ int i;
  REAL ***spot;
  REAL **row = (REAL **)(name + rows);
  for ( spot = name ; rows > 0 ; rows--,spot++ )
  { *spot = row;
     for ( i = 0 ; i < cols ; i++,spacename += levels, row++ )
        *row = spacename;
  }
  return name;
}

REAL **** mat4d_setup(name,spacename,rows,cols,levels,tiers)
REAL ****name;
REAL *spacename;
int rows,cols,levels,tiers;
{ int i,j;
  REAL ***row = (REAL ***)(name + rows);
  REAL **col = (REAL **)(name + rows + rows*cols);
  REAL ****spot;
  for (spot=name ; rows > 0 ; rows--,spot++ )
  { *spot = row;
     for ( i = 0 ; i < cols ; i++, row++ )
     { *row = col;
        for ( j = 0 ; j < levels ; j++,spacename += tiers, col++ )
          *col = spacename;
     }
  }
  return name;
}
/* end local declaration routines */

/******************************************************************
*
* Function: ivector(), vector()
*
* Purpose: allocate integer or real vector with given index range.
*/

int *ivector(lo,hi)
int lo,hi;
/* allocates a int vector with range [lo..hi] */
{
  int *v;

  v = (int *)mycalloc((unsigned)(hi-lo+1),sizeof(int));
  if ( !v ) nrerror("allocation failure in ivector().");
  return v-lo;
}

REAL *vector(lo,hi)
int lo,hi;
/* allocates a REAL vector with range [lo..hi] */
{
  REAL *v;

  v = (REAL *)mycalloc((unsigned)(hi-lo+1),sizeof(REAL));
  if ( !v ) nrerror("allocation failure in vector().");
  return v-lo;
}

void free_ivector(v,lo,hi)
int *v,lo,hi;
{
  myfree((char *)(v+lo));
}

void free_vector(v,lo,hi)
REAL *v;
int lo,hi;
{
  myfree((char *)(v+lo));
}

/******************************************************************
*
* Function: free_matrixN()
*
* Purpose: Deallocate storage allocated by kb_dmatrixN().
*/

void free_matrix(m)
REAL **m;
{
  if ( !m ) return;
  myfree((char *)m[-1]);  /* using private pointer */
  myfree((char *)(m-1));
}

void free_matrix3(m)
REAL ***m;
{
  if ( !m ) return;
  myfree((char *)m[-1]);
  myfree((char *)(m-1));
}

void free_matrix4(m)
REAL ****m;
{
  if ( !m ) return;
  myfree((char *)m[-1]);
  myfree((char *)(m-1));
}

/******************************************************************
*
* Function: vector_add()
*
* Purpose: add vector b to vector a 
*/
void vector_add(a,b,n)
REAL *a,*b;
int n;
{ for(;n!=0;n--) *(a++) += *(b++);
}

/******************************************************************
*
* Function: vector_add_smul()
*
* Purpose: add scalar multiple of vector b to vector a 
*/
void vector_add_smul(a,b,c,n)
REAL *a,*b;
REAL c;
int n;
{ for(;n!=0;n--) *(a++) += c*(*(b++));
}

/******************************************************************
*
* Function: vector_sub()
*
* Purpose: subtract vector b from vector a  
*/

void vector_sub(a,b,n)
REAL *a,*b;
int n;
{ for(;n!=0;n--) *(a++) -= *(b++);
}

/******************************************************************
*
* Function: vnormal()
*
* Purpose: given 3 points, find cross product of sides 
*/
void vnormal(a,b,c,n)
REAL *a,*b,*c,*n;
{
  REAL aa[MAXCOORD],bb[MAXCOORD];
  int i;

  for ( i = 0 ; i < SDIM ; i++ )
  { aa[i] = a[i] - c[i];
    bb[i] = b[i] - c[i];
  }
  cross_prod(aa,bb,n);
}
  
/******************************************************************
*
* Function: cross_product()
*
* Purpose; Find 3D cross product of a and b, return in c
*/
void cross_prod(a,b,c)
REAL *a,*b,*c;
{
  c[0] = a[1]*b[2] - a[2]*b[1];
  c[1] = a[2]*b[0] - a[0]*b[2];
  c[2] = a[0]*b[1] - a[1]*b[0];
} 

/******************************************************************
*
* Function: triple_prod()
*
* Purpose: Find scalar triple product in 3D.
*/
REAL triple_prod(a,b,c)
REAL *a,*b,*c;
{
  return  a[0]*(b[1]*c[2] - b[2]*c[1]) - a[1]*(b[0]*c[2] - b[2]*c[0])
             + a[2]*(b[0]*c[1] - b[1]*c[0]);
}

/******************************************************************
*
* Function: dot(), dotdf(), dotf()
*
* Purpose: dot products of various REALs and floats.
*
*/
/* dot product of REALS */
REAL dot(a,b,n)
REAL *a,*b;
int n;  /* number of items */
{
  REAL x = 0.0;
  for (  ; --n >= 0 ;  ) x += (*(a++))*(*(b++));
  return x;
}

/* dot product for doubles and floats */
REAL dotdf(a,b,n)
REAL *a;
float *b;
int n;  /* number of items */
{
  REAL x = 0.0;
  for (  ; --n >= 0 ;  ) x += (*(a++))*(*(b++));
  return x;
}

/* dot product for floats */
REAL dotf(a,b,n)
float *a,*b;
int n;  /* number of items */
{
  REAL x = 0.0;
  for (  ; --n >= 0 ;  ) x += (*(a++))*(*(b++));
  return x;
}


/******************************************************************
*
* Function: matvec_mul()
*
* Purpose:  matrix times vector multiplication, c = a * b 
*/
void matvec_mul(a,b,c,rows,cols)
REAL **a,*b,*c;
int rows,cols;
{
  int i,j;

  for ( i = 0 ; i < rows ; i++ )
  { c[i] = 0.0;
    for ( j = 0 ; j < cols ; j++ )
      c[i] += a[i][j]*b[j];
  }
}


/******************************************************************
*
* Function: vec_mat_mul()
*
* Purpose: vector times matrix multiplication, c = a * b 
*/
void vec_mat_mul(a,b,c,rows,cols)
REAL *a,**b,*c;
int rows,cols;
{
  int i,j;

  for ( i = 0 ; i < cols ; i++ )
  { c[i] = 0.0;
    for ( j = 0 ; j < rows ; j++ )
      c[i] += a[j]*b[j][i];
  }
}

/******************************************************************
*
* Function: mat_mult()
*
* Purpose: matrix by matrix multiplication,  c = a * b 
*          a is imax x jmax, b is jmax x kmax, c is imax x kmax
*       a, b, and c need not be distinct.
*/
void mat_mult(a,b,c,imax,jmax,kmax)
REAL **a,**b,**c;  /* not assumed distinct */
int imax,jmax,kmax;
{
  REAL **temp;  /* temporary storage, if needed */
  int i,j,k;

  if ( (a == c) || (b == c) )
  { temp = dmatrix(0,imax-1,0,kmax-1);  /* temporary storage */
    for ( i = 0 ; i < imax ; i++ )
      for ( j = 0 ; j < jmax ; j++ )
        for ( k = 0 ; k < kmax ; k++ )
          temp[i][k] += a[i][j]*b[j][k];
    matcopy(c,temp,imax,kmax);
    free_matrix(temp);
  }
  else
  { for ( i = 0 ; i < imax ; i++ )
      for ( k = 0 ; k < kmax ; k++ )
      { c[i][k] = 0.0;
        for ( j = 0 ; j < jmax ; j++ )
        c[i][k] += a[i][j]*b[j][k];
      }
  }
}

/******************************************************************
*
* Function: tr_mat_mul()
*
* Purpose: matrix transpose by matrix multiplication 
*          output: c = aT*b 
*          a is imax x jmax, b is imax x kmax, c is jmax x kmax 
*       a, b, and c need not be distinct.
*/
void tr_mat_mul(a,b,c,imax,jmax,kmax)
REAL **a,**b,**c;  /* not assumed distinct */
int imax,jmax,kmax; 
{
  REAL **temp;  /* temporary storage, if needed */
  int i,j,k;

  if ( (a == c) || (b == c) )
  { temp = dmatrix(0,jmax-1,0,kmax-1);  /* temporary storage */
    for ( j = 0 ; j < jmax ; j++ )
      for ( k = 0 ; k < kmax ; k++ )
        for ( i = 0 ; i < imax ; i++ )
          temp[j][k] += a[i][j]*b[i][k];
    matcopy(c,temp,jmax,kmax);
    free_matrix(temp);
  }
  else
  { REAL *s;
    for ( j = 0 ; j < jmax ; j++ )
      for ( k = 0, s = c[j] ; k < kmax ; k++,s++ )
      { *s = 0.0;
         for ( i = 0 ; i < imax ; i++ )
           *s += a[i][j]*b[i][k];
      }
  }
}

/******************************************************************
*
* Function: mat_mul_tr()
*
* Purpose: matrix by matrix transpose multiplication,  c = a * bT 
*       a is imax x jmax, b is kmax x jmax, c is imax x kmax
*       a, b, and c need not be distinct.
*/  
void mat_mul_tr(a,b,c,imax,jmax,kmax)
REAL **a,**b,**c;  /* not assumed distinct */
int imax,jmax,kmax;
{
  REAL **temp;  /* temporary storage, if needed */
  int i,j,k;

  if ( (a == c) || (b == c) )
  { temp = dmatrix(0,imax-1,0,kmax-1);  /* temporary storage */
    for ( i = 0 ; i < imax ; i++ )
      for ( j = 0 ; j < jmax ; j++ )
        for ( k = 0 ; k < kmax ; k++ )
          temp[i][k] += a[i][j]*b[k][j];
    matcopy(c,temp,imax,kmax);
    free_matrix(temp);
  }
  else
  { for ( i = 0 ; i < imax ; i++ )
      for ( k = 0 ; k < kmax ; k++ )
      { c[i][k] = 0.0;
        for ( j = 0 ; j < jmax ; j++ )
          c[i][k] += a[i][j]*b[k][j];
      }
  }
}

/******************************************************************
*
* Function: mat_tsquare()
*
* Purpose: matrix times own transpose, b = a*aT
*          a and b must be different. 
*/
void  mat_tsquare(a,b,n,m)
REAL **a; /* original */
REAL **b; /* square  b = a*aT */
int n,m; /* a is nxm, b is nxn */
{
  int i,j;
  if ( a == b )  kb_error(3771,"mat_tsquare: a and b same (internal error).\n",
    RECOVERABLE);
  for ( i = 0 ; i < n ; i++ )
    for ( j = 0 ; j <= i ; j++ )
      b[i][j] = b[j][i] = dot(a[i],a[j],m);
}

/******************************************************************
*
* Function: quadratic_form()
*
* Purpose: quadratic form evaluation, a*b*c; only uses lower triangle 
*/
REAL quadratic_form(a,b,c,n)
REAL *a,**b,*c;
int n; /* size */ 
{ int i,j;
  REAL sum = 0.0;
  REAL temp;

  for ( i = 0 ; i < n ; i++ )
  { temp = b[i][0]*c[0];
    for ( j = 1 ; j <= i ; j++ )
      temp += b[i][j]*c[j];
    for (  ; j < n ; j++ )
      temp += b[j][i]*c[j];
    sum += a[i]*temp;
  }

  return sum;
}


/******************************************************************
*
* Function: mat_inv()
*
* Purpose: in-place matrix inverse by gauss-jordan 
* returns -1 for singular matrix, 1 for nonsingular 
*/

#define SWAP(a,b) {REAL temp = (a); (a) = (b); (b) = temp; }
#define SMALL 10

int mat_inv(a,n)
REAL **a;     /* matrix to invert in place */
int n;        /* size of matrix */
{
  int *indxc,*indxr,*ipiv;
  int i,icol=0,irow=0,j,k,l,ll;
  REAL big,dum,pivinv;
  int retval = 1;  /* default return value is success */
  int temp1[SMALL],temp2[SMALL],temp3[SMALL]; /* avoid alloc for small sizes */

  if ( n <= SMALL )
  { indxc = temp1; indxr = temp2; ipiv = temp3; }
  else
  { /* large size */
    indxc = ivector(0,n-1);
    indxr = ivector(0,n-1);
    ipiv  = ivector(0,n-1);
  }
  for ( j = 0 ; j < n ; j++ ) ipiv[j] = -1;
  for ( i = 0 ; i < n ; i++ )
  { big = 0.0;
    for ( j = 0 ; j < n ; j++ )
      if ( ipiv[j] != 0 )
         for ( k = 0 ; k < n ; k++ )
         { if ( ipiv[k] == -1 )
           { if ( fabs(a[j][k]) >= big )
             { big = fabs(a[j][k]);
               irow = j;
               icol = k;
             }
           }
           else if ( ipiv[k] > 0 ) { retval = -1; goto mat_inv_exit; }
         }
      ++(ipiv[icol]);

      if ( irow != icol )
         for ( l = 0 ; l < n ; l++ ) SWAP(a[irow][l],a[icol][l])
      indxr[i] = irow;
      indxc[i] = icol;
      if ( a[icol][icol] == 0.0 ) { retval = -1; goto mat_inv_exit; }
      pivinv = 1/a[icol][icol];
      a[icol][icol] = 1.0;
      for ( l = 0 ; l < n ; l++ ) a[icol][l] *= pivinv;
      for ( ll = 0  ; ll < n ; ll++ )
        if ( ll != icol )
        { dum = a[ll][icol];
          a[ll][icol] = 0.0;
          for ( l = 0 ; l < n ; l++ ) a[ll][l] -= a[icol][l]*dum;
        }
  }
  for ( l = n-1 ; l >= 0 ; l-- )
  { if ( indxr[l] != indxc[l] )
       for ( k = 0 ; k < n ; k++ )
          SWAP(a[k][indxr[l]],a[k][indxc[l]])
  }

mat_inv_exit:
  if ( n > SMALL )
  { free_ivector(ipiv,0,n-1);
    free_ivector(indxr,0,n-1);
    free_ivector(indxc,0,n-1);
  }
  return retval;
}

/******************************************************************
*
* Function: det_adjoint()
*
* Purpose: calculates determinant in place and leaves adjoint transpose 
*/
REAL  det_adjoint(a,n)
REAL **a;     /* matrix to change in place */
int n;        /* size of matrix */
{
  int *indxc,*indxr,*ipiv;
  int i,icol=0,irow=0,j,k,l,ll;
  REAL big,dum,pivinv,piv;
  int temp1[SMALL],temp2[SMALL],temp3[SMALL]; /* avoid alloc for small sizes */
  REAL det = 1.0;  /* will multiply by pivots */

  if ( n <= 0 ) kb_error(1205,"Internal error: Matrix size not positive.",RECOVERABLE);

  if ( n == 1 ) { det = a[0][0]; a[0][0] = 1.0; return det; }
  if ( n == 2 )
  { REAL temp;
    det = a[0][0]*a[1][1] - a[0][1]*a[1][0];
    temp = a[0][0]; a[0][0] = a[1][1]; a[1][1] = temp;
    a[0][1] = -a[0][1]; a[1][0] = -a[1][0];
    return det;
  }

  if ( n <= SMALL )
  { indxc = temp1; indxr = temp2; ipiv = temp3; }
  else
  { /* large size */
    indxc = ivector(0,n-1);
    indxr = ivector(0,n-1);
    ipiv  = ivector(0,n-1);
  }
  for ( j = 0 ; j < n ; j++ ) ipiv[j] = -1;
  for ( i = 0 ; i < n-1 ; i++ )
  { big = 0.0;
    for ( j = 0 ; j < n ; j++ )
      if ( ipiv[j] != 0 )
         for ( k = 0 ; k < n ; k++ )
         { if ( ipiv[k] == -1 )
           { if ( fabs(a[j][k]) >= big )
             { big = fabs(a[j][k]);
               irow = j;
               icol = k;
             }
           }
           else if ( ipiv[k] > 0 )
           { kb_error(1206,"Internal: ipiv > 0.\n",WARNING); det = 0.0; goto det_exit; }
         }
      ++(ipiv[icol]);

      if ( irow != icol )
      { for ( l = 0 ; l < n ; l++ ) SWAP(a[irow][l],a[icol][l])
        det = -det;
      }
      indxr[i] = irow;
      indxc[i] = icol;
      det *= a[icol][icol];  /* build determinant */
      if ( a[icol][icol] == 0.0 ) { goto det_lowrank; }
      pivinv = 1/a[icol][icol];
      a[icol][icol] = 1.0;
      for ( l = 0 ; l < n ; l++ ) a[icol][l] *= pivinv;
      for ( ll = 0  ; ll < n ; ll++ )
        if ( ll != icol )
        { dum = a[ll][icol];
          a[ll][icol] = 0.0;
          for ( l = 0 ; l < n ; l++ ) a[ll][l] -= a[icol][l]*dum;
        }
  }
  /* special treatment for last pivot; works even if zero */
  for ( j = 0 ; j < n ; j++ )
     if ( ipiv[j] != 0 ) { irow = icol = j; break; }
  indxr[n-1] = irow;
  indxc[n-1] = icol;
  piv = a[icol][icol];
  a[icol][icol] = 1.0;
  for ( l = 0 ; l < n ; l++ ) a[icol][l] *= det;
  for ( ll = 0  ; ll < n ; ll++ )
    if ( ll != icol )
    { dum = a[ll][icol];
      a[ll][icol] = 0.0;
      for ( l = 0 ; l < n ; l++ ) 
         a[ll][l] = a[ll][l]*piv*det - a[icol][l]*dum;
    }
  det *= piv;

  for ( l = n-1 ; l >= 0 ; l-- )
  { if ( indxr[l] != indxc[l] )
      for ( k = 0 ; k < n ; k++ )
        SWAP(a[k][indxr[l]],a[k][indxc[l]])
  }

det_exit:
  if ( n > SMALL )
  { free_ivector(ipiv,0,n-1);
    free_ivector(indxr,0,n-1);
    free_ivector(indxc,0,n-1);
  }
  return det;

det_lowrank: /* rank less than n-1, so adjoint = 0 */
  for ( i = 0 ; i < n ; i++ )
     for ( j = 0 ; j < n ; j++ )
        a[i][j] = 0.0;
  det = 0.0;
  goto det_exit;
  
}


/******************************************************************
*
* Function: determinant()
*
* Purpose: calculates determinant; no change in matrix for 3x3 or smaller 
*           otherwise calls det_adjoint()
*/
REAL  determinant(a,n)
REAL **a;     /* matrix to change in place */
int n;        /* size of matrix */
{
  if ( n == 1 ) { return a[0][0];  }
  if ( n == 2 ) { return  a[0][0]*a[1][1] - a[0][1]*a[1][0]; }
  if ( n == 3 )
     { return a[0][0]*(a[1][1]*a[2][2] - a[1][2]*a[2][1])
              - a[0][1]*(a[1][0]*a[2][2] - a[1][2]*a[2][0])
              + a[0][2]*(a[1][0]*a[2][1] - a[1][1]*a[2][0]);
     }
  return det_adjoint(a,n);  /* other cases */
}

/******************************************************************
*
* Function: print_matrix()
*
*/
void print_matrix(a,rows,cols)
REAL **a;
int rows,cols;
{
  int i,j;

  for ( i = 0 ; i < rows ; i++ )
    { msg[0] = 0;
      for ( j = 0 ; j < cols ; j++ )
        sprintf(msg+strlen(msg),"%10.6f ",(DOUBLE)a[i][j]);
      strcat(msg,"\n");
      outstring(msg);
    }
}

/******************************************************************
*
* Function: exterior_product()
*
* Purpose: conversion of k vectors to a k-vector 
*          components in index lexicographic order 
*/
void exterior_product(v,w,k,n)
REAL **v;  /* list of k vectors */
REAL *w;    /* returned k-vector */
int k;      /* number of vectors */
int n;      /* space dimension */
{
  /* anticipate only small k, so just brute force */
  int i1,i2,i3;

  switch ( k )
    {
      case 1:  for ( i1 = 0 ; i1 < n ; i1++ ) *(w++) = v[0][i1];
               break;

      case 2:  for ( i1 = 0 ; i1 < n ; i1++ )
                 for ( i2 = i1+1 ; i2 < n ; i2++ )
                    *(w++) = v[0][i1]*v[1][i2] - v[0][i2]*v[1][i1];
               break;

      case 3:  for ( i1 = 0 ; i1 < n ; i1++ )
                 for ( i2 = i1+1 ; i2 < n ; i2++ )
                   for ( i3 = i2+1 ; i3 < n ; i3++ )
                      *(w++) = v[0][i1]*v[1][i2]*v[2][i3]
                             + v[0][i2]*v[1][i3]*v[2][i1] 
                             + v[0][i3]*v[1][i1]*v[2][i2] 
                             - v[0][i1]*v[1][i3]*v[2][i2] 
                             - v[0][i3]*v[1][i2]*v[2][i1] 
                             - v[0][i2]*v[1][i1]*v[2][i3] ;
               break;

      default: sprintf(errmsg,"Exterior product of %d vectors.\n",k);
               kb_error(1207,errmsg,RECOVERABLE);

               break;
    }
}

/**********************************************************************
*
*  function: kernel_basis()
*
*  purpose:  Find basis for kernel of matrix (nullspace of rows)
*/

int kernel_basis(a,ker,imax,jmax)
REAL **a;  /* the matrix, will be altered */
REAL **ker; /* for basis vectors in columns */
int imax,jmax;  /* rows and columns of a */
{
  int i,j,k;
  int pivrow[20];    /* pivot row in column */
  int n; /* nullity */

  for ( j = 0 ; j < jmax ; j++ ) pivrow[j] = -1;  /* mark as no pivot in col */

  /* get row echelon form, pivot largest in each row */
  for ( i = 0 ; i < imax ; i++ )
  { int piv = -1;
    REAL b,big,p;

    /* find largest element in row */
    big = 0.0;
    for ( j = 0 ; j < jmax ; j++ )
      if ( fabs(a[i][j]) > big )
      { big = fabs(a[i][j]);
        piv = j;
      }
    if ( piv == -1 ) continue; /* row of zeros */
    pivrow[piv] = i;

    /* pivot step */
    p = a[i][piv];
    for ( j = 0 ; j < jmax ; j++ )
      a[i][j] /= p;
    for ( k = 0 ; k < imax ; k++ )
    { if ( k == i ) continue;
      b = a[k][piv];
      for ( j = 0 ; j < jmax ; j++ )
           a[k][j] -= b*a[i][j];
    }
  }         

  /* now find kernel basis */
  for ( j = 0, n = 0 ; j < jmax ; j++ )
  { if ( pivrow[j] >= 0 ) continue;  /* column has leading 1 */
    /* column j is parameter column */
    for ( k = 0 ; k < jmax ; k++ )
    { if ( pivrow[k] >= 0 )
         ker[k][n] = -a[pivrow[k]][j];
      else if ( k == j )
         ker[k][n] = 1.0;
      else ker[k][n] = 0.0;
    }
    n++;
  }
  return n; /* nullity */
}

/**********************************************************************
*
*  function: kernel_basis_rows()
*
*  purpose:  Find basis for kernel of matrix (nullspace of rows)
*                Returns basis rowwise.
*      basis vectors normalized, but not orthohormal.
*/

int kernel_basis_rows(a,ker,imax,jmax)
REAL **a;  /* the matrix, will be altered */
REAL **ker; /* for basis vectors in rows */
int imax,jmax;  /* rows and columns of a */
{
  int i,j,k;
  int pivrow[20];    /* pivot row in column */
  int pivcol[20];    /* pivot column  in row */
  int n; /* nullity */
  int  detsign=1;  /* to try to keep orientation of normal positive */

  for ( j = 0 ; j < jmax ; j++ ) pivrow[j] = -1;  /* mark as no pivot in col */

  /* get row echelon form, pivot largest in each row */
  for ( i = 0 ; i < imax ; i++ )
  { int piv = -1;
    REAL b,big,p;

    /* find largest element in row */
    big = 0.0;
    for ( j = 0 ; j < jmax ; j++ )
      if ( fabs(a[i][j]) > big )
      { big = fabs(a[i][j]);
        piv = j;
      }
    if ( piv == -1 ) continue; /* row of zeros */
    pivrow[piv] = i; pivcol[i] = piv;

    /* pivot step */
    p = a[i][piv];   if ( p < 0 ) detsign = -detsign;
    for ( j = 0 ; j < jmax ; j++ )
      a[i][j] /= p;
    for ( k = 0 ; k < imax ; k++ )
    { if ( k == i ) continue;
      b = a[k][piv];
      for ( j = 0 ; j < jmax ; j++ )
         a[k][j] -= b*a[i][j];
    }
  }         

  /* now find kernel basis */
  for ( j = 0, n = 0 ; j < jmax ; j++ )
  { int sign;
    if ( pivrow[j] >= 0 ) continue;  /* column has leading 1 */
    /* column j is parameter column */
    pivcol[imax+n] = j;
    /* get sign for pos det */
    for (sign = detsign, k=0 ; k <= imax+n ; k++ )
      for ( i = k+1 ; i <= imax+n ; i++ )
        if ( pivcol[k] > pivcol[i] ) sign = -sign;
    for ( k = 0 ; k < jmax ; k++ )
    { if ( pivrow[k] >= 0 )
        ker[n][k] = -sign*a[pivrow[k]][j];
      else if ( k == j )
        ker[n][k] = sign;
      else ker[n][k] = 0.0;
    }
    n++;
  }
  /* normalize */
  for ( i = 0 ; i < n ; i ++ )
  { REAL mag;
    mag = sqrt(SDIM_dot(ker[i],ker[i]));
    for ( j = 0 ; j < SDIM ; j++ )
      ker[i][j] /= mag;
  } 

  return n; /* nullity */
}


/*********************************************************************
*
* function: matrix_index()
*
* purpose:  return number of negative eigenvalues of matrix
*           Does not destroy original.  For symmetric matrices.
*
*/

int matrix_index(M,n)
REAL **M;  /* square matrix */
int n;  /* size */
{ REAL **a = dmatrix(0,n-1,0,n-1);
  REAL *tempptr;
  int row,col,prow=0;
  REAL maxp;
  int i,j;
  int indx = 0;
  REAL temp;
  REAL *firstrow;  /* for proper freeing after swapping rows */

  firstrow = a[0];
  matcopy(a,M,n,n);


  /* basically, gauss elimination to lower triangular form
     with partial pivoting.  
  */
  for ( col = 0 ; col < n ; col++ )
  { /* find max pivot in diagonal */
    maxp = 0.0;
    for ( row = col ; row < n ; row++ )
       if ( fabs(a[row][row]) > maxp ) { maxp = fabs(a[row][row]); prow = row; }
    if ( maxp == 0.0 ) continue;
    if ( prow != col )
    { /* swap rows and colums to keep symmetric */
      tempptr = a[prow]; a[prow] = a[col]; a[col] = tempptr;
      for ( j = col; j < n ; j++ )
      { temp = a[j][col]; a[j][col] = a[j][prow]; a[j][prow] = temp; }
    }
    if ( a[col][col] < 0.0 ) indx++;
    for ( row = col+1 ; row < n ; row++ )
      for ( i = col+1 ; i < n ; i++ )
      { a[row][i] -= a[row][col]/a[col][col]*a[col][i];
      }
  }
  a[0] = firstrow; free_matrix(a);
  return indx;
}


/****************************************************************************
*
* function: jacobi_eigenpairs()
*
* purpose: find eigenpairs of small dense symmetric matrix by jacobi rotations
*          From Numerical Recipes.
*
* output: eigenvalues in d[], sorted in descending order, and
*         corresponding eigenvectors in columns of v[][].
*/
void jacobi_eigenpairs(a,n,d,v)
REAL **a;  /* input matrix, destroyed */
int n; /* size */
REAL *d; /* for return of eigenvalues */
REAL **v;  /* for return of eigenvectors */
{ REAL sm,h,tresh,dum,g,t,theta,tau,s,c;
  int i,iq,ip,nrot,j;
  REAL *z,*b;

  z = (REAL*)temp_calloc(n,sizeof(REAL));
  b = (REAL*)temp_calloc(n,sizeof(REAL));

  /* initialize v to identity */
  for ( ip = 0 ; ip < n ; ip++ )
  { for ( iq = 0 ; iq < n ; iq++ ) v[ip][iq] = 0.0;
    v[ip][ip] = 1.0;
  }

  for ( ip = 0 ; ip < n ; ip++ )
  { b[ip] = a[ip][ip];
    d[ip] = b[ip];
    z[ip] = 0.0;
  }

  nrot = 0;
  for ( i = 1 ; i < 10 ; i++ )
  { sm = 0.0;
    for ( ip = 0 ; ip < n-1; ip++ )
      for ( iq = ip + 1 ; iq < n ; iq++ )
        sm += fabs(a[ip][iq]);

    if ( sm == 0.0 ) goto jacobi_exit; /* normal exit */

    if ( i < 4 ) tresh = .2*sm/n/n;
    else tresh = 0.0;

    for ( ip = 0 ; ip < n-1 ; ip++ )
     for ( iq = ip+1 ; iq < n ; iq++ )
     { g = 100*fabs(a[ip][iq]);
       dum = fabs(d[ip]);
       if ( (i > 4) && (dum+g == dum) && (fabs(d[iq])+g == fabs(d[iq])) )
         a[ip][iq] = 0.0;
       else if ( fabs(a[ip][iq]) > tresh ) 
       { h = d[iq] - d[ip];
         if ( fabs(h) + g == fabs(h) ) t = a[ip][iq]/h;
         else 
         { theta = .5*h/a[ip][iq];
           t = 1.0/(fabs(theta) + sqrt(1 + theta*theta));
           if ( theta < 0.0 ) t = -t;
         }
         c = 1.0/sqrt(1 + t*t);
         s = t*c;
         tau = s/(1+c);
         h = t*a[ip][iq];
         z[ip] -= h;
         z[iq] += h;
         d[ip] -= h;
         d[iq] += h;
         a[ip][iq] = 0.0;
         for ( j = 0 ; j <= ip-1 ; j++ )
         { g = a[j][ip];
           h = a[j][iq];
           a[j][ip] = g - s*(h+g*tau);
           a[j][iq] = h + s*(g-h*tau);
         }
         for ( j = ip+1 ; j <= iq-1 ; j++ )
         { g = a[ip][j];
           h = a[j][iq];
           a[ip][j] = g - s*(h+g*tau);
           a[j][iq] = h + s*(g-h*tau);
         }
         for ( j = iq+1 ; j < n ; j++ )
         { g = a[ip][j];
           h = a[iq][j];
           a[ip][j] = g - s*(h+g*tau);
           a[iq][j] = h + s*(g-h*tau);
         }
         for ( j = 0 ; j < n ; j++ )
         { g = v[j][ip];
           h = v[j][iq];
           v[j][ip] = g - s*(h+g*tau);
           v[j][iq] = h + s*(g-h*tau);
         }
         nrot++;
      } /* end if */

     } /* end iq */
      /* end ip */

     for ( ip = 0 ; ip < n ; ip++ )
     { b[ip] += z[ip];
       d[ip] = b[ip];
       z[ip] = 0.0;
     }
  } /* end i */

  printf("50 iterations should never happen.\n");
  { temp_free((char*)z); temp_free((char*)b); return; }

jacobi_exit:
  /* sort eigenpairs in descending order, insertion sort */
  for ( i = 0 ; i < n-1 ; i++ )
  { REAL p;
    int k;
    k = i;
    p = d[i];
    for ( j = i + 1 ; j < n ; j++ )
    { if ( d[j] >= p ) 
      { k = j; p = d[j]; }
    }
    if ( k != i )
    { d[k] = d[i];
      d[i] = p;
      for ( j = 0 ; j < n ; j++ ) 
      { p = v[j][i]; v[j][i] = v[j][k]; v[j][k] = p; }
    }
  }

  temp_free((char*)z); temp_free((char*)b); 
  return; 
}

/**********************************************************************
*
* function: det_hess(a,h,n)
*
* Purpose: find hessian of determinant as function of entries
*
* Returns  h[i1][j1][i2][j2] as d^2 det(a)/da[i1][j1]/da[i2][j2]
*/

void det_hess(a,h,n)
REAL **a;
REAL ****h;
int n;  /* size */
{ int i1,i2,jj1,j2,k;

  /* copy original matrix into h lots of times */
  for ( i1 = 0 ; i1 < n ; i1++ )
    for ( jj1 = 0 ; jj1 < n ; jj1++ )
     for ( i2 = 0 ; i2 < n ; i2++ )
      for ( j2 = 0 ; j2 < n ; j2++ )
         h[i1][jj1][i2][j2] = a[i2][j2];

  /* replace element row and column with identity stuff */
  for ( i1 = 0 ; i1 < n ; i1++ )
    for ( jj1 = 0 ; jj1 < n ; jj1++ )
    { for ( k = 0 ; k < n ; k++ )
      { h[i1][jj1][i1][k] = 0.0;
         h[i1][jj1][k][jj1] = 0.0;
      }
      h[i1][jj1][i1][jj1] = 1.0;
    }

  /* find adjoints */
  for ( i1 = 0 ; i1 < n ; i1++ )
    for ( jj1 = 0 ; jj1 < n ; jj1++ )
    { det_adjoint(h[i1][jj1],n);
      h[i1][jj1][jj1][i1] = 0.0; /* need fixup */
    }

  /* transpose to get back to hessian */
  for ( i1 = 0 ; i1 < n ; i1++ )
    for ( jj1 = 0 ; jj1 < n ; jj1++ )
     for ( i2 = 1 ; i2 < n ; i2++ )
      for ( j2 = 0 ; j2 < i2 ; j2++ )
      { REAL tmp =  h[i1][jj1][i2][j2];
         h[i1][jj1][i2][j2] = h[i1][jj1][j2][i2];
         h[i1][jj1][j2][i2] = tmp;
      }
}

/**********************************************************************
*
* function: gram_schmidt()
*
* purpose: orthonormalize rows of a matrix
*
* return: number of independent rows
*/

int gram_schmidt(mat,rows,cols)
REAL **mat;
int rows, cols;
{ int i,j,k;
  REAL d;
  for ( i = 0 ; i < rows ; i++ )
  { for ( j = 0 ; j < i ; j++ )
     { REAL c = dot(mat[i],mat[j],cols);
        for ( k = 0 ; k < cols ; k++ ) mat[i][k] -= c*mat[j][k];
     }
     d = dot(mat[i],mat[i],cols);
     if ( d < 1e-25 ) 
     { rows--; 
        for ( k = 0 ; k < cols ; k++ ) mat[i][k] = mat[rows][k];
        i--;
     }
     else
     { d = 1/sqrt(d);
        for ( k = 0 ; k < cols ; k++ ) mat[i][k] *= d;
     }
  }
  return rows;
} 

