/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: plugin_cb.c
 */

#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "widget.h"
#include "task.h"
#include "plugin_cb.h"
#include "thing.h"
#include "readable.h"
#include "pixmap.h"
#include "logging.h"
#include "help.h"

/*
 *
 *   void on_plugin_selection_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine responds to plugin selection with creating the
 *      task given and creating the acceptable object selection dialog
 *      window to proceed with acceptable parameter and option selections.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - contains code corresponding to task to initiate
 *
 *   Exit:
 *      The task context is created with the given plugin and task code.
 *      This window is hidden and another standard selection window is displayed.
 *
 */
void on_plugin_selection_button_clicked(GtkButton * button, gpointer user_data)
{
	GList *window_list;
	GtkWidget *clist;
	GtkWidget *next_window;
	GtkWidget *selection_window;
	task_action_t action;
	plugin_handle_t plugin;
	plugin_handle_t prev_plugin;

	/*
	 * Hide the selection window and either redisplay the existing task window
	 * if it exists and the plugin selection did not change. If the task window
	 * does not exist, create it. However, if the plugin selections changed and
	 * the task window existed then destroy the current task window and create
	 * a new one.
	 */

	action = GPOINTER_TO_UINT(user_data);
	selection_window = gtk_widget_get_toplevel(GTK_WIDGET(button));
	clist = lookup_widget(GTK_WIDGET(button), "selection_window_clist");
	plugin = GPOINTER_TO_UINT(get_single_select_current_row_data(GTK_CLIST(clist)));
	window_list = get_window_list(selection_window);
	next_window = gtk_object_get_data(GTK_OBJECT(selection_window), "next_window_id");
	prev_plugin =
	    GPOINTER_TO_UINT(gtk_object_get_data(GTK_OBJECT(selection_window), "previous_plugin"));

	if (window_list == NULL)
		window_list = g_list_append(window_list, selection_window);

	if (plugin != prev_plugin) {
		gint rc;
		task_handle_t handle;

		rc = evms_create_task(plugin, action, &handle);

		if (rc != SUCCESS) {
			log_error("%s: evms_create_task() returned error code %d.\n", __FUNCTION__,
				  rc);

			display_results_window(rc, NULL, _("Unable to create the task context."),
					       NULL, FALSE, selection_window);
		} else {
			gint count;
			gchar *next_button_text = NULL;

			/*
			 * If we already created the next window for selecting
			 * the acceptable objects (and maybe even other follow-on 
			 * windows like for options), make sure we destroy them
			 * all and clean up the list since our plugin selections
			 * have apperently changed and so all the following panels
			 * need to be recreated.
			 */

			if (next_window != NULL)
				destroy_window_list(g_list_find(window_list, next_window));

			/*
			 * If we have no options then set the Next button label to
			 * indicate the actual task action rather than "Next".
			 */

			if (evms_get_option_count(handle, &count) == SUCCESS && count == 0)
				next_button_text = get_task_action_string(handle);

			next_window =
			    create_standard_selection_window(_("Select Plugin Acceptable Objects"),
							     next_button_text,
							     get_acceptable_objects_help_text(),
							     on_acceptable_objects_clist_realize,
							     on_acceptable_objects_button_clicked,
							     on_button_clicked_display_prev_window,
							     on_button_clicked_destroy_window_list,
							     on_acceptable_objects_clist_select_row,
							     on_acceptable_objects_clist_unselect_row,
							     GUINT_TO_POINTER(handle));

			window_list = g_list_append(window_list, next_window);

			set_window_list(next_window, window_list);
			set_window_list(selection_window, window_list);

			gtk_object_set_data(GTK_OBJECT(selection_window), "next_window_id",
					    next_window);
			gtk_object_set_data(GTK_OBJECT(selection_window), "previous_plugin",
					    GUINT_TO_POINTER(plugin));

			/*
			 * Connect the destroy signal to the next window. If the window is
			 * destroyed, so is the task context.
			 */

			gtk_signal_connect(GTK_OBJECT(next_window), "destroy",
					   GTK_SIGNAL_FUNC(on_task_window_destroy),
					   GUINT_TO_POINTER(handle));

			if (next_button_text)
				g_free(next_button_text);

			gtk_widget_show(next_window);
			gtk_widget_hide(selection_window);
		}
	} else {
		gtk_widget_show(next_window);
		gtk_widget_hide(selection_window);
	}
}

/*
 *
 *   void on_plugin_selection_clist_realize (GtkCList *, evms_plugin_code_t,
 *                                           plugin_search_flags_t flags,
 *                                           task_action_t action)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of plugins desired for a certain task action.
 * 
 *   Entry:
 *      clist  - address of the selections GtkCList widget
 *      code   - the plugin code to enumerate and add to the list
 *      flags  - plugin search flags
 *      action - the code for the task action we want to perform
 *
 *   Exit:
 *      Selection list populated with plugins of the given type
 *
 */
void on_plugin_selection_clist_realize(GtkCList * clist, evms_plugin_code_t code,
				       plugin_search_flags_t flags, task_action_t action)
{
	gint rc;
	handle_array_t *plugins;

	rc = evms_get_plugin_list(code, flags, &plugins);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		gint i;
		gboolean is_selected = (plugins->count == 1);

		for (i = 0; i < plugins->count; i++) {
			if (can_plugin_do_action(plugins->handle[i], action) == TRUE)
				add_thing_to_selection_list(clist, plugins->handle[i], is_selected);
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(plugins);
	}

	/*
	 * We won't need the standard three columns (pixmap, size, description) for
	 * the plugin list so hide the size column.
	 */

	gtk_clist_set_column_visibility(clist, SL_SIZE_COLUMN, FALSE);
}

/*
 *
 *   void on_plugin_function_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine starts a task for a plug-in specific function.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      See description.
 *
 */
void on_plugin_function_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	engine_handle_t handle;
	function_info_t *info;

	handle = GPOINTER_TO_UINT(user_data);
	info = gtk_object_get_user_data(GTK_OBJECT(menuitem));

	if (info)
		initiate_task(NULL, handle, info->function, info->title, info->verb, info->help);
	else
		log_error("%s: menu item is missing function info!", __FUNCTION__);
}

/*
 *
 *   function_info_t *copy_function_info (function_info_t *)
 *   
 *   Description:
 *      This routine produces a copy of the given function_info_t
 *      structure.
 * 
 *   Entry:
 *      info - the function_info_t structure to copy
 *
 *   Exit:
 *      See description.
 *
 */
function_info_t *copy_function_info(function_info_t * info)
{
	function_info_t *new_info;

	new_info = g_new0(function_info_t, 1);

	new_info->function = info->function;

	if (info->title)
		new_info->title = g_strdup(info->title);

	if (info->verb)
		new_info->verb = g_strdup(info->verb);

	if (info->help)
		new_info->help = g_strdup(info->help);

	log_debug("%s: returning %p.\n", __FUNCTION__, new_info);

	return new_info;
}

/*
 *
 *   void free_function_info_copy (function_info_t *)
 *   
 *   Description:
 *      This routine frees the strings and the copy
 *      of the function_info_t structure allocated 
 *      with copy_function_info.
 * 
 *   Entry:
 *      info - the function_info_t structure to free
 *
 *   Exit:
 *      The function_info_t copy is freed.
 *
 */
void free_function_info_copy(function_info_t * info)
{
	log_debug("%s: Freeing function info %p.\n", __FUNCTION__, info);

	g_free(info->title);
	g_free(info->verb);
	g_free(info->help);
	g_free(info);
}

/*
 *
 *   void on_popup_menu_item_destroy (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine frees the copy of a function_info_t associated
 *      with a popup menu item when the menu item is destroyed.
 *
 *   Entry:
 *      widget    - the menu item being destroyed
 *      user_data - address of a function_info_t for this menu item
 *
 *   Exit:
 *      The function_info_t copy is freed.
 *
 */
void on_popup_menu_item_destroy(GtkWidget * widget, gpointer user_data)
{
	free_function_info_copy(user_data);
}

/*
 *
 *   void add_plugin_specific_functions_to_menu (GtkMenu *, engine_handle_t)
 *   
 *   Description:
 *      This routine queries the plugin for additional functions specific
 *      to the plugin for the thing in question. It then appends menu items
 *      for each function to the menu supplied.
 * 
 *   Entry:
 *      menu   - the menu to append plugin functions to
 *      handle - the handle of the thing the functions are for
 *
 *   Exit:
 *      Plug-in specific functions for the object are appended to the given menu
 *
 */
void add_plugin_specific_functions_to_menu(GtkMenu * menu, engine_handle_t handle)
{
	gint rc;
	function_info_array_t *functions;

	rc = evms_get_plugin_functions(handle, &functions);

	if (rc == SUCCESS) {
		gint i;
		gchar *label;
		GtkWidget *item;
		function_info_t *info;

		for (i = 0; i < functions->count; i++) {
			info = copy_function_info(&(functions->info[i]));

			label = g_strdup_printf("%s...", info->title);
			item = gtk_menu_item_new_with_label(label);
			gtk_menu_append(menu, item);

			gtk_signal_connect(GTK_OBJECT(item), "activate",
					   GTK_SIGNAL_FUNC(on_plugin_function_menu_item_activate),
					   GUINT_TO_POINTER(handle));

			gtk_signal_connect(GTK_OBJECT(item), "destroy",
					   GTK_SIGNAL_FUNC(on_popup_menu_item_destroy), info);

			gtk_object_set_user_data(GTK_OBJECT(item), info);

			/* Gray out plug-in functions that are inactive */
			if (functions->info[i].flags & EVMS_FUNCTION_FLAGS_INACTIVE)
				gtk_widget_set_sensitive(GTK_WIDGET(item), FALSE);

			g_free(label);
		}

		evms_free(functions);
	}
}

/*
 *
 *   void on_thing_with_tasks_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of things that have any plug-in specific functions/tasks
 *      available for them.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - contains type of thing to enumerate (object, segment, etc.)
 *
 *   Exit:
 *      Selection list populated with things with tasks available
 *
 */
void on_thing_with_tasks_clist_realize(GtkWidget * widget, gpointer user_data)
{
	gint rc = 0;
	gchar *col_title;
	GtkCList *clist = GTK_CLIST(widget);
	object_type_t type = GPOINTER_TO_UINT(user_data);
	handle_array_t *things;

	switch (type) {
	case VOLUME:
		col_title = _("Logical Volume");
		rc = evms_get_volume_list(0, 0, 0, &things);
		break;

	case CONTAINER:
		col_title = _("Storage Container");
		rc = evms_get_container_list(0, 0, 0, &things);
		break;

	default:
		col_title = _("Storage Object");
		rc = evms_get_object_list(0, 0, 0, 0, 0, &things);
		break;
	}

	if (rc != SUCCESS) {
		log_error("%s: evms_get_xxxx_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		guint i;
		gboolean is_selected = (things->count == 1);
		function_info_array_t *functions;

		set_selection_window_clist_column_titles(clist, _("Size"), col_title, NULL);

		for (i = 0; i < things->count; i++) {
			if (evms_get_plugin_functions(things->handle[i], &functions) == 0) {
				add_thing_to_selection_list(clist, things->handle[i], is_selected);
				evms_free(functions);
			}
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(things);
	}
}

/*
 *
 *   void on_plugin_with_tasks_clist_realize (GtkCList *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of plugins that have plugin specific tasks available for
 *      them.
 * 
 *   Entry:
 *      clist     - address of the selections GtkCList widget
 *      user_data - not used
 *
 *   Exit:
 *      Selection list populated with plugins that have plugin
 *      supplied functions applicable to them.
 *
 */
void on_plugin_with_tasks_clist_realize(GtkCList * clist, gpointer user_data)
{
	gint rc;
	handle_array_t *plugins;

	set_selection_window_clist_column_titles(clist, NULL, _("Plug-in"), NULL);

	rc = evms_get_plugin_list(0, 0, &plugins);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		gint i;
		gboolean is_selected = (plugins->count == 1);
		function_info_array_t *functions;

		for (i = 0; i < plugins->count; i++) {
			if (evms_get_plugin_functions(plugins->handle[i], &functions) == 0) {
				add_thing_to_selection_list(clist, plugins->handle[i], is_selected);
				evms_free(functions);
			}
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(plugins);
	}

	/*
	 * We won't need the standard three columns (pixmap, size, description) for
	 * the plugin list so hide the size column.
	 */

	gtk_clist_set_column_visibility(clist, SL_SIZE_COLUMN, FALSE);
}

/*
 *
 *   void on_thing_plugin_tasks_clist_realize (GtkCList *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of plugin tasks for a given handle/thing.
 * 
 *   Entry:
 *      clist     - address of the selections GtkCList widget
 *      user_data - contains the handle to query plugin functions
 *
 *   Exit:
 *      Selection list populated with task descriptions for the
 *      given handle.
 *
 */
void on_thing_plugin_tasks_clist_realize(GtkCList * clist, gpointer user_data)
{
	gint rc;
	engine_handle_t handle = GPOINTER_TO_UINT(user_data);
	function_info_array_t *functions;

	set_selection_window_clist_column_titles(clist, NULL, _("Task Description"), NULL);

	rc = evms_get_plugin_functions(handle, &functions);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_plugin_functions() returned error code %d.\n", __FUNCTION__,
			  rc);
	} else {
		gint i;
		gint row;
		gchar *text[] = { "", "", "", "" };
		GdkBitmap *mask;
		GdkPixmap *pixmap;

		for (i = 0; i < functions->count; i++) {
			text[SL_DESC_COLUMN] = functions->info[i].title;

			row = clist_append_row(clist, text);

			if (row != -1) {
				get_exec_16_pixmap(&pixmap, &mask);
				gtk_clist_set_pixmap(clist, row, 0, pixmap, mask);

				gtk_clist_set_row_data_full(clist, row,
							    copy_function_info(&
									       (functions->
										info[i])),
							    (GtkDestroyNotify)
							    free_function_info_copy);

				/* If the plugin function is inactive, make the row unselectable */
				if (functions->info[i].flags & EVMS_FUNCTION_FLAGS_INACTIVE)
					gtk_clist_set_selectable(clist, row, FALSE);
			}
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(functions);
	}

	gtk_clist_set_column_visibility(clist, SL_SIZE_COLUMN, FALSE);
}

/*
 *
 *   void on_thing_plugin_tasks_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the selected plugin function with the
 *      thing that it was to act upon.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - contains handle for thing to act upon
 *
 *   Exit:
 *      This plugin task is initiated.
 *
 */
void on_thing_plugin_tasks_button_clicked(GtkButton * button, gpointer user_data)
{
	GList *window_list;
	GtkWidget *clist;
	GtkWidget *next_window;
	GtkWidget *selection_window;
	function_info_t *info;
	task_action_t prev_function;

	selection_window = gtk_widget_get_toplevel(GTK_WIDGET(button));
	clist = lookup_widget(GTK_WIDGET(button), "selection_window_clist");
	window_list = get_window_list(selection_window);
	info = get_single_select_current_row_data(GTK_CLIST(clist));
	next_window = gtk_object_get_data(GTK_OBJECT(selection_window), "next_window_id");
	prev_function =
	    GPOINTER_TO_UINT(gtk_object_get_data
			     (GTK_OBJECT(selection_window), "previous_function"));

	if (info->function != prev_function) {
		gint rc;

		if (next_window != NULL)
			destroy_window_list(g_list_find(window_list, next_window));

		rc = initiate_task(selection_window, GPOINTER_TO_UINT(user_data),
				   info->function, info->title, info->verb, info->help);

		if (rc == SUCCESS) {
			gtk_widget_hide(selection_window);
			gtk_object_set_data(GTK_OBJECT(selection_window), "previous_function",
					    GUINT_TO_POINTER(info->function));
		}
	} else {
		gtk_widget_show(next_window);
		gtk_widget_hide(selection_window);
	}
}

/*
 *
 *   void on_plugin_function_context_selection_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine creates another selection window containing the tasks
 *      available by the plugin for the thing selected.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - not used
 *
 *   Exit:
 *      This window is hidden and another standard selection window is displayed
 *      containing the functions available for the selected thing by its plugin.
 *
 */
void on_plugin_function_context_selection_button_clicked(GtkButton * button, gpointer user_data)
{
	GList *window_list;
	GtkWidget *clist;
	GtkWidget *next_window;
	GtkWidget *selection_window;
	engine_handle_t thing;
	engine_handle_t prev_thing;

	selection_window = gtk_widget_get_toplevel(GTK_WIDGET(button));
	clist = lookup_widget(GTK_WIDGET(button), "selection_window_clist");
	thing = GPOINTER_TO_UINT(get_single_select_current_row_data(GTK_CLIST(clist)));
	window_list = get_window_list(selection_window);
	next_window = gtk_object_get_data(GTK_OBJECT(selection_window), "next_window_id");
	prev_thing =
	    GPOINTER_TO_UINT(gtk_object_get_data(GTK_OBJECT(selection_window), "previous_thing"));

	if (window_list == NULL)
		window_list = g_list_append(window_list, selection_window);

	if (thing != prev_thing) {
		/*
		 * If we already created the next window for selecting
		 * the plugin task (and maybe even other follow-on 
		 * windows like for options), make sure we destroy them
		 * all and clean up the list since our selection
		 * has apperently changed and so all the following panels
		 * need to be recreated.
		 */

		if (next_window != NULL)
			destroy_window_list(g_list_find(window_list, next_window));

		next_window = create_standard_selection_window(_("Select Plug-in Task"),
							       NULL,
							       NULL,
							       on_thing_plugin_tasks_clist_realize,
							       on_thing_plugin_tasks_button_clicked,
							       on_button_clicked_display_prev_window,
							       on_button_clicked_destroy_window_list,
							       NULL, NULL, GUINT_TO_POINTER(thing));

		window_list = g_list_append(window_list, next_window);

		set_window_list(next_window, window_list);
		set_window_list(selection_window, window_list);

		gtk_object_set_data(GTK_OBJECT(selection_window), "next_window_id", next_window);
		gtk_object_set_data(GTK_OBJECT(selection_window), "previous_thing",
				    GUINT_TO_POINTER(thing));
	}

	gtk_widget_show(next_window);
	gtk_widget_hide(selection_window);
}
