/*----------------------------------------------------------------------
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: evmsn.c
 *---------------------------------------------------------------------*/

/*----------------------------------------------------------------------
 * Change History:
 *
 * 9/2001  John Stiles   Initial version.
 *
 *---------------------------------------------------------------------*/


/* Identify this file. */
#define EVMSN_C    1

/*----------------------------------------------------------------------
 * Necessary include files
 *---------------------------------------------------------------------*/
#include <sys/types.h>
#include <string.h>        /* strcat, strcpy    */
#include <stdlib.h>        /* malloc, free      */
#include <ctype.h>         /* toupper           */
#include <stdio.h>         /* printf            */
#include <unistd.h>        /* geteuid           */
#include <getopt.h>        /* getopt_long       */


#include "evmsn.h"

/*----------------------------------------------------------------------
 * Private Global Variables
 *---------------------------------------------------------------------*/
static engine_mode_t open_mode   = ENGINE_READWRITE;
static debug_level_t debug_level = DEFAULT;
static char          *log_name   = NULL;
static engine_message_functions_t  message_calls;


/*----------------------------------------------------------------------
 * There are No Private Constants
 * There are No Private Type Definitions
 * There are No Global Variables.
 *---------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 * Local Functions
 *---------------------------------------------------------------------*/

static void show_help (char **argv)
{
    printf ("\nEnterprise Volume Management System %s\n", VERSION);
    printf ("International Business Machines  %s\n\n", DATE);
   
    printf ("Usage: %s [OPTION...] \n\n", argv[0]);

    printf ("  -d, --log-level=LEVEL\t\tLogging output level where LEVEL is one of the following:\n");
    printf ("  \t\t\t\t\tcritical | serious | error | warning | default |\n"); 
    printf ("  \t\t\t\t\tdetails | debug | extra | entry-exit | everything\n");
    printf ("  -?, --help\t\t\tShow this help message\n");
    printf ("  -m, --mode=ENGINE-MODE\tEngine mode where ENGINE-MODE is one of the following:\n");
    printf ("  \t\t\t\t\treadwrite | readonly\n");
    printf ("  -l, --log-name=FILENAME\tFile to send logging output to\n\n");
}

/*
 *
 *   static int parse_options (int argc, char **argv)
 *
 *   Description:
 *      This routine parses the command line options. It then sets the
 *      appropriate program variables accordingly.
 *       
 *   Entry:
 *      argc - count of command line arguments
 *      argv - array of program command line argument strings
 *
 *   Exit:
 *      Returns 0 is parsing was successful or -1 if a problem was encountered.
 *
 */
static int parse_options (int argc, char **argv)
{

    int                  c;
    int                  rc=0;
    char                *short_opts = "h?d:m:l:";
    struct option        long_opts[] =
    {
           {"help",      no_argument,       NULL, 'h'},
           {"log-level", required_argument, NULL, 'd'},
           {"mode",      required_argument, NULL, 'm'},
           {"log-name",  required_argument, NULL, 'l'},
           {NULL,        0,                 NULL,  0 }
    };

    while (rc == 0 && (c = getopt_long (argc, argv, short_opts, long_opts, NULL)) != EOF)
    {
           switch (c)
           {
               case 'm':
                   if (strcasecmp (optarg, "readwrite") == 0)
                       open_mode = ENGINE_READWRITE;
                   else if (strcasecmp (optarg, "readonly") == 0)
                       open_mode = ENGINE_READONLY;
                   else
                   {
                       printf (_("Unrecognized mode argument -- \"%s\"\n\n"), optarg);
                       rc = -1;
                   }

                   break;
                          
               case 'd':                    
                   if (strcasecmp (optarg, "critical") == 0)
                       debug_level = CRITICAL;
                   else if (strcasecmp (optarg, "serious") == 0)
                       debug_level = SERIOUS;
                   else if (strcasecmp (optarg, "error") == 0)
                       debug_level = ERROR;
                   else if (strcasecmp (optarg, "warning") == 0)
                       debug_level = WARNING;
                   else if (strcasecmp (optarg, "default") == 0)
                       debug_level = DEFAULT;
                   else if (strcasecmp (optarg, "details") == 0)
                       debug_level = DETAILS;
                   else if (strcasecmp (optarg, "debug") == 0)
                       debug_level = DEBUG;
                   else if (strcasecmp (optarg, "extra") == 0)
                       debug_level = EXTRA;
                   else if (strcasecmp (optarg, "entry-exit") == 0)
                       debug_level = ENTRY_EXIT;
                   else if (strcasecmp (optarg, "everything") == 0)
                       debug_level = EVERYTHING;
                   else
                   {
                       printf (_("Unrecognized debug level argument -- \"%s\"\n\n"), optarg);
                       rc = -1;
                   }

                   break;

               case 'l':
                   log_name = strdup (optarg);
                   break;

               default:
                   printf (_("Unrecognized option -- \"%c\"\n\n"), c);
               case 'h':
               case '?':
                   rc = -1;
                   break;
           }
    }

    if (rc != 0)
        show_help (argv);
 
    return rc;
}

int simple_message( char *msg_text, int *msg_answer, char * * choices )
{
  int i, j, startx, starty, screen_cols, screen_lines, keyval = 0,
      start_index = 0, select_index = 0, num_objects = 0, item_count = 0,
      msg_lines;
  char *holdbuf, *curtext;
  size_t msg_length;

  log_debug( "%s: got message: %s\n", __FUNCTION__, msg_text );

  if ( GL_screen_context == SC_NONE ) {
    screen_cols = GL_screen_cols - XPOS_BOX2TEXT - XPOS_BOX2RIGHT - 2;
    screen_lines = CALC_SCREEN2_LINES;
    startx = XPOS_BOX2TEXT - 1;
    starty = YPOS_BOX2TOP + 1;
  }
  else {
    screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT - 2;
    screen_lines = CALC_SCREEN3_LINES;
    startx = XPOS_BOX3TEXT - 1;
    starty = YPOS_BOX3TOP + 1;
  }

  holdbuf = malloc( screen_cols + 3 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return ENOMEM;
  }

  // first blank the lines
  BLANK_BUFFER( holdbuf, screen_cols + 2 )
  for ( i = starty; i < starty + 1 + screen_lines; i++ )
    print_string( startx, i, COLOR_DEFAULT, holdbuf );

  startx++;
  starty++;
  screen_lines--;
  print_string( startx, starty, COLOR_DEFAULT, _( "Message:  " ));
  starty++;
  screen_lines--;

  msg_lines = screen_lines - 1;
  curtext = msg_text;
  while ( msg_lines ) {
    BLANK_BUFFER( holdbuf, screen_cols )
    msg_length = strlen( curtext );
    if ( msg_length > screen_cols - 2 )
      msg_length = screen_cols - 2;

    for ( i = 0; i < msg_length; i++ ) {
      if ( curtext[i] == '\n' ) {
        i++;
        if ( curtext[i] == '\0' )
          msg_lines = 1;
        break;
      }
      else  if ( curtext[i] == '\0' ) {
        msg_lines = 1;
        break;
      }
      else
        holdbuf[i] = curtext[i];
    }

    curtext += i;
    print_string( startx, starty, COLOR_DEFAULT, holdbuf );
    starty++;
    screen_lines--;
    msg_lines--;
  }
  screen_lines += 2;

  if ( msg_answer != NULL && choices != NULL ) {
    log_debug( "%s: message has answer, initially %i\n", __FUNCTION__, *msg_answer );
    while ( choices[num_objects] != NULL )
      num_objects++;
    log_debug( "%s: counted %i choices\n", __FUNCTION__, num_objects );

    do {

      // first blank the lines
      BLANK_BUFFER( holdbuf, screen_cols )
      for ( i = starty; i < starty + screen_lines - 1; i++ )
        print_string( startx, i, COLOR_DEFAULT, holdbuf );

      if ( start_index + screen_lines > num_objects )
        item_count = num_objects - start_index + starty;
      else
        item_count = screen_lines - 1 + starty;

      for ( i = starty, j = start_index; i < item_count; i++, j++ ) {

        BLANK_BUFFER( holdbuf, screen_cols )
        if ( strlen( choices[j] ) > screen_cols - 1 )
          memcpy( holdbuf, choices[j], screen_cols - 2 );
        else
          memcpy( holdbuf, choices[j], strlen( choices[j] ));

        if ( select_index == j )
          print_string( startx, i, COLOR_BAR, holdbuf );
        else
          print_string( startx, i, COLOR_TITLE, holdbuf );
      }

      draw_status_line( _( "Select a choice and press enter." ));

      keyval = key_scroller( &start_index, &select_index, num_objects, screen_lines - 2 );

      switch ( keyval ) {

        case KEY_ENTER :
          *msg_answer = select_index;
          break;

        default:
          keyval = KEY_CURSORACT;
          break;
      }

    }  while ( keyval == KEY_CURSORACT );
  }
  else {
    starty += 2;
    print_string( startx, starty, COLOR_DEFAULT, info_press_any_keyc );
    get_inputc();
  }

  free( holdbuf );
  return 0;
}


/*----------------------------------------------------------------------
 * Public Functions
 *---------------------------------------------------------------------*/

/*----------------------------------------------------------------------
 *
 *   Function Name:  main
 *
 *   Descriptive Name:  guess
 *
 *   Input:  int argc    - The number of parameters on the command line.
 *           char * argv - An array of pointers to the text of each
 *                          parameter found on the command line.
 *
 *   Output: If Success :  The program's exit code will be 0.
 *
 *           If Failure :  The program's exit code will be the failing
 *                          function's return code.
 *
 *   Error Handling:  If an error occurs, all memory allocated by this
 *                     program is freed, and an error message is output
 *                     to the user.
 *
 *   Side Effects:  The state of the disks, partitions, and/or volumes
 *                   in the system may be altered.  Nodes may be
 *                   created or removed from the /dev tree.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
int main( int argc, char * argv[] )
{
  int  keyval, rc = 0;
  char buf[] = "                                                                         ";

  rc = parse_options (argc, argv);
  
  if (rc == 0)
  {    
      // Initialize Globals
      GL_use_color = 0;
      GL_screen_cols  = 0;
      GL_screen_lines  = 0;
      GL_screen_context = SC_NONE;
    
      // Start up the screen environment
      kb_video_setup();
      update_screen_info();                  /* set up for initial screen size.*/
    
      // Set up the messaging functions
      message_calls.user_message = &simple_message;
      message_calls.user_communication = NULL;
    
      // Draw initial screen
      draw_frame();
      draw_fnkeys( KEYS_DEFAULT );
      keyval = KEY_VOLUMES;                     /* initial view.  */
    
      // Presently, only allow root access
      if ( geteuid() == 0) {
         evms_version_t version;
          
         draw_status_line ( _("EVMS is examining your system. Please wait..."));
     
         evms_get_api_version (&version);
          
         if (REQUIRED_ENGINE_VER_MAJOR == version.major &&        
             ((version.minor > REQUIRED_ENGINE_VER_MINOR) ||
             (REQUIRED_ENGINE_VER_MINOR == version.minor && REQUIRED_ENGINE_VER_PATCHLEVEL <= version.patchlevel))) {
          
             if ( (rc = evms_open_engine( open_mode, &message_calls, debug_level, log_name )) == 0 ) {
        
                rc = control_loop( keyval );
        
                evms_close_engine();
             }
             else
             {
                sprintf( buf, _( "Open EVMS Engine failed, rc == %i " ), rc );
                print_string( 4, 13, COLOR_DEFAULT, buf );
                print_string( 6, 14, COLOR_DEFAULT, info_press_any_keyx );
                draw_status_line( info_press_any_keyx );
                keyval = get_inputc();             
             }
         }
         else
         {
             sprintf( buf, _( "Version %u.%u.%u or later of the EVMS Engine API required. Found version %u.%u.%u." ), 
                               REQUIRED_ENGINE_VER_MAJOR, REQUIRED_ENGINE_VER_MINOR, REQUIRED_ENGINE_VER_PATCHLEVEL,
                               version.major, version.minor, version.patchlevel);
             print_string( 2, 13, COLOR_DEFAULT, buf );
             print_string( 6, 14, COLOR_DEFAULT, info_press_any_keyx );
             draw_status_line( info_press_any_keyx );
             keyval = get_inputc();
             rc = EPERM;
         }
      }
      else {  // Not root...
          print_string( 4, 13, COLOR_DEFAULT, _( "This command must be run with root privilege." ));
          print_string( 6, 14, COLOR_DEFAULT, info_press_any_keyx );
          draw_status_line( info_press_any_keyx );
          keyval = get_inputc();
      }
    
      // Terminate the screen environment
      kb_video_shutdown();
  }
  
  return  rc;
}

