/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: selection_cb.c
 */ 
 
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "help.h"
#include "selection_cb.h"

/*
 *
 *   void on_selection_window_clist_select_row (GtkCList *, gint, gint, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine is called whenever a row is selected. 
 *      All we do is ensure that the "Next" button is 
 *      sensitive to allow the operation to be initiated.
 * 
 *   Entry:
 *      clist     - address of GtkCList object that received select-row signal
 *      row       - index of row selected
 *      column    - index of column selected on current row
 *      event     - address of structure to obtain additional context information
 *      user_data - address of user data bound to clist (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_selection_window_clist_select_row (GtkCList *clist, gint row, gint column,
                                           GdkEventButton *event, gpointer user_data)
{
    GtkWidget *next_button = lookup_widget (GTK_WIDGET (clist), "selection_window_next_button");

    if (!(GTK_WIDGET_IS_SENSITIVE (next_button)))
    {    
        gtk_widget_set_sensitive (next_button, TRUE);
        gtk_widget_grab_default (next_button);
    }
}

/*
 *
 *   void on_selection_window_clist_unselect_row (GtkCList *, gint, gint, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine is called whenever a row is unselected. 
 *      All we do is ensure that the "Next" button is 
 *      not sensitive to disallow the operation from being 
 *      initiated until a choice is made.
 * 
 *   Entry:
 *      clist     - address of GtkCList object that received unselect-row signal
 *      row       - index of row deselected
 *      column    - index of column deselected on current row
 *      event     - address of structure to obtain additional context information
 *      user_data - address of user data bound to clist (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_selection_window_clist_unselect_row (GtkCList *clist, gint row, gint column,
                                             GdkEventButton *event, gpointer user_data)
{
    GtkWidget *next_button = lookup_widget (GTK_WIDGET (clist), "selection_window_next_button");

    if (g_list_length (clist->selection) == 0 && GTK_WIDGET_IS_SENSITIVE (next_button))
        gtk_widget_set_sensitive (next_button, FALSE);
}

/*
 *
 *   void on_selection_window_clist_select_row_extra (GtkCList *, gint, gint, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine is called whenever a row is selected. 
 *      All we do is ensure that the "Next" button is 
 *      sensitive to allow the operation to be initiated.
 *      Before we make it sensitive though we ensure the
 *      selection_window_entry field has viable characters.
 *
 *      NOTE: This callback is not part of the default selection
 *      window handlers. It is connected when someone creates a
 *      a GtkEntry field that requires coordination with a selection
 *      from the selection_window_clist.
 * 
 *   Entry:
 *      clist     - address of GtkCList object that received select-row signal
 *      row       - index of row selected
 *      column    - index of column selected on current row
 *      event     - address of structure to obtain additional context information
 *      user_data - address of user data bound to clist (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_selection_window_clist_select_row_extra (GtkCList *clist, gint row, gint column,
                                                 GdkEventButton *event, gpointer user_data)
{
    GtkWidget *entry = lookup_widget (GTK_WIDGET (clist), "selection_window_entry");
    GtkWidget *next_button = lookup_widget (GTK_WIDGET (clist), "selection_window_next_button");

    if (!(editable_field_is_blank (GTK_EDITABLE (entry))) &&
        !(GTK_WIDGET_IS_SENSITIVE (next_button)))
          gtk_widget_set_sensitive (next_button, TRUE);
}

/*
 *
 *   void on_selection_window_entry_changed (GtkEditable *, gpointer)
 *
 *   Description:
 *      This routine is called whenever the editable text
 *      entry field in the selection window has changed.
 *      If the entry is void of characters or only contains
 *      whitespace then we disable the next button. If it
 *      selection_window_clist has been selected and the
 *      entry now has text then we ensure that the next 
 *      button is enabled.
 *
 *      NOTE: This callback is not part of the default selection
 *      window handlers. It is connected when someone creates a
 *      a GtkEntry field that requires coordination with a selection
 *      from the selection_window_clist.
 * 
 *   Entry:
 *      editable  - address of GtkEditable descendant object that 
 *                  received changed signal
 *      user_data - address of user data bound to widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_selection_window_entry_changed (GtkEditable *editable, gpointer user_data)
{    
    GtkWidget *next_button = lookup_widget (GTK_WIDGET (editable), "selection_window_next_button");

    if (editable_field_is_blank (editable))
    {
       if (GTK_WIDGET_IS_SENSITIVE (next_button))
           gtk_widget_set_sensitive (next_button, FALSE);
    }
    else
    {
        GtkCList *clist = (GtkCList *)lookup_widget (GTK_WIDGET (editable), "selection_window_clist");
        
        if (g_list_length (clist->selection) > 0)
        {
           if (!(GTK_WIDGET_IS_SENSITIVE (next_button)))
               gtk_widget_set_sensitive (next_button, TRUE);
        }
    }
}

/*
 *
 *   void inactivate_next_button (GtkWidget *)
 *   
 *   Description:
 *      This routine makes the "Next" button insensitive if it
 *      is currently sensitive to user input.
 * 
 *   Entry:
 *      widget - a widget in the same window as the next button
 *
 *   Exit:
 *      "Next" button is so insensitive as to hurt your feelings
 *
 */
void inactivate_next_button (GtkWidget *widget)
{
    GtkWidget *next_button;
    
    next_button = lookup_widget (widget, "selection_window_next_button");
        
    if (GTK_WIDGET_IS_SENSITIVE (next_button))
        gtk_widget_set_sensitive (next_button, FALSE);                                
}

/*
 *
 *   void on_empty_selection_clist_set_label_text (GtkWidget *, gchar *)
 *   
 *   Description:
 *      This routine changes the selection window instruction label
 *      text to indicate that the selection list contains no items
 *      for selection.
 * 
 *   Entry:
 *      widget  - a widget in the same window as the selection window label
 *      message - optional user-specified message
 *
 *   Exit:
 *      The selection window instruction label text is changed.
 *
 */
void on_empty_selection_clist_set_label_text (GtkWidget *widget, gchar *message)
{
    GtkWidget *label;
    
    label = lookup_widget (widget, "selection_window_instruction_label");
    
    if (label)
    {    
        if (message)
            gtk_label_set_text (GTK_LABEL (label), message);
        else
            gtk_label_set_text (GTK_LABEL (label), _("No items were found to populate the selection list"));
    }
}

/*
 *
 *   void on_options_window_help_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine displays help on the options/configuration window.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      See description.
 *
 */
void on_options_window_help_button_clicked (GtkButton *button, gpointer user_data)
{
    GtkWidget *parent;
    
    parent = gtk_widget_get_toplevel (GTK_WIDGET (button));
    display_help_window (GTK_WINDOW (parent), options_window_help_text);
}

/*
 *
 *   void on_selection_window_help_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine displays help on the standard selection window.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - contains address of text to display
 *
 *   Exit:
 *      See description.
 *
 */
void on_selection_window_help_button_clicked (GtkButton *button, gpointer user_data)
{
    if (user_data == NULL)
        display_popup_window (_("No help available"), _("No help available"));
    else
    {
        GtkWidget *parent;
        
        parent = gtk_widget_get_toplevel (GTK_WIDGET (button));
        display_help_window (GTK_WINDOW (parent), user_data);
    }
}
