/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils
 * File: evms_vgremove.c
 *
 *	Emulates LVM's 'vgremove' utility using the EVMS Engine. All options
 *	and several status messages are based on the original vgremove command
 *	from Heinz Mauelshagen and Sistina Software (www.sistina.com).
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <libgen.h>
#include <ctype.h>
#include <frontend.h>

typedef struct cmd_options_s {
	int debug;
	int help;
	int verbose;
	int version;
} cmd_options_t;

static char * cmd = NULL;
static cmd_options_t opts;

#include "helpers/get_container_handle.c"
#include "helpers/open_engine.c"
#include "helpers/remove_duplicates.c"


static int showheader( void )
{
	// VERSION and DATE are defined in the top-level make.rules
	printf("Enterprise Volume Management System\n");
	printf("International Business Machines  %s\n", DATE);
	printf("LVM Emulation Utilities %s\n\n", VERSION);
	printf("%s --  Volume Group Remove\n",cmd);
	return 0;
}


static int showhelp( void )
{
	showheader();
	printf("Synopsis:\n");
	printf("---------\n\n");
	printf("%s\n", cmd);
	printf("\t[-d/--debug]\n");
	printf("\t[-h/--help]\n");
	printf("\t[-v/--verbose]\n");
	printf("\t[-V/--version]\n");
	printf("\tVolumeGroupName [VolumeGroupName...]\n");
	return 0;
}


static int parse_options( int		argc,
			char		** argv )
{
	int		c;
	char		* short_opts = "dh?vV";
	struct option	long_opts[] = {
				{ "debug",	no_argument, NULL, 'd'},
				{ "help",	no_argument, NULL, 'h'},
				{ "verbose",	no_argument, NULL, 'v'},
				{ "version",	no_argument, NULL, 'V'},
				{ NULL, 0, NULL, 0} };

    while ( (c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF ) {
		switch (c) {
		case 'd':
			opts.debug++;
			opts.verbose++;
			break;
		case 'h':
		case '?':
			opts.help++;
			break;
		case 'v':
			opts.verbose++;
			break;
		case 'V':
			opts.version++;
			break;
		default:
			printf("%s -- unrecognized option \"%c\"\n\n", cmd, c);
			return EINVAL;
		}
	}

	return 0;
}


int main( int argc, char *argv[] )
{
	object_handle_t		container;
	char			* vg_names[256] = {0};
	int			number_of_vgs;
	int			log_level = DEFAULT;
	int			i, rc;

	memset(&opts, 0, sizeof(cmd_options_t));
	cmd = basename(argv[0]);

	// Get the command line options.
	rc = parse_options(argc, argv);
	if (rc) {
		showhelp();
		return rc;
	}
	if ( opts.help ) {
		showhelp();
		return 0;
	}
	if ( opts.version ) {
		showheader();
		return 0;
	}
	if ( opts.verbose ) {
		log_level = DEBUG;
	}
	if ( opts.debug ) {
		log_level = ENTRY_EXIT;
	}

	// Check for volume group name.
	if ( optind == argc ) {
		printf("%s -- please enter a volume group name\n\n", cmd);
		return EINVAL;
	}
	number_of_vgs = argc - optind;

	// Copy the VG names from the command line, detecting any duplicates.
	rc = remove_duplicates(&argv[optind], vg_names, number_of_vgs);
	if (rc) {
		printf("%s -- Duplicate VGs specified. Please only specify each VG once\n", cmd);
		return rc;
	}

	// Open the EVMS Engine.
	rc = open_engine(ENGINE_READWRITE, log_level);
	if (rc) {
		return rc;
	}

	// Now we can actually remove each volume group, one by one. For now, if
	// any one fails, exit without committing, so nothing is changed.
	for ( i = 0; i < number_of_vgs; i++ ) {
		// Get the handle for the group to be deleted.
		rc = get_container_handle(vg_names[i], &container);
		if (rc) {
			evms_close_engine();
			return rc;
		}

		// Perform the removal.
		rc = evms_destroy(container);
		if (rc) {
			printf("%s -- Error removing container %s (%d)\n", cmd, vg_names[i], rc);
			evms_close_engine();
			return rc;
		}
		printf("%s -- volume group \"%s\" successfully removed\n", cmd, vg_names[i]);
	}

	// Write everything to disk.
	rc = evms_commit_changes(NULL);
	if (rc) {
		printf("%s -- Error committing changes to disk (%d)\n", cmd, rc);
		evms_close_engine();
		return rc;
	}

	if ( opts.verbose ) {
		printf("%s -- All VG changes committed to disk\n", cmd);
	}

	evms_close_engine();

	return 0;
}

