/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils
 * File: evms_vgcreate.c
 *
 *	Emulates LVM's 'vgcreate' utility using the EVMS Engine. All options
 *	and several status messages are based on the original vgcreate command
 *	from Heinz Mauelshagen and Sistina Software (www.sistina.com).
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <libgen.h>
#include <ctype.h>
#include <frontend.h>

typedef struct cmd_options_s {
	int debug;
	int help;
	unsigned long pe_size;
	int verbose;
	int version;
} cmd_options_t;

static char * cmd = NULL;
static cmd_options_t opts;

// Must be included after above definitions.
#include "helpers/get_lvmregmgr.c"
#include "helpers/get_number.c"
#include "helpers/get_segment_handle.c"
#include "helpers/is_in_container.c"
#include "helpers/open_engine.c"
#include "helpers/remove_duplicates.c"
#include "helpers/revert_volume.c"


static int showheader( void )
{
	// VERSION and DATE are defined in the top-level make.rules
	printf("Enterprise Volume Management System\n");
	printf("International Business Machines  %s\n", DATE);
	printf("LVM Emulation Utilities %s\n\n", VERSION);
	printf("%s --  initialize a volume group for use by EVMS\n",cmd);
	return 0;
}

static int showhelp( void )
{
	showheader();
	printf("\n%s [-A|--autobackup {y|n}] [-d|--debug] [-l|--maxlogicalvolumes MaxLogicalVolumes]\n", cmd);
	printf("\t[-p|--maxphysicalvolumes MaxPhysicalVolumes] [-h|--help]\n");
	printf("\t[-s|--physicalextentsize PhysicalExtentSize[kKmMgGtT]] [-v|--verbose]\n");
        printf("\t[-V|--version] VolumeGroupName PhysicalVolume [PhysicalVolume...]\n");
	return 0;
}


static int vgcreate_end(option_array_t	* a,
			handle_array_t	* b)
{
	if (a) free(a);
	if (b) free(b);
	evms_close_engine();
	return 0;
}


static int parse_options( int argc, char ** argv )
{
	int		c;
	char		* short_opts = "A:dh?l:p:s:vV";
	struct option	long_opts[] = {
				{ "autobackup",		required_argument, NULL, 'A'},
				{ "debug",		no_argument,       NULL, 'd'},
				{ "help",		no_argument,       NULL, 'h'},
				{ "maxlogicalvolumes",	required_argument, NULL, 'l'},
				{ "maxphysicalvolumes",	required_argument, NULL, 'p'},
				{ "physicalextentsize",	required_argument, NULL, 's'},
				{ "verbose",		no_argument,       NULL, 'v'},
				{ "version",		no_argument,       NULL, 'V'},
				{ NULL, 0, NULL, 0} };

	// Default PE size
	opts.pe_size = 8192;

	while ( (c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF ) {
		switch (c) {
		case 'A':
			// -A is ignored by EVMS.
			if ( optarg[0] != 'n' || optarg[0] != 'y' ) {
				printf( "%s -- ERROR option A argument \"%s\"\n", cmd, optarg);
				return EINVAL;
			}
			break;
		case 'd':
			opts.debug++;
			opts.verbose++;
			break;
		case 'h':
		case '?':
			opts.help++;
			break;
		case 'l':
			// -l is ignored by EVMS. Max LVs is always 256.
		case 'p':
			// -p is ignored by EVMS. Max PVs is always 256.
			break;
		case 's':
			if ( (opts.pe_size = get_number(optarg)) <= 0 ) {
				printf("%s -- ERROR option s argument \"%s\"\n", cmd, optarg);
				return EINVAL;
			}
			opts.pe_size = (opts.pe_size * 1024) / EVMS_VSECTOR_SIZE;
			if ( opts.pe_size < 16 ||
			     opts.pe_size > 33554432 ||
			     (opts.pe_size & (opts.pe_size-1)) != 0 ) {
				printf("%s -- invalid physical extent size %lu\n", cmd, opts.pe_size/2);
				printf("%s -- must be power of 2 and between 16 and 33554432\n\n", cmd);
				return EINVAL;
			}
			break;
		case 'v':
			opts.verbose++;
			break;
		case 'V':
			opts.version++;
			break;
		default:
			printf("%s -- unrecognized option \"%c\"\n\n", cmd, c);
			return EINVAL;
		}
	}

	return 0;
}


int set_vgcreate_options(option_array_t	* creation_options,
			char		* vg_name,
			unsigned long	pe_size )
{
	// Set the VG name option.
	creation_options->count++;
	creation_options->option[1].is_number_based	= 1;
	creation_options->option[1].number		= 0;	// VG Name
	creation_options->option[1].value.s = malloc(strlen(vg_name)+1);
	strcpy(creation_options->option[1].value.s, vg_name);
	if ( opts.verbose ) {
		printf("%s -- Setting VG Name option: %s\n", cmd, vg_name);
	}

	// Set the PE size option.
	creation_options->count++;
	creation_options->option[0].is_number_based	= 1;
	creation_options->option[0].number		= 1;	// PE Size
	creation_options->option[0].value.ui32		= pe_size;
	if ( opts.verbose ) {
		printf("%s -- Setting PE Size option: %ld sectors\n", cmd, pe_size);
	}

	return 0;
}


int main( int argc, char * argv[] )
{
	option_array_t		* creation_options	= NULL;
	handle_array_t		* pv_handles		= NULL;
	plugin_handle_t		lvmregmgr		= 0;
	object_handle_t		new_container		= 0;
	char			* vg_name		= NULL;
	char			* pv_names[256]		= {0};
	int			number_of_pvs		= 0;
	int			log_level		= DEFAULT;
	int			i,rc;

	memset(&opts, 0, sizeof(cmd_options_t));
	cmd = basename(argv[0]);

	// Get the command line options.
	rc = parse_options(argc, argv);
	if (rc) {
		showhelp();
		return rc;
	}
	if ( opts.help ) {
		showhelp();
		return 0;
	}
	if ( opts.version ) {
		showheader();
		return 0;
	}
	if ( opts.verbose ) {
		log_level = DEBUG;
	}
	if ( opts.debug ) {
		log_level = ENTRY_EXIT;
	}

	// Memory for the segment and option arrays.
	creation_options = (option_array_t *)malloc(1000);
	pv_handles = (handle_array_t*)malloc(2000);
	if ( !creation_options || !pv_handles ) {
		printf("%s -- Memory allocation error.\n", cmd);
		vgcreate_end(creation_options, pv_handles);
		return ENOMEM;
	}
	memset(creation_options, 0, 1000);
	memset(pv_handles, 0, 2000);


	// Check for volume group name.
	if ( optind == argc ) {
		printf("%s -- please enter a volume group name and physical volumes\n", cmd);
		vgcreate_end(creation_options, pv_handles);
		showhelp();
		return EINVAL;
	}
	vg_name = argv[optind];
	optind++;

	set_vgcreate_options(creation_options, vg_name, opts.pe_size);

	// Check for PV names
	if ( optind == argc) {
		printf("%s -- please enter physical volume name(s)\n", cmd);
		vgcreate_end(creation_options, pv_handles);
		showhelp();
		return EINVAL;
	}
	number_of_pvs = argc - optind;

	// Copy the PV names from the command line, detecting any duplicates.
	rc = remove_duplicates( &argv[optind], pv_names, number_of_pvs);
	if (rc) {
		printf("%s -- Duplicate PVs specified. Please only specify each PV once\n", cmd);
		vgcreate_end(creation_options, pv_handles);
		return rc;
	}

	// Open the EVMS Engine.
	rc = open_engine(ENGINE_READWRITE, log_level);
	if (rc) {
		vgcreate_end(creation_options, pv_handles);
		return rc;
	}

	// Get the handle for the LVM region manager.
	rc = get_lvmregmgr(&lvmregmgr);
	if (rc) {
		vgcreate_end(creation_options, pv_handles);
		return rc;
	}

	// Get the handles for each PV specified on the command line.
	for ( i = 0; i < number_of_pvs; i++ ) {
		rc = get_segment_handle(pv_names[i], &(pv_handles->handle[i]));
		if (rc) {
			printf("%s -- Could not find handles for all specified segments\n", cmd);
			vgcreate_end(creation_options, pv_handles);
			return rc;
		}

		// If this segment is a top-level compatibility-volume, revert
		// it before using it in the create.
		rc = revert_volume_from_segment(pv_handles->handle[i]);
		if (rc) {
			vgcreate_end(creation_options, pv_handles);
			return rc;
		}
	}
	pv_handles->count = number_of_pvs;


	// Create the new volume group.
	if ( opts.verbose ) {
		printf("%s -- Creating container %s using %d segments\n", cmd, vg_name, pv_handles->count);
	}
	rc = evms_create_container(lvmregmgr, pv_handles, creation_options, &new_container);
	if (rc) {
		printf("%s -- Error creating a new container (%d)\n", cmd, rc);
		vgcreate_end(creation_options,pv_handles);
		return rc;
	}
	if ( opts.verbose ) {
		printf("%s -- New container created\n", cmd);
	}

	// Write everything to disk.
	rc = evms_commit_changes(NULL);
	if (rc) {
		printf("%s -- Error committing changes to disk (%d)\n", cmd, rc);
		vgcreate_end(creation_options, pv_handles);
		return rc;
	}
	if ( opts.verbose ) {
		printf("%s -- New container committed to disk\n", cmd);
	}

	printf("%s -- volume group \"%s\" successfully created and activated\n\n", cmd, vg_name);

	vgcreate_end(creation_options, pv_handles);
	return 0;
}

