/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: commit.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>

#include "ptables.h"
#include "defsegmgr.h"
#include "checks.h"
#include "display.h"
#include "segs.h"
#include "commit.h"
#include "os2dlat.h"
#include "mbr.h"
#include "bsd.h"
#include "unixware.h"
#include "solarisX86.h"


int  DisplayDiskSeg(void * Object, TAG ObjectTag, u_int32_t ObjectSize, void * ObjectHandle, void * Parameters ) ;


/*
 *  Test if the mbr sector is zero cleared. Called by commit code to see
 *  if we need to refresh the boot sector code.
 */
static BOOLEAN isa_null_mbr_sector( void *mbr_sector )
{
    int i;
    u_int32_t  *ui_ptr = (u_int32_t *) mbr_sector;

    LOGENTRY();

    if (ui_ptr) {

        for (i=0; i<128; i++, ui_ptr++) {

            if ( *ui_ptr != 0) {
                return FALSE;
            }

        }

    }

    LOGEXIT();
    return TRUE;
}



/*
 *   Create an MBR partition table for the specified disk. The seglist should
 *   contain a list of segments on this disk, ordered by starting LBA.
 */
static int Build_MBR_PartitionTable( LOGICALDISK *ld, Master_Boot_Record *boot_sector , DISKSEG *mbr )
{
    DISKSEG           *seg;
    SEG_PRIVATE_DATA  *pdata;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
    Partition_Record  *part;

    int                i;
    int                rc;

    chs_t              chs;

    BOOLEAN            ptable_entry_in_use[4] = {FALSE,FALSE,FALSE,FALSE};

    lba_t              start;
    u_int32_t          size;



    LOGENTRY();

    /*
     *  Clear all the partition table entries in case we dont find any
     *  partitions hanging off the disk we can just fall through and the
     *  partition table will be set to all NULL entries.
     */
    for (i=0; i<4; i++)  memset(&boot_sector->Partition_Table[i], 0, sizeof(Partition_Record));


    /*
     *  Walk through the segment list, finding all the primary paritions that belong
     *  in the MBR partition table. Also find the extended partition.
     */
    rc = GoToStartOfList( ld->parent_objects );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( ld->parent_objects, sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE, (void **) &seg );
        if (rc == DLIST_SUCCESS) {

            do {
                 if (seg != NULL) {

                     pdata = (SEG_PRIVATE_DATA *)seg->private_data;

                     // if this seg belongs to the mbr partition table ...
                     if  ( ( pdata->prev_ebr == mbr ) ||
                           ( pdata->ebr      == mbr )) {

                         if ( ptable_entry_in_use[pdata->ptable_index]==FALSE ) {

                            part = &boot_sector->Partition_Table[pdata->ptable_index];

                            if ( seg->start == disk_pdata->extd_partition_lba ) {
                                SYS_IND(part)  = disk_pdata->extd_partition_sys_ind;
                                BOOT_IND(part) = 0;
                                size           = disk_pdata->extd_partition_size;
                                start          = disk_pdata->extd_partition_lba;
                            }
                            else {
                                SYS_IND(part)  = pdata->sys_id;
                                BOOT_IND(part) = pdata->boot_ind;
                                size           = seg->size;
                                start          = seg->start;
                            }

                            START_LBA(part)   = CPU_TO_DISK32(start);
                            NR_SECTS(part)    = CPU_TO_DISK32(size);


                            LBA_to_Ptable_CHS(ld, start, &chs);

                            START_CYL(part)   = (unsigned char) ( chs.cylinder  & 0x000000ff);
                            START_HEADS(part) = (unsigned char) ( chs.head      & 0x000000ff);
                            START_SECT(part)  = (unsigned char) ( chs.sector    & 0x0000003f ) |
                                                (( chs.cylinder >> 2 ) & 0xC0) ;

                            LBA_to_Ptable_CHS(ld, (start + size - 1 ) , &chs);

                            END_CYL(part)     = (unsigned char) ( chs.cylinder  & 0x000000ff);
                            END_HEADS(part)   = (unsigned char) ( chs.head      & 0x000000ff);
                            END_SECT(part)    = (unsigned char) ( chs.sector    & 0x0000003f ) |
                                                    (( chs.cylinder >> 2 ) & 0xC0) ;

                            ptable_entry_in_use[pdata->ptable_index] = TRUE;

                         }
                         else {  // multiple partition records claim the same ptable entry.
                             rc = EIO;
                         }
                     }

                     if (rc == DLIST_SUCCESS) {

                         rc = GetNextObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG, (void **) &seg );

                     }
                 }

            } while ( rc==DLIST_SUCCESS  && seg != NULL );

            /* look for normal termination return codes and modify rc */
            if (rc == DLIST_EMPTY || rc == DLIST_END_OF_LIST ) {
                rc = 0;
            }
        }
    }

    // if there are embedded partitions then we need to find the primary
    // partitions that they consumed and add these partitions to the
    // mbr ptable
    if ((rc == 0) && (disk_pdata->embedded_partition_count)){

        rc = GoToStartOfList( disk_pdata->container_segs );
        if (rc == DLIST_SUCCESS) {

            rc = GetObject( disk_pdata->container_segs, sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE, (void **) &seg );

            while (rc == DLIST_SUCCESS) {

                pdata = (SEG_PRIVATE_DATA *)seg->private_data;

                // if this seg belongs to the mbr partition table ...
                if  ( ( pdata->prev_ebr == mbr ) ||
                      ( pdata->ebr      == mbr )) {

                    if ( ptable_entry_in_use[pdata->ptable_index]==FALSE ) {

                        part = &boot_sector->Partition_Table[pdata->ptable_index];

                        SYS_IND(part)     = pdata->sys_id;
                        BOOT_IND(part)    = pdata->boot_ind;

                        START_LBA(part)   = CPU_TO_DISK32(seg->start);
                        NR_SECTS(part)    = CPU_TO_DISK32(seg->size);


                        LBA_to_Ptable_CHS(ld, seg->start, &chs);

                        START_CYL(part)   = (unsigned char) ( chs.cylinder  & 0x000000ff);
                        START_HEADS(part) = (unsigned char) ( chs.head      & 0x000000ff);
                        START_SECT(part)  = (unsigned char) ( chs.sector    & 0x0000003f ) |
                                                (( chs.cylinder >> 2 ) & 0xC0) ;

                        LBA_to_Ptable_CHS(ld, (seg->start + seg->size - 1 ) , &chs);

                        END_CYL(part)     = (unsigned char) ( chs.cylinder  & 0x000000ff);
                        END_HEADS(part)   = (unsigned char) ( chs.head      & 0x000000ff);
                        END_SECT(part)    = (unsigned char) ( chs.sector    & 0x0000003f ) |
                                                    (( chs.cylinder >> 2 ) & 0xC0) ;

                        ptable_entry_in_use[pdata->ptable_index] = TRUE;
                    }
                    else {

                        part = &boot_sector->Partition_Table[pdata->ptable_index];

                        if ( ( seg->start != DISK_TO_CPU32(START_LBA(part))) ||
                             ( seg->size  |= DISK_TO_CPU32(NR_SECTS(part)))) {

                            rc = EIO; // dam ... too many partitions point to same partition record

                        }

                    }

                }

                rc = GetNextObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG, (void **) &seg );
            }

        }

        /* look for normal termination return codes and modify rc */
        if (rc == DLIST_EMPTY || rc == DLIST_END_OF_LIST ) {
            rc = 0;
        }

    }


    LOGEXITRC();
    return rc;
}


/*
 *   Create an EBR partition table for the specified disk.
 */
static int Build_EBR_PartitionTable( LOGICALDISK *ld, Extended_Boot_Record *ebr, DISKSEG *ebr_seg)
{
    DISKSEG           *seg;
    SEG_PRIVATE_DATA  *pdata;
    Partition_Record  *part;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
    int                rc;
    chs_t              chs;
    sector_count_t     size;
    lba_t              start_lba;
    BOOLEAN            ptable_entry_in_use[4] = {FALSE,FALSE,FALSE,FALSE};

    LOGENTRY();
    LOG_DEBUG("LBA= %08d\n", ebr_seg->start );

    /*
     *  Walk through the segment list, finding all the segments that belong
     *  in this EBR partition table.
     */
    rc = GoToStartOfList( ld->parent_objects );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE, (void **) &seg );
        if (rc == DLIST_SUCCESS) {


            do {

                 if ( seg != NULL ) {

                     pdata = (SEG_PRIVATE_DATA *)seg->private_data;

                     // if owner of this seg is this EBR partition table then add the seg as a partition
                     if  ( pdata->ebr == ebr_seg ) {

                         if ( ptable_entry_in_use[pdata->ptable_index]==FALSE ) {

                            part = &ebr->Partition_Table[pdata->ptable_index];

                            SYS_IND(part)    = pdata->sys_id;
                            BOOT_IND(part)   = pdata->boot_ind;
                            NR_SECTS(part)   = CPU_TO_DISK32(seg->size);

                            /*
                             * logical partition's starting LBA is relative to their EBR
                             * however, an ebr's starting LBA is relative to the starting LBA
                             * of the extended partition.
                             */
                            if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) {
                                start_lba  = seg->start - ebr_seg->start;
                                size       = seg->size;
                            }
                            else {
                                start_lba  = seg->start - disk_pdata->extd_partition_lba;
                                size       = pdata->ebr_sector_count;

                                if ( DISK_TO_CPU32(NR_SECTS(part)) < 0) {
                                   return EIO;
                                }
                            }

                            START_LBA(part)   = CPU_TO_DISK32(start_lba);
                            NR_SECTS(part)    = CPU_TO_DISK32(size);


                            LBA_to_Ptable_CHS(ld, seg->start, &chs);

                            START_CYL(part)   = (unsigned char) ( chs.cylinder  & 0x000000ff);
                            START_HEADS(part) = (unsigned char) ( chs.head      & 0x000000ff);
                            START_SECT(part)  = (unsigned char) ( chs.sector    & 0x0000003f ) |
                                               (( chs.cylinder >> 2 ) & 0xC0) ;

                            LBA_to_Ptable_CHS(ld, (seg->start + size - 1 ) , &chs);

                            END_CYL(part)     = (unsigned char) ( chs.cylinder  & 0x000000ff);
                            END_HEADS(part)   = (unsigned char) ( chs.head      & 0x000000ff);
                            END_SECT(part)    = (unsigned char) ( chs.sector    & 0x0000003f ) |
                                                   (( chs.cylinder >> 2 ) & 0xC0) ;

                            ptable_entry_in_use[pdata->ptable_index] = TRUE;

                         }
                         else {  // multiple partition records claim the same ptable entry.
                             rc = EIO;
                         }
                     }

                     if (rc == DLIST_SUCCESS) {

                         rc = GetNextObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG, (ADDRESS *) &seg );

                     }
                 }

            } while ( rc==DLIST_SUCCESS );


            // look for normal termination return codes and modify rc
            if (rc == DLIST_EMPTY || rc == DLIST_END_OF_LIST ) {
                rc = 0;
            }
        }
    }

    LOGEXITRC();
    return rc;
}




/*
 *   Create an MBR for the specified disk.
 */
static int  Write_MasterBootRecord( LOGICALDISK *ld )
{
    Master_Boot_Record         *boot_sector = malloc(EVMS_VSECTOR_SIZE);
    struct plugin_functions_s  *DevFncs;
    DISKSEG                    *mbr;
    DISK_PRIVATE_DATA          *disk_pdata = get_disk_private_data(ld);
    int                         rc;


    LOGENTRY();

    if ( boot_sector && disk_pdata ) {

        DevFncs = (struct plugin_functions_s *) ld->plugin->functions.plugin;

        mbr = get_mbr_from_seglist(ld->parent_objects);
        if (mbr) {

            rc = DevFncs->read( ld, 0, 1, (void *) boot_sector );
            if (rc == 0) {

                // if mbr boot sector is NULL then fill boot sector
                // code with simple boot sector program.
                if (isa_null_mbr_sector( (void*)boot_sector) == TRUE) {
                    memcpy(boot_sector, os2_mbr_sector, sizeof(os2_mbr_sector) );
                }

                boot_sector->Signature = CPU_TO_DISK16(MSDOS_DISKMAGIC);  // just to be sure

                rc = Build_MBR_PartitionTable( ld, boot_sector, mbr );

                if (rc == 0) {

                    LOG_DEBUG("     Committing MBR to disk ...\n");
                    DisplayPartitionTable( ld, &boot_sector->Partition_Table[0], TRUE);

                    rc = DevFncs->write( ld, 0, 1, (void *) boot_sector );

                    if ( (rc ==0) && ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES )) {
                        rc = Write_Dlat_Sector( ld, mbr );
                    }

                }

            }

        }
        else {
            rc = EINVAL;
        }
    }
    else {
        rc = ENOMEM;
    }

    if (boot_sector) free(boot_sector);

    LOGEXITRC();
    return rc;
}


/*
 *   Create an EBR for the specified disk.
 */
static int  Write_ExtendedBootRecord_Chain( LOGICALDISK *ld, DISKSEG *mbr )
{
    Master_Boot_Record         *sector_buffer = calloc(1, ld->geometry.bytes_per_sector );
    struct plugin_functions_s  *DevFncs;
    int                         i;
    int                         rc = 0;
    DISKSEG                    *ebr;
    SEG_PRIVATE_DATA           *pdata;
    DISK_PRIVATE_DATA          *disk_pdata = get_disk_private_data(ld);


    LOGENTRY();

    if ( sector_buffer && mbr ) {

        sector_buffer->Signature = CPU_TO_DISK16(MSDOS_DISKMAGIC);

        DevFncs = (struct plugin_functions_s  *)ld->plugin->functions.plugin;

        //  get first EBR in seglist by chaining forward from mbr meta data segment
        pdata  = (SEG_PRIVATE_DATA *) mbr->private_data;
        ebr    = pdata->next_ebr;

        // safety check ... bail if no EBR segment found
        if ( ebr == NULL ) {
            LOG_DEBUG("disk doesnt have an extended partition\n");
            LOGEXIT();
            rc = 0;
        }

        // walk the EBR chain
        while ( rc==0 && ebr != NULL) {

            pdata = (SEG_PRIVATE_DATA *) ebr->private_data;

            if ( pdata ) {

                // clear partition table entries in the EBR sector
                for (i=0; i<4; i++)  memset(&sector_buffer->Partition_Table[i], 0, sizeof(Partition_Record));

                // build the EBR partition table
                rc = Build_EBR_PartitionTable( ld, sector_buffer, ebr );
                if (rc == 0) {

                    LOG_DEBUG("     Committing EBR seg %s\n", ebr->name );
                    DisplayPartitionTable( ld, &sector_buffer->Partition_Table[0], FALSE );

                    rc = DevFncs->write( ld, ebr->start, 1, (void *) sector_buffer );

                    if (rc==0) {

                        if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {
                            rc = Write_Dlat_Sector( ld, ebr );
                        }

                        // walk forward in ebr chain
                        ebr = pdata->next_ebr;
                    }

                }
            }
            else {
                rc = ENOMEM;
            }
        }
    }
    else {
        rc = ENOMEM;
    }

    if (sector_buffer) free(sector_buffer);


    LOGEXITRC();
    return rc;
}


/*
 *   Create an MBR partition table for the specified disk. The seglist should
 *   contain a list of segments on this disk, ordered by starting LBA.
 */
static int Commit_Embedded_Partition_Tables( LOGICALDISK *ld, DISK_PRIVATE_DATA *disk_pdata, DISKSEG *mbr )
{
    DISKSEG           *seg;
    SEG_PRIVATE_DATA  *pdata;
    int                rc;


    LOGENTRY();

    /*
     *  Walk through the segment list, finding all the primary paritions that belong
     *  in the MBR partition table. Also find the extended partition.
     */
    rc = GoToStartOfList( disk_pdata->container_segs );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( disk_pdata->container_segs, sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE, (void **) &seg );
        while (rc == DLIST_SUCCESS) {

            pdata = (SEG_PRIVATE_DATA *)seg->private_data;

            switch ( pdata->sys_id ) {

                case BSD_PARTITION:
                case NETBSD_PARTITION:
                case OPENBSD_PARTITION:
                    rc = do_bsd_partition_commit(ld, seg);
                    break;
                case UNIXWARE_PARTITION:
                    rc = do_unixware_partition_commit(ld, seg);
                    break;
                case SOLARIS_X86_PARTITION:
                    rc = do_solaris_x86_partition_commit(ld, seg);
                    break;
                default:
                    rc = 0;
                    break;
            }

            if (rc == DLIST_SUCCESS) {
                rc = GetNextObject( disk_pdata->container_segs,sizeof(DISKSEG),SEGMENT_TAG, (void **) &seg );
            }

        }

    }

    /* look for normal termination return codes and modify rc */
    if (rc == DLIST_EMPTY || rc == DLIST_END_OF_LIST ) {
        rc = 0;
    }


    LOGEXITRC();
    return rc;
}


/*
 *  Called by SEG_COMMIT() to write out partition tables and dlat tables for
 *  the specified disk.
 */
int  Commit_MSDOS_Partition_Tables(LOGICALDISK *ld, DISK_PRIVATE_DATA *disk_pdata )
{
    int rc;

    LOGENTRY();

    rc = Write_MasterBootRecord(ld);
    if (rc==0) {

        rc = Write_ExtendedBootRecord_Chain(ld, get_mbr_from_seglist(ld->parent_objects));
        if (rc) {
            POPUP_MSG(NULL, NULL,
            "Error, a problem occurred while attempting to commit logical drives on disk %s.\n"
            "The segment changes were not committed.\n"
            "The return code = %d.\n",
            ld->name, rc );
        }

    }
    else {
        POPUP_MSG(NULL, NULL,
        "Error, a problem occurred while attempting to commit the master boot record on disk %s.\n"
        "The segment changes were not committed.\n"
        "The return code = %d.\n",
        ld->name, rc );
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Called by SEG_COMMIT() to write out partition tables and dlat tables for
 *  the specified disk.
 */
int  Commit_Disk_Partition_Tables(LOGICALDISK *ld)
{
    int rc=EINVAL;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);

    LOGENTRY();

    if (disk_pdata) {

        if ( (disk_pdata->flags & DISK_HAS_FORCED_LBA_ADDRESSING) == 0 ) {

            rc = Commit_MSDOS_Partition_Tables(ld, disk_pdata);
            if (rc==0) {

                rc = Commit_Embedded_Partition_Tables(ld, disk_pdata, get_mbr_from_seglist(ld->parent_objects));
                if (rc) {
                    POPUP_MSG(NULL, NULL,
                    "Error, a problem occurred while attempting to commit an embedded partition table on disk %s.\n"
                    "The segment changes were not committed.\n"
                    "The return code = %d.\n",
                    ld->name, rc );
                }

            }

        }

    }

    LOGEXITRC();
    return rc;
}
