#include <gtk/gtk.h>
#include <string.h>
#include "entity.h"
#include "gtk-common.h"
#include "gtk-widget-attr.h"



static void
rendgtk_slider_onchange_cb (GtkAdjustment *adjustment, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function;

    function = enode_attrib_str (node, "onchange", NULL);
    enode_call_ignore_return (node, function, "");
}

static void
rendgtk_slider_render (ENode *node)
{
    GtkWidget *slider;
    GtkObject *adjustment = NULL;

    gfloat min = 0.0;
    gfloat max = 100.0;
    gfloat value = 0.0;
    gfloat page_incr;			/* Hardcoded at 10% for now. */
    int digits = 0;

    EBuf *min_buf;
    EBuf *max_buf;
    EBuf *value_buf;
    EBuf *digits_buf;

    min_buf = enode_attrib (node, "min", NULL);
    if (ebuf_not_empty (min_buf))
	min = erend_get_float (min_buf);

    max_buf = enode_attrib (node, "max", NULL);
    if (ebuf_not_empty (max_buf))
	max = erend_get_float (max_buf);

    value_buf = enode_attrib (node, "value", NULL);
    if (ebuf_not_empty (value_buf))
	value = erend_get_float (value_buf);

    page_incr = (max - min) / 10;

    adjustment = gtk_adjustment_new (value, min, max, 0.0, page_incr, 0.0);

    /* 0 is successful match for strncmp. */
    if (0 == strncmp ("vslider", node->element->str, strlen ("vslider")))
	slider = gtk_vscale_new (GTK_ADJUSTMENT (adjustment));
    else
	slider = gtk_hscale_new (GTK_ADJUSTMENT (adjustment));

    /* Set the number of digits to display. */
    digits_buf = enode_attrib (node, "digits", NULL);
    if (ebuf_not_empty (digits_buf))
	digits = erend_get_integer (digits_buf);

    if (digits < 0)
	gtk_scale_set_draw_value (GTK_SCALE (slider), FALSE);
    else {
	gtk_scale_set_draw_value (GTK_SCALE (slider), TRUE);
	gtk_scale_set_digits (GTK_SCALE (slider), digits);
    }

    enode_set_kv (node, "top-widget", slider);
    enode_set_kv (node, "bottom-widget", slider);
    enode_set_kv (node, "adjust-widget", adjustment);

    gtk_signal_connect (adjustment, "value-changed",
			rendgtk_slider_onchange_cb, node);

    enode_attribs_sync (node);
    rendgtk_show_cond (node, slider);
}


static void
rendgtk_slider_destroy (ENode *node)
{
   GtkObject *adjustment;

   adjustment = enode_get_kv (node, "adjust-widget");
   /* gtk_object_unref (adjustment); */  /* HELP! should i be freeing this? */

   rendgtk_element_destroy (node);
}


static gint
rendgtk_slider_value_set (ENode *node, EBuf *attr, EBuf *value)
{
    GtkAdjustment *adjustment;
    float new_val;

    adjustment = enode_get_kv (node, "adjust-widget");

    if (!adjustment)
	return FALSE;

    new_val = erend_get_float (value);

    gtk_adjustment_set_value (adjustment, new_val);
    return TRUE;
}


static gint
rendgtk_slider_max_set (ENode *node, EBuf *attr, EBuf *value)
{
    GtkAdjustment *adjustment;
    float new_val;

    adjustment = enode_get_kv (node, "adjust-widget");

    if (!adjustment)
	return FALSE;

    new_val = erend_get_float (value);

    adjustment->upper = new_val;
    adjustment->page_increment = (adjustment->upper - adjustment->lower) / 10;
    gtk_adjustment_changed (adjustment);
    return TRUE;
}


static gint
rendgtk_slider_min_set (ENode *node, EBuf *attr, EBuf *value)
{
    GtkAdjustment *adjustment;
    float new_val;

    adjustment = enode_get_kv (node, "adjust-widget");

    if (!adjustment)
	return FALSE;

    new_val = erend_get_float (value);

    adjustment->lower = new_val;
    adjustment->page_increment = (adjustment->upper - adjustment->lower) / 10;
    gtk_adjustment_changed (adjustment);
    return TRUE;
}


static gint
rendgtk_slider_digits_set (ENode *node, EBuf *attr, EBuf *value)
{
    GtkScale *slider;
    gint digits;

    slider = enode_get_kv (node, "bottom-widget");
    
    if (!slider)
	return FALSE;

    digits = erend_get_integer (value);

    if (digits < 0)
	gtk_scale_set_draw_value (slider, FALSE);
    else {
	gtk_scale_set_draw_value (slider, TRUE);
	gtk_scale_set_digits (slider, digits);
    }
    return TRUE;
}


static void
rendgtk_slider_value_get (ENode *node, gchar *value)
{
    EBuf *val = NULL;
    GtkAdjustment *adjustment;

    adjustment = enode_get_kv (node, "adjust-widget");

    if (!adjustment)
	return;
 
    val = ebuf_new ();
    ebuf_sprintf (val, "%f", adjustment->value);

    enode_attrib_quiet (node, value, val);
}


static void
rendgtk_slider_min_get (ENode *node, gchar *value)
{
    EBuf *val = NULL;
    GtkAdjustment *adjustment;

    adjustment = enode_get_kv (node, "adjust-widget");

    if (!adjustment)
	return;

    val = ebuf_new ();
    ebuf_sprintf (val, "%f", adjustment->lower);

    enode_attrib_quiet (node, value, val);
}


static void
rendgtk_slider_max_get (ENode *node, gchar *value)
{
    EBuf *val = NULL;
    GtkAdjustment *adjustment;

    adjustment = enode_get_kv (node, "adjust-widget");

    if (!adjustment)
	return;

    val = ebuf_new ();
    ebuf_sprintf (val, "%f", adjustment->upper);

    enode_attrib_quiet (node, value, val);
}


void
slider_renderer_register (void)
{
    Element *element;
    ElementAttr *e_attr;


    /**** the vertical slider. ****/

    element = g_new0 (Element, 1);
    element->render_func = rendgtk_slider_render;
    element->destroy_func = rendgtk_slider_destroy;
    element->parent_func = rendgtk_box_pack;
    element->tag = "vslider";
    element->description = "Create a new vertical sliding value adjuster";
    element_register (element);
    rendgtk_widget_attr_register (element, GTK_TYPE_SCALE);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onchange";
    e_attr->description = "Sets the function to call on value change.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(slider)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);
    
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "value";
    e_attr->description = "The current value of the slider";
    e_attr->value_desc = "float";
    e_attr->get_attr_func = rendgtk_slider_value_get;
    e_attr->set_attr_func = rendgtk_slider_value_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "min";
    e_attr->description = "The minimum value of the slider";
    e_attr->value_desc = "float";
    e_attr->get_attr_func = rendgtk_slider_min_get;
    e_attr->set_attr_func = rendgtk_slider_min_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "max";
    e_attr->description = "The maximum value of the slider";
    e_attr->value_desc = "float";
    e_attr->get_attr_func = rendgtk_slider_max_get;
    e_attr->set_attr_func = rendgtk_slider_max_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "digits";
    e_attr->description = "The number of digits to display.  -1 to disable";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_slider_digits_set;
    element_register_attrib (element, e_attr);


    /**** the horizontal slider. ****/

    element = g_new0 (Element, 1);
    element->render_func = rendgtk_slider_render;
    element->destroy_func = rendgtk_element_destroy;
    element->parent_func = rendgtk_box_pack;
    element->tag = "hslider";
    element->description = "Create a new horizontal sliding value adjuster";
    element_register (element);
    rendgtk_widget_attr_register (element, GTK_TYPE_SCALE);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onchange";
    e_attr->description = "Sets the function to call on value change.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(slider)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);
    
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "value";
    e_attr->description = "The current value of the slider";
    e_attr->value_desc = "float";
    e_attr->get_attr_func = rendgtk_slider_value_get;
    e_attr->set_attr_func = rendgtk_slider_value_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "min";
    e_attr->description = "The minimum value of the slider";
    e_attr->value_desc = "float";
    e_attr->get_attr_func = rendgtk_slider_min_get;
    e_attr->set_attr_func = rendgtk_slider_min_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "max";
    e_attr->description = "The maximum value of the slider";
    e_attr->value_desc = "float";
    e_attr->get_attr_func = rendgtk_slider_max_get;
    e_attr->set_attr_func = rendgtk_slider_max_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "digits";
    e_attr->description = "The number of digits to display.";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_slider_digits_set;
    element_register_attrib (element, e_attr);
}

