#include <gmodule.h>
#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <glib.h>

#ifdef HAVE_SYS_DIR_H
#include <sys/dir.h>
#endif				/* HAVE_SYS_DIR_H */

#include "entity.h"
#include "modulegen.h"


/* This is used to build a configuration file containing a list of what
 * elements a renderer module supports.  This will then be used at runtime
 * (see renderers.c) to load this library on demand */

static void
write_conffile (gchar * modulefile)
{
    FILE *fp;
    GSList *elementlist;
    GSList *elem_tmp;
    gchar *filename;

    GModule *module;

    InitFunc initfunc;		/* Call to init this module. */

    module = eutils_load_module (modulefile);
    if (!module) {
	g_warning
	    ("Unable to load module %s!  You will be unable to use this module!",
	     modulefile);
	return;
    }

    if (!g_module_symbol (module, "renderer_init", (gpointer *) & initfunc)) {
	g_warning ("while loading %s: %s\n", modulefile, g_module_error ());
	g_print ("Unable to find init func in module %s, skipping it.\n",
		 modulefile);
	return;
    }

    filename =
	g_strconcat (econfig_get_attr ("config-location"), "/config/",
		     modulefile, ".conf", NULL);

    fp = fopen (filename, "w");

    if (!fp) {
	g_warning ("Unable to open file '%s' for writing: %s", filename,
		   g_strerror (errno));
	return;
    }
    g_free (filename);

    initfunc (RENDERER_REGISTER);

    fprintf (fp, "<dynaload-library library=\"%s\" type=\"clib\"", modulefile);
    if (entity_mainloop)
	fprintf (fp, " mainloop=\"%s\"", entity_mainloop->loopname);
    fprintf (fp, ">\n");

    /* Write out all the elements in this module */
    elementlist = element_list ();
    elem_tmp = elementlist;

    while (elem_tmp) {
	Element *elem = elem_tmp->data;

	fprintf (fp, "  <dynaload-element tag=\"%s\"/>\n", elem->tag);
	elem_tmp = elem_tmp->next;
    }

    g_slist_free (elementlist);
    fprintf (fp, "</dynaload-library>\n");

    fclose (fp);

    /* Unload them all */
    element_unregister_all ();

    /* Reset mainloop.  We could free it, but it doesn't really matter here. */
    entity_mainloop = NULL;
}

static int
lafile (const struct dirent *d)
{
    gchar *filename = d->d_name;
    gint len = strlen (filename);

    if (len >= 3)
	return strcmp (".la", filename + (len - 3)) == 0 ? 1 : 0;
    else
	return 0;
}

gchar *
modulegen_get_lib_from_la (gchar * lafilename)
{
    static gchar *libfilename = NULL;

    if (NULL == libfilename)
	libfilename = g_malloc (1024);

    libfilename = &lafilename[3];	/* clip lib from begining */
    libfilename[strlen (libfilename) - 3] = '\0';	/* truncate ".la" */

    return (libfilename);
}


/* Output a module configuration file for a given set of modules as passed in 
 * on entity -m <modlist> command line */
/* IAN: TODO: Should parse module names rather than searching dir, or at
 * least do both perhaps.. */
void
modulegen (gchar * modnames)
{
    gchar *sofile;
    DIR *dir;
    struct dirent *entry;
    gchar *dirname;

    dirname =
	g_strconcat (econfig_get_attr ("config-location"), "/config", NULL);
    mkdir (dirname, 0755);
    g_free (dirname);

    dirname = g_strconcat (econfig_get_attr ("config-location"), "/clib", NULL);
    dir = opendir (dirname);

    /* If we can't open that dir, that's ok, we'll just spew a warning and
     * exit.. */
    if (!dir) {
	g_warning ("Could not open directory '%s' for listing: %s", dirname,
		   g_strerror (errno));
	g_print ("No modules installed ?\n");
	exit (0);
    }
    g_free (dirname);

    while ((entry = readdir (dir))) {
	if (lafile (entry)) {
	    sofile = modulegen_get_lib_from_la (entry->d_name);

	    g_print ("Processing module %s\n", sofile);
	    write_conffile (sofile);
	}
    }

    if (entity_mainloop && entity_mainloop->exitall)
	entity_mainloop->exitall (0);
    exit (0);
}





