/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: enigma.cc,v 1.26.2.3 2003/10/06 19:16:17 dheck Exp $
 */
#include "config.h"             // for DEFAULT_DATA_PATH
#include "enigma.hh"
#include "system.hh"
#include "px/px.hh"

#include <algorithm>
#include <fstream>
#include <iostream>
#include <functional>
#include <cassert>
#include <ctime>
#include <sys/types.h>
#include <dirent.h>

using namespace std;
using namespace px;
using namespace enigma;

namespace
{
    class FontAlloc {
    public:
        Font *acquire (const std::string &name) {
	    string png, bmf;
	    using enigma::FindFile;
	    if (FindFile (string("fonts/")+name+".png", png) &&
		FindFile (string("fonts/")+name+".bmf", bmf))
	    {
		return px::LoadBitmapFont(png.c_str(), bmf.c_str());
	    }
	    else
		return 0;
        }
        void release(Font *f) { delete f; }
    };

    class ImageAlloc {
    public:
        Surface *acquire(const std::string &name) {
            return px::LoadImage(name.c_str());
        }
        void release(Surface *s) { delete s; }
    };

    typedef Cache<Surface*, ImageAlloc> ImageCache;
}

// ---------------------
//      Direction :
// ---------------------

Direction
enigma::reverse(Direction d)
{
    static Direction rdir[] = { NODIR, EAST, NORTH, WEST, SOUTH };
    return rdir[d+1];
}

Direction
enigma::rotate(Direction d, bool clockwise)
{
    static Direction rdir[2][5] = {
        { NODIR, SOUTH, EAST, NORTH, WEST }, // anticlockwise
        { NODIR, NORTH, WEST, SOUTH, EAST }  // clockwise
    };
    return rdir[clockwise][d+1];
}

Direction
direction_fromto(GridPos source, GridPos target) 
{
    // source and target have to be adjacent

    int       dx = target.x-source.x;
    int       dy = target.y-source.y;
    Direction d  = NODIR;

    if (dx == 0) {
        if      (dy == -1) d = NORTH;
        else if (dy ==  1) d = SOUTH;
    }
    else if (dy == 0) {
        if      (dx == -1) d = WEST;
        else if (dx ==  1) d = EAST;
    }

    assert(d != NODIR); // source and target have to be adjacent!
    return d;
}

string
enigma::to_suffix(Direction d)
{
    static const char *sfx[] = { "", "-w", "-s", "-e", "-n" };
    return sfx[d+1];
}



// ----------------------
//      DirectionBits
// ----------------------

DirectionBits
enigma::rotate(DirectionBits d, bool clockwise) 
{
    if (clockwise) {
        d = DirectionBits(((d>>1) | (d<<3)) & ALL_DIRECTIONS);
    } else {
        d = DirectionBits(((d>>3) | (d<<1)) & ALL_DIRECTIONS);
    }
    return d;
}

//----------------------------------------------------------------------
// Value implementation
//----------------------------------------------------------------------
Value::Value(const char* str)
    : type(STRING)
{
    val.str = new char[strlen(str)+1];
    strcpy(val.str, str);
}
Value::Value(const string& str)
    : type(STRING)
{
    val.str = new char[str.length()+1];
    strcpy(val.str, str.c_str());
}

void Value::assign(const char* s)
{
    clear();
    type = STRING;
    val.str = new char[strlen(s)+1];
    strcpy(val.str, s);
}

void Value::clear()
{
    if (type == STRING)
	delete[] val.str;
    type = NIL;
}

Value::Value(const Value& other) : type(NIL)
{
    this->operator=(other);
}

Value&
Value::operator=(const Value& other)
{
    if (this != &other) {
        if (other.type == STRING) {
            assign(other.val.str);
        } else {
	    clear();
            type = other.type;
            val = other.val;
        }
    }
    return *this;
}

double Value::get_double() const throw()
{
    assert(type == DOUBLE);
    return val.dval;
}

const char* Value::get_string() const throw()
{
    assert(type == STRING);
    return val.str;
}

Buffer& enigma::operator<<(Buffer& buf, const Value& val)
{
    buf << Uint8(val.get_type());

    switch (val.get_type()) {
    case Value::NIL:
        break;
    case Value::DOUBLE:
        buf << val.get_double();
        break;
    case Value::STRING:
        {
            const char* str = val.get_string();
            buf << (Uint16)strlen(str);
            buf.write(str, strlen(str));
        } break;
    }
    return buf;
}

Buffer& enigma::operator>>(Buffer& buf, Value& val)
{
    Uint8 type = Value::NIL;
    buf >> type;

    switch (type) {
    case Value::NIL:
        // ## fixme
        break;
    case Value::DOUBLE:
        {
            double tmp;
            if (buf >> tmp)
                val = Value(tmp);
        } break;
    case Value::STRING:
        {
            Uint16 len;
            if (buf >> len) {
                char* tmp = new char[len+1];
                tmp[len] = 0;
                if (buf.read(tmp, len))
                    val.assign(tmp);
                delete[] tmp;
            }
        } break;
    }
    return buf;
}

int
enigma::to_int(const Value &v)
{
    switch (v.get_type()) {
    case Value::DOUBLE: return int(v.get_double());
    case Value::STRING: return atoi(v.get_string());
    default: return 0;
    }
}

bool
enigma::to_bool(const Value &v)
{
    return (v.get_type() != Value::NIL);
}

double
enigma::to_double(const Value &v)
{
    switch (v.get_type()) {
    case Value::DOUBLE: return v.get_double();
    case Value::STRING: return atof(v.get_string());
    default: return 0;
    }
}

const char *
enigma::to_string(const Value &v)
{
    static char buf[30];
    switch (v.get_type()) {
    case Value::NIL: return "";
    case Value::DOUBLE:
        snprintf(buf, sizeof(buf), "%f", v.get_double());
        return buf;
    case Value::STRING: return v.get_string();
    default: return 0;
    }
}

Direction 
enigma::to_direction (const Value &v)
{
    int val = Clamp(to_int(v), 0, 3);
    return static_cast<Direction>(val);
}

ostream& enigma::operator<<(ostream& os, const Value& val)
{
    switch (val.get_type()) {
    case Value::NIL:   os << "nil"; break;
    case Value::DOUBLE: os << val.get_double(); break;
    case Value::STRING: os << val.get_string(); break;
    }
    return os;
}

//----------------------------------------------------------------------
// Timer
//----------------------------------------------------------------------

namespace enigma
{
    class Alarm {
        TimeHandler * handler;
        double        interval;
        double        timeleft;
        bool          repeatp;
        bool          removed;
    public:
        Alarm(TimeHandler* h, double interval, bool repeatp);
        void          tick(double dtime);
        bool expired() const { return removed || timeleft <= 0; }
        void remove() { removed = true; }

        bool operator==(const Alarm &a) const {
            return a.handler == handler;
        }
    };

    struct Timer::Rep {
        std::list<TimeHandler*>  handlers;
        std::list<Alarm>         alarms;
	~Rep() { }
    };
}

Alarm::Alarm(TimeHandler* h, double i, bool r)
    : handler(h), interval(i), timeleft(i), repeatp(r), removed(false)
{}

void
Alarm::tick(double dtime)
{
    if (!removed) {
        timeleft -= dtime;
        if (repeatp) {
            while (timeleft <= 0) {
                handler->alarm();
                timeleft += interval;
            }
        } else if (timeleft <= 0) {
            handler->alarm();
        }
    }
}

Timer::Timer() : rep(*new Rep)
{}

Timer::~Timer() {
    delete &rep;
}

void
Timer::deactivate(TimeHandler* th)
{
    // This doesn't work: Some objects deactivate themselves from their
    // `tick' method!
    //    rep.handlers.remove(th);

    std::list<TimeHandler*>::iterator i;

    i = find(rep.handlers.begin(), rep.handlers.end(), th);
    if (i != rep.handlers.end()) {
        *i = 0;
    }
}

void
Timer::activate(TimeHandler *th)
{
    if (find(rep.handlers.begin(), rep.handlers.end(), th) == rep.handlers.end())
        rep.handlers.push_back(th);
}

void
Timer::set_alarm(TimeHandler *th, double interval, bool repeatp)
{
    if (interval > 0)
        rep.alarms.push_back(Alarm(th, interval, repeatp));
}

void
Timer::remove_alarm(TimeHandler *th)
{
    // does not work (crashes if alarm_n removed alarm_n+1)
    // rep.alarms.remove(Alarm(th,0,0));

    list<Alarm>::iterator e = rep.alarms.end();
    // only the ``handler'' entries are compared:
    list<Alarm>::iterator a = find(rep.alarms.begin(), e, Alarm(th, 0, 0));
    if (a != e)
        a->remove(); // only mark for removal!
}


void
Timer::tick(double dtime)
{
    rep.handlers.remove(0);     // remove inactive entries
    for_each(rep.handlers.begin(), rep.handlers.end(),
             bind2nd(mem_fun(&TimeHandler::tick), dtime));

    // explicit loop to allow remove_alarm() to be called from inside alarm()
    for (list<Alarm>::iterator i=rep.alarms.begin(); i != rep.alarms.end(); ) {
        list<Alarm>::iterator n = next(i);

        i->tick(dtime);
        i = n;
    }
    rep.alarms.remove_if(mem_fun_ref(&Alarm::expired));
}


void
Timer::clear()
{
    rep.handlers.clear();
    rep.alarms.clear();
}


//----------------------------------------------------------------------
// Data path
//----------------------------------------------------------------------
namespace
{
    class DataPathList {
    public:
        DataPathList(const string &pathspec=DEFAULT_DATA_PATH);

        void set_path(const string &pathspec);
        const string &get_path() const;

        bool find_file(const string &filename, string &dest) const;

	std::list <string> find_files (const string &folder, 
				       const string &filename) const;

    private:
	// Variables
        string path;
        vector<string> datapaths;
    };

    DataPathList datapaths;
}

DataPathList::DataPathList(const string &pathspec)
{
    set_path(pathspec);
}

void 
DataPathList::set_path(const string &pathspec)
{
    path=pathspec;
    datapaths.clear();
    split_copy(pathspec, ':', back_inserter(datapaths));

    for (unsigned i=0; i<datapaths.size(); ++i)
	datapaths[i] = sysdep::expand_path(datapaths[i]);
}

const string &
DataPathList::get_path() const 
{ 
    return path; 
}

bool 
DataPathList::find_file(const string &filename, string &dest) const
{
    for (unsigned i=0; i<datapaths.size(); ++i) {
	string complete_name = datapaths[i] + sysdep::path_separator + filename;
	if (sysdep::FileExists(complete_name))
	{
	    dest = complete_name;
	    return true;
	}
    }
    return false;
}

std::list <string> 
DataPathList::find_files(const string &folder, const string &filename) const
{
    std::list <string> matches;

    for (unsigned i=0; i<datapaths.size(); ++i) {
	string complete_name = datapaths[i] + sysdep::path_separator + folder;
	if (sysdep::FolderExists(complete_name))
	{
	    DIR *dir = opendir( complete_name.c_str());
	    struct dirent *entry;
	    while((entry = readdir(dir)) != NULL)
	    {
		if( strcmp( entry->d_name, ".") != 0 && strcmp( entry->d_name, "..") != 0)
		{
		    string tmp_name = complete_name + sysdep::path_separator 
			+ string(entry->d_name) + sysdep::path_separator + filename;
		    if (sysdep::FileExists (tmp_name))
			matches.push_back (tmp_name);
		}
	    }
	    closedir(dir);
	}
    }
    return matches;
}

string enigma::GetDataPath() 
{
    return datapaths.get_path();
}

void enigma::SetDataPath(const string &p) 
{
    datapaths.set_path(p);
}

bool
enigma::FindFile (const string &fname, string &dst_fname)
{
    return datapaths.find_file(fname, dst_fname);
}

string
enigma::FindDataFile(const string &filename)
{
    string found_file;
    if (!datapaths.find_file(filename, found_file)) {
        Log << "File not found: " << filename << endl;
        return filename;
    }
    return found_file;
}

string
enigma::FindDataFile(const string &path, const string &filename)
{
    return FindDataFile(path+"/"+filename);
}

std::list <string>
enigma::FindDataFiles(const string &path, const string &filename)
{
    return datapaths.find_files(path, filename);
}

//----------------------------------------------------------------------
// Resource management
//----------------------------------------------------------------------
namespace
{
    px::Cache<px::Font *, FontAlloc>        font_cache;
    px::Cache<px::Surface*, ImageAlloc>     image_cache;
}

px::Font *
enigma::LoadFont (const char *name)
{
    string png = string("fonts/") + name + ".png";
    string bmf = string("fonts/") + name + ".bmf";
    return px::LoadBitmapFont(FindDataFile(png).c_str(),
                              FindDataFile(bmf).c_str());
}

px::Font *
enigma::GetFont(const char *name)
{
    return font_cache.get(name);
}

px::Surface *
enigma::LoadImage(const char *name)
{
    string filename = FindDataFile(string("gfx/") + name + ".png");
    return px::LoadImage(filename.c_str());
}

px::Surface *
enigma::GetImage(const char *name)
{
    string filename = FindDataFile(string("gfx/") + name + ".png");
    Surface *s = image_cache.get(filename);
    return s;
}


//----------------------------------------
// Random numbers
//----------------------------------------
void  enigma::Randomize ()
{
    srand (time(NULL));
}

void   enigma::Randomize (unsigned seed)
{
    srand (seed);
}

int    enigma::IntegerRand (int min, int max)
{
    int r = int((max-min+1) * (rand()/(RAND_MAX+1.0)));
    return r+min;
}

double enigma::DoubleRand (double min, double max)
{
    return min + double(rand())/RAND_MAX * (max-min);
}

#define MAX_DATE_LENGTH 256
const char *enigma::date(const char *format) { // format see 'man strftime'
    static char *result = 0;
    char         buffer[MAX_DATE_LENGTH];

    time_t t;
    time(&t);

    struct tm *tm  = localtime(&t);
    strftime(buffer, MAX_DATE_LENGTH, format, tm);

    if (result) free(result);
    result = strdup(buffer);

    return result;
}
