#!/bin/sh

set -e

# cron support - runs silently using a log file

#  Copyright (C) 2008, 2009  Neil Williams <codehelp@debian.org>
#
#  This package is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

while [ -n "$1" ]; do
case "$1" in
	-b|--base-path)
		shift
		BASE=$1
		shift
	;;
	-t|--tmpfs)
		shift
		TMPFS=$1
		shift
	;;
	-e|--edos-only)
		shift
		EDOS=1
	;;
	*)
		echo "Unrecognised option: $1" 1>&2
		echo 1>&2
		exit 3
	;;
esac
done

if [ -z "$BASE" ]; then
	echo "No base path specified, ensure you use '-b'" 1>&2
	exit 5
fi

if [ ! -d "$BASE" ]; then
	echo "Unable to find base directory: $BASE" 1>&2
	exit 4
fi

if [ -n "$TMPFS" -a ! -d "$TMPFS" ]; then
	echo "Unable to find tmpfs directory: $TMPFS" 1>&2
	exit 4
else 
	if [ -n "$TMPFS" ]; then
		# fails under set -e if TMPFS is not writeable by us.
		touch $TMPFS/cron-check-stamp
		rm $TMPFS/cron-check-stamp
		export TMPDIR=$TMPFS
	fi
fi

# some udebs simply don't exist even though a binary udeb
# depends on them.
add_equivs()
{
	TMPFILE=$1
	echo >> $TMPFILE
	echo "Package: mouse-modules" >> $TMPFILE
	echo "Version: 0.0.1" >> $TMPFILE
	echo >> $TMPFILE
	echo "Package: ppp-modules" >> $TMPFILE
	echo "Version: 0.0.1" >> $TMPFILE
	echo >> $TMPFILE
	echo "Package: jfs-modules" >> $TMPFILE
	echo "Version: 0.0.1" >> $TMPFILE
	echo >> $TMPFILE
	echo "Package: xfs-modules" >> $TMPFILE
	echo "Version: 0.0.1" >> $TMPFILE
	echo >> $TMPFILE
}

STAMP=`date +%s | tr -d '\n'`
LOGFILE="$BASE/grip/logs/$STAMP.log"
if [ ! -d $BASE/grip/logs/ ]; then
	mkdir -p $BASE/grip/logs/
fi
cd $BASE
if [ -z "$EDOS" ]; then
	echo "Updating the filter repository:" >> $LOGFILE
	date >> $LOGFILE
	echo >> $LOGFILE
	nice -n 7 reprepro -v -b $BASE/grip update 2>/dev/null >> $LOGFILE
	echo "Building updated packages for Grip" >> $LOGFILE
	date >> $LOGFILE
	nice -n 7 em_autogrip -b $BASE >$LOGFILE 2>&1
	echo >> $LOGFILE
	date >> $LOGFILE
	# needs improvements in edos-debcheck output to work
	# See http://bugs.debian.org/540797
	# echo "Looking for missing dependencies in grip:" >> $LOGFILE
	# nice -n 7 em_autogrip -b $BASE --edos 2>/dev/null >>$LOGFILE
	# date >> $LOGFILE
	# echo "Adding any identified dependencies:" >> $LOGFILE
	# nice -n 7 em_autogrip -b $BASE 2>/dev/null >>$LOGFILE
	# pull the udeb updates into grip directly but ignore STDERR
	echo "Applying Grip overrides" >> $LOGFILE
	date >> $LOGFILE
	/usr/share/emdebian-tools/grip-overridearch.pl -b $BASE -s unstable 2>/dev/null >> $LOGFILE
fi
# get the archlist from the perl module.
ARCHLIST=`BASE=$BASE perl -MDebian::Packages::Compare -e 'set_base("$ENV{BASE}"); $a=&get_archlist("unstable", "grip"); $s=join (" ", @$a); $s=~s/source//; chomp($s); print "$s\n";'`
if [ -z "$ARCHLIST" ]; then
	ARCHLIST="i386 amd64 arm armel powerpc mips mipsel"
fi
echo "Beginning dependency checks." >> $LOGFILE
date >> $LOGFILE
for ARCH in $ARCHLIST; do
	echo >> $LOGFILE
	echo "Checking $ARCH" >> $LOGFILE
	echo >> $LOGFILE
	# collate Packages files into a temp file
	TMPFILE=`mktemp /tmp/grip-cron-XXXXXX`
	cat grip/dists/unstable/main/binary-$ARCH/Packages > $TMPFILE
	CHECK=`tail -n 1 $TMPFILE|tr -d 'n'`
	if [ -n "$CHECK" ]; then
		echo >> $TMPFILE
	fi
	cat grip/dists/unstable/main/debian-installer/binary-$ARCH/Packages >> $TMPFILE
	CHECK=`tail -n 1 $TMPFILE|tr -d 'n'`
	if [ -n "$CHECK" ]; then
		echo >> $TMPFILE
	fi
	if [ -f "grip/dists/unstable/dev/binary-$ARCH/Packages" ]; then
		cat grip/dists/unstable/dev/binary-$ARCH/Packages >> $TMPFILE
		CHECK=`tail -n 1 $TMPFILE|tr -d 'n'`
		if [ -n "$CHECK" ]; then
			echo >> $TMPFILE
		fi
	fi
	if [ -f "grip/dists/unstable/doc/binary-$ARCH/Packages" ]; then
		cat grip/dists/unstable/doc/binary-$ARCH/Packages >> $TMPFILE
		CHECK=`tail -n 1 $TMPFILE|tr -d 'n'`
		if [ -n "$CHECK" ]; then
			echo >> $TMPFILE
		fi
	fi
	if [ $ARCH = "mipsel" ]; then
		add_equivs $TMPFILE
	elif [ $ARCH = "mips" ]; then
		add_equivs $TMPFILE
	elif [ $ARCH = "arm" ]; then
		add_equivs $TMPFILE
	elif [ $ARCH = "armel" ]; then
		add_equivs $TMPFILE
	fi
	# edos-debcheck insists on setting an error exit value.
	nice -n 7 edos-debcheck -explain -failures < $TMPFILE 2>/dev/null >> $LOGFILE || true
	rm $TMPFILE
done
if [ -z "$EDOS" ]; then
	echo >> $LOGFILE
	echo "Starting testing update:" >> $LOGFILE
	date >> $LOGFILE
	echo >> $LOGFILE
	nice -n 7 em_autogrip -b $BASE -t --add-new 2>/dev/null >>$LOGFILE || true
	echo "Checking testing update:" >> $LOGFILE
	date >> $LOGFILE
	nice -n 7 em_autogrip -b $BASE -t 2>/dev/null >>$LOGFILE || true
	/usr/share/emdebian-tools/grip-overridearch.pl -b $BASE -s testing 2>/dev/null >> $LOGFILE
fi
for ARCH in $ARCHLIST; do
	TMPFILE=`mktemp /tmp/grip-cron-XXXXXX`
	echo >> $LOGFILE
	echo "Checking $ARCH" >> $LOGFILE
	echo >> $LOGFILE
	cat grip/dists/testing/main/binary-$ARCH/Packages > $TMPFILE
	cat grip/dists/testing/main/debian-installer/binary-$ARCH/Packages >> $TMPFILE
	if [ -f "grip/dists/testing/dev/binary-$ARCH/Packages" ]; then
		cat grip/dists/testing/dev/binary-$ARCH/Packages >> $TMPFILE
		echo >> $TMPFILE
	fi
	if [ -f "grip/dists/testing/doc/binary-$ARCH/Packages" ]; then
		cat grip/dists/testing/doc/binary-$ARCH/Packages >> $TMPFILE
		echo >> $TMPFILE
	fi
	if [ $ARCH = "mipsel" ]; then
		add_equivs $TMPFILE
	elif [ $ARCH = "mips" ]; then
		add_equivs $TMPFILE
	elif [ $ARCH = "arm" ]; then
		add_equivs $TMPFILE
	elif [ $ARCH = "armel" ]; then
		add_equivs $TMPFILE
	fi
	nice -n 7 edos-debcheck -explain -failures < $TMPFILE 2>/dev/null >> $LOGFILE  || true
	rm $TMPFILE
done
if [ -z "$EDOS" ]; then
	echo "Checking britney status:" >> $LOGFILE
	date >> $LOGFILE
	nice -n 7 em_autogrip -b $BASE --britney 2>/dev/null >> $LOGFILE  || true
fi
echo "Checking duplicates for unstable:" >> $LOGFILE
date >> $LOGFILE
nice -n 7 /usr/share/emdebian-tools/emgrip-dupes -b $BASE 2>/dev/null >> $LOGFILE  || true
echo "Checking duplicates for testing:" >> $LOGFILE
date >> $LOGFILE
nice -n 7 /usr/share/emdebian-tools/emgrip-dupes -b $BASE -s testing 2>/dev/null >> $LOGFILE  || true
echo "Checking for stable-proposed-updates:" >> $LOGFILE
if [ -d $BASE/grip/dists/stable-proposed-updates ]; then
	date >> $LOGFILE
	echo "Checking duplicates in stable-proposed-updates:" >> $LOGFILE
	nice -n 7 /usr/share/emdebian-tools/emgrip-dupes -b $BASE -s stable-proposed-updates 2>/dev/null >> $LOGFILE  || true
	echo "Updating stable-proposed-updates." >> $LOGFILE
	nice -n 7 /usr/share/emdebian-tools/proposed-updates.pl -d $BASE 2>/dev/null >> $LOGFILE  || true
fi
if [ -n "$TMPDIR" ]; then
	# clean up after edos-debcheck
	rm -f $TMPDIR/distcheck*
fi
date >> $LOGFILE
