// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2006-2007 Benoit Jacob <jacob@math.jussieu.fr>
//
// Eigen is free software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Eigen; if not, write to the Free Software Foundation, Inc., 51
// Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. This exception does not invalidate any other reasons why a work
// based on this file might be covered by the GNU General Public License.

#include <QtGui/QApplication>
#include <QtGui/QMessageBox>

#include "camera.h"

#include <cstdlib>
#include <ctime>

using namespace std;

const Vector3d AXIS_X( 1, 0, 0 );
const Vector3d AXIS_Y( 0, 1, 0 );
const Vector3d AXIS_Z( 0, 0, 1 );
const double ROTATION_SPEED = 0.005;
const double TRANSLATION_SPEED = 0.04;

double myrand( double boundary1, double boundary2 )
{
    return boundary1 + (boundary2 - boundary1) * rand() / RAND_MAX;
}

void GLWidget::mousePressEvent( QMouseEvent *event )
{
    m_lastPos = event->pos();
}

void GLWidget::mouseMoveEvent( QMouseEvent *event )
{
    QPoint delta = event->pos() - m_lastPos;
    m_lastPos = event->pos();
    if( event->buttons() & Qt::LeftButton )
    {
        m_cameraMatrix.prerotate3( - delta.x() * ROTATION_SPEED, AXIS_Y );
        m_cameraMatrix.prerotate3( - delta.y() * ROTATION_SPEED, AXIS_X );
    }
    if( event->buttons() & Qt::RightButton )
    {
        m_cameraMatrix.prerotate3( - delta.x() * ROTATION_SPEED, AXIS_Z );
        m_cameraMatrix.pretranslate( delta.y() * TRANSLATION_SPEED * AXIS_Z );
    }
    if( event->buttons() & ( Qt::LeftButton | Qt::RightButton ) )
        update();
}

void GLWidget::wheelEvent( QWheelEvent *event )
{
    m_cameraMatrix.pretranslate( event->delta()
                                 * TRANSLATION_SPEED / 10 * AXIS_Z );
    update();
}

void GLWidget::compileLists()
{
    int i, j;
    Vector3d vertices[4] = { Vector3d( -1, -1,  0 ),
                             Vector3d(  1, -1,  0 ),
                             Vector3d(  0,  1,  0 ),
                             Vector3d(  0,  0,  1 ) };
    int ind[4][3] = { { 1, 0, 2 }, { 1, 2, 3}, { 2, 0, 3}, { 0, 1, 3 } };
    m_objectList = glGenLists(1);
    glNewList( m_objectList, GL_COMPILE );
    glBegin( GL_TRIANGLES );
    for( i = 0; i < 4; i++ )
    {
        Vector3d normal = cross( vertices[ ind[i][1] ] - vertices[ ind[i][0] ],
                                 vertices[ ind[i][2] ] - vertices[ ind[i][0] ] );
        normal.normalize();
        glNormal3dv( normal.array() );
        for( j = 0; j < 3; j++ )
            glVertex3dv( vertices[ ind[i][j] ].array() );
    }
    glEnd();
    glEndList();
    m_sceneList = glGenLists(1);
    glNewList( m_sceneList, GL_COMPILE );
    for( i = 0; i < 100; i++ )
    {
        glPushMatrix();
        Vector3d pos;
        pos.loadRandomUnit();
        glTranslated( 5 * pos.x(), 5 * pos.y(), 5 * pos.z() );
        glRotated( myrand( 0, 360), 1, myrand( -2, 2 ), myrand( -2, 2 ) );
        glColor3d( myrand( 0, 1 ), myrand( 0, 1 ), myrand( 0, 1 ) );
        glCallList( m_objectList );
        glPopMatrix();
    }
    glEndList();
}

void GLWidget::initializeGL()
{
    glClearColor( 0.0, 0.0, 0.0, 1.0 );
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    glShadeModel(GL_FLAT);
    glDisable(GL_NORMALIZE);
    glEnable(GL_COLOR_MATERIAL);
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);

    Vector4f white( 1, 1, 1, 1 );
    glLightfv( GL_LIGHT0, GL_AMBIENT,  (0.3f * white).array() );
    glLightfv( GL_LIGHT0, GL_DIFFUSE,  (0.8f * white).array() );
    glLightfv( GL_LIGHT0, GL_SPECULAR, white.array() );

    m_cameraMatrix.loadIdentity();
    compileLists();
}

void GLWidget::paintGL()
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glLoadMatrixd( m_cameraMatrix.array() );
    Vector4f lightpos( 0, 1, 1, 0 );
    glLightfv( GL_LIGHT0, GL_POSITION, lightpos.array() );
    glCallList( m_sceneList );
}

void GLWidget::resizeGL(int width, int height)
{
    glViewport(0, 0, width, height);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective( 40.0, static_cast<GLfloat>(width) / height, 0.5, 50.0 );
    glMatrixMode(GL_MODELVIEW);
}

int main(int argc, char *argv[])
{
    srand( (unsigned int) time(NULL) );
    QApplication app(argc, argv);
    GLWidget w;
    w.show();
    return app.exec();
}

#include "camera.moc"
