/* Copyright (C) 2004 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYING distributed with the source files.

*/

#include <gtkmm/main.h>
#include <gdkmm/pixbuf.h>

#include "addressbook.h"
#include "socket_notify.h"

#ifdef HAVE_STRINGSTREAM
#include <sstream>
#else
#include <strstream>
#endif

#ifdef ENABLE_NLS
#include <libintl.h>
#endif

SocketNotifyDialog::SocketNotifyDialog(const int size, 
				       const std::pair<std::string, unsigned int>& fax_pair,
				       Gtk::Window& window, bool modal):
                                               Gtk::Window(Gtk::WINDOW_TOPLEVEL),
					       standard_size(size), is_modal(modal),
					       number_button(gettext("Tel number: ")),
					       send_button(gettext("Send fax")),
					       queue_button(gettext("Queue fax")),
					       button_box(Gtk::BUTTONBOX_END, standard_size/3),
					       parent(window) {


  fax_name.second = fax_pair.first;
#ifdef HAVE_STRINGSTREAM
  std::ostringstream strm;
  strm << gettext("PRINT JOB: ") << fax_pair.second;

  fax_name.first = strm.str();
  strm << gettext(" has been received on socket.\n"
		  "Send or queue fax?");
  label.set_text(strm.str());
#else
  std::ostrstream strm1;
  std::ostrstream strm2;

  strm1 << gettext("PRINT JOB: ") << fax_pair.second << std::ends;
  const char* text_p = strm1.str();
  fax_name.first = text_p;
  delete[] text_p;

  strm2 << gettext("PRINT JOB: ") << fax_pair.second
	<< gettext(" has been received on socket.\n"
		   "Send or queue fax?")
	<< std::ends;

  text_p = strm2.str();
  label.set_text(text_p);
  delete[] text_p;
#endif

  number_entry.set_size_request(standard_size * 7, standard_size);

  button_box.add(queue_button);
  button_box.add(send_button);

  number_box.pack_start(number_button, false, false, standard_size/3);
  number_box.pack_start(number_entry, true, true, standard_size/3);

  window_box.pack_start(label, true, true, standard_size/2);
  window_box.pack_start(number_box, false, false, standard_size/3);
  window_box.pack_start(button_box, false, false, standard_size/3);

  queue_button.set_flags(Gtk::CAN_DEFAULT);
  send_button.set_flags(Gtk::CAN_DEFAULT);
  number_button.set_flags(Gtk::CAN_DEFAULT);

  number_button.signal_clicked().connect(sigc::mem_fun(*this, &SocketNotifyDialog::addressbook_slot));
  send_button.signal_clicked().connect(sigc::bind(sigc::mem_fun(*this, &SocketNotifyDialog::selected),
				       true));
  queue_button.signal_clicked().connect(sigc::bind(sigc::mem_fun(*this, &SocketNotifyDialog::selected),
					false));

  add(window_box);

  set_title(gettext("efax-gtk: print job received on socket"));
  set_type_hint(Gdk::WINDOW_TYPE_HINT_DIALOG);

  if (is_modal) {
    set_transient_for(parent);
    parent.set_sensitive(false);
    set_modal(true);
  }

  set_border_width(standard_size/2);
  number_button.grab_focus();

  set_icon(prog_config.window_icon_r);

  show_all();
}

void SocketNotifyDialog::addressbook_slot(void) {

  if (!AddressBook::get_is_address_list()) {
    AddressBook* dialog_p = new AddressBook(standard_size, *this);

    dialog_p->accepted.connect(sigc::mem_fun(*this, &SocketNotifyDialog::set_number_slot));
    // there is no memory leak -- AddressBook will delete its own memory
    // when it is closed
  }
}

void SocketNotifyDialog::set_number_slot(const Glib::ustring& number) {

  if (!number.empty()) {
    number_entry.set_text(number);
    // reset this window as sensitive to make grab_focus() work
    set_sensitive(true);
    queue_button.grab_focus();
  }
}

void SocketNotifyDialog::selected(bool send) {
  if (is_modal) parent.set_sensitive(true); // do this before we emit send()
  hide_all();
  if (send) send_signals();
  // this dialog is self-owning and it is safe to call `delete this'
  delete this;
}

bool SocketNotifyDialog::on_delete_event(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

void SocketNotifyDialog::send_signals(void) {

  fax_name_sig(fax_name);
  fax_number_sig(number_entry.get_text());
  sendfax_sig();
}
