/*  $Header: /cvsroot/dvipdfmx/src/pdfspecial.c,v 1.33 2004/04/11 05:37:59 hirata Exp $

    This is dvipdfmx, an eXtended version of dvipdfm by Mark A. Wicks.

    Copyright (C) 2002 by Jin-Hwan Cho and Shunsaku Hirata,
    the dvipdfmx project team <dvipdfmx@project.ktug.or.kr>
    
    Copyright (C) 1998, 1999 by Mark A. Wicks <mwicks@kettering.edu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
*/

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <ctype.h>
#include <math.h>
#include <string.h>

#include "system.h"
#include "mem.h"
#include "error.h"
#include "mfileio.h"

#include "numbers.h"
#include "dvi.h"
#include "pdflimits.h"
#include "pdfobj.h"
#include "pdfdoc.h"
#include "pdfdev.h"
#include "pdfparse.h"

#include "pdfximage.h"

#include "pdfspecial.h"

static void add_reference     (const char *ident, pdf_obj *object_ref);
static void add_object        (const char *ident, pdf_obj *object);
static void release_reference (const char *ident);

static pdf_obj *lookup_object (const char *ident);

static void do_content (char **start, char *end, double x_user, double y_user);
static void do_literal (char **start, char *end, double x_user, double y_user);
static void do_fstream (char **start, char *end);

static void do_image (char **start, char *end, double x_user, double y_user);
static void do_bxobj (char **start, char *end, double x_user, double y_user);
static void do_exobj (char **start, char *end);
static void do_uxobj (char **start, char *end, double x_user, double y_user);

static int verbose = 0;

void
pdf_special_set_verbose (void)
{
  verbose++;
}

static char *__save;
#define SAVE(s,e) { __save = (s); }
#define DUMP_RESTORE(s,e) { (s) = __save; dump((s),(e)); }

static void
do_bop (char **start, char *end)
{
  if (*start < end)
    pdf_doc_bop(*start, end - *start);
  *start = end;
}

static void
do_eop (char **start, char *end)
{
  if (*start < end)
    pdf_doc_eop(*start, end - *start);
  *start = end;
}

static void
do_put (char **start, char *end)
{
  pdf_obj *thisobj;
  pdf_obj *data;
  char    *ident;

  SAVE(*start, end);

  skip_white(start, end);
  ident = parse_opt_ident(start, end);
  if (!ident) {
    WARN("Special put: Nonexistent object reference.");
    DUMP_RESTORE(*start, end);
    return;
  }

  thisobj = lookup_object(ident);
  if (!thisobj) {
    WARN("Named object \"@%s\" doesn't exist.", ident);
    DUMP_RESTORE(*start, end);
    RELEASE(ident);
    return;
  }
  RELEASE(ident);

  skip_white(start, end);
  switch (PDF_OBJ_TYPEOF(thisobj)) {
  case PDF_DICT:
    data = parse_pdf_dict(start, end);
    if (data) {
      if (PDF_OBJ_DICTTYPE(data)) {
	pdf_merge_dict(thisobj, data);
      } else {
	WARN("PDF special put: Expecting a dictionary.");
	DUMP_RESTORE(*start, end);
      }
      pdf_release_obj(data);
    }
    break;
  case PDF_STREAM:
    data = parse_pdf_object(start, end);
    if (data) {
      pdf_obj *stream_dict;

      if (PDF_OBJ_DICTTYPE(data)) {
	stream_dict = pdf_stream_dict(thisobj);
	pdf_merge_dict(stream_dict, data);
      } else if (PDF_OBJ_STREAMTYPE(data)) {
	/*
	 * pdf_merge_stream() not implemented.
	 */
	WARN("PDF special put: PDF stream not supported yet.");
      } else {
	WARN("Incompatible type in special pdf:put.");
	DUMP_RESTORE(*start, end);
      }
      pdf_release_obj(data);
    }
    break;
  case PDF_ARRAY:
    while (*start < end && 
	   (data = parse_pdf_object(start, end)) != NULL) {
      pdf_add_array(thisobj, data);
      skip_white(start, end);
    }
    if (*start < end) {
      WARN("PDF special: put: invalid object.  Rest of line ignored.");
      DUMP_RESTORE(*start, end);
    }
    break;
  default:
    WARN("Special put:  Invalid destination object type.");
    break;
  }

  return;
}

/*
 * The following must be consecutive and
 * starting at 0.  Order must match
 * dimensions array below
 */

#define WIDTH  0
#define HEIGHT 1
#define DEPTH  2
#define SCALE  3
#define XSCALE 4
#define YSCALE 5
#define ROTATE 6
#define BBOX   7

static struct {
  const char *strkey;
  int         key;
  int         hasunits;
} dimensions[] = {
  {"width",  WIDTH,  1},
  {"height", HEIGHT, 1},
  {"depth",  DEPTH,  1},
  {"scale",  SCALE,  0},
  {"xscale", XSCALE, 0},
  {"yscale", YSCALE, 0},
  {"rotate", ROTATE, 0},
  {"bbox",   BBOX,   0}
};
#define NUM_DIMENSION_KEYS (sizeof(dimensions)/sizeof(dimensions[0]))

void
transform_info_clear (transform_info *info)
{
  /* Physical dimensions */
  info->width  = 0.0;
  info->height = 0.0;
  info->depth  = 0.0;

  info->user_bbox = 0;
  info->u_llx = 0.0;
  info->u_lly = 0.0;
  info->u_urx = 0.0;
  info->u_ury = 0.0;

  /* Transformation */
  info->scale  = 0.0;
  info->xscale = 0.0;
  info->yscale = 0.0;
  info->rotate = 0.0;

  info->xoffset = 0.0;
  info->yoffset = 0.0;

  info->clip = 0;
}

static int
validate_transform_info (transform_info *p)
{
  if (p->width  != 0.0 &&
      (p->scale != 0.0 || p->xscale != 0.0)) {
    WARN("Can't supply both width and scale.");
    p->scale = p->xscale = 1.0;
    return 0;
  } else if (p->height != 0.0 &&
	     (p->scale != 0.0 || p->yscale != 0.0)) {
    WARN("Can't supply both height and scale.");
    p->scale = p->yscale = 1.0;
    return 0;
  } else  if (p->scale   != 0.0 &&
	      (p->xscale != 0.0 || p->yscale != 0.0)) {
    WARN("Can't supply overall scale along with axis scales.");
    p->xscale = p->yscale = p->scale;
    return 0;
  } else if (p->scale != 0.0) {
    p->xscale = p->scale;
    p->yscale = p->scale;
  }

  return 1;
}

static int
parse_one_dim_word (char **start, char *end)
{
  int   result = -1;
  char *dim_str;
  int   i;

  SAVE(*start, end);

  skip_white(start, end);
  dim_str = parse_ident(start, end);
  if (!dim_str) {
    WARN("Expecting a keyword here, e.g., height, width, etc.");
    DUMP_RESTORE(*start, end);
  } else {
    for (i = 0; i< NUM_DIMENSION_KEYS; i++) {
      if (!strcmp(dimensions[i].strkey, dim_str)) {
	result = dimensions[i].key;
	break;
      }
    }
    if (i >= NUM_DIMENSION_KEYS) {
      WARN("Invalid keyword \"%s\".", dim_str);
      DUMP_RESTORE(*start, end);
    }
    RELEASE(dim_str);
  }

  return result;
}

static struct {
  const char *key;
  double      units;
  int         is_true_unit;
} unit_in_bps[] = {
  {"pt",     (72.0/72.27), 0},
  {"in",     (72.0),       0},
  {"cm",     (72.0/2.54),  0},
  {"mm",     (72.0/25.4),  0},
  {"bp",     1.0,          0},
  {"truept", (72.0/72.27), 1},
  {"truein", (72.0),       1},
  {"truecm", (72.0/2.54),  1},
  {"truemm", (72.0/25.4),  1},
  {"truebp", 1.0,          1}
};
#define NUM_KNOWN_UNITS (sizeof(unit_in_bps)/sizeof(unit_in_bps[0]))

static double
parse_one_unit (char **start, char *end)
{
  double result = -1.0;
  char  *unit_str;
  int    i;

  SAVE(*start, end);

  skip_white(start, end);
  unit_str = parse_c_ident(start, end);
  if (!unit_str) {
    WARN("Expecting a unit here (e.g., in, cm, pt).");
    DUMP_RESTORE(*start, end);
  } else {
    for (i = 0; i < NUM_KNOWN_UNITS; i++) {
      if (!strcmp(unit_in_bps[i].key, unit_str)) {
	result = unit_in_bps[i].units;
	break;
      }
    }
    if (i >= NUM_KNOWN_UNITS) {
      WARN("Unknown unit of measurement \"%s\".", unit_str);
      DUMP_RESTORE(*start, end);
    } else {
      if (unit_in_bps[i].is_true_unit)
	result = unit_in_bps[i].units / dvi_tell_mag();
      else
	result = unit_in_bps[i].units;
    }
    RELEASE(unit_str);
  }

  return result;
}

double
parse_length (char **start, char *end)
{
  double val = 0.0, one_bp = 0.0;
  char  *nextptr;

  SAVE(*start, end);

  /* Our units does not start with "e". */
  val = strtod(*start, &nextptr);
  if (nextptr == *start) {
    ERROR("Error in length specification: %s", *start);
  } else if (nextptr < end && nextptr != NULL) {
    one_bp = parse_one_unit(&nextptr, end);
    if (one_bp <= 0.0)
      ERROR("Unknown unit specified in: %s", nextptr);
    else if (nextptr != NULL &&
	     nextptr < end && !isspace(*nextptr) &&
	     !strchr("([</,", *nextptr)) {
      ERROR("Unknown unit specified in: %s", *start);
    }
  } else {
    one_bp = 1.0;
  }

  *start = nextptr;
  return (val * one_bp);
}

static int
parse_dimension (char **start, char *end, transform_info *info)
{
  skip_white(start, end);
  while (*start < end && isalpha(**start)) {
    int   key;
    char *tmp;

    key = parse_one_dim_word(start, end);
    if (key < 0) {
      ERROR("Expecting a dimension/transformation keyword (e.g., width, height).");
    }
    skip_white(start, end);
    switch (key) {
    case WIDTH:
      info->width  = parse_length(start, end);
      break;
    case HEIGHT:
      info->height = parse_length(start, end);
      break;
    case DEPTH:
      info->depth  = parse_length(start, end);
      break;
    case SCALE:
      tmp = parse_number(start, end);
      info->scale = atof(tmp);
      RELEASE(tmp);
      break;
    case XSCALE:
      tmp = parse_number(start, end);
      info->xscale = atof(tmp);
      RELEASE(tmp);
      break;
    case YSCALE:
      tmp = parse_number(start, end);
      info->yscale = atof(tmp);
      RELEASE(tmp);
      break;
    case ROTATE:
      tmp = parse_number(start, end);
      info->rotate = atof(tmp)*M_PI/180.0;
      RELEASE(tmp);
      break;
    case BBOX:
      {
	char *llx = NULL, *lly = NULL, *urx = NULL, *ury = NULL;

	if (!(llx = parse_number(start, end)) ||
	    !(lly = parse_number(start, end)) ||
	    !(urx = parse_number(start, end)) ||
	    !(ury = parse_number(start, end))) {
	  ERROR("Expecting four numbers following \"bbox\" specification.");
	}
	info->u_llx = atof(llx);
	info->u_lly = atof(lly);
	info->u_urx = atof(urx);
	info->u_ury = atof(ury);
	info->user_bbox = 1; /* Flag to indicate that user specified a bbox */
	RELEASE(llx);
	RELEASE(lly);
	RELEASE(urx);
	RELEASE(ury);
      }
      break;
    default:
      ERROR("parse_dimension: Invalid key");
    }
    skip_white(start, end);
  }

  return 0;
}

static void
do_pagesize (char **start, char *end)
{
  *start = end;
  return;

#if 0
  /* Handled differently before bop. */

  pdf_rect mediabox;
  transform_info info;

  SAVE(*start, end);

  transform_info_clear(&info);

  skip_white(start, end);
  if (parse_dimension(start, end, &info) < 0) {
    WARN("Special: pagesize: Failed to find a valid set of dimensions.");
    DUMP_RESTORE(*start, end);
    return;
  }
  validate_transform_info(&info);

  if (info.scale  != 0.0 ||
      info.xscale != 0.0 || info.yscale != 0.0) {
    WARN("Scale meaningless for pagesize.");
    DUMP_RESTORE(*start, end);
    return;
  } else if (info.width == 0.0 ||
	     info.depth + info.height == 0.0) {
    WARN("Page cannot have a zero dimension.");
    DUMP_RESTORE(*start, end);
    return;
  }

  /* Since these are physical dimensions, they need to be scaled by
   * mag.  "Virtual" dimensions are scaled by a transformation
   * matrix.  Physical dimensions (e.g, page size and annotations)
   * cannot
   */
  mediabox.llx =  0.0;
  mediabox.lly = -dvi_tell_mag() * info.depth;
  mediabox.urx =  dvi_tell_mag() * info.width;
  mediabox.ury =  dvi_tell_mag() * info.height;
  pdf_doc_set_mediabox(pdf_doc_current_page_no(), &mediabox);
#endif
}

static void
do_ann (char **start, char *end)
{
  pdf_obj *annot_dict;
  pdf_rect rect;
  char    *ident;
  transform_info info;

  SAVE(*start, end);

  transform_info_clear(&info);

  skip_white(start, end);
  ident = (**start == '@') ? parse_opt_ident(start, end) : NULL;

  skip_white(start, end);
  if (parse_dimension(start, end, &info) < 0) {
    WARN("Ignoring annotation with invalid dimension.");
    DUMP_RESTORE(*start, end);
    return;
  }

  annot_dict = parse_pdf_tainted_dict(start, end);
  if (!annot_dict) {
    WARN("Ignoring annotation with invalid dictionary.");
    DUMP_RESTORE(*start, end);
  } else {
    rect.llx = dev_phys_x();
    rect.lly = dev_phys_y() - dvi_tell_mag() * info.depth;
    rect.urx = rect.llx + dvi_tell_mag() * info.width;
    rect.ury = rect.lly + dvi_tell_mag() * info.height;

    pdf_doc_add_annot(pdf_doc_current_page_no(), &rect, annot_dict); /* This add reference. */
    /* Object is replaced by reference.
     * Modification of annot attribute is not allowed.
     */
    if (ident)
      add_reference(ident, pdf_ref_obj(annot_dict));
    pdf_release_obj(annot_dict);
  }

  if (ident)
    RELEASE(ident);
}

pdf_obj *pending_annot_dict = NULL;

static void
do_bann (char **start, char *end)
{
  if (pending_annot_dict)
    WARN("Can't begin an annotation when one is pending.");
  else {
    skip_white(start, end);
    pending_annot_dict = parse_pdf_tainted_dict(start, end);
    if (!pending_annot_dict)
      WARN("Ignoring annotation with invalid dictionary.");
    else {
      pdf_doc_begin_annot(pending_annot_dict);
      /*
       * If this object has a named reference, we file it away for
       * later.  Otherwise we release it.
       */
      /* Did someone modified this leaving the above comment unchanged? */
    }
  }
}

static void
do_eann (char **start, char *end)
{
  if (!pending_annot_dict)
    WARN("Tried to end an annotation without starting one!");
  else {
    pdf_doc_end_annot();
    pdf_release_obj(pending_annot_dict);
  }
  pending_annot_dict = NULL;
}

static int
parse_colorspec (char **start, char *end, pdf_color *colorspec)
{
  pdf_obj *color;

  skip_white(start, end);
  color = parse_pdf_object(start, end);
  if (color == NULL) {
    return -1;
  }
  switch (PDF_OBJ_TYPEOF(color)) {
  case PDF_ARRAY:
    {
      int components;

      components = pdf_array_length(color);
      {
	int i;

	for (i = 0; i < components; i++) {
	  if (!PDF_OBJ_NUMBERTYPE(pdf_get_array(color, i))) {
	    WARN("A number expected for color component value.");
	    pdf_release_obj(color);
	    return -1;
	  }
	}
      }
      switch (components) {
      case 1:
	colorspec->space_id  = PDF_COLOR_DEV_GRAY;
	colorspec->values[0] = pdf_number_value(pdf_get_array(color, 0));
	break;
      case 3:
	colorspec->space_id  = PDF_COLOR_DEV_RGB;
	colorspec->values[0] = pdf_number_value(pdf_get_array(color, 0));
	colorspec->values[1] = pdf_number_value(pdf_get_array(color, 1));
	colorspec->values[2] = pdf_number_value(pdf_get_array(color, 2));
	break;
      case 4:
	colorspec->space_id  = PDF_COLOR_DEV_CMYK;
	colorspec->values[0] = pdf_number_value(pdf_get_array(color, 0));
	colorspec->values[1] = pdf_number_value(pdf_get_array(color, 1));
	colorspec->values[2] = pdf_number_value(pdf_get_array(color, 2));
	colorspec->values[3] = pdf_number_value(pdf_get_array(color, 3));
	break;
      default:
	WARN("Expecting either RGB or CMYK color array.");
      }
    }
    break;
  case PDF_NUMBER:
    colorspec->space_id  = PDF_COLOR_DEV_GRAY;
    colorspec->values[0] = pdf_number_value(pdf_get_array(color, 0));
    break;
  default:
    WARN("Expecting a number or an array for color specification.");
    pdf_release_obj(color);
    return -1;
  }
  pdf_release_obj(color);

  return 0;
}

static void
do_bgcolor (char **start, char *end)
{
  pdf_color colorspec;

  SAVE(*start, end);

  if (parse_colorspec(start, end, &colorspec) < 0) {
    WARN("No valid color specified? special pdf:bgcolor ignored.");
    DUMP_RESTORE(*start, end);
    return;
  }

  dev_bg_dev_color(&colorspec);
}

static void
do_bcolor (char **start, char *end)
{
  pdf_color colorspec;

  SAVE(*start, end);

  if (parse_colorspec(start, end, &colorspec) < 0) {
    WARN("Special: bcolor: No valid color specified?");
    DUMP_RESTORE(*start, end);
    return;
  }

  dev_begin_dev_color(&colorspec);
}

static void
do_scolor (char **start, char *end)
{
  pdf_color colorspec;

  SAVE(*start, end);

  if (parse_colorspec(start, end, &colorspec) < 0) {
    WARN("Special: bcolor: No valid color specified?");
    DUMP_RESTORE(*start, end);
    return;
  }

  dev_set_def_dev_color(&colorspec);
}

static void do_ecolor(void)
{
  dev_end_color();
}

static void
do_bxform (char **start, char *end, double x_user, double y_user)
{
  transform_info info;

  transform_info_clear(&info);

  SAVE(*start, end);

  skip_white(start, end);
  if (parse_dimension(start, end, &info) < 0) {
    WARN("Error in transformation parameters. Special will be ignored.");
    DUMP_RESTORE(*start, end);
    return;
  }
  validate_transform_info(&info);

  if (info.width  != 0.0  ||
      info.height != 0.0 || info.depth != 0.0)
    WARN("bxform: width, height, and depth are meaningless. (ignored)");
  if (info.scale != 0.0) {
    info.xscale = info.scale;
    info.yscale = info.scale;
  }
  if (info.xscale == 0.0)
    info.xscale = 1.0;
  if (info.yscale == 0.0)
    info.yscale = 1.0;

  dev_begin_transform(info.xscale, info.yscale, info.rotate, x_user, y_user);
}

static void
do_exform (void)
{
  dev_end_transform();
}

static void
do_tounicode (char **start, char *end)
{
  char *cmap_name;

  skip_white(start, end);
  /*
   * This should be PDF name or PDF string.
   */
  cmap_name = parse_ident(start, end);
  set_tounicode_cmap(cmap_name);
  RELEASE(cmap_name);
}

static void
do_outline (char **start, char *end)
{
  static int lowest_level = 255;
  pdf_obj *item_dict, *tmp;
  int level, is_open;
  int current_depth;

  SAVE(*start, end);

  skip_white(start, end);
  tmp = parse_pdf_object(start, end);
  if (!PDF_OBJ_NUMBERTYPE(tmp)) {
    WARN("Special: outline: Expecting number for object level.");
    DUMP_RESTORE(*start, end);
    return;
  }

  item_dict = NULL;
  is_open   = -1;

  level = (int) pdf_number_value(tmp);
  pdf_release_obj(tmp);

  /*
   * What is this? Starting at level 3 and can go down to level 1?
   *
   * Here is the original comment:
   *  Make sure we know where the starting level is
   */
  lowest_level = MIN(lowest_level, level);

  level  +=  1 - lowest_level;

  skip_white(start, end);
  if (*start < end && **start != '<') {
    char *opt;
    /*
     * Should we use key-value (open true/open false) pair here?
     */
    opt = parse_ident(start, end);
    if (!opt) {
      WARN("Unknown option in special pdf:out.");
      DUMP_RESTORE(*start, end);
      return;
    }
    if (!strcmp(opt, "opened")) {
      is_open = 1;
    } else if (!strcmp(opt, "closed")) {
      is_open = 0;
    } else {
      WARN("Unknown option \"%s\" in special pdf:out.", opt);
      DUMP_RESTORE(*start, end);
      return;
    }
    RELEASE(opt);
  }

  item_dict = parse_pdf_tainted_dict(start, end);
  if (!item_dict) {
    WARN("Ignoring invalid dictionary.");
    DUMP_RESTORE(*start, end);
    return;
  }
  current_depth = pdf_doc_bookmarks_depth();
  if (current_depth > level) {
    while (current_depth-- > level)
      pdf_doc_bookmarks_up();
  } else if (current_depth < level) {
    while (current_depth++ < level)
      pdf_doc_bookmarks_down();
  }

  pdf_doc_bookmarks_add(item_dict, is_open);
}

static void
do_article (char **start, char *end)
{
  char    *ident;
  pdf_obj *info_dict;

  SAVE(*start, end);

  skip_white (start, end);
  ident = parse_opt_ident(start, end);
  if (!ident) {
    WARN("Article name expected.");
    DUMP_RESTORE(*start, end);
    return;
  }

  info_dict = parse_pdf_tainted_dict(start, end);
  if (!info_dict) {
    WARN("Ignoring invalid dictionary.");
    DUMP_RESTORE(*start, end);
    return;
  }

  pdf_doc_start_article(ident, pdf_link_obj(info_dict));
  add_object(ident, info_dict);

  RELEASE(ident);
}

static void
do_bead (char **start, char *end)
{
  pdf_obj *article;
  pdf_obj *article_info;
  char    *article_name;
  pdf_rect rect;
  long     page_no;
  transform_info info;

  SAVE(*start, end);

  skip_white(start, end);
  if (**start != '@') {
    WARN("Article reference expected.");
    DUMP_RESTORE(*start, end);
    return;
  }

  article_name = parse_ident(start, end);
  if (!article_name) {
    WARN("Article reference expected.");
    WARN("Which article does this go with?");
    DUMP_RESTORE(*start, end);
    return;
  }

  /* If okay so far, try to get a bounding box */
  transform_info_clear(&info);
  skip_white(start, end);
  if (parse_dimension(start, end, &info) < 0) {
    WARN("Special: thread: Error in bounding box specification for this bead.");
    DUMP_RESTORE(*start, end);
    return;
  } else if (info.scale  != 0.0 ||
	     info.xscale != 0.0 || info.yscale != 0.0) {
    WARN("Scale meaningless for annotations.");
    DUMP_RESTORE(*start, end);
    return;
  } else  if (info.width == 0.0 || info.depth + info.height == 0.0) {
    WARN("Special thread: Rectangle has a zero dimension.");
    DUMP_RESTORE(*start, end);
    return;
  }
  validate_transform_info(&info);

  rect.llx = dev_phys_x();
  rect.lly = dev_phys_y() - info.depth;
  rect.urx = dev_phys_x() + info.width;
  rect.ury = dev_phys_y() + info.height;

  skip_white (start, end);
  if (**start != '<') {
    article_info = pdf_new_dict();
  } else {
    article_info = parse_pdf_tainted_dict(start, end);
    if (!article_info) {
      WARN("Special: thread: Error in dictionary.");
      DUMP_RESTORE(*start, end);
      return;
    }
  }

  /* Does this article exist yet */
  article = lookup_object(article_name);
  if (article) {
    pdf_merge_dict (article, article_info);
    pdf_release_obj(article_info);
  } else {
    pdf_doc_start_article(article_name, pdf_link_obj(article_info));
    add_object(article_name, article_info);
  }
  page_no = pdf_doc_current_page_no();
  pdf_doc_add_bead(article_name, NULL, page_no, &rect);
}

static void
do_image (char **start, char *end, double x_user, double y_user)
{
  int      xobj_id;
  char    *ident;
  pdf_obj *fspec;
  transform_info info;

  SAVE(*start, end);

  transform_info_clear(&info);

  skip_white(start, end);
  if (**start == '@') {
    ident = parse_opt_ident(start, end);
  } else {
    ident = NULL;
  }

  skip_white(start, end);
  if (parse_dimension(start, end, &info) < 0) {
    WARN("Error in parsing dimensions of encapsulated image. special pdf:image ignored.");
    DUMP_RESTORE(*start, end);
    if (ident)
      RELEASE(ident);
    return;
  }
  validate_transform_info(&info);

  skip_white(start, end);
  fspec = parse_pdf_string(start, end);
  if (!fspec) {
    WARN("Missing filename.");
    DUMP_RESTORE(*start, end);
    if (ident)
      RELEASE(ident);
    return;
  }

  xobj_id = pdf_ximage_findresource(pdf_string_value(fspec));
  if (xobj_id < 0) {
    WARN("Could not find image resource: \"%s\"", pdf_string_value(fspec));
  } else {
    pdf_ximage_put_image(xobj_id, &info, x_user, y_user);
  }
  pdf_release_obj(fspec);

  if (ident) {
    if (xobj_id >= 0)
      add_reference(ident, pdf_ximage_getresource(xobj_id));
    RELEASE(ident);
  }
}

/* Use do_names instead. */
static void
do_dest (char **start, char *end)
{
  pdf_obj *name;
  pdf_obj *array;

  SAVE(*start, end);

  skip_white(start, end);
  name = parse_pdf_string(start, end);
  if (!name) {
    WARN("PDF string expected and not found. special pdf:dest ignored.");
    DUMP_RESTORE(*start, end);
    return;
  }

  array = parse_pdf_object(start, end);
  if (!array) {
    WARN("Destination not specified. special pdf:dest ignored.");
    DUMP_RESTORE(*start, end);
    pdf_release_obj(name);
    return;
  }
  pdf_doc_add_names("Dests", name, pdf_ref_obj(array));
  pdf_release_obj(array);
}

static void
do_names (char **start, char *end)
{
  pdf_obj *category, *key, *value;

  SAVE(*start, end);

  category = parse_pdf_name(start, end);
  if (!category) {
    WARN("PDF name expected but not found. special pdf:names ignored.");
    DUMP_RESTORE(*start, end);
    return;
  }

  key = parse_pdf_string(start, end);
  if (!key) {
    pdf_release_obj(category);
    WARN("PDF string expected but not found. special pdf:names ignored.");
    DUMP_RESTORE(*start, end);
    return;
  }

  value = parse_pdf_object(start, end);
  if (!value) {
    pdf_release_obj(category);
    pdf_release_obj(key);
    WARN("PDF object expected but not found. special pdf:names ignored.");
    DUMP_RESTORE(*start, end);
    return;
  }

  if (pdf_doc_add_names(pdf_name_value(category), key, value) < 0) {
    WARN("Special \"pdf:names\" ignored.");
    pdf_release_obj(key);
    pdf_release_obj(value);
  }

  pdf_release_obj(category);
}

static void
do_docinfo (char **start, char *end)
{
  pdf_obj *docinfo, *dict;

  SAVE(*start, end);

  dict = parse_pdf_tainted_dict(start, end);
  if (!dict) {
    WARN("Dictionary expected but not found. special:docinfo ignored.");
    DUMP_RESTORE(*start, end);
  } else {
    docinfo = pdf_doc_docinfo();
    pdf_merge_dict(docinfo, dict);
    pdf_release_obj(dict);
  }
}

static void
do_docview (char **start, char *end)
{
  pdf_obj *catalog, *docview;
  pdf_obj *pref_old, *pref_add;

  SAVE(*start, end);

  docview = parse_pdf_dict(start, end);
  if (!docview) {
    WARN("Dictionary expected but not found. special:docview ignored.");
    DUMP_RESTORE(*start, end);
  } else {
    catalog  = pdf_doc_catalog();
    /* Avoid overriding whole ViewerPreferences */
    pref_old = pdf_lookup_dict(catalog, "ViewerPreferences");
    pref_add = pdf_lookup_dict(docview, "ViewerPreferences");
    if (pref_old && pref_add) {
      pdf_merge_dict (pref_old, pref_add);
      pdf_remove_dict(docview, "ViewerPreferences");
    }
    pdf_merge_dict (catalog, docview);
    pdf_release_obj(docview);
  }
}


static void
do_close (char **start, char *end)
{
  char *ident;

  SAVE(*start, end);

  skip_white(start, end);
  ident = parse_opt_ident(start, end);
  if (ident) {
    release_reference(ident);
    RELEASE(ident);
  } else {
    /* Close all? */
  }
}

static void
do_obj (char **start, char *end)
{
  pdf_obj *object;
  char    *ident;

  SAVE(*start, end);

  skip_white(start, end);
  ident = parse_opt_ident(start, end);
  if (!ident) {
    WARN("Could not find a reference name. special:object ignored.");
    DUMP_RESTORE(*start, end);
  } else {
    object = parse_pdf_object(start, end);
    if (!object) {
      WARN("Could not find an object. special:object ignored.");
      DUMP_RESTORE(*start, end);
    } else {
      add_object(ident, object);
    }
    RELEASE(ident);
  }
}

static void
do_content (char **start, char *end, double x_user, double y_user)
{
  int len;

  if (*start < end) {
    len = sprintf(work_buffer, " q 1 0 0 1 %.2f %.2f cm ", x_user, y_user);
    pdf_doc_add_to_page(work_buffer, len);
    pdf_doc_add_to_page(*start, end - (*start));
    pdf_doc_add_to_page(" Q", 2);
  }
  *start = end;
}

static void
do_literal (char **start, char *end, double x_user, double y_user)
{
  int len, direct = 0;

  skip_white(start, end);
  while (*start < end) {
    if (*start <= end - 7 &&
	!strncmp(*start, "reverse", 7)) {
      x_user *= -1.0;
      y_user *= -1.0;
      *start += 7;
    } else if (*start <= end - 6 &&
	       !strncmp(*start, "direct", 6)) {
      direct = 1;
      *start += 6;
    } else {
      break;
    }
    skip_white(start, end);
  }

  skip_white(start, end);
  /* It seems that "literal" used as if it were "translate" command... */
  if (direct) {
    pdf_doc_add_to_page(" ", 1);
  } else {
    len = sprintf(work_buffer,
		  " 1 0 0 1 %.2f %.2f cm ", x_user, y_user);
    pdf_doc_add_to_page(work_buffer, len);
  }
  if (*start < end)
    pdf_doc_add_to_page(*start, end - (*start));

  *start = end;
}

/*
 * FSTREAM: Create a PDF stream object from an existing file.
 *
 *  pdf: fstream @objname (filename) [PDF_DICT]
 */
static void
do_fstream (char **start, char *end)
{
  pdf_obj *fstream;
  long     stream_len;
  char    *ident, *filename, *fullname;
  pdf_obj *tmp;
  FILE    *fp;

  SAVE(*start, end);

  skip_white(start, end);
  ident = parse_opt_ident(start, end);
  if (!ident) {
    WARN("Could not find a reference name. special pdf:fstream ignored.");
    DUMP_RESTORE(*start, end);
    return;
  }

  skip_white(start, end);
  tmp = parse_pdf_string(start, end);
  if (!tmp) {
    WARN("Missing filename. special pdf:fstream ignored.");
    DUMP_RESTORE(*start, end);
    RELEASE(ident);
    return;
  }
  filename = pdf_string_value(tmp);
  if (!filename) {
    WARN("Missing filename. special pdf:fstream ignored.");
    DUMP_RESTORE(*start, end);
    RELEASE(ident);
    return;
  }

  MESG("(FILE:%s", filename);

  fstream  = pdf_new_stream(STREAM_COMPRESS);

  fullname = kpse_find_file(filename, kpse_program_binary_format, 0);
  pdf_release_obj(tmp);
  if (!fullname) {
    WARN("Could not find file. special pdf:fstream ignored.", filename);
    DUMP_RESTORE(*start, end);
    RELEASE(ident);
    pdf_release_obj(fstream);
    return;
  }

  fp = MFOPEN(fullname, FOPEN_RBIN_MODE);
  if (!fp) {
    WARN("Could not open file. special pdf:fstream ignored.", filename);
    DUMP_RESTORE(*start, end);
    RELEASE(ident);
    pdf_release_obj(fstream);
    return;
  }

  stream_len = 0;
  {
    long len;

    while ((len = fread(work_buffer, sizeof(char), WORK_BUFFER_SIZE, fp)) > 0) {
      pdf_add_stream(fstream, work_buffer, len);
      stream_len += len;
    }
  }

  MFCLOSE(fp);

  if (verbose > 1)
    MESG("[%s]", fullname);

  /*
   * Optional dict.
   *
   *  TODO: check Length, Filter...
   */
  skip_white(start, end);
  if (**start == '<') {
    pdf_obj *stream_dict;

    stream_dict = pdf_stream_dict(fstream);

    tmp = parse_pdf_dict(start, end);
    if (!tmp) {
      WARN("Parsing dictionary failed. special:fstream ignored.");
      DUMP_RESTORE(*start, end);
      RELEASE(ident);
      pdf_release_obj(fstream);
      return;
    }
    if (pdf_lookup_dict(tmp, "Length")) {
      WARN("Please don't use /Length key...");
      pdf_remove_dict(tmp, "Length");
    } else if (pdf_lookup_dict(tmp, "Filter")) {
      WARN("Please don't use /Filter key...");
      pdf_remove_dict(tmp, "Filter");
    }
    pdf_merge_dict(stream_dict, tmp);
    pdf_release_obj(tmp);
  }

  /* Users should explicitly close this. */
  add_object(ident, fstream);
  RELEASE(ident);

  MESG(")");

  return;
}

static int
is_pdf_special (char **start, char *end)
{
  skip_white(start, end);
  if (*start <= end - strlen("pdf:") &&
      !strncmp(*start, "pdf:", strlen("pdf:"))) {
    *start += strlen("pdf:");
    return 1;
  }

  return 0;
}

#define ANN     1
#define OUTLINE 2
#define ARTICLE 3
#define DEST    4
#define DOCINFO 7
#define DOCVIEW 8
#define OBJ     9
#define CONTENT 10
#define PUT     11
#define CLOSE   12
#define BOP     13
#define EOP     14
#define BEAD    15
#define EPDF    16
#define IMAGE   17
#define BCOLOR  18
#define ECOLOR  19
#define BGRAY   20
#define EGRAY   21
#define BGCOLOR 22
#define BXFORM  23
#define EXFORM  24
#define PAGE_SIZE 25
#define BXOBJ   26
#define EXOBJ   27
#define UXOBJ   28
#define SCOLOR  29
#define BANN    30
#define EANN    31
#define LINK_ANNOT   32
#define NOLINK_ANNOT 33
#define TOUNICODE 34
#define LITERAL 35
#define FSTREAM 36
#define NAMES   37

static struct pdfmark
{
  const char *string;
  int value;
} pdfmarks[] = {
  {"ann",        ANN},
  {"annot",      ANN},
  {"annotate",   ANN},
  {"annotation", ANN},
  {"out",     OUTLINE},
  {"outline", OUTLINE},
  {"art",     ARTICLE},
  {"article", ARTICLE},
  {"bead",   BEAD},
  {"thread", BEAD},
  {"dest",   DEST},
  {"docinfo", DOCINFO},
  {"docview", DOCVIEW},
  {"obj",    OBJ},
  {"object", OBJ},
  {"content", CONTENT},
  {"put",   PUT},
  {"close", CLOSE},
  {"bop", BOP},
  {"eop", EOP},
  {"epdf",  EPDF},
  {"image", IMAGE},
  {"img",   IMAGE},
  {"bc",         BCOLOR},
  {"bcolor",     BCOLOR},
  {"begincolor", BCOLOR},
  {"link",   LINK_ANNOT},
  {"nolink", NOLINK_ANNOT},
  {"sc",        SCOLOR},
  {"scolor",    SCOLOR},
  {"setcolor",  SCOLOR},
  {"ec",        ECOLOR},
  {"ecolor",    ECOLOR},
  {"endcolor",  ECOLOR},
  {"bg",        BGRAY},
  {"bgray",     BGRAY},
  {"begingray", BGRAY},
  {"eg",        EGRAY},
  {"egray",     EGRAY},
  {"endgray",   EGRAY},
  {"bgcolor",   BGCOLOR},
  {"bgc",       BGCOLOR},
  {"bbc",       BGCOLOR},
  {"bbg",       BGCOLOR},
  {"pagesize",  PAGE_SIZE},
  {"beginann",  BANN},
  {"bann",      BANN},
  {"bannot",    BANN},
  {"eann",   EANN},
  {"endann", EANN},
  {"eannot", EANN},
  {"begintransform", BXFORM},
  {"begintrans",     BXFORM},
  {"btrans",         BXFORM},
  {"bt",             BXFORM},
  {"endtransform",   EXFORM},
  {"endtrans",       EXFORM},
  {"etrans",         EXFORM},
  {"et",             EXFORM},
  {"beginxobj", BXOBJ},
  {"bxobj",     BXOBJ},
  {"endxobj",   EXOBJ},
  {"exobj",     EXOBJ},
  {"usexobj",   UXOBJ},
  {"uxobj",     UXOBJ},
  {"tounicode", TOUNICODE},
  {"literal",   LITERAL},
  {"fstream",   FSTREAM},
  {"names",     NAMES}
};

static int
parse_pdfmark (char **start, char *end)
{
  char *save;
  int i;
  if (verbose > 2) {
    MESG("\nparse_pdfmark:");
    dump (*start, end);
  }
  skip_white(start, end);
  if (*start >= end) {
    WARN("Special ignored...no pdfmark found.");
    return -1;
  }

  save = *start;
  while (*start < end && isalpha (**start))
    (*start)++;
  for (i=0; i<sizeof(pdfmarks)/sizeof(struct pdfmark); i++) {
    if (*start-save == strlen (pdfmarks[i].string) &&
        !strncmp (save, pdfmarks[i].string,
                  strlen(pdfmarks[i].string)))
      return pdfmarks[i].value;
  }
  *start = save;
  WARN("Expecting pdfmark (and didn't find one).");
  dump(*start, end);
  return -1;
}

struct named_object
{
  char    *name;
  pdf_obj *object_ref;
  pdf_obj *object;
} *named_objects = NULL;

static long num_named_objects = 0;
static long max_named_objects = 0;

static void
add_object (const char *ident, pdf_obj *object)
{
  int i;

  if (num_named_objects >= max_named_objects) {
    max_named_objects += NAMED_OBJ_ALLOC_SIZE;
    named_objects = RENEW(named_objects,
			  max_named_objects, struct named_object);
  }

  for (i = 0; i < num_named_objects; i++)
    if (!strcmp(named_objects[i].name, ident))
      break;

  if (i == num_named_objects) {
    named_objects[i].name = NEW(strlen(ident)+1, char);
    strcpy(named_objects[i].name, ident);
    named_objects[i].object    = NULL;
    named_objects[i].object_ref = NULL;
    num_named_objects++;
  }

  if (named_objects[i].object_ref) {
    WARN("Object already has reference...");
    pdf_release_obj(named_objects[i].object_ref);
  }
  named_objects[i].object     = object;
  named_objects[i].object_ref = NULL;
}

static void
add_reference (const char *name, pdf_obj *object_ref)
{
  int i;

  if (!PDF_OBJ_INDIRECTTYPE(object_ref))
    ERROR("Passed non-reference...");

  if (num_named_objects >= max_named_objects) {
    max_named_objects += NAMED_OBJ_ALLOC_SIZE;
    named_objects = RENEW(named_objects,
			  max_named_objects, struct named_object);
  }

  for (i = 0; i < num_named_objects; i++)
    if (!strcmp(named_objects[i].name, name))
      break;

  if (i == num_named_objects) {
    named_objects[i].name = NEW(strlen(name)+1, char);
    strcpy(named_objects[i].name, name);
    named_objects[i].object    = NULL;
    named_objects[i].object_ref = NULL;
    num_named_objects++;
  }

  if (named_objects[i].object) {
    WARN("Object already defined. Flushing it...");
    pdf_release_obj(named_objects[i].object);
  }

  named_objects[i].object_ref = object_ref;
  named_objects[i].object     = NULL;
}

/*
 * The following routine returns copies, not the original object.
 */
pdf_obj *
lookup_reference (const char *name)
{
  int i;

  /* First check for builtins first */
  if (!strcmp(name, "ypos")) {
    return pdf_new_number(ROUND(dev_phys_y(), .001));
  } else if (!strcmp(name, "xpos")) {
    return pdf_new_number(ROUND(dev_phys_x(), .001));
  } else if (!strcmp(name, "thispage")) {
    return pdf_doc_this_page_ref();
  } else if (!strcmp(name, "prevpage")) {
    return pdf_doc_prev_page_ref();
  } else if (!strcmp(name, "nextpage")) {
    return pdf_doc_next_page_ref();
  } else if (!strcmp(name, "pages")) {
    return pdf_ref_obj(pdf_doc_page_tree());
  } else if (!strcmp(name, "names")) {
    return pdf_ref_obj(pdf_doc_names());
  } else if (!strcmp(name, "resources")) {
    return pdf_ref_obj(pdf_doc_current_page_resources());
  } else if (!strcmp(name, "catalog")) {
    return pdf_ref_obj(pdf_doc_catalog());
  } else if (strlen(name) > 4 &&
	     !strncmp(name, "page", 4) && is_an_int(name + 4)) {
    return pdf_doc_ref_page(atoi(name + 4));
  }

  for (i = 0; i < num_named_objects; i++) {
    if (!strcmp(named_objects[i].name, name)) {
      break;
    }
  }
  /*
   * "obj @foo [@bar]" creates null object @bar when @bar is not
   * defined yet, and the value of @foo is something like [10 0 R]
   * where the indirect object #10 is a null object. Clearly it
   * does not work as intended...
   */
  if (i == num_named_objects) {
    named_objects[i].name = NEW(strlen(name)+1, char);
    strcpy(named_objects[i].name, name);
    named_objects[i].object = pdf_new_null();
    num_named_objects++;
  }

  if (!named_objects[i].object_ref)
    named_objects[i].object_ref = pdf_ref_obj(named_objects[i].object);

  return pdf_link_obj(named_objects[i].object_ref);
}

static pdf_obj *
lookup_object (const char *name)
{
  int i;

  if (!strcmp(name, "ypos")) {
    return pdf_new_number(ROUND(dev_phys_y(), .001));
  } else if (!strcmp(name, "xpos")) {
    return pdf_new_number(ROUND(dev_phys_x(), .001));
  } else if (!strcmp(name, "thispage")) {
    return pdf_doc_this_page();
  } else if (!strcmp(name, "pages")) {
    return pdf_doc_page_tree();
  } else if (!strcmp(name, "names")) {
    return pdf_doc_names();
  } else if (!strcmp(name, "resources")) {
    return pdf_doc_current_page_resources();
  } else if (!strcmp(name, "catalog")) {
    return pdf_doc_catalog();
  }

  for (i = 0; i < num_named_objects; i++) {
    if (!strcmp(named_objects[i].name, name)) {
      break;
    }
  }
  if (i == num_named_objects)
    return NULL;
  else if (!named_objects[i].object) {
    WARN("lookup_object: Referenced object not defined or already closed.");
  }

  return named_objects[i].object;
}

static void
release_reference (const char *name)
{
  int i;

  for (i = 0; i < num_named_objects; i++) {
    if (!strcmp(named_objects[i].name, name)) {
      break;
    }
  }

  if (i == num_named_objects) {
    WARN("release_reference: tried to release nonexistent reference.");
  } else if (named_objects[i].object != NULL) {
    pdf_release_obj(named_objects[i].object);
    named_objects[i].object = NULL;
  } else {
    WARN("release_reference: @%s: trying to close an object twice?", name);
  }
}

void
pdf_special_init (void)
{
  /* no-op */
}

void
pdf_special_close (void)
{
  /*
   * Flush out any pending objects that weren't properly closeed.
   * Threads never get closed.  Is this a bug?
   */
  if (named_objects) {
    int i;

    for (i = 0; i < num_named_objects; i++) {
      if (named_objects[i].object_ref) {
	pdf_release_obj(named_objects[i].object_ref);
	named_objects[i].object_ref = NULL;
      }
      if (named_objects[i].object) {
	pdf_release_obj(named_objects[i].object);
	named_objects[i].object = NULL;
      }
      RELEASE(named_objects[i].name);
      named_objects[i].name = NULL;
    }
    RELEASE(named_objects);
  }
}

int
pdf_parse_special (char *buffer, UNSIGNED_QUAD size,
		   double x_user, double y_user)
{
  char *start = buffer, *end = buffer + size;

  if (is_pdf_special(&start, end)) {
    switch (parse_pdfmark(&start, end)) {
    case ANN:
      do_ann(&start, end);
      break;
    case BANN:
      do_bann(&start, end);
      break;
    case LINK_ANNOT:
      dev_link_annot(1);
      break;
    case NOLINK_ANNOT:
      dev_link_annot(0);
      break;
    case EANN:
      do_eann(&start, end);
      break;
    case OUTLINE:
      do_outline(&start, end);
      break;
    case ARTICLE:
      do_article(&start, end);
      break;
    case BEAD:
      do_bead(&start, end);
      break;
    case DEST:
      do_dest(&start, end);
      break;
    case DOCINFO:
      do_docinfo(&start, end);
      break;
    case DOCVIEW:
      do_docview(&start, end);
      break;
    case OBJ:
      do_obj(&start, end);
      break;
    case CONTENT:
      do_content(&start, end, x_user, y_user);
      break;
    case PUT:
      do_put(&start, end);
      break;
    case CLOSE:
      do_close(&start, end);
      break;
    case BOP:
      do_bop(&start, end);
      break;
    case EOP:
      do_eop(&start, end);
      break;
    case EPDF:
      do_image(&start, end, x_user, y_user);
      break;
    case IMAGE:
      do_image(&start, end, x_user, y_user);
      break;
    case BGCOLOR:
      do_bgcolor(&start, end);
      break;
    case SCOLOR:
      do_scolor(&start, end);
      break;
    case BCOLOR:
    case BGRAY:
      do_bcolor(&start, end);
      break;
    case ECOLOR:
    case EGRAY:
      do_ecolor();
      break;
    case BXFORM:
      do_bxform(&start, end, x_user, y_user);
      break;
    case EXFORM:
      do_exform();
      break;
    case PAGE_SIZE:
      do_pagesize(&start, end);
      break;
    case BXOBJ:
      do_bxobj(&start, end, x_user, y_user);
      break;
    case EXOBJ:
      do_exobj(&start, end);
      break;
    case UXOBJ:
      do_uxobj(&start, end, x_user, y_user);
      break;
    case TOUNICODE:
      do_tounicode(&start, end);
      break;
    case LITERAL:
      do_literal(&start, end, x_user, y_user);
      break;
    case FSTREAM:
      do_fstream(&start, end);
      break;
    case NAMES:
      do_names(&start, end);
      break;
    default:
      dump(start, end);
      WARN("Invalid pdf special ignored.");
      break;
    }
    skip_white(&start, end);
    if (start < end) {
      WARN("Unparsed material at end of special ignored.");
      dump(start, end);
    }
    return 1;
  } else
    return 0;
}

#define SPECIAL_OBJNAME_MAX 256
/* Grab page content */
static void
do_bxobj (char **start, char *end, double x_user, double y_user)
{
  char    *ident;
  pdf_rect cropbox;
  transform_info p;

  transform_info_clear(&p);

  skip_white(start, end);
  ident = parse_opt_ident(start, end);
  if (!ident) {
    WARN("A form XObject must have named. special pdf:bxobj ignored.");
    RELEASE(ident);
    return;
  }
  if (strlen(ident) > SPECIAL_OBJNAME_MAX) {
    WARN("Object reference name too long. (truncated)");
    ident[SPECIAL_OBJNAME_MAX] = '\0';
  }

  skip_white(start, end);
  if (parse_dimension(start, end, &p) < 0) {
    WARN("Failed to find a valid dimension. special pdf:bxobj ignored.");
    RELEASE(ident);
    return;
  }
  validate_transform_info(&p);

  if (p.scale  != 0.0  ||
      p.xscale != 0.0 || p.yscale != 0.0) {
    WARN("Scale information is meaningless. special pdf:bxobj ignored.");
    RELEASE(ident);
    return;
  }

#if 0
  /* The following code can't be removed.
   *
   * A XForm with zero dimension result in a non-invertible transformation
   * matrix. And it may result in unpredictable behaviour. It might be an
   * error in Acrobat. Bounding box with zero dimention may cause division
   * by zero.
   */
  /* Removed because ConTeXt sometimes uses an empty object. */
#endif
  if (p.width == 0.0 ||
      p.depth + p.height == 0.0) {
    WARN("Special: bxobj: Bounding box has a zero dimension.");
    WARN("width:%g, height:%g, depth:%g", p.width, p.height, p.depth);
    RELEASE(ident);
    return;
  }

  cropbox.llx = x_user;
  cropbox.lly = y_user - p.depth;
  cropbox.urx = x_user + p.width;
  cropbox.ury = y_user + p.height;

  /*
   * ident only used by pdfspecial :-(
   * Tell pdf_doc to remember identifier of the current form object
   * and it's bounding box.
   */
  pdf_doc_start_grabbing(ident, x_user, y_user, &cropbox);
}

static void
do_exobj (char **start, char *end)
{
  int         xobj_id;
  pdf_obj    *xform;
  xform_info  info;
  char       *ident;

  skip_white(start, end);
  if (*start < end) {
    pdf_obj *garbage;
    pdf_obj *resource;

    /* An extra dictionary after exobj must be merged to /Resources???
     * Please use pdf:put @resources (before pdf:exobj) instead.
     */
    garbage = parse_pdf_dict(start, end);
    if (garbage) {
      resource = pdf_doc_current_page_resources();
      pdf_merge_dict(resource, garbage);
#if 0
      WARN("Garbage after special pdf:exobj.");
      pdf_release_obj(garbage);
#endif
    }
  }

  pdf_ximage_init_form_info(&info);
  pdf_doc_current_form_info(&ident, &(info.bbox));

  xform = pdf_doc_end_grabbing();

  /* Both pdfximage and pdfspecial remember the name of this form
   * object. uxobj use pdfximage to put images, other pdf specials
   * use pdfspecial to link form XObject resource. (pdfximage can
   * be used for this purpose, though)
   */
  xobj_id = pdf_ximage_defineresource(ident, PDF_XOBJECT_TYPE_FORM, &info, xform);
  
  add_reference(ident, pdf_ximage_getresource(xobj_id));
  RELEASE(ident);
}

static void
do_uxobj (char **start, char *end, double x_user, double y_user)
{
  int   xobj_id;
  char *ident;
  transform_info p;

  transform_info_clear(&p);

  skip_white(start, end);
  ident = parse_opt_ident(start, end);
  if (!ident)
    ERROR("Special: usexobj: No XObject specified.");

  skip_white(start, end);
  if (*start < end) {
    if (parse_dimension(start, end, &p) < 0)
      transform_info_clear(&p);
    validate_transform_info(&p);
  }

  xobj_id = pdf_ximage_findresource(ident);
  if (xobj_id < 0)
    WARN("Specified XObject \"%s\" doesn't exist. special pdf:uxobj ignored.", ident);
  else {
    pdf_ximage_put_image(xobj_id, &p, x_user, y_user);
  }
  RELEASE(ident);
}
