/* 
    Copyright (C) 2000 Marcus Metzler (mocm@convergence.de)
    for convergence integrated media

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
 
#include <stdint.h>
#include <strstream.h>
#include <time.h>
#include <ost/audio.h>
#include <ost/video.h>
#include "dvb_formats.h"

#include "rtuxzap.hh"
#include "channel.hh"
extern "C"{
#ifdef HAVE_LIBLIRC_CLIENT
#include <lirc/lirc_client.h>
#endif
#include <curses.h>
}


DVB dvb;
ChannelList chlist[NLIST+1];
int clist = 0;
int lastclist = 0;
int lastlc[NLIST+1];
int currlc[NLIST+1];
int it = 0;
int oit = 1;
int dev = 0;
int o=0;
int audio=0;
time_t mytime=0;
int rmute = 0;
boolean enable_dvr = false;
boolean enable_rec = false;
boolean enable_replay = false;
boolean wide = false;
boolean vt_block = false;
uint16_t current_apid;
uint16_t current_vpid;
int nswitch = 0;

static char apref[4] = {'E','N','G',0};
//static char *apref = NULL;

#ifdef HAVE_LIBLIRC_CLIENT
static struct lirc_config *config;
#endif
int volume = 200;
int has_decoder = 1;
#ifdef PTHREADS
int save_format = C_TS;
char *save_dir = "/tmp/";
char *save_name = "record";
#endif

void clear_display()
{
	int y = 2;
	int x = 40;
	
	clear();
	mvprintw(0,0,"Welcome to RTuxZap");
	mvprintw(y, 0,   "e  : toggle recording");
	if (has_decoder)
		mvprintw(y++, x, "p  : toggle playback");
	else y++;
	mvprintw(y, 0,   "r  : return to last program");
	mvprintw(y++, x, "w  : toggle wide screen setting");
	mvprintw(y, 0,   "z  : toggle random zapping");
	mvprintw(y++, x, "y  : decrease random interval");
	mvprintw(y, 0,   "x  : increase random interval");
	mvprintw(y++, x, "q  : quit"); 
	if (has_decoder){
		mvprintw(y, 0,   "+  : increase volume");
		mvprintw(y++, x, "-  : decrease volume");
	}
	mvprintw(y, 0,   "d|v: one program down in current list");
	mvprintw(y++, x, "u|^: one program up in current list");
	mvprintw(y, 0,   "f|>: next list");
	mvprintw(y++, x, "g|<: previous list");
	mvprintw(y, 0,   "a  : next audio channel");
	mvprintw(y++, x, "t  : show time and date");
	mvprintw(y, 0,   "v  : block all keys for videotext");
	mvprintw(y++, x, "o  : start/restart OSD");
	mvprintw(y++, 0,   "0 ... 9: input program number in list");
	mvprintw(y, 0,   "| followed by <,>,v,^ refers to arrow keys");
	refresh();
}

void killprint(int o)
{
	if (mytime &&time(0) > mytime){
		if (o) dvb.osd.Hide();
		mytime = 0;
	}
}

void myprint(char *str, int o, int t=1)
{
	mytime = time(0) - 1;
	killprint(o);
	clear_display();
	if (o){
		dvb.osd.Clear();
		dvb.osd.Text(0, 0, 20, 1, str);
		dvb.osd.Show();
		mytime = time(0)+t;
	}
	mvprintw(15,0,str);
}


void set_volume(int vol)
{
	int ans;
	audioMixer_t am;
	int audio_fd;
        char path[256];
	ostrstream str(path,256);
	
	if (enable_rec || !has_decoder) return;
	str << "/dev/ost/audio" << dev ;
	str << ends;

	if (vol > 255) vol = 255;
	if (vol < 0  ) vol = 0;
	volume = vol;

	am.volume_left = vol;
	am.volume_right = vol;

	if ((audio_fd = open(path, O_RDWR)) < 0){
	    perror("mixer");
	    return;
	}

	if ( (ans = ioctl(audio_fd,AUDIO_SET_MIXER, &am) < 0)){
		perror("AUDIO SET MIXER: ");
		return;
	}

	close (audio_fd);

	ostrstream fstr;
	fstr << "Volume " << volume << ends;
	myprint(fstr.str(), o, 2);
}

void set_format(videoFormat_t format)
{
        int ans;
	int fd;

        char path[256];
	ostrstream str(path,256);
	
	if (enable_rec || !has_decoder) return;
	str << "/dev/ost/video" << dev ;
	str << ends;

	if ((fd = open(path, O_RDWR)) < 0){
	    perror("display");
	    return;
	}

        if ( (ans = ioctl(fd,VIDEO_SET_FORMAT, format) < 0)){
                perror("VIDEO SET FORMAT: ");
        }
	close(fd);
}

void SetMute(int m){
	int audio_fd;
        char path[256];
	ostrstream str(path,256);
	audioMixer_t am;

	if (enable_rec|| !has_decoder) return;
	rmute = m;
	str << "/dev/ost/audio" << dev ;
	str << ends;

	if ((audio_fd = open(path, O_RDWR)) < 0){
	    perror("mixer");
	    return;
	}
	if (m){
		am.volume_left = 0;
		am.volume_right = 0;
	} else {
		am.volume_left = volume;
		am.volume_right = volume;
	}
	if ( ioctl(audio_fd,AUDIO_SET_MIXER, &am) < 0){
		perror("AUDIO SET MIXER: ");
	}
		
	close (audio_fd);
}


void stop_curses(void)
{
    clear();
    refresh();
    endwin();
}

void curses_init(void)     /* for kbd input */
{
    initscr();
    refresh();
    cbreak();
    noecho();
    nonl();
    intrflush(stdscr, FALSE);
    keypad(stdscr, TRUE);
    nodelay(stdscr, TRUE);

    atexit(stop_curses);
}

int lirc_zap_init()
{
#ifdef HAVE_LIBLIRC_CLIENT
	int fd;
	
	if (-1 == (fd = lirc_init("rtuxzap",0))) {
		cerr << "no infrared remote support available" << endl;
		return -1;
	}
	if (0 != lirc_readconfig(NULL,&config,NULL)) {
		lirc_deinit();
		return -1;
	}
	fcntl(fd,F_SETFL,O_NONBLOCK);
	fcntl(fd,F_SETFD,FD_CLOEXEC);
	
	return fd;
#else
	cerr << "no infrared remote not supported" << endl;
	return -1;
#endif
}


static void get_dvbfav(char *path, int nb, int len)
{
        const char *home = getenv("HOME");
        const char *file = ".dvbfav";

	ostrstream str(path,len);
	
	str << home << "/" << file << nb ;
	if (dev)
		str << "." << dev;
	str << ends; 
	lastlc[nb] = 0;
	currlc[nb] = 0;

}

static void load_ch(int nb)
{
	char filen[80];

	get_dvbfav(filen,nb,80);
	cout << _("Reading channel list from ") << filen << endl;
	chlist[nb].init(filen);
}

static int cnum (gChannel & ch)
{
	char buf[80];
        ostrstream str(buf, 80);

        uint16_t onid  =  uint16_t((ch.DVB_Num() & 
				    0xFFFF000000000000ULL) >> 48);
	uint16_t satid =  uint16_t((ch.DVB_Num() & 
				    0x0000FFFF00000000ULL) >> 32);
        uint16_t tpid  =  uint16_t((ch.DVB_Num() & 
				    0x00000000FFFF0000ULL) >> 16);
	ushort sid     =  ushort   (ch.DVB_Num() & 0x000000000000FFFFULL);


	int chnr;
	int found=0;
	for (chnr=0; chnr< dvb.num[CHAN]; chnr++) {
		if (dvb.chans[chnr].pnr==sid   && 
		    (dvb.chans[chnr].onid==onid || onid == 0xffff) && 
		    (dvb.chans[chnr].satid==satid || satid == 0xffff)  && 
                    (dvb.chans[chnr].tpid==tpid)) {
		  found = 1;
		  break;
                }
	}
	if (!found) chnr = -1;
	return chnr;
}


void set_channel()
{
	ostrstream fstr;
	int num = cnum(chlist[clist](currlc[clist]));
	
	if (num < 0) return;

	char buf[80];
	ostrstream str(buf, 80);

	SetMute(1);

	if (clist < NLIST)
		fstr << "F " << clist << "  Prog: " 
		     << currlc[clist] 
		     << ends;
	else 
		fstr << "DVB Prog: " 
		     << currlc[clist] 
		     << ends;
	
	myprint(fstr.str(), o, 2);

	audio = 0;
	dvb.SetChannel(num,apref,&current_apid, &current_vpid);
	str << _("Setting ") 
	    << dvb.chans[num].name << " ("<< num  << ")" << ends;
	myprint(str.str(), 0, 2);
	
	mytime = time(0)+2;
	SetMute(0);
	nswitch++;
}
int set_audio(int audion)
{
	int cchan= cnum(chlist[clist](currlc[clist]));
	int anum = dvb.chans[cchan].apidnum;

	if (enable_rec) return 0;
	if (!dvb.chans[cchan].checked){
		dvb.check_pids(&dvb.chans[cchan]);
		set_channel();
		dvb.scan_sdt(&dvb.chans[cchan]);
	}

	if (anum < 2) {
		ostrstream fstr;
		fstr << "Only one Audio Channel" << ends;
		myprint(fstr.str(), o, 2);
		return 0;
	}

	if (audion >= anum) audion = 0;
	audio = audion;
	ostrstream fstr;
	char name[4];
	memcpy(name, dvb.chans[cchan].apids_name +(audion*4), 4);
	fstr << "Audio " << audion ;
	if (name[0]) fstr << " " << name  ;
	fstr << ends;
	myprint(fstr.str(), o, 2);

	dvb.set_apid(dvb.chans[cchan].apids[audion]);  
	current_apid = dvb.chans[cchan].apids[audion];  
	return 0;
}


#ifdef PTHREADS

void get_recfile(char *name, int l, int dev, int type)
{
	int dirlen;
	int fnlen;
	ostrstream str(name,l);

	dirlen = strlen(save_dir);
	fnlen = strlen(save_name);
	
	if (dirlen+fnlen+5 > l) {
		perror("recording file name too long");
		exit(1);
	}
	
	str << save_dir <<  save_name;

	switch(type){
	case C_TS:
		str << ".ts" << ends;
		return;

	case C_MPEG:
		str << ".mpeg" << ends;
		return;

	default:
		perror("unknown type for recording file");
		exit(1);
	}	

}

void get_repfile(char *name, int l, int dev, int type)
{

	get_recfile(name, l, dev, type);
}


int init_record(dvr_info_t *rdinfo)
{
	char name[80];
	char dvrs[80];
	ostrstream sdvr(dvrs,80);
	int num = cnum(chlist[clist](currlc[clist]));

	rdinfo->stop = 0;
	rdinfo->dev = dev;
	rdinfo->length = 0;
	rdinfo->type = save_format;
	rdinfo->pida = current_apid;
	rdinfo->pidv = current_vpid;

	sdvr << "/dev/ost/dvr" << rdinfo->dev << ends;

	if ((rdinfo->fd_dvr = open(dvrs, O_RDONLY| O_NONBLOCK)) < 0){
		perror("DVR DEVICE");
		return -1;
	}
	
	get_recfile(name, 80, dev, save_format);
	rdinfo->filefd = open(name,O_WRONLY|O_CREAT|O_TRUNC, 
			       S_IRUSR|S_IWUSR|S_IRGRP
			       |S_IWGRP|S_IROTH|S_IWOTH);
	if (rdinfo->filefd < 0) {
		cerr << "can't open " << name << endl;
		return -1;
	}
	return 0;
}

int init_replay(dvr_info_t *pdinfo)
{
	char name[80];

	if (!has_decoder) return -1;
	pdinfo->stop = 0;
	pdinfo->dev = dev;
	pdinfo->length = 0;
	pdinfo->type = save_format;

	get_recfile(name, 80, dev, save_format);
	pdinfo->filefd = open(name,O_RDONLY);
	if (pdinfo->filefd < 0) {
		cerr << "can't open " << name << endl;
		return -1;
	}
	return 0;
}
#endif

#define FILELEN 256
int main (int argc, char **argv)
{
	char filen[FILELEN];
	int c,channel = 0;
	int lfd= -1;
	int dummy1, dummy2;
	struct pollfd pfd[2];
	boolean mrand = false;
	int rstep = 3; 
	time_t rtime;

#ifdef PTHREADS
        pthread_t threadp;
        pthread_t threadr;
	dvr_info_t pdinfo;
	dvr_info_t rdinfo;
#endif

#ifdef ENABLE_NLS
	setlocale(LC_ALL,"");
	bindtextdomain("rtuxzap", LOCALE_DIR);
	textdomain("rtuxzap");
#endif
	lfd = lirc_zap_init();
	filen[0]='\0';

	pfd[0].fd = STDIN_FILENO;
	pfd[0].events = POLLIN;
	pfd[1].fd = lfd;
	pfd[1].events = POLLIN;


	for (;;) {
		if (-1 == (c = getopt(argc, argv, "red:c:owf:s:")))
			break;
		switch (c) {
		case 'c':
			channel = atoi(optarg);
			break;
		case 'r':
			mrand = true;
			break;
		case 'd':
			dev = atoi(optarg);
			break;

		case 'o':
			o = 1;
			break;

		case 'e':
			enable_dvr = true;
			break;

		case 'w':
			wide = true;
			break;

		case 'f':
			strcpy(filen,optarg);
			break;

		case 's':
			save_format = atoi(optarg);
			if (save_format != C_TS && save_format != C_MPEG){
				fprintf(stderr,
					"unknown format %d using %d (TS)\n",
					save_format, C_TS);
				save_format = C_TS;
			}
					
			break;

		case 'h':
		default:
			cerr << _("usage: ") << argv[0] << _("  [ options ] ") 
			     <<	endl << endl << _("options:") << endl
			     << endl << _("    -c <n>    set channel") << endl
			     << endl << _("    -d <n>    set devices")
			     << "(0 =/dev/ost/xxx0)" << endl
			     << endl << _("    -o        use OSD") << endl
			     << endl << _("    -e        enable dvr") << endl
			     << endl << _("    -w        set 16:9") << endl
			     << endl << _("    -f <f>    use <f> as dvbrc") 
			     << endl
			     << endl << _("    -r        random zap") << endl
			     << endl << _("    -s        save format") 
			     << " (" << C_TS << "= TS, " << C_MPEG << "= MPEG)"
			     << endl;
			exit(1);
		}
	}
	fprintf(stderr, "Using device %d\n",dev);	

	if (wide)
		set_format(VIDEO_FORMAT_16_9);
	else
		set_format(VIDEO_FORMAT_4_3);

	dvb.init("","",dev);
	if (enable_dvr) dvb.enable_DVR();
	else dvb.disable_DVR();
	if (open_av( &dummy1, &dummy2, dev, 0) < 0){
		has_decoder = 0;
		o = 0;
		fprintf(stderr, "card has no decoder\n");
	} else {
		has_decoder =1;
		fprintf(stderr, "card has decoder\n");
	}

	if (o) dvb.use_osd(dev);
	close(dummy1);
	close(dummy2);

	if (get_dvbrc(filen,dvb,dev,FILELEN)) {
		cout << _("Found DVB input file ") << filen << endl;
	} else {
		cout << _("Could not find DVB input file ") 
		     << filen << endl;
	}

	for (int i = 0; i < NLIST; i++){
		load_ch(i);
	}
	
	for (int l = 0; l < dvb.num[CHAN]; l++){
                int k = 0, i = 0, j = 0;

                while (dvb.tps[k].id != dvb.chans[l].tpid)
                        k++;
                while (dvb.tps[k].satid != dvb.sats[i].id)
                        i++;
                while (dvb.sats[i].lnbid != dvb.lnbs[j].id)
                        j++;

		gChannel ch(dvb.chans[l], dvb.tps[k], dvb.lnbs[j]);
                chlist[NLIST].AddChannel(ch);
	}
	
	
	curses_init();
	mvprintw(0,0,"Welcome to RTuxZap");
	set_volume(volume);
	killprint(o);
	refresh();


	set_channel();
	int progn = -1;
	if (mrand){
		rtime = time(0);
		srand(time(0));
	}
	for (;;){
		int sw;
		int cc = 0;
		char *code;
		char *c;
		time_t key_time;

		refresh();
		killprint(o);

		if (mrand && rtime && rtime < time(0)){
			clist = rand()%NLIST; 
			currlc[clist] = rand()%(chlist[clist].NumChan());
			set_channel();
			rtime = time(0)+rstep;
		}
		
		if (progn >= 0 && time(0) > key_time){
			if (progn >= 0 && progn < chlist[clist].NumChan()){
				lastlc[clist] = currlc[clist];
				currlc[clist] = progn;
				set_channel();
			}
			lastclist = clist;
			progn = -1;
		}

#ifdef PTHREADS		
		if (pdinfo.stop && enable_replay){
			enable_replay = false;
			ostrstream fstr;
			
			fstr << "replay stopped";
			fstr << ends;
			myprint(fstr.str(), o, 3);
		}

		if (rdinfo.stop && enable_rec){
			enable_rec = false;
			ostrstream fstr;
			
			fstr << "recording stopped because of error";
			fstr << ends;
			myprint(fstr.str(), o, 3);
		}
#endif
		if (poll(pfd,2,1000)){
			if (pfd[0].revents & POLLIN){
				cc = getch();
			}
#ifdef HAVE_LIBLIRC_CLIENT
			if (pfd[1].revents & POLLIN){
				if ( lirc_nextcode(&code) == 0 &&
				     lirc_code2char(config,code,&c) ==0 &&
				     c!=NULL){
					refresh();
					cc = int(c[0]); 
					
					free(code);
				}
			}
#endif
			if (cc){
				ostrstream fstr;
					
				if ( mrand ) mrand = false;

#ifdef PTHREADS
				if (!mytime && cc != 'e' && enable_rec){
					fstr << "recording: ";
					fstr << rdinfo.length/1024/1024;
					fstr << " MB";
					fstr << ends;
					myprint(fstr.str(), o, 1);
				}  
		
				if ( enable_replay && cc != 'p'){
					uint64_t l;
/*					switch_t sw;
					av_settings av;

					init_av(&av);
					av.nopat = 1;
					sw.av = &av;
					sw.filefd = pdinfo.filefd;
					sw.ts = 1;
					sw.fdv = pdinfo.fdv;
					sw.fda = pdinfo.fda;
					av_switches(cc, &sw, &l);
*/
					cc = 0;
					
				}  

#endif
				switch(cc){
				case 'e':
					if(enable_dvr) break;
					enable_rec = !enable_rec;
					
					if (enable_rec){
						dvb.enable_DVR();
						dvb.set_apid(current_apid);  
						dvb.set_vpid(current_vpid);  
#ifdef PTHREADS
						if (init_record(&rdinfo) ||
							c_dvr_thread_record(
								&threadr, 
								&rdinfo))
							enable_rec = false;
#endif
					} else {
						dvb.disable_DVR();
#ifdef PTHREADS						
						rdinfo.stop = 1;

#endif
						set_channel();
					}
					
					fstr << "recording";
					if (enable_rec)
						fstr << " started";
					else 
						fstr << " stopped";
					fstr << ends;
					myprint(fstr.str(), o, 3);
					
					break;

				case 'p':
					if(enable_dvr) break;
					enable_replay = !enable_replay;
					
					if (enable_replay){
#ifdef PTHREADS
						if (init_replay(&pdinfo)||
						    c_dvr_thread_play(&threadp,
								      &pdinfo))
							enable_replay = false;
#endif
					} else {
#ifdef PTHREADS						
						pdinfo.stop = 1;
#endif
					}

					fstr << "replay";
					if (enable_replay)
						fstr << " started";
					else 
						fstr << " stopped";
					fstr << ends;
					myprint(fstr.str(), o, 3);
					break;

				case 'w':
					wide = !wide;

					if (wide)
						set_format(VIDEO_FORMAT_16_9);
					else 
						set_format(VIDEO_FORMAT_4_3);
					
					fstr << "wide screen";
					if (wide)
						fstr << " enabled";
					else 
						fstr << " disabled";
					fstr << ends;
					myprint(fstr.str(), o, 2);
					break;

				case 'z':
					if (enable_rec | vt_block) break;
					mrand = !mrand;

					if (mrand){
						rtime = time(0);
						srand(time(0));
					} else 	rtime = 0;
					fstr << "random";
					if (mrand)
						fstr << " enabled";
					else 
						fstr << " disabled";
					fstr << ends;
					myprint(fstr.str(), o, 2);
					break;
					
				case 'y':
					rstep--;
					if (rstep < 2) rstep = 2;
					fstr << "random step: ";
					fstr << rstep;
					fstr << ends;
					myprint(fstr.str(), o, 2);
					break;

				case 'x':
					rstep++;
					if (rstep > 10) rstep = 10;
					fstr << "random step: ";
					fstr << rstep;
					fstr << ends;
					myprint(fstr.str(), o, 2);
					break;

				case 'm':
					if (enable_rec) break;
					SetMute(!rmute);
					break;

				case 'q':
				case 'Q':
					if (enable_rec) break;
					exit(0);
					break;
					
				case '+':
					SetMute(0);
					lastclist = clist;
					set_volume(volume+1);
					break;

				case '-':
					SetMute(0);
					lastclist = clist;
					set_volume(volume-1);
					break;

				case 'd':
				case 258:
					if (enable_rec | vt_block) break;
					lastlc[clist] = currlc[clist];
					currlc[clist]--;
					if (currlc[clist] < 0) 
						currlc[clist] =  
							chlist[clist].NumChan()
							-1;
					set_channel();
					break;

				case 'u':
				case 259:
					if (enable_rec | vt_block) break;
					lastlc[clist] = currlc[clist];
					currlc[clist]++;
					if (currlc[clist] >=
					    chlist[clist].NumChan())
						currlc[clist] = 0; 
					set_channel();
					break;

				case 261:
				case 'f':
					if (enable_rec | vt_block) break;
					lastclist = clist;
					clist++;
					if (clist > NLIST) clist = 0;
					set_channel();
					break;
					
				case 260:
				case 'g':
					if (enable_rec | vt_block) break;
					lastclist = clist;
					clist = clist;
					clist--;
					if (clist < 0) clist = NLIST;
					set_channel();
					break;
					
				case 'a':
					if (enable_rec) break;
					if (vt_block) vt_block = false;
					set_audio(audio+1);
					break;

				case 'r':
					if (enable_rec | vt_block) break;
					if (clist != lastclist){
						sw = clist;
						clist = lastclist;
						lastclist = sw;
					} else {
						sw = lastlc[clist];
						lastlc[clist] = currlc[clist];
						currlc[clist] = sw;
					}
					set_channel();
					break;
				
				case 't':
				{
					time_t t;
					
					t = time(0);
					fstr << ctime(&t)
					     << ends;
					myprint(fstr.str(), o, 2);
					break;
				}

				case 'v':
					if (vt_block) vt_block = false;
					else vt_block = true;
					break;

				case 'o':
					if (o) {
						dvb.close_osd();
						dvb.use_osd(dev);
					}
					break;

				case '0'...'9':
					if (enable_rec| vt_block) break;
					if (progn >= 0 && 
					    10*progn < chlist[clist].NumChan())
						progn = 10*progn+(cc-48);
					else
						progn = cc-48;
					
					fstr << "Prog: "
					     << progn << ends;
					myprint(fstr.str(), o, 1);
					key_time = time(0)+1;
					break;
				}
			}
		}
	}
#ifdef HAVE_LIBLIRC_CLIENT
	lirc_freeconfig(config);
	lirc_deinit();
#endif
	exit (0);
}

