import java.io.*;

/**
 * Class for extracting information from MP3 files
 *
 * @version $Id: MP3Parser.java,v 1.8 2002/09/22 23:13:29 blsecres Exp $
 * @author Ben Secrest &lt;blsecres@users.sourceforge.net&gt;
 */
public class MP3Parser implements FileParser {
    /** File extensions for MP3 files */
    private final static String[] extensions = {"mp3"};

    /** Mime types for MP3 files */
    private final static String[] mimeTypes = {"audio/mpeg"};

    /** Type of file this parser handles */
    private final static String fileType = "MP3";

    /** MP3 file magic signature */
    private final static byte[][] magic = {{(byte) 0xff, (byte) 0xfb}};

    /** MP3 headers are always at the beginning of the file */
    private final static boolean magicOffset = false;

    /** MP3 headers are case sensitive */
    private final static boolean magicCase = true;

    /** MP3 FileMagic structure */
    private final static FileMagic mp3Magic = new FileMagic(magic, magicOffset,
	    magicCase);

    /** Search string for beginning of id3 tag */
    private final static byte[] id3Tag = {(byte) 'T', (byte) 'A', (byte) 'G'};

    /**
     * Size of buffer to read data into.
     */
    private static final int INPUT_BUFFER_SIZE = 1024;

    /** Size of an id3 tag */
    private static final int ID3_SIZE = 128 - id3Tag.length;

    /** Size of an id3 tag string */
    private static final int ID3_STRING_LENGTH = 30;

    /** Offset of song title */
    private static final int SONG_OFFSET = 0 * ID3_STRING_LENGTH;

    /** Offset of artist */
    private static final int ARTIST_OFFSET = 1 * ID3_STRING_LENGTH;

    /** Offset of album */
    private static final int ALBUM_OFFSET = 2 * ID3_STRING_LENGTH;

    /** The highest level of logging to use */
    private static final int LOGLEVEL = 9;

    /** Determines if this parser will search for song title */
    private boolean wantSong;

    /** Determines if this parser will search for artist */
    private boolean wantArtist;

    /** Determines if this parser will search for album */
    private boolean wantAlbum;

    /** Determines if the parser will provide file type */
    private boolean wantFileType;

    /** Determines if the parser will provide parser information */
    private boolean wantParser;

    /** The object to use to log messages */
    private IGLog log;


    /**
     * Constructor
     */
    public MP3Parser() {
	log = null;
	wantSong = wantArtist = wantAlbum = wantFileType = wantParser = false;
    }


    /**
     * Perform parsing on the given file source.
     * @param file The IGFile object to open a file for and fill in attributes
     * @throws IOException if an error occurs while reading a file
     * @throws FileNotFoundException if the file reference in the IGFile object
     *		does not exist
     */
    public void parse(IGFile file) throws IOException, FileNotFoundException {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "MP3Parser.parser(IGFile["
		    + file.getLocation() + "])");

	try {
	    parse(file, new FileInputStream(file.getLocation()));
	} catch (StreamResetException sre) {
	    throw new IOException("Stream Reset Exception shouldn't happen");
	}
    }


    /**
     * Perform parsing on an open stream.
     * @param file The IGFile to fill in attributes for
     * @param stream The data source for parsing
     * @throws IOException if an error occurs while reading data
     */
    public void parse(IGFile file, InputStream stream)
	    throws IOException, StreamResetException {
	if (log == null)
	    // FIXME
	    return;

	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "MP3Parser.parser(IGFile["
		    + file.getLocation() + "], InputStream, String)");

	if (LOGLEVEL >= IGLog.FILE)
	    log.addResource(IGLog.FILE, "PROCESS_FILE",
		    new String[]{file.getLocation()});

	boolean foundTag = false;	// found the id3 tag
	byte[] buffer = new byte[INPUT_BUFFER_SIZE]; // buffer to read data into
	byte[] id3Buffer = new byte[ID3_SIZE];       // buffer for id3 data
	int bytesRead;			// number of bytes read
	int findTagPosition = 0;	// position in tag search
	int bufferPosition;		// position in stream buffer
	int id3Position = 0;		// position in id3 buffer
	long fileSize;			// file size for skipping data
	
	try {
	    fileSize = Long.parseLong(file.getString(IGKey.FILE_SIZE));
	} catch (NumberFormatException nfe) {
	    fileSize = 0;
	}

	if (LOGLEVEL >= IGLog.SECTION)
	    log.addResource(IGLog.SECTION, "FP_BEGIN_PARSE",
		    new String[]{fileType});

	/*
	 * skip over audio data, only interested in id3 tag which resides at
	 * the end of the file
	 */
	if (fileSize > ID3_SIZE)
	    stream.skip(fileSize - (ID3_SIZE + id3Tag.length));

	/*
	 * read data, id3 tag is at the end of the file so read until there's
	 * no more data
	 */
	while ((bytesRead = stream.read(buffer)) != -1) {
	    bufferPosition = 0;	// track position in current buffer
	    if (! foundTag)
		while (bufferPosition < bytesRead) {
		    // search for beginning of id3 tag
		    if (id3Tag[findTagPosition] == buffer[bufferPosition])
			findTagPosition++;
		    else
			findTagPosition = 0;

		    bufferPosition++;

		    if (findTagPosition == id3Tag.length) {
			foundTag = true;
			break;
		    }
		}

	    if (foundTag)
		// copy data following tag leader into buffer for analysis
		while ((bufferPosition + id3Position) < bytesRead
			&& id3Position < id3Buffer.length) {
		    id3Buffer[id3Position] = buffer[bufferPosition
					     + id3Position];
		    id3Position++;
		}
	}

	if (wantSong)
	    file.put(IGKey.TITLE,
		    getString(id3Buffer, SONG_OFFSET, ID3_STRING_LENGTH));
	if (wantArtist)
	    file.put(IGKey.AUTHOR,
		    getString(id3Buffer, ARTIST_OFFSET, ID3_STRING_LENGTH));
	if (wantAlbum)
	    file.put(IGKey.DESCRIPTION,
		    getString(id3Buffer, ALBUM_OFFSET, ID3_STRING_LENGTH));
	if (wantFileType)
	    file.put(IGKey.FILE_TYPE, new String(fileType));
	if (wantParser)
	    file.put(IGKey.PARSER, getClass().getName());

	if (LOGLEVEL >= IGLog.SECTION)
	    log.addResource(IGLog.SECTION, "FP_FINISH_PARSE",
		    new String[]{fileType});

	stream.close();
    }


    /**
     * Set the desired attributes to extract
     * @param wanted A set of bits describing preferences
     */
    public void setWantedItems(IGKeySet wanted) {
	wantSong = wanted.wants(IGKey.TITLE);
	wantArtist = wanted.wants(IGKey.AUTHOR);
	wantAlbum = wanted.wants(IGKey.DESCRIPTION);
	wantFileType = wanted.wants(IGKey.FILE_TYPE);
	wantParser = wanted.wants(IGKey.PARSER);
    }


    /**
     * Set the logger to use with this module
     * @param logObj The object to use for logging data
     */
    public void setLog(IGLog logObj) {
	log = logObj;
    }


    /**
     * Get the file extensions this parser can handle
     * @return String array of file extensions
     */
    public String[] getExtensions() {
	return extensions;
    }


    /**
     * Get the mime types this parser can handle
     * @return String array of mime types
     */
    public String[] getMimeTypes() {
	return mimeTypes;
    }


    /**
     * Get the file magic for files this parser can handle
     * @return Array of byte arrays containing magic signature
     */
    public FileMagic getMagic() {
	return mp3Magic;
    }


    /**
     * Extract a string from a byte buffer
     * @param array The array to extract the string from
     * @param offset The starting position of the string
     * @param len The maximum length of the string
     * @return The String representation of the bytes
     */
    private static String getString(byte[] array, int offset, int len) {
	int i;
	/*
	 * find the position of the first null character, use this to create a
	 * new String
	 */
	for (i = 0; i < len && (i + offset) < array.length; i++)
	    if (array[i + offset] == 0)
		break;

	return (new String(array, offset, i));
    }
}
