// Copyright (C) 2007-2009 Garth N. Wells
// Licensed under the GNU LGPL Version 2.1.
//
// Modified by Anders Logg, 2007-2009.
//
// First added:  2007-03-13
// Last changed: 2009-06-19

#ifndef __SPARSITY_PATTERN_H
#define __SPARSITY_PATTERN_H

#include <vector>
#include "dolfin/common/Set.h"
#include "GenericSparsityPattern.h"

namespace dolfin
{

  /// This class implements the GenericSparsityPattern interface.
  /// It is used by most linear algebra backends, except for Epetra
  /// which uses a special/native implementation.

  class SparsityPattern: public GenericSparsityPattern
  {
  public:

    enum Type {sorted, unsorted};

    /// Create empty sparsity pattern
    SparsityPattern(Type type);

    /// Destructor
    ~SparsityPattern();

    /// Initialize sparsity pattern for a generic tensor
    void init(uint rank, const uint* dims);

    /// Insert non-zero entries
    void insert(const uint* num_rows, const uint * const * rows);

    /// Return rank
    uint rank() const;

    /// Return global size for dimension i
    uint size(uint i) const;

    /// Return local range for dimension dim
    virtual std::pair<uint, uint> local_range(uint dim) const ;

    /// Return total number of nonzeros in local_range for dimension 0
    uint num_nonzeros() const;

    /// Fill array with number of nonzeros for diagonal block in local_range for dimension 0
    /// For matrices, fill array with number of nonzeros per local row for diagonal block
    void num_nonzeros_diagonal(uint* num_nonzeros) const;

    /// Fill array with number of nonzeros for off-diagonal block in local_range for dimension 0
    /// For matrices, fill array with number of nonzeros per local row for off-diagonal block
    void num_nonzeros_off_diagonal(uint* num_nonzeros) const;

    /// Finalize sparsity pattern
    void apply();

    /// Return informal string representation (pretty-print)
    std::string str() const;

    /// Return underlying sparsity pattern (diagonal)
    const std::vector<Set<uint> >& diagonal_pattern() const;

    /// Return underlying sparsity pattern (off-diagional)
    const std::vector<Set<uint> >& off_diagonal_pattern() const;

  private:

    // Sort entries for each row
    void sort();

    // Print some useful information
    void info_statistics() const;

    // Sparsity pattern type (sorted/unsorted)
    const Type type;

    // Whether or not pattern has been sorted
    bool _sorted;

    // Shape of tensor
    std::vector<uint> shape;

    // Local range
    uint row_range_min;
    uint row_range_max;
    uint col_range_min;
    uint col_range_max;

    // Sparsity patterns for diagonal and off-diagonal blocks
    std::vector<Set<uint> > diagonal;
    std::vector<Set<uint> > off_diagonal;

    // Sparsity pattern for non-local entries stored as [i, j, i, j, ...]
    std::vector<uint> non_local;

  };

}
#endif
