/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */



#include <diagnostics/extensions/memory/annotations.hpp>

#if DIAGNOSTICS_SWITCH_MEMORY_ANALYSIS_ENABLED == 1


#  include <diagnostics/extensions/memory/allocation_database.hpp>

#  include <diagnostics/frame/logging_implementation.hpp>

#  include <diagnostics/util/to_string.hpp>

#  include <diagnostics/frame/logging_mutex.hpp>

#  include <algorithm>

using ::diagnostics::internal::to_string;
using ::diagnostics::internal::Logging_Implementation;

UNNAMED_NAMESPACE_BEGIN;

char const * block_name(bool const log,bool const check)
{
    if(log) 
	return "MEMORY: Logging";
    else if(check)
	return "MEMORY: Balance";
    else 
	return "MEMORY: Analysis";
}

UNNAMED_NAMESPACE_END;

DIAGNOSTICS_NAMESPACE_BEGIN;
MEMORY_NAMESAPCE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;

Allocation_Balance_Guard::Allocation_Balance_Guard
(char const * const base_file_name,
 char const * const file_name,
 Level_t const level,
 int const line,						     
 bool const log,
 bool const check,
 Type_t const type) 
    : m_base_file_name(base_file_name),
      m_file_name(file_name),
	  m_line(line),
	  m_level(level),
	  m_log(log),
	  m_check(check),
	  m_type(type),
      m_creation_tick(Allocation_Database::creation_tick()+1)
    
{
	DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
    // obtain current balance
    Allocation_Database::balance(m_entering_total_size,m_entering_total_number);

    // log the block entry
    Logging_Implementation::instance()->log
		(::diagnostics::LEVEL_AUDIT,
		 TYPE_BLOCK_ENTER,
		 0,
		 block_name(log,check),
		 m_base_file_name,
		 m_file_name,
		 line);
	
    // activate @ref Allocation_Database (log==true -> all changes are logged)
    Allocation_Database::analysis_on(log);
}

Allocation_Balance_Guard::~Allocation_Balance_Guard()
{
	DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
    if(m_check) assert_balance(m_line,m_type);
    Allocation_Database::analysis_off(m_log);

    // log the block exit
    Logging_Implementation::instance()->log
		(::diagnostics::LEVEL_AUDIT,
		 TYPE_BLOCK_EXIT,
		 0,
		 block_name(m_log,m_check),
		 m_base_file_name,
		 m_file_name,
		 m_line);
}


void Allocation_Balance_Guard::assert_balance(int const line,Type_t const type)
{
	DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);

    ::std::size_t size;
    Tick_t number;
    Allocation_Database::balance(m_creation_tick,size,number);
    if(number>0) {
	Logging_Implementation::instance()->log
	    (m_level,
	     type,
	     0,
	     "MEMORY: Leak: bytes="+
	     to_string(size) + " blocks=" + to_string(number),
	     m_base_file_name,
	     m_file_name,
	     line);
	Tick_t ticks[128];
	Tick_t * ticks_begin(ticks);
	Tick_t * const ticks_end(Allocation_Database::unallocated_blocks(m_creation_tick,ticks,ticks+128));
	::std::sort(ticks_begin,ticks_end);
	::std::string tick_report("MEMORY: Leak: ticks=");
	for(;ticks_begin!=ticks_end;++ticks_begin){
	    tick_report+=' ';
	    tick_report+=to_string(*ticks_begin);
	}
	Logging_Implementation::instance()->log
	    (m_level,
	     type,
	     0,
	     tick_report,
	     m_base_file_name,
	     m_file_name,
	     line);
    }
    ::std::size_t total_size;
    Tick_t total_number;
    Allocation_Database::balance(total_size,total_number);
    // compensating for leak
    total_size-=size; 
    total_number-=number; 
    // total_number<=m_number_blocks (we do not assert -- but if this
    // condition is not met something wired happened (probably,
    // multi-threading)).
    if(total_number!=m_entering_total_number) {
	Logging_Implementation::instance()->log
	    (m_level,
	     type,
	     0,
	     "MEMORY: Antileak: bytes="
	     + to_string(m_entering_total_size-total_size) + " blocks=" 
	     + to_string(m_entering_total_number-total_number),
	     m_base_file_name,
	     m_file_name,
	     line);
    }
}

void Allocation_Balance_Guard::log_balance(int const line)
{
	DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
    ::std::size_t size;
    Tick_t number;
    Allocation_Database::balance(m_creation_tick,size,number);
    Logging_Implementation::instance()->log
		(m_level,
		 TYPE_TRACE,
		 0,
		 "MEMORY: Balance: bytes="
		 + to_string(size) + " blocks=" 
		 + to_string(number),
		 m_base_file_name,
		 m_file_name,
		 line);}


void Allocation_Balance_Guard::log_total_balance(int const line)
{
	DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
    ::std::size_t size;
    Tick_t number;
    Allocation_Database::balance(size,number);
    Logging_Implementation::instance()->log
		(m_level,
		 TYPE_TRACE,
		 0,
		 "MEMORY: Total_Balance: bytes="
		 + to_string(size) + " blocks=" 
		 + to_string(number),
		 m_base_file_name,
		 m_file_name,
		 m_line);
}

INTERNAL_NAMESPACE_END;
MEMORY_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

#endif

// vim:ts=4:sw=4
